/*
 * elfspe - A wrapper to allow direct execution of SPE binaries
 * Copyright (C) 2005 IBM Corp.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License,
 * or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 *  License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this library; if not, write to the Free Software Foundation,
 *   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#define _XOPEN_SOURCE 600
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include "libspe.h"
#include "spe.h"

static void *spe = NULL;

#ifndef LS_SIZE
#define LS_SIZE                       0x40000   /* 256K (in bytes) */
#define LS_ADDR_MASK                  (LS_SIZE - 1)
#endif /* LS_SIZE */

static void handler (int signr ) __attribute__ ((noreturn));
static void
handler (int signr)
{
  if (spe)
    spe_cleanup (spe);

  fprintf (stderr, "Killed by signal %d\n", signr);
  exit (128 + signr);
}

struct spe_regs {
    unsigned int r3[4];	
    unsigned int r4[4];
    unsigned int r5[4];
};

/* spe_copy_argv - setup C99-style argv[] region for SPE main().
 *
 * Duplicate command line arguments and set up argv pointers 
 * to be absolute LS offsets, allowing SPE program to directly 
 * reference these strings from its own LS.
 *
 * An SPE entry function (crt0.o) can take the data from spe_regs 
 * and copy the argv region from EA to the top of LS (largest addr), 
 * and set $SP per ABI requirements.  Currently, this looks something
 * like:
 *
 *   +-----+ 0
 *   | txt |
 *   |-----|
 *   |     |
 *   |  ^  |
 *   |  |  |
 *   |  |  |
 *   |stack|
 *   |-----| ls_dst (LS_SIZE-nbytes)
 *   | args|
 *   +-----+ LS_SIZE
 */
static int
spe_copy_argv(int argc, char **argv, struct spe_regs *ret)
{
    unsigned int *argv_offsets;
    void *start = NULL;
    char *ptr, *end;
    union {
	unsigned long long ull;
	unsigned int ui[2];
    } addr64;
    int spe_arg_max = 4096;
    int i, nbytes = 0;

    memset(ret, 0, sizeof(struct spe_regs));
    if ((argc <= 0) || getenv("SPE_NO_ARGS")) {
	return 0;
    }
    if (getenv("SPE_ARG_MAX")) {
	int v;
	v = strtol((const char *) getenv("SPE_ARG_MAX"), (char **) NULL,
		   0);
	if ((v >= 0) && (v <= spe_arg_max))
	    spe_arg_max = v & ~(15);
    }
    for (i = 0; i < argc; i++) {
	nbytes += strlen(argv[i]) + 1;
    }
    nbytes += (argc * sizeof(unsigned int));
    nbytes = (nbytes + 15) & ~(15);
    if (nbytes > spe_arg_max) {
	return 2;
    }
    posix_memalign(&start, 16, nbytes);
    if (!start) {
	return 3;
    }
    addr64.ull = (unsigned long long) ((unsigned long) start);
    memset(start, 0, nbytes);
    ptr = (char *)start;
    end = (char *)start + nbytes;
    argv_offsets = (unsigned int *) ptr;
    ptr += (argc * sizeof(unsigned int));
    for (i = 0; i < argc; i++) {
	int len = strlen(argv[i]) + 1;
	argv_offsets[i] = LS_SIZE - ((unsigned int) (end - ptr));
	argv_offsets[i] &= LS_ADDR_MASK;
	memcpy(ptr, argv[i], len);
	ptr += len;
    }
    ret->r3[0] = argc;
    ret->r4[0] = LS_SIZE - nbytes;
    ret->r4[1] = addr64.ui[0];
    ret->r4[2] = addr64.ui[1];
    ret->r4[3] = nbytes;
    return 0;
}

int
main (int argc, char **argv)
{
  spe_gid_t gid;
  struct spe_regs params;
  int rc;

  spe_program_handle_t *spe_handle;

  signal (SIGSEGV, handler);
  signal (SIGBUS, handler);
  signal (SIGILL, handler);
  signal (SIGFPE, handler);
  signal (SIGTERM, handler);
  signal (SIGHUP, handler);
  signal (SIGINT, handler);

  if (argc < 2)
    {
      fprintf (stderr, "Usage: elfspe [spe-elf]\n");
      exit (1);
    }

  spe_handle = spe_open_image (argv[1]);
  if (!spe_handle)
    {
      perror (argv[1]);
      exit (1);
    }

  if (spe_copy_argv(argc-1, &argv[1], &params)) 
    {
	perror ("spe_copy_argv");
	exit (1);
    }
   
  gid = spe_create_group(0,0,0);
  if (!gid)
    {
      perror ("create_group");
      exit (1);
    }

  spe = spe_setup (gid, spe_handle, &params, NULL, SPE_USER_REGS);
  if (!spe)
    {
      perror ("spe_setup");
      exit (1);
    }

  rc = do_spe_run (spe);
  spe_cleanup (spe);

  return rc & 0xff;
}

