/*
 * @(#)DirClassMetaDataReaderUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.util.Vector;

import junit.framework.Test;
import junit.framework.TestCase;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.compiler.ModifiedClass;
import net.sourceforge.groboutils.junit.v1.iftc.CxFactory;
import net.sourceforge.groboutils.junit.v1.iftc.InterfaceTestSuite;


/**
 * Tests the DirClassMetaDataReader class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 22, 2003
 */
public class DirClassMetaDataReaderUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = DirClassMetaDataReaderUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public DirClassMetaDataReaderUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1() throws Exception
    {
        try
        {
            new DirClassMetaDataReader( null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testConstructor2() throws Exception
    {
        try
        {
            new DirClassMetaDataReader(
                createIAnalysisModule( "A", "B", "C" ), null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testConstructor3() throws Exception
    {
        try
        {
            new DirClassMetaDataReader( null,
                createDirMetaDataIO( createIAnalysisModule( "A", "B", "C" ) ) );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException ex)
        {
            // test exception
        }
    }
    
    
    public void testGetClassSignatures1() throws Exception
    {
        IAnalysisModule am = createIAnalysisModule( "A", "B", "C" );
        DirMetaDataIO dmdio = createDirMetaDataIO( am );
        String sig1 = addClassRecord( dmdio, THIS_CLASS, am );
        
        DirClassMetaDataReader dcmdr = new DirClassMetaDataReader(
            am, dmdio );
        String[] sigs = dcmdr.getClassSignatures();
        assertNotNull(
            "Returned null signatures.",
            sigs );
        assertEquals(
            "Returned incorrect number of signatures.",
            1,
            sigs.length );
        assertEquals(
            "Did not return the correct sig.",
            sig1,
            sigs[0] );
    }
    
    
    public void testReadClass1() throws Exception
    {
        IAnalysisModule am = createIAnalysisModule( "A", "B", "C" );
        DirMetaDataIO dmdio = createDirMetaDataIO( am );
        String sig1 = addClassRecord( dmdio, THIS_CLASS, am );
        
        DirClassMetaDataReader dcmdr = new DirClassMetaDataReader(
            am, dmdio );
        ClassRecord cr = dcmdr.readClass( sig1 );
        assertNotNull(
            "Returned null class record.",
            cr );
        assertEquals(
            "Returned incorrect class record.",
            sig1,
            cr.getClassSignature() );
    }
    
    
    public void testClose1() throws Exception
    {
        IAnalysisModule am = createIAnalysisModule( "A", "B", "C" );
        DirMetaDataIO dmdio = createDirMetaDataIO( am );
        DirClassMetaDataReader dcmdr = new DirClassMetaDataReader(
            am, dmdio );
        
        dcmdr.close();
        
        assertFalse(
            "Reader closed the underlying IO object.",
            dmdio.isClosed() );
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    protected static IAnalysisModule createIAnalysisModule( String name,
            String unit, String mime )
    {
        return CCCreatorUtil.createIAnalysisModule( name, unit, mime );
    }
    
    protected static DirMetaDataIO createDirMetaDataIO( IAnalysisModule am )
            throws IOException
    {
        DirMetaDataIO dmd = new DirMetaDataIO(
            CCCreatorUtil.createNewDirectory() );
        dmd.putAnalysisModuleSet(
            new AnalysisModuleSet( new IAnalysisModule[] { am } ) );
        return dmd;
    }
    
    protected String addClassRecord( DirMetaDataIO dmd, Class c,
            IAnalysisModule am )
            throws IOException
    {
        ModifiedClass mc = CCCreatorUtil.createModifiedClass( c );
        ClassRecord cr = mc.createClassRecord( new AnalysisModuleSet(
            new IAnalysisModule[] { am } ) );
        dmd.putClassRecord( am, cr );
        return cr.getClassSignature();
    }
    
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        InterfaceTestSuite suite = IClassMetaDataReaderUTestI.suite();
        suite.addTestSuite( THIS_CLASS );
        suite.addFactory( new CxFactory( "A" ) {
            Vector v = new Vector();
            public Object createImplObject() throws IOException {
                IAnalysisModule am = createIAnalysisModule( "a", "u",
                    "text/plain" );
                DirMetaDataIO dmd = createDirMetaDataIO( am );
                v.addElement( dmd );
                return new DirClassMetaDataReader( am, dmd );
            }
        } );
        suite.addFactory( new CxFactory( "B" ) {
            Vector v = new Vector();
            public Object createImplObject() throws IOException {
                IAnalysisModule am = createIAnalysisModule( "a", "u",
                    "text/plain" );
                DirMetaDataReader dmdr = CCCreatorUtil.createDirMetaDataReader(
                    CCCreatorUtil.createNewDirectory(),
                    new Class[0],
                    new IAnalysisModule[] { am } );
                DirClassMetaDataReader dcmdr = (DirClassMetaDataReader)
                    dmdr.getClassReader( am );
                // prevent the closing of the DMDR
                v.addElement( dmdr );
                return dcmdr;
            }
        } );
        suite.addFactory( new CxFactory( "C" ) {
            Vector v = new Vector();
            public Object createImplObject() throws IOException {
                IAnalysisModule am = createIAnalysisModule( "a", "u",
                    "text/plain" );
                DirMetaDataReader dmdr = CCCreatorUtil.createDirMetaDataReader(
                    CCCreatorUtil.createNewDirectory(),
                    new Class[] { THIS_CLASS, String.class, Short.class },
                    new IAnalysisModule[] { am } );
                DirClassMetaDataReader dcmdr = (DirClassMetaDataReader)
                    dmdr.getClassReader( am );
                // prevent the closing of the DMDR
                v.addElement( dmdr );
                return dcmdr;
            }
        } );
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

