/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <ogmrip-container.h>
#include <ogmrip-fs.h>
#include <ogmrip-mpeg.h>
#include <ogmrip-mplayer.h>
#include <ogmrip-plugin.h>
#include <ogmrip-settings.h>

#include <ogmjob-queue.h>

#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <unistd.h>
#include <string.h>


G_BEGIN_DECLS

#define OGMRIP_TYPE_MPEG          (ogmrip_mpeg_get_type ())
#define OGMRIP_MPEG(obj)          (G_TYPE_CHECK_INSTANCE_CAST ((obj), OGMRIP_TYPE_MPEG, OGMRipMpeg))
#define OGMRIP_MPEG_CLASS(klass)  (G_TYPE_CHECK_CLASS_CAST ((klass), OGMRIP_TYPE_MPEG, OGMRipMpegClass))
#define OGMRIP_IS_MPEG(obj)       (G_TYPE_CHECK_INSTANCE_TYPE ((obj), OGMRIP_TYPE_MPEG))
#define OGMRIP_IS_MPEG_CLASS(obj) (G_TYPE_CHECK_CLASS_TYPE ((klass), OGMRIP_TYPE_MPEG))

typedef struct _OGMRipMpeg      OGMRipMpeg;
typedef struct _OGMRipMpegClass OGMRipMpegClass;

typedef enum
{
  OGMRIP_MPEG_DVD,
  OGMRIP_MPEG_VCD,
  OGMRIP_MPEG_SVCD
} OGMRipMpegFormat;

struct _OGMRipMpeg
{
  OGMRipContainer parent_instance;

  OGMRipMpegFormat format;
  guint spu;

  GSList *files;
};

struct _OGMRipMpegClass
{
  OGMRipContainerClass parent_class;
};

enum
{
  PROP_0,
  PROP_FORMAT
};

static void ogmrip_mpeg_get_property (GObject         *gobject,
                                      guint           property_id,
                                      GValue          *value,
                                      GParamSpec      *pspec);
static void ogmrip_mpeg_set_property (GObject         *gobject,
                                      guint           property_id,
                                      const GValue    *value,
                                      GParamSpec      *pspec);

static gint ogmrip_mpeg_run          (OGMJobSpawn     *spawn);
static void ogmrip_mpeg_set_options  (OGMRipContainer *container,
                                      const gchar     *section);

static gboolean have_dvdauthor;

G_DEFINE_TYPE (OGMRipMpeg, ogmrip_mpeg, OGMRIP_TYPE_CONTAINER)

static void
ogmrip_mpeg_append_audio_file (OGMRipContainer *mpeg, const gchar *filename, GPtrArray *argv)
{
  struct stat buf;

  if (g_stat (filename, &buf) == 0 && buf.st_size > 0)
    g_ptr_array_add (argv, g_strdup (filename));
}

static void
ogmrip_mpeg_foreach_audio (OGMRipContainer *mpeg, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  input = ogmrip_codec_get_output (codec);
  ogmrip_mpeg_append_audio_file (mpeg, input, argv);
}

static void
ogmrip_mpeg_foreach_audio_file (OGMRipContainer *mpeg, OGMRipFile *file, GPtrArray *argv)
{
  gchar *filename;

  filename = ogmrip_file_get_filename (file);
  if (filename && ogmrip_file_get_type (file) == OGMRIP_FILE_TYPE_AUDIO)
    ogmrip_mpeg_append_audio_file (mpeg, filename, argv);
  g_free (filename);
}

static gchar **
ogmrip_mpeg_rawvideo_command (OGMRipContainer *mpeg, const gchar *output)
{
  GPtrArray *argv;
  OGMRipVideoCodec *video;
  const gchar *filename;

  g_return_val_if_fail (OGMRIP_IS_MPEG (mpeg), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("rawvideo"));
  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  video = ogmrip_container_get_video (mpeg);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_ptr_array_add (argv, g_strdup (filename));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_mpeg_mplex_command (OGMRipContainer *mpeg, const gchar *input)
{
  GPtrArray *argv;
  const gchar *output;

  g_return_val_if_fail (OGMRIP_IS_MPEG (mpeg), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mplex"));

  g_ptr_array_add (argv, g_strdup ("-f"));
  switch (OGMRIP_MPEG (mpeg)->format)
  {
    case OGMRIP_MPEG_DVD:
      g_ptr_array_add (argv, g_strdup ("8"));
      break;
    case OGMRIP_MPEG_VCD:
      g_ptr_array_add (argv, g_strdup ("1"));
      break;
    case OGMRIP_MPEG_SVCD:
      g_ptr_array_add (argv, g_strdup ("4"));
      break;
  }

  output = ogmrip_container_get_output (mpeg);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  g_ptr_array_add (argv, g_strdup (input));

  ogmrip_container_foreach_audio (mpeg, 
      (OGMRipContainerCodecFunc) ogmrip_mpeg_foreach_audio, argv);
  ogmrip_container_foreach_file (mpeg, 
      (OGMRipContainerFileFunc) ogmrip_mpeg_foreach_audio_file, argv);

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_spumux_command (OGMRipMpeg *mpeg, const gchar *xml_file)
{
  GPtrArray *argv;
  const gchar *filename;

  g_return_val_if_fail (OGMRIP_IS_MPEG (mpeg), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("spumux.sh"));

  g_ptr_array_add (argv, g_strdup ("--mode"));
  switch (mpeg->format)
  {
    case OGMRIP_MPEG_DVD:
      g_ptr_array_add (argv, g_strdup ("dvd"));
      break;
    case OGMRIP_MPEG_VCD:
      g_ptr_array_add (argv, g_strdup ("vcd"));
      break;
    case OGMRIP_MPEG_SVCD:
      g_ptr_array_add (argv, g_strdup ("svcd"));
      break;
  }

  g_ptr_array_add (argv, g_strdup ("--spu"));
  g_ptr_array_add (argv, g_strdup_printf ("%u", mpeg->spu));
  mpeg->spu ++;

  g_ptr_array_add (argv, g_strdup ("--tmp"));
  g_ptr_array_add (argv, g_strdup (ogmrip_fs_get_tmp_dir ()));

  g_ptr_array_add (argv, g_strdup (xml_file));

  filename = ogmrip_container_get_output (OGMRIP_CONTAINER (mpeg));
  g_ptr_array_add (argv, g_strdup (filename));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_dvdauthor_command (OGMRipMpeg *mpeg, gboolean toc)
{
  GPtrArray *argv;
  const gchar *input;

  g_return_val_if_fail (OGMRIP_IS_MPEG (mpeg), NULL);

  input = ogmrip_container_get_output (OGMRIP_CONTAINER (mpeg));

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("dvdauthor"));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strndup (input, strlen (input) - 5));

  if (toc)
    g_ptr_array_add (argv, g_strdup ("-T"));
  else
  {
    g_ptr_array_add (argv, g_strdup ("-t"));
    g_ptr_array_add (argv, g_strdup (input));
  }

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_subp2png_command (const gchar *input)
{
  GPtrArray *argv;

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("subp2png"));
  g_ptr_array_add (argv, g_strdup ("--expand"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar **
ogmrip_subptools_command (const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("subptools"));

  g_ptr_array_add (argv, g_strdup ("--convert"));
  g_ptr_array_add (argv, g_strdup ("spumux"));

  g_ptr_array_add (argv, g_strdup ("--input"));
  g_ptr_array_add (argv, g_strdup (input));

  g_ptr_array_add (argv, g_strdup ("--output"));
  g_ptr_array_add (argv, g_strdup (output));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static gchar *
ogmrip_mpeg_create_srt_file (OGMRipMpeg *mpeg, const gchar *sub_file,
    gint charset, OGMJobContainer *queue)
{
  OGMRipVideoCodec *video;

  const gchar *cset;
  guint width, height;
  gchar *str, *spumux_file;
  gssize len, written;
  gint fd;

  fd = ogmrip_fs_open_tmp ("spumux.XXXXXX", &spumux_file, NULL);
  if (fd < 0)
    return NULL;

  mpeg->files = g_slist_append (mpeg->files, spumux_file);

  video = ogmrip_container_get_video (OGMRIP_CONTAINER (mpeg));
  ogmrip_video_codec_get_raw_size (video, &width, &height);

  switch (charset)
  {
    case OGMRIP_CHARSET_UTF8:
      cset = "UTF-8";
      break;
    case OGMRIP_CHARSET_ASCII:
      cset = "ASCII";
      break;
      default:
      cset = "ISO8859-1";
      break;
  }

  str = g_strdup_printf ("<subpictures>"
      "<stream>"
      "<textsub "
      "filename=\"%s\" "
      "characterset=\"%s\" "
      "fontsize=\"18.0\" "
      "font=\"Vera.ttf\" "
      "horizontal-alignment=\"center\" "
      "vertical-alignment=\"bottom\" "
      "left-margin=\"60\" "
      "right-margin=\"60\" "
      "top-margin=\"20\" "
      "bottom-margin=\"30\" "
      "subtitle-fps=\"25\" "
      "movie-fps=\"25\" "
      "movie-width=\"%u\" "
      "movie-height=\"%u\"/>"
      "</stream>"
      "</subpictures>", sub_file, cset, width, height);

  len = strlen (str);
  written = write (fd, str, len);
  close (fd);

  g_free (str);

  if (written != len)
  {
    g_free (spumux_file);
    return NULL;
  }

  return spumux_file;
}

static gchar *
ogmrip_mpeg_create_vobsub_file (OGMRipMpeg *mpeg, const gchar *sub_file, OGMJobContainer *queue)
{
  OGMJobSpawn *child;
  gchar **argv, *xml_file;

  xml_file = g_strconcat (sub_file, ".xml", NULL);
  mpeg->files = g_slist_append (mpeg->files, xml_file);

  argv = ogmrip_subp2png_command (sub_file);

  if (argv)
  {
    gchar *spumux_file;

    child = ogmjob_exec_newv (argv);
    ogmjob_container_add (queue, child);
    g_object_unref (child);

    spumux_file = ogmrip_fs_mktemp ("spumux.XXXXXX", NULL);
    mpeg->files = g_slist_append (mpeg->files, spumux_file);

    argv = ogmrip_subptools_command (xml_file, spumux_file);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      ogmjob_container_add (queue, child);
      g_object_unref (child);

      return spumux_file;
    }
  }

  return NULL;
}

static void
ogmrip_mpeg_append_subp (OGMRipMpeg *mpeg, const gchar *xml_file, OGMJobContainer *queue)
{
  gchar **argv;

  argv = ogmrip_spumux_command (mpeg, xml_file);
  if (argv)
  {
    OGMJobSpawn *child;

    child = ogmjob_exec_newv (argv);
    ogmjob_container_add (queue, child);
    g_object_unref (child);
  }
}

static void
ogmrip_mpeg_foreach_subp (OGMRipMpeg *mpeg,
    OGMRipCodec *codec, guint demuxer, gint language, OGMJobContainer *queue)
{
  const gchar *filename;
  struct stat buf;

  filename = ogmrip_codec_get_output (codec);
  if (g_stat (filename, &buf) == 0 && buf.st_size > 0)
  {
    gchar *xml_file = NULL;
    gint format;

    format = ogmrip_plugin_get_subp_codec_format  (G_TYPE_FROM_INSTANCE (codec));

    if (format == OGMRIP_FORMAT_SRT)
    {
      gint charset;

      charset = ogmrip_subp_codec_get_charset (OGMRIP_SUBP_CODEC (codec));
      xml_file = ogmrip_mpeg_create_srt_file (mpeg, filename, charset, queue);
    }
    else if (format == OGMRIP_FORMAT_VOBSUB)
      xml_file = ogmrip_mpeg_create_vobsub_file (mpeg, filename, queue);

    if (xml_file)
      ogmrip_mpeg_append_subp (mpeg, xml_file, queue);
  }
}

static void
ogmrip_mpeg_foreach_subp_file (OGMRipMpeg *mpeg, OGMRipFile *file, OGMJobContainer *queue)
{
  const gchar *filename;

  filename = ogmrip_file_get_filename (file);
  if (filename && ogmrip_file_get_type (file) == OGMRIP_FILE_TYPE_SUBP)
  {
    gchar *xml_file = NULL;
    gint format;

    format = ogmrip_file_get_format (file);

    if (format == OGMRIP_FORMAT_SRT)
    {
      gint charset;

      charset = ogmrip_subp_file_get_charset (OGMRIP_SUBP_FILE (file));
      xml_file = ogmrip_mpeg_create_srt_file (mpeg, filename, charset, queue);
    }
    else if (format == OGMRIP_FORMAT_VOBSUB)
      xml_file = ogmrip_mpeg_create_vobsub_file (mpeg, filename, queue);

    if (xml_file)
      ogmrip_mpeg_append_subp (mpeg, xml_file, queue);
  }
}

static void
ogmrip_mpeg_remove_png_files (const gchar *basename)
{
  GDir *dir;

  dir = g_dir_open (ogmrip_fs_get_tmp_dir (), 0, NULL);
  if (dir)
  {
    GPatternSpec *pspec;
    gchar *pattern, *str;
    const gchar *name;

    str = g_path_get_basename (basename);
    pattern = g_strconcat (str, "*.png", NULL);
    g_free (str);

    pspec = g_pattern_spec_new (pattern);
    g_free (pattern);

    while ((name = g_dir_read_name (dir)))
    {
      if (g_pattern_match (pspec, strlen (name), name, NULL))
      {
        str = g_build_filename (ogmrip_fs_get_tmp_dir (), name, NULL);
        g_unlink (str);
        g_free (str);
      }
    }

    g_pattern_spec_free (pspec);
    g_dir_close (dir);
  }
}

static void
ogmrip_mpeg_remove_subp (OGMRipMpeg *mpeg,
    OGMRipCodec *codec, guint demuxer, gint language)
{
  if (ogmrip_plugin_get_subp_codec_format  (G_TYPE_FROM_INSTANCE (codec)) == OGMRIP_FORMAT_VOBSUB)
    ogmrip_mpeg_remove_png_files (ogmrip_codec_get_output (codec));
}

static void
ogmrip_mpeg_remove_subp_file (OGMRipMpeg *mpeg, OGMRipFile *file)
{
  if (ogmrip_file_get_type (file) == OGMRIP_FILE_TYPE_SUBP &&
      ogmrip_file_get_format (file) == OGMRIP_FORMAT_VOBSUB)
    ogmrip_mpeg_remove_png_files (ogmrip_file_get_filename (file));
}

static void
ogmrip_mpeg_class_init (OGMRipMpegClass *klass)
{
  OGMRipSettings *settings;

  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;
  OGMRipContainerClass *container_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->get_property = ogmrip_mpeg_get_property;
  gobject_class->set_property = ogmrip_mpeg_set_property;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->run = ogmrip_mpeg_run;

  container_class = OGMRIP_CONTAINER_CLASS (klass);
  container_class->set_options = ogmrip_mpeg_set_options;

  settings = ogmrip_settings_get_default ();

  g_object_class_install_property (gobject_class, PROP_FORMAT,
      g_param_spec_uint (OGMRIP_MPEG_PROP_FORMAT,
        "format property", "Set format", 0, 2, 0, G_PARAM_READWRITE));
}

static void
ogmrip_mpeg_init (OGMRipMpeg *mpeg)
{
}

static void
ogmrip_mpeg_get_property (GObject *gobject, guint property_id, GValue *value, GParamSpec *pspec)
{
  switch (property_id)
  {
    case PROP_FORMAT:
      g_value_set_uint (value, OGMRIP_MPEG (gobject)->format);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_mpeg_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
  {
    case PROP_FORMAT:
      OGMRIP_MPEG (gobject)->format = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_mpeg_set_options (OGMRipContainer *container, const gchar *section)
{
  OGMRipSettings *settings;

  settings = ogmrip_settings_get_default ();
  if (settings)
  {
    gchar *key;

    key = ogmrip_settings_build_section (settings, OGMRIP_MPEG_SECTION, OGMRIP_MPEG_PROP_FORMAT, NULL);
    ogmrip_settings_set_property_from_key (settings, G_OBJECT (container), OGMRIP_MPEG_PROP_FORMAT, section, key);
    g_free (key);
  }
}

static gboolean
ogmrip_mpeg_timeout (OGMRipMpeg *mpeg)
{
  g_signal_emit_by_name (mpeg, "progress", -1.0);

  return TRUE;
}

static gint
ogmrip_mpeg_run (OGMJobSpawn *spawn)
{
  GError *error = NULL;
  OGMJobSpawn *queue, *child;
  gchar **argv, *filename;
  gint result;

  result = OGMJOB_RESULT_ERROR;

  filename = ogmrip_fs_mktemp ("rawvideo.XXXXXX", &error);
  if (!filename)
  {
    ogmjob_spawn_propagate_error (spawn, error);
    return OGMJOB_RESULT_ERROR;
  }

  queue = ogmjob_queue_new ();
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), queue);
  g_object_unref (queue);

  argv = ogmrip_mpeg_rawvideo_command (OGMRIP_CONTAINER (spawn), filename);
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
    g_object_unref (child);

    argv = ogmrip_mpeg_mplex_command (OGMRIP_CONTAINER (spawn), filename);
    if (argv)
    {
      guint source;

      child = ogmjob_exec_newv (argv);
      ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
      g_object_unref (child);

      OGMRIP_MPEG (spawn)->spu = 0;
      ogmrip_container_foreach_subp (OGMRIP_CONTAINER (spawn),
          (OGMRipContainerCodecFunc) ogmrip_mpeg_foreach_subp, queue);
      ogmrip_container_foreach_file (OGMRIP_CONTAINER (spawn), 
          (OGMRipContainerFileFunc) ogmrip_mpeg_foreach_subp_file, queue);

      if (OGMRIP_MPEG (spawn)->format == OGMRIP_MPEG_DVD && have_dvdauthor)
      {
        argv = ogmrip_dvdauthor_command (OGMRIP_MPEG (spawn), FALSE);
        if (argv)
        {
          child = ogmjob_exec_newv (argv);
          ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
          g_object_unref (child);

          argv = ogmrip_dvdauthor_command (OGMRIP_MPEG (spawn), TRUE);
          if (argv)
          {
            child = ogmjob_exec_newv (argv);
            ogmjob_container_add (OGMJOB_CONTAINER (queue), child);
            g_object_unref (child);
          }
        }
      }

      source = g_timeout_add (500, (GSourceFunc) ogmrip_mpeg_timeout, spawn);
      result = OGMJOB_SPAWN_CLASS (ogmrip_mpeg_parent_class)->run (spawn);
      g_source_remove (source);
    }
  }

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), queue);

  ogmrip_container_foreach_subp (OGMRIP_CONTAINER (spawn),
      (OGMRipContainerCodecFunc) ogmrip_mpeg_remove_subp, NULL);
  ogmrip_container_foreach_file (OGMRIP_CONTAINER (spawn), 
      (OGMRipContainerFileFunc) ogmrip_mpeg_remove_subp_file, NULL);

  g_slist_foreach (OGMRIP_MPEG (spawn)->files,
      (GFunc) ogmrip_fs_unref, GINT_TO_POINTER (TRUE));
  g_slist_free (OGMRIP_MPEG (spawn)->files);
  OGMRIP_MPEG (spawn)->files = NULL;

  ogmrip_fs_unref (filename, TRUE);

  return result;
}

/**
 * ogmrip_mpeg_new:
 * @output: The output file
 *
 * Creates a new #OGMRipMpeg.
 *
 * Returns: The new #OGMRipMpeg
 */
OGMJobSpawn *
ogmrip_mpeg_new (const gchar *output)
{
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_MPEG, "output", output, NULL);
}

static OGMRipContainerPlugin mpeg_plugin =
{
  NULL,
  G_TYPE_NONE,
  "mpeg",
  N_("Mpeg Stream (MPG)"),
  TRUE,
  FALSE,
  G_MAXINT,
  G_MAXINT,
  NULL
};

static gint formats[] = 
{
  OGMRIP_FORMAT_MPEG1,
  OGMRIP_FORMAT_MPEG2,
  OGMRIP_FORMAT_AC3,
  OGMRIP_FORMAT_COPY,
  -1,
  -1,
  -1
};

OGMRipContainerPlugin *
ogmrip_init_plugin (GError **error)
{
  OGMRipSettings *settings;
  gboolean have_mplex;
  gchar *fullname;
  gint fmt = 4;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif /* ENABLE_NLS */

  fullname = g_find_program_in_path ("mplex");
  have_mplex = fullname != NULL;
  g_free (fullname);

  if (!have_mplex)
  {
    g_set_error (error, OGMRIP_PLUGIN_ERROR, OGMRIP_PLUGIN_ERROR_REQ, _("MPlex is missing"));
    return NULL;
  }

  fullname = g_find_program_in_path ("dvdauthor");
  have_dvdauthor = fullname != NULL;
  g_free (fullname);

  fullname = g_find_program_in_path ("spumux");
  if (!fullname)
    mpeg_plugin.max_subp = 0;
  g_free (fullname);

  fullname = g_find_program_in_path ("subp2png");
  if (fullname)
    formats[fmt++] = OGMRIP_FORMAT_VOBSUB;
  g_free (fullname);

  fullname = g_build_filename (g_get_home_dir (), ".spumux", "Vera.ttf", NULL);
  if (g_file_test (fullname, G_FILE_TEST_IS_REGULAR))
    formats[fmt++] = OGMRIP_FORMAT_SRT;
  g_free (fullname);

  settings = ogmrip_settings_get_default ();
  if (settings)
  {
    GObjectClass *klass;

    klass = g_type_class_ref (OGMRIP_TYPE_MPEG);
    ogmrip_settings_install_key_from_property (settings, klass,
        OGMRIP_MPEG_SECTION, OGMRIP_MPEG_PROP_FORMAT, OGMRIP_MPEG_PROP_FORMAT);
    g_type_class_unref (klass);
  }

  mpeg_plugin.type = OGMRIP_TYPE_MPEG;
  mpeg_plugin.formats = formats;

  return &mpeg_plugin;
}

