/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  assertUnreachable,
  HttpStatusCode,
  MerchantAuthMethod,
  opFixedSuccess,
} from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useChallengeHandler,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../components/form/FormProvider.js";
import { Input } from "../../components/form/Input.js";
import { SolveMFAChallenges } from "../../components/SolveMFA.js";
import { useSessionContext } from "../../context/session.js";
import { FOREVER_REFRESHABLE_TOKEN } from "../login/index.js";

const TALER_SCREEN_ID = 82;

interface Form {
  password: string;
  repeat: string;
}
interface Props {
  onCancel: () => void;
  id: string;
  onReseted: () => void;
}

export function ResetAccount({
  onCancel,
  onReseted,
  id: instanceId,
}: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn } = useSessionContext();

  const [value, setValue] = useState<Partial<Form>>({
    // password: "asd",
    // repeat: "asd",
  });

  const errors: FormErrors<Form> = {
    password: !value.password ? i18n.str`Required` : undefined,
    repeat: !value.repeat
      ? i18n.str`Required`
      : value.password !== value.repeat
        ? i18n.str`Doesn't match`
        : undefined,
  };

  function valueHandler(s: (d: Partial<Form>) => Partial<Form>): void {
    const next = s(value);
    const v: Form = {
      password: next.password ?? "",
      repeat: next.repeat ?? "",
    };
    setValue(v);
  }
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const mfa = useChallengeHandler();

  const reset = safeFunctionHandler(
    async (password: string, challengeIds: string[]) => {
      const forgot = await lib
        .subInstanceApi(instanceId)
        .instance.forgotPasswordSelfProvision(
          { method: MerchantAuthMethod.TOKEN, password },
          { challengeIds },
        );

      if (forgot.type === "fail") {
        return forgot;
      }
      const login = await lib.instance.createAccessToken(
        instanceId,
        value.password!,
        FOREVER_REFRESHABLE_TOKEN(i18n.str`Password reset`),
      );
      if (login.type === "fail") return login;
      return opFixedSuccess(login.body);
    },
  );
  reset.onSuccess = (suc) => {
    logIn(instanceId, suc.access_token);
    onReseted();
  };
  reset.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Accepted:
        mfa.onChallengeRequired(fail.body);
        return i18n.str`Second factor authentication required.`;
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.Forbidden:
        return i18n.str`Forbidden.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      default:
        assertUnreachable(fail);
    }
  };

  const retry = reset.lambda((ids: string[]) => [reset.args![0], ids]);
  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={retry}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <LocalNotificationBannerBulma notification={notification} />

      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>
                  Resetting access to the instance "{instanceId}"
                </i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              <FormProvider<Form>
                name="settings"
                errors={errors}
                object={value}
                valueHandler={valueHandler}
              >
                <Input<Form>
                  label={i18n.str`New password`}
                  inputType="password"
                  name="password"
                />
                <Input<Form>
                  label={i18n.str`Repeat password`}
                  inputType="password"
                  name="repeat"
                />
              </FormProvider>
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              <button type="button" class="button" onClick={onCancel}>
                <i18n.Translate>Cancel</i18n.Translate>
              </button>
              <ButtonBetterBulma type="submit" onClick={reset}>
                <i18n.Translate>Reset</i18n.Translate>
              </ButtonBetterBulma>
            </footer>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
