import { assert } from '../../../../../common/util/util.js';
import { kValue } from '../../../../util/constants.js';
import { FP, FPInterval } from '../../../../util/floating_point.js';
import { sparseScalarF64Range } from '../../../../util/math.js';
import { reinterpretU64AsF64 } from '../../../../util/reinterpret.js';

/** Simplified version of kSparseVectorF64Range that only uses values from sparseScalarF64Range() */
export const kSparseVectorAFValues = {
  2: sparseScalarF64Range().map(f => Array(2).fill(f)),
  3: sparseScalarF64Range().map(f => Array(3).fill(f)),
  4: sparseScalarF64Range().map(f => Array(4).fill(f)),
} as const;

/** Simplified version of kSparseMatrixF64Range that only uses values from sparseScalarF64Range() */
export const kSparseMatrixAFValues = {
  2: {
    2: sparseScalarF64Range().map(f => Array(2).fill(Array(2).fill(f))),
    3: sparseScalarF64Range().map(f => Array(2).fill(Array(3).fill(f))),
    4: sparseScalarF64Range().map(f => Array(2).fill(Array(4).fill(f))),
  },
  3: {
    2: sparseScalarF64Range().map(f => Array(3).fill(Array(2).fill(f))),
    3: sparseScalarF64Range().map(f => Array(3).fill(Array(3).fill(f))),
    4: sparseScalarF64Range().map(f => Array(3).fill(Array(4).fill(f))),
  },
  4: {
    2: sparseScalarF64Range().map(f => Array(4).fill(Array(2).fill(f))),
    3: sparseScalarF64Range().map(f => Array(4).fill(Array(3).fill(f))),
    4: sparseScalarF64Range().map(f => Array(4).fill(Array(4).fill(f))),
  },
} as const;

/** Line format of af_data_gen/main.cpp outputs for unary operations */
interface RawUnaryData {
  readonly input: number;
  readonly expected: readonly number[];
}

/** Line format of af_data_gen/main.cpp outputs for binary operations */
interface RawBinaryData {
  readonly lhs: number;
  readonly rhs: number;
  readonly expected: readonly number[];
}

/**
 * Externally generated values for binary operations, addition (+) and subtraction (-).
 * These values are generated using af_data_gen/main.cpp and manually copied in
 * here. This is done to get around the limitations of AbstractFloats in WGSL
 * and TS numbers both being f64 internally.
 *
 * This contains entries covering performing the operations over the cartesian product of
 * kInterestingF64Values with itself. Thus, can be used to create a lookup table
 * to replace using the FPTraits framework for calculating intervals where
 * sparseScalarF64Range, etc are being used to generate inputs.
 */
// prettier-ignore
const kAdditionRawValues:  readonly RawBinaryData[] = [{ lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: -10.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: -1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: -0.125, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 0.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 0.125, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 10.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.max, expected: [ 0.0 ] },
  { lhs: -10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: -10.0, rhs: -10.0, expected: [ -20 ] },
  { lhs: -10.0, rhs: -1.0, expected: [ -11 ] },
  { lhs: -10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0xc024400000000000n) /* -10.125 */ ] },
  { lhs: -10.0, rhs: kValue.f64.negative.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: 0.0, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0xc023c00000000000n) /* -9.875 */ ] },
  { lhs: -10.0, rhs: 1.0, expected: [ -9 ] },
  { lhs: -10.0, rhs: 10.0, expected: [ 0.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: -1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: -1.0, rhs: -10.0, expected: [ -11 ] },
  { lhs: -1.0, rhs: -1.0, expected: [ -2 ] },
  { lhs: -1.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0xbff2000000000000n) /* -1.125 */ ] },
  { lhs: -1.0, rhs: kValue.f64.negative.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: 0.0, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0xbfec000000000000n) /* -0.875 */ ] },
  { lhs: -1.0, rhs: 1.0, expected: [ 0.0 ] },
  { lhs: -1.0, rhs: 10.0, expected: [ 9 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: -0.125, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: -0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0xc024400000000000n) /* -10.125 */ ] },
  { lhs: -0.125, rhs: -1.0, expected: [ reinterpretU64AsF64(0xbff2000000000000n) /* -1.125 */ ] },
  { lhs: -0.125, rhs: -0.125, expected: [ reinterpretU64AsF64(0xbfd0000000000000n) /* -0.25 */ ] },
  { lhs: -0.125, rhs: kValue.f64.negative.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: 0.0, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: 0.125, expected: [ 0.0 ] },
  { lhs: -0.125, rhs: 1.0, expected: [ reinterpretU64AsF64(0x3fec000000000000n) /* 0.875 */ ] },
  { lhs: -0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0x4023c00000000000n) /* 9.875 */ ] },
  { lhs: -0.125, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.max, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.max, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.max, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x8020000000000000n) /* -4.45015e-308 */ ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.min, expected: [ reinterpretU64AsF64(0x801fffffffffffffn) /* -4.45015e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.max, expected: [ reinterpretU64AsF64(0x8010000000000001n) /* -2.22507e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: 0.0, expected: [ kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.min, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.max, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.max, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x801fffffffffffffn) /* -4.45015e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ reinterpretU64AsF64(0x801ffffffffffffen) /* -4.45015e-308 */, kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.0, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, reinterpretU64AsF64(0x800ffffffffffffen) /* -2.22507e-308 */, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.subnormal.min, kValue.f64.positive.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x8010000000000001n) /* -2.22507e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ reinterpretU64AsF64(0x8000000000000002n) /* -9.88131e-324 */, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.0, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0, reinterpretU64AsF64(0x000ffffffffffffen) /* 2.22507e-308 */, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: 0.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: 0.0, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: 0.0, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: 0.0, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.max ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: 0.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: 0.0, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.min ] },
  { lhs: 0.0, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: 0.0, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: 0.0, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, reinterpretU64AsF64(0x800ffffffffffffen) /* -2.22507e-308 */, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.0, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min, reinterpretU64AsF64(0x0000000000000002n) /* 9.88131e-324 */ ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min, kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x0010000000000001n) /* 2.22507e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0, reinterpretU64AsF64(0x000ffffffffffffen) /* 2.22507e-308 */, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.0, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min, kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.max, reinterpretU64AsF64(0x001ffffffffffffen) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x001fffffffffffffn) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.min, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.min, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.min, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.max, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.positive.subnormal.min, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: 0.0, expected: [ kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x0010000000000001n) /* 2.22507e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x001fffffffffffffn) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x0020000000000000n) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.min, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.min, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: 0.125, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: 0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0xc023c00000000000n) /* -9.875 */ ] },
  { lhs: 0.125, rhs: -1.0, expected: [ reinterpretU64AsF64(0xbfec000000000000n) /* -0.875 */ ] },
  { lhs: 0.125, rhs: -0.125, expected: [ 0.0 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: 0.0, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: 0.125, expected: [ reinterpretU64AsF64(0x3fd0000000000000n) /* 0.25 */ ] },
  { lhs: 0.125, rhs: 1.0, expected: [ reinterpretU64AsF64(0x3ff2000000000000n) /* 1.125 */ ] },
  { lhs: 0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0x4024400000000000n) /* 10.125 */ ] },
  { lhs: 0.125, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: 1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: 1.0, rhs: -10.0, expected: [ -9 ] },
  { lhs: 1.0, rhs: -1.0, expected: [ 0.0 ] },
  { lhs: 1.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0x3fec000000000000n) /* 0.875 */ ] },
  { lhs: 1.0, rhs: kValue.f64.negative.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: 0.0, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0x3ff2000000000000n) /* 1.125 */ ] },
  { lhs: 1.0, rhs: 1.0, expected: [ 2 ] },
  { lhs: 1.0, rhs: 10.0, expected: [ 11 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: 10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: 10.0, rhs: -10.0, expected: [ 0.0 ] },
  { lhs: 10.0, rhs: -1.0, expected: [ 9 ] },
  { lhs: 10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0x4023c00000000000n) /* 9.875 */ ] },
  { lhs: 10.0, rhs: kValue.f64.negative.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: 0.0, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0x4024400000000000n) /* 10.125 */ ] },
  { lhs: 10.0, rhs: 1.0, expected: [ 11 ] },
  { lhs: 10.0, rhs: 10.0, expected: [ 20 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.min, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.max, rhs: -10.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: -1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: -0.125, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 0.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 0.125, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 10.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
] as const;

// prettier-ignore
const kMultiplicationRawValues:  readonly RawBinaryData[] = [
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: kValue.f64.negative.min, rhs: -10.0, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: kValue.f64.negative.min, rhs: -1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.min, rhs: -0.125, expected: [ reinterpretU64AsF64(0x7fbfffffffffffffn) /* 2.24712e+307 */ ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x400fffffffffffffn) /* 4 */ ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, reinterpretU64AsF64(0x400ffffffffffffdn) /* 4 */ ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, reinterpretU64AsF64(0x3ccfffffffffffffn) /* 8.88178e-16 */ ] },
  { lhs: kValue.f64.negative.min, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.min, expected: [ reinterpretU64AsF64(0xbccfffffffffffffn) /* -8.88178e-16 */, 0.0 ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.max, expected: [ reinterpretU64AsF64(0xc00ffffffffffffdn) /* -4 */, 0.0 ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0xc00fffffffffffffn) /* -4 */ ] },
  { lhs: kValue.f64.negative.min, rhs: 0.125, expected: [ reinterpretU64AsF64(0xffbfffffffffffffn) /* -2.24712e+307 */ ] },
  { lhs: kValue.f64.negative.min, rhs: 1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 10.0, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: -10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: -10.0, rhs: -10.0, expected: [ 100 ] },
  { lhs: -10.0, rhs: -1.0, expected: [ 10.0 ] },
  { lhs: -10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0x3ff4000000000000n) /* 1.25 */ ] },
  { lhs: -10.0, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x0044000000000000n) /* 2.22507e-307 */ ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, reinterpretU64AsF64(0x0043fffffffffffen) /* 2.22507e-307 */, reinterpretU64AsF64(0x0043ffffffffffffn) /* 2.22507e-307 */ ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, reinterpretU64AsF64(0x000000000000000an) /* 4.94066e-323 */ ] },
  { lhs: -10.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ reinterpretU64AsF64(0x800000000000000an) /* -4.94066e-323 */, 0.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ reinterpretU64AsF64(0x8043ffffffffffffn) /* -2.22507e-307 */, reinterpretU64AsF64(0x8043fffffffffffen) /* -2.22507e-307 */, 0.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x8044000000000000n) /* -2.22507e-307 */ ] },
  { lhs: -10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0xbff4000000000000n) /* -1.25 */ ] },
  { lhs: -10.0, rhs: 1.0, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: 10.0, expected: [ -100 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: -1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: -1.0, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: -1.0, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: -1.0, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: -1.0, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.min ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: -1.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.max ] },
  { lhs: -1.0, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: -1.0, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: -0.125, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0x7fbfffffffffffffn) /* 2.24712e+307 */ ] },
  { lhs: -0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0x3ff4000000000000n) /* 1.25 */ ] },
  { lhs: -0.125, rhs: -1.0, expected: [ 0.125 ] },
  { lhs: -0.125, rhs: -0.125, expected: [ reinterpretU64AsF64(0x3f90000000000000n) /* 0.015625 */ ] },
  { lhs: -0.125, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, reinterpretU64AsF64(0x0001ffffffffffffn) /* 2.78134e-309 */, reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: -0.125, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */, reinterpretU64AsF64(0x8001ffffffffffffn) /* -2.78134e-309 */, 0.0 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */ ] },
  { lhs: -0.125, rhs: 0.125, expected: [ reinterpretU64AsF64(0xbf90000000000000n) /* -0.015625 */ ] },
  { lhs: -0.125, rhs: 1.0, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0xbff4000000000000n) /* -1.25 */ ] },
  { lhs: -0.125, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0xffbfffffffffffffn) /* -2.24712e+307 */ ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0x400fffffffffffffn) /* 4 */ ] },
  { lhs: kValue.f64.negative.max, rhs: -10.0, expected: [ reinterpretU64AsF64(0x0044000000000000n) /* 2.22507e-307 */ ] },
  { lhs: kValue.f64.negative.max, rhs: -1.0, expected: [ kValue.f64.positive.min ] },
  { lhs: kValue.f64.negative.max, rhs: -0.125, expected: [ reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.max, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: 0.125, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */ ] },
  { lhs: kValue.f64.negative.max, rhs: 1.0, expected: [ kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: 10.0, expected: [ reinterpretU64AsF64(0x8044000000000000n) /* -2.22507e-307 */ ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0xc00fffffffffffffn) /* -4 */ ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.min, expected: [ 0.0, reinterpretU64AsF64(0x400ffffffffffffdn) /* 4 */ ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -10.0, expected: [ 0.0, reinterpretU64AsF64(0x0043fffffffffffen) /* 2.22507e-307 */, reinterpretU64AsF64(0x0043ffffffffffffn) /* 2.22507e-307 */ ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -1.0, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -0.125, expected: [ 0.0, reinterpretU64AsF64(0x0001ffffffffffffn) /* 2.78134e-309 */, reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.125, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */, reinterpretU64AsF64(0x8001ffffffffffffn) /* -2.78134e-309 */, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 1.0, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 10.0, expected: [ reinterpretU64AsF64(0x8043ffffffffffffn) /* -2.22507e-307 */, reinterpretU64AsF64(0x8043fffffffffffen) /* -2.22507e-307 */, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0xc00ffffffffffffdn) /* -4 */, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.min, expected: [ 0.0, reinterpretU64AsF64(0x3ccfffffffffffffn) /* 8.88178e-16 */ ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -10.0, expected: [ 0.0, reinterpretU64AsF64(0x000000000000000an) /* 4.94066e-323 */ ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -1.0, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -0.125, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.125, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 1.0, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 10.0, expected: [ reinterpretU64AsF64(0x800000000000000an) /* -4.94066e-323 */, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0xbccfffffffffffffn) /* -8.88178e-16 */, 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.min, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: -10.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: -1.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: -0.125, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.max, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.min, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: 0.125, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: 1.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: 10.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.max, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0xbccfffffffffffffn) /* -8.88178e-16 */, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -10.0, expected: [ reinterpretU64AsF64(0x800000000000000an) /* -4.94066e-323 */, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -1.0, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -0.125, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.125, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 1.0, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 10.0, expected: [ 0.0, reinterpretU64AsF64(0x000000000000000an) /* 4.94066e-323 */ ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.max, expected: [ 0.0, reinterpretU64AsF64(0x3ccfffffffffffffn) /* 8.88178e-16 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0xc00ffffffffffffdn) /* -4 */, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -10.0, expected: [ reinterpretU64AsF64(0x8043ffffffffffffn) /* -2.22507e-307 */, reinterpretU64AsF64(0x8043fffffffffffen) /* -2.22507e-307 */, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -1.0, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -0.125, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */, reinterpretU64AsF64(0x8001ffffffffffffn) /* -2.78134e-309 */, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.125, expected: [ 0.0, reinterpretU64AsF64(0x0001ffffffffffffn) /* 2.78134e-309 */, reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 1.0, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 10.0, expected: [ 0.0, reinterpretU64AsF64(0x0043fffffffffffen) /* 2.22507e-307 */, reinterpretU64AsF64(0x0043ffffffffffffn) /* 2.22507e-307 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.max, expected: [ 0.0, reinterpretU64AsF64(0x400ffffffffffffdn) /* 4 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0xc00fffffffffffffn) /* -4 */ ] },
  { lhs: kValue.f64.positive.min, rhs: -10.0, expected: [ reinterpretU64AsF64(0x8044000000000000n) /* -2.22507e-307 */ ] },
  { lhs: kValue.f64.positive.min, rhs: -1.0, expected: [ kValue.f64.negative.max ] },
  { lhs: kValue.f64.positive.min, rhs: -0.125, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.min, rhs: 0.125, expected: [ reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: kValue.f64.positive.min, rhs: 1.0, expected: [ kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: 10.0, expected: [ reinterpretU64AsF64(0x0044000000000000n) /* 2.22507e-307 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0x400fffffffffffffn) /* 4 */ ] },
  { lhs: 0.125, rhs: kValue.f64.negative.min, expected: [ reinterpretU64AsF64(0xffbfffffffffffffn) /* -2.24712e+307 */ ] },
  { lhs: 0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0xbff4000000000000n) /* -1.25 */ ] },
  { lhs: 0.125, rhs: -1.0, expected: [ -0.125 ] },
  { lhs: 0.125, rhs: -0.125, expected: [ reinterpretU64AsF64(0xbf90000000000000n) /* -0.015625 */ ] },
  { lhs: 0.125, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */ ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ reinterpretU64AsF64(0x8002000000000000n) /* -2.78134e-309 */, reinterpretU64AsF64(0x8001ffffffffffffn) /* -2.78134e-309 */, 0.0 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: 0.125, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, reinterpretU64AsF64(0x0001ffffffffffffn) /* 2.78134e-309 */, reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: 0.125, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x0002000000000000n) /* 2.78134e-309 */ ] },
  { lhs: 0.125, rhs: 0.125, expected: [ reinterpretU64AsF64(0x3f90000000000000n) /* 0.015625 */ ] },
  { lhs: 0.125, rhs: 1.0, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0x3ff4000000000000n) /* 1.25 */ ] },
  { lhs: 0.125, rhs: kValue.f64.positive.max, expected: [ reinterpretU64AsF64(0x7fbfffffffffffffn) /* 2.24712e+307 */ ] },
  { lhs: 1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.min ] },
  { lhs: 1.0, rhs: -10.0, expected: [ -10.0 ] },
  { lhs: 1.0, rhs: -1.0, expected: [ -1.0 ] },
  { lhs: 1.0, rhs: -0.125, expected: [ -0.125 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.max ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: 1.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: 1.0, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.min ] },
  { lhs: 1.0, rhs: 0.125, expected: [ 0.125 ] },
  { lhs: 1.0, rhs: 1.0, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: 10.0, expected: [ 10.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max ] },
  { lhs: 10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: 10.0, rhs: -10.0, expected: [ -100 ] },
  { lhs: 10.0, rhs: -1.0, expected: [ -10.0 ] },
  { lhs: 10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0xbff4000000000000n) /* -1.25 */ ] },
  { lhs: 10.0, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x8044000000000000n) /* -2.22507e-307 */ ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ reinterpretU64AsF64(0x8043ffffffffffffn) /* -2.22507e-307 */, reinterpretU64AsF64(0x8043fffffffffffen) /* -2.22507e-307 */, 0.0 ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ reinterpretU64AsF64(0x800000000000000an) /* -4.94066e-323 */, 0.0 ] },
  { lhs: 10.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, reinterpretU64AsF64(0x000000000000000an) /* 4.94066e-323 */ ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, reinterpretU64AsF64(0x0043fffffffffffen) /* 2.22507e-307 */, reinterpretU64AsF64(0x0043ffffffffffffn) /* 2.22507e-307 */ ] },
  { lhs: 10.0, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x0044000000000000n) /* 2.22507e-307 */ ] },
  { lhs: 10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0x3ff4000000000000n) /* 1.25 */ ] },
  { lhs: 10.0, rhs: 1.0, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: 10.0, expected: [ 100 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.max, rhs: -10.0, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.max, rhs: -1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.max, rhs: -0.125, expected: [ reinterpretU64AsF64(0xffbfffffffffffffn) /* -2.24712e+307 */ ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0xc00fffffffffffffn) /* -4 */ ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.min, expected: [ reinterpretU64AsF64(0xc00ffffffffffffdn) /* -4 */, 0.0 ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.max, expected: [ reinterpretU64AsF64(0xbccfffffffffffffn) /* -8.88178e-16 */, 0.0 ] },
  { lhs: kValue.f64.positive.max, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.0, reinterpretU64AsF64(0x3ccfffffffffffffn) /* 8.88178e-16 */ ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.0, reinterpretU64AsF64(0x400ffffffffffffdn) /* 4 */ ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x400fffffffffffffn) /* 4 */ ] },
  { lhs: kValue.f64.positive.max, rhs: 0.125, expected: [ reinterpretU64AsF64(0x7fbfffffffffffffn) /* 2.24712e+307 */ ] },
  { lhs: kValue.f64.positive.max, rhs: 1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 10.0, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
] as const;

// prettier-ignore
const kSubtractionRawValues:  readonly RawBinaryData[] = [
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.min, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.min, rhs: -10.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: -1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: -0.125, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 0.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 0.125, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 1.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: 10.0, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.infinity, kValue.f64.negative.min ] },
  { lhs: -10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: -10.0, rhs: -10.0, expected: [ 0.0 ] },
  { lhs: -10.0, rhs: -1.0, expected: [ -9 ] },
  { lhs: -10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0xc023c00000000000n) /* -9.875 */ ] },
  { lhs: -10.0, rhs: kValue.f64.negative.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: 0.0, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.min, expected: [ -10.0 ] },
  { lhs: -10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0xc024400000000000n) /* -10.125 */ ] },
  { lhs: -10.0, rhs: 1.0, expected: [ -11 ] },
  { lhs: -10.0, rhs: 10.0, expected: [ -20 ] },
  { lhs: -10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: -1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: -1.0, rhs: -10.0, expected: [ 9 ] },
  { lhs: -1.0, rhs: -1.0, expected: [ 0.0 ] },
  { lhs: -1.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0xbfec000000000000n) /* -0.875 */ ] },
  { lhs: -1.0, rhs: kValue.f64.negative.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: 0.0, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.min, expected: [ -1.0 ] },
  { lhs: -1.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0xbff2000000000000n) /* -1.125 */ ] },
  { lhs: -1.0, rhs: 1.0, expected: [ -2 ] },
  { lhs: -1.0, rhs: 10.0, expected: [ -11 ] },
  { lhs: -1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: -0.125, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: -0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0x4023c00000000000n) /* 9.875 */ ] },
  { lhs: -0.125, rhs: -1.0, expected: [ reinterpretU64AsF64(0x3fec000000000000n) /* 0.875 */ ] },
  { lhs: -0.125, rhs: -0.125, expected: [ 0.0 ] },
  { lhs: -0.125, rhs: kValue.f64.negative.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: 0.0, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: kValue.f64.positive.min, expected: [ -0.125 ] },
  { lhs: -0.125, rhs: 0.125, expected: [ reinterpretU64AsF64(0xbfd0000000000000n) /* -0.25 */ ] },
  { lhs: -0.125, rhs: 1.0, expected: [ reinterpretU64AsF64(0xbff2000000000000n) /* -1.125 */ ] },
  { lhs: -0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0xc024400000000000n) /* -10.125 */ ] },
  { lhs: -0.125, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.max, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.max, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.max, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.max, expected: [ 0.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min ] },
  { lhs: kValue.f64.negative.max, rhs: 0.0, expected: [ kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.min, expected: [ reinterpretU64AsF64(0x8010000000000001n) /* -2.22507e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.subnormal.max, expected: [ reinterpretU64AsF64(0x801fffffffffffffn) /* -4.45015e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x8020000000000000n) /* -4.45015e-308 */ ] },
  { lhs: kValue.f64.negative.max, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.max, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.max, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.subnormal.min, kValue.f64.positive.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.min, 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, reinterpretU64AsF64(0x800ffffffffffffen) /* -2.22507e-308 */, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.0, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ reinterpretU64AsF64(0x801ffffffffffffen) /* -4.45015e-308 */, kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x801fffffffffffffn) /* -4.45015e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.subnormal.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0, reinterpretU64AsF64(0x000ffffffffffffen) /* 2.22507e-308 */, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.negative.subnormal.max, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.0, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ reinterpretU64AsF64(0x8000000000000002n) /* -9.88131e-324 */, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min, kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.min, expected: [ reinterpretU64AsF64(0x8010000000000001n) /* -2.22507e-308 */, kValue.f64.negative.max ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.negative.subnormal.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: 0.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: 0.0, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: 0.0, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: 0.0, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: 0.0, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.min ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: 0.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: 0.0, rhs: 0.0, expected: [ 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, 0.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.max ] },
  { lhs: 0.0, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: 0.0, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: 0.0, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: 0.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x0010000000000001n) /* 2.22507e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min, kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min, reinterpretU64AsF64(0x0000000000000002n) /* 9.88131e-324 */ ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.0, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, reinterpretU64AsF64(0x800ffffffffffffen) /* -2.22507e-308 */, 0.0, kValue.f64.positive.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.min ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.subnormal.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x001fffffffffffffn) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.max, reinterpretU64AsF64(0x001ffffffffffffen) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.min, kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.0, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.negative.subnormal.max, 0.0, reinterpretU64AsF64(0x000ffffffffffffen) /* 2.22507e-308 */, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.negative.subnormal.min, 0.0, kValue.f64.positive.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.negative.max, kValue.f64.negative.subnormal.max ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.subnormal.max, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.min, rhs: -10.0, expected: [ 10.0 ] },
  { lhs: kValue.f64.positive.min, rhs: -1.0, expected: [ 1.0 ] },
  { lhs: kValue.f64.positive.min, rhs: -0.125, expected: [ 0.125 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.max, expected: [ reinterpretU64AsF64(0x0020000000000000n) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x001fffffffffffffn) /* 4.45015e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.positive.min, reinterpretU64AsF64(0x0010000000000001n) /* 2.22507e-308 */ ] },
  { lhs: kValue.f64.positive.min, rhs: 0.0, expected: [ kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.positive.subnormal.max, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.positive.subnormal.min, kValue.f64.positive.min ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.min, expected: [ 0.0 ] },
  { lhs: kValue.f64.positive.min, rhs: 0.125, expected: [ -0.125 ] },
  { lhs: kValue.f64.positive.min, rhs: 1.0, expected: [ -1.0 ] },
  { lhs: kValue.f64.positive.min, rhs: 10.0, expected: [ -10.0 ] },
  { lhs: kValue.f64.positive.min, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: 0.125, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: 0.125, rhs: -10.0, expected: [ reinterpretU64AsF64(0x4024400000000000n) /* 10.125 */ ] },
  { lhs: 0.125, rhs: -1.0, expected: [ reinterpretU64AsF64(0x3ff2000000000000n) /* 1.125 */ ] },
  { lhs: 0.125, rhs: -0.125, expected: [ reinterpretU64AsF64(0x3fd0000000000000n) /* 0.25 */ ] },
  { lhs: 0.125, rhs: kValue.f64.negative.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.negative.subnormal.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: 0.0, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.subnormal.max, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: kValue.f64.positive.min, expected: [ 0.125 ] },
  { lhs: 0.125, rhs: 0.125, expected: [ 0.0 ] },
  { lhs: 0.125, rhs: 1.0, expected: [ reinterpretU64AsF64(0xbfec000000000000n) /* -0.875 */ ] },
  { lhs: 0.125, rhs: 10.0, expected: [ reinterpretU64AsF64(0xc023c00000000000n) /* -9.875 */ ] },
  { lhs: 0.125, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: 1.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: 1.0, rhs: -10.0, expected: [ 11 ] },
  { lhs: 1.0, rhs: -1.0, expected: [ 2 ] },
  { lhs: 1.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0x3ff2000000000000n) /* 1.125 */ ] },
  { lhs: 1.0, rhs: kValue.f64.negative.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: 0.0, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.min, expected: [ 1.0 ] },
  { lhs: 1.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0x3fec000000000000n) /* 0.875 */ ] },
  { lhs: 1.0, rhs: 1.0, expected: [ 0.0 ] },
  { lhs: 1.0, rhs: 10.0, expected: [ -9 ] },
  { lhs: 1.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: 10.0, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max ] },
  { lhs: 10.0, rhs: -10.0, expected: [ 20 ] },
  { lhs: 10.0, rhs: -1.0, expected: [ 11 ] },
  { lhs: 10.0, rhs: -0.125, expected: [ reinterpretU64AsF64(0x4024400000000000n) /* 10.125 */ ] },
  { lhs: 10.0, rhs: kValue.f64.negative.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.negative.subnormal.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: 0.0, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.subnormal.max, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.min, expected: [ 10.0 ] },
  { lhs: 10.0, rhs: 0.125, expected: [ reinterpretU64AsF64(0x4023c00000000000n) /* 9.875 */ ] },
  { lhs: 10.0, rhs: 1.0, expected: [ 9 ] },
  { lhs: 10.0, rhs: 10.0, expected: [ 0.0 ] },
  { lhs: 10.0, rhs: kValue.f64.positive.max, expected: [ kValue.f64.negative.min ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.min, expected: [ kValue.f64.positive.max, kValue.f64.positive.infinity ] },
  { lhs: kValue.f64.positive.max, rhs: -10.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: -1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: -0.125, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.negative.subnormal.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 0.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.subnormal.max, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.min, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 0.125, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 1.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: 10.0, expected: [ kValue.f64.positive.max ] },
  { lhs: kValue.f64.positive.max, rhs: kValue.f64.positive.max, expected: [ 0.0 ] },
] as const;

// prettier-ignore
const kFractRawValues:  readonly RawUnaryData[] = [
  { input: kValue.f64.negative.min, expected: [ 0.0 ] },
  { input: -10.0, expected: [ 0.0 ] },
  { input: -1.0, expected: [ 0.0 ] },
  { input: -0.125, expected: [ reinterpretU64AsF64(0x3fec000000000000n) /* 0.875 */ ] },
  { input: kValue.f64.negative.max, expected: [ 1.0 ] },
  { input: kValue.f64.negative.subnormal.min, expected: [ 1.0 ] },
  { input: kValue.f64.negative.subnormal.max, expected: [ 1.0 ] },
  { input: 0.0, expected: [ 0.0 ] },
  { input: kValue.f64.positive.subnormal.min, expected: [ 0.0, kValue.f64.positive.subnormal.min ] },
  { input: kValue.f64.positive.subnormal.max, expected: [ 0.0, kValue.f64.positive.subnormal.max ] },
  { input: kValue.f64.positive.min, expected: [ kValue.f64.positive.min ] },
  { input: 0.125, expected: [ 0.125 ] },
  { input: 1.0, expected: [ 0.0 ] },
  { input: 10.0, expected: [ 0.0 ] },
  { input: kValue.f64.positive.max, expected: [ 0.0 ] },
  { input: -10.0000999999999997669, expected: [ reinterpretU64AsF64(0x3fefff2e48e8a720n) /* ~0.9999 */ ] },
  { input: -2, expected: [ 0.0 ] },
  { input: -1, expected: [ 0.0 ] },
  { input: -0.5, expected: [ reinterpretU64AsF64(0x3fe0000000000000n) /* 0.5 */ ] },
  { input: 0.5, expected: [ reinterpretU64AsF64(0x3fe0000000000000n) /* 0.5 */ ] },
  { input: 1, expected: [ 0.0 ] },
  { input: 2, expected: [ 0.0 ] },
  { input: 10.0000999999999997669, expected: [ reinterpretU64AsF64(0x3f1a36e2eb1c0000n) /* ~0.0001 */ ] },
  { input: 3937509.87755102012306, expected: [ reinterpretU64AsF64(0x3fec14e5e0800000n) /* ~0.877551 */ ] },
] as const;

/** Table mapping addition binary inputs to expectation intervals */
const kAdditionTable: Readonly<Map<string, FPInterval>> = new Map<string, FPInterval>(
  kAdditionRawValues.map(value => [
    `${value.lhs} ${value.rhs}`,
    FP.abstract.spanIntervals(...value.expected.map(e => FP.abstract.correctlyRoundedInterval(e))),
  ])
);

/** External interface for fetching addition expectation intervals. */
export function getAdditionAFInterval(lhs: number, rhs: number): FPInterval {
  assert(kAdditionTable.has(`${lhs} ${rhs}`), `${lhs} + ${rhs} not defined in lookup table`);
  return kAdditionTable.get(`${lhs} ${rhs}`) as FPInterval;
}

/** Table mapping multiplication binary inputs to expectation intervals */
const kMultiplicationTable: Readonly<Map<string, FPInterval>> = new Map<string, FPInterval>(
  kMultiplicationRawValues.map(value => [
    `${value.lhs} ${value.rhs}`,
    FP.abstract.spanIntervals(...value.expected.map(e => FP.abstract.correctlyRoundedInterval(e))),
  ])
);

/** External interface for fetching multiplication expectation intervals. */
export function getMultiplicationAFInterval(lhs: number, rhs: number): FPInterval {
  assert(kMultiplicationTable.has(`${lhs} ${rhs}`), `${lhs} * ${rhs} not defined in lookup table`);
  return kMultiplicationTable.get(`${lhs} ${rhs}`) as FPInterval;
}

/** Table mapping subtraction binary inputs to expectation intervals */
const kSubtractionTable: Readonly<Map<string, FPInterval>> = new Map<string, FPInterval>(
  kSubtractionRawValues.map(value => [
    `${value.lhs} ${value.rhs}`,
    FP.abstract.spanIntervals(...value.expected.map(e => FP.abstract.correctlyRoundedInterval(e))),
  ])
);

/** External interface for fetching subtraction expectation intervals. */
export function getSubtractionAFInterval(lhs: number, rhs: number): FPInterval {
  assert(kSubtractionTable.has(`${lhs} ${rhs}`), `${lhs} * ${rhs} not defined in lookup table`);
  return kSubtractionTable.get(`${lhs} ${rhs}`) as FPInterval;
}

/**
 * Table mapping fract unary inputs to expectation intervals. Directly exposed, since the keys are
 * numbers, so do not need the caller to know how to pack multiple values into a key correctly.
 */
export const kFractTable: Readonly<Map<number, FPInterval>> = new Map<number, FPInterval>(
  kFractRawValues.map(value => [
    value.input,
    FP.abstract.spanIntervals(...value.expected.map(e => FP.abstract.correctlyRoundedInterval(e))),
  ])
);
