/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "2D.h"
#include "PathAnalysis.h"
#include "PathHelpers.h"

namespace mozilla {
namespace gfx {

static double CubicRoot(double aValue) {
  if (aValue < 0.0) {
    return -CubicRoot(-aValue);
  } else {
    return pow(aValue, 1.0 / 3.0);
  }
}

struct PointD : public BasePoint<double, PointD> {
  typedef BasePoint<double, PointD> Super;

  PointD() : Super() {}
  PointD(double aX, double aY) : Super(aX, aY) {}
  MOZ_IMPLICIT PointD(const Point& aPoint) : Super(aPoint.x, aPoint.y) {}

  Point ToPoint() const {
    return Point(static_cast<Float>(x), static_cast<Float>(y));
  }
};

struct BezierControlPoints {
  BezierControlPoints() = default;
  BezierControlPoints(const PointD& aCP1, const PointD& aCP2,
                      const PointD& aCP3, const PointD& aCP4)
      : mCP1(aCP1), mCP2(aCP2), mCP3(aCP3), mCP4(aCP4) {}

  PointD mCP1, mCP2, mCP3, mCP4;
};

void FlattenBezier(const BezierControlPoints& aPoints, PathSink* aSink,
                   double aTolerance);

Path::Path() = default;

Path::~Path() = default;

Float Path::ComputeLength() {
  EnsureFlattenedPath();
  return mFlattenedPath->ComputeLength();
}

Point Path::ComputePointAtLength(Float aLength, Point* aTangent) {
  EnsureFlattenedPath();
  return mFlattenedPath->ComputePointAtLength(aLength, aTangent);
}

void Path::EnsureFlattenedPath() {
  if (!mFlattenedPath) {
    mFlattenedPath = new FlattenedPath();
    StreamToSink(mFlattenedPath);
  }
}

// This is the maximum deviation we allow (with an additional ~20% margin of
// error) of the approximation from the actual Bezier curve.
const Float kFlatteningTolerance = 0.0001f;

void FlattenedPath::MoveTo(const Point& aPoint) {
  MOZ_ASSERT(!mCalculatedLength);
  FlatPa