/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import DELIMITER = require( './../../../path/delimiter' );
import DELIMITER_POSIX = require( './../../../path/delimiter-posix' );
import DELIMITER_WIN32 = require( './../../../path/delimiter-win32' );
import SEP = require( './../../../path/sep' );
import SEP_POSIX = require( './../../../path/sep-posix' );
import SEP_WIN32 = require( './../../../path/sep-win32' );

/**
* Interface describing the `path` namespace.
*/
interface Namespace {
	/**
	* Platform-specific path delimiter.
	*
	* @example
	* var IS_WINDOWS = require( `@stdlib/assert/is-windows` );
	*
	* var PATH;
	* var paths;
	*
	* if ( IS_WINDOWS ) {
	*     PATH = 'C:\\Windows\\system32;C:\\Windows;C:\\Program Files\\node\\';
	*     paths = PATH.split( ns.DELIMITER );
	*     // returns ['C:\\Windows\\system32','C:\\Windows','C:\\Program Files\\node\\']
	* } else {
	*     PATH = '/usr/bin:/bin:/usr/sbin:/sbin:/usr/local/bin';
	*     paths = PATH.split( ns.DELIMITER );
	*     // returns ['/usr/bin','/bin','/usr/sbin','/sbin','/usr/local/bin']
	* }
	*/
	DELIMITER: typeof DELIMITER;

	/**
	* POSIX path delimiter.
	*
	* @example
	* var PATH = '/usr/bin:/bin:/usr/sbin:/sbin:/usr/local/bin';
	* var paths = PATH.split( ns.DELIMITER_POSIX );
	* // returns ['/usr/bin','/bin','/usr/sbin','/sbin','/usr/local/bin']
	*/
	DELIMITER_POSIX: typeof DELIMITER_POSIX;

	/**
	* Windows path delimiter.
	*
	* @example
	* var PATH = 'C:\\Windows\\system32;C:\\Windows;C:\\Program Files\\node\\';
	* var paths = PATH.split( ns.DELIMITER_WIN32 );
	* // returns ['C:\\Windows\\system32','C:\\Windows','C:\\Program Files\\node\\']
	*/
	DELIMITER_WIN32: typeof DELIMITER_WIN32;

	/**
	* Platform-specific path segment separator.
	*
	* @example
	* var IS_WINDOWS = require( `@stdlib/assert/is-windows` );
	*
	* var parts;
	* if ( IS_WINDOWS ) {
	*     parts = 'foo\\bar\\baz'.split( ns.SEP );
	*     // returns ['foo','bar','baz']
	* } else {
	*     parts = 'foo/bar/baz'.split( ns.SEP );
	*     // returns ['foo','bar','baz']
	* }
	*/
	SEP: typeof SEP;

	/**
	* POSIX path segment separator.
	*
	* @example
	* var parts = 'foo/bar/baz'.split( ns.SEP_POSIX );
	* // returns ['foo','bar','baz']
	*/
	SEP_POSIX: typeof SEP_POSIX;

	/**
	* Windows path segment separator.
	*
	* @example
	* var parts = 'foo\\bar\\baz'.split( ns.SEP_WIN32 );
	* // returns ['foo','bar','baz']
	*/
	SEP_WIN32: typeof SEP_WIN32;
}

/**
* Standard string path constants.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
