/*
 * Copyright (C) 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <diagnostics/extensions/stacktrace/descriptor.hpp>
#include <diagnostics/extensions/stacktrace/signal.hpp>
#include <diagnostics/extensions/stacktrace/process.hpp>
#include <diagnostics/extensions/stacktrace/raw_process.hpp>

#include <cerrno>
#include <poll.h>
#include <sys/wait.h>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;

POSIX::ProcessResult::ProcessResult()
: Status(-1), Signal(-1)
{
}

POSIX::ProcessResult::~ProcessResult()
{
}

// Read from OUTPUT and ERROR and store the result in RESULT.
static void
run(pid_t pid,
    POSIX::Descriptor &output, POSIX::Descriptor &error,
    POSIX::ProcessResult &result)
{
  int nfds = 2;
  struct pollfd fds[2];
  fds[0].fd = output.GetRaw();
  fds[0].events = POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;
  fds[1].fd = error.GetRaw();
  fds[1].events = POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;

  std::string *strings[2];
  strings[0] = &result.Output;
  strings[1] = &result.Error;

  result.Output.clear();
  result.Error.clear();

  char buf[1024];
  while (nfds > 0) {
    // Always use both descriptors, the kernel will skip them if necessary.
    POSIX::Error::Unless(poll(fds, 2, -1) >= 1);
    for (int i = 0; i < 2; ++i) {
      if (fds[i].revents & POLLNVAL) {
        throw POSIX::Error(EBADF);
      }

      if (fds[i].revents
                 & (POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI)) {
        int ret = read(fds[i].fd, buf, sizeof(buf));
        switch (ret) {
        case 0:
          // EOF, release this descriptor.
          fds[i].events = 0;
          --nfds;
          break;
        case -1:
          if (errno != EAGAIN && errno != EWOULDBLOCK) {
            throw POSIX::Error();
          }
          break;
        default:
          strings[i]->append(buf, ret);
        }
      } else if (fds[i].revents & (POLLERR | POLLHUP )) {
        // Can also be set when POLLIN etc. is set.  In this case, we
        // could safe a poll syscall when the read was smaller than
        // the whole buffer.
        fds[i].events = 0;
        --nfds;
      }
    }
  }

  int status;
  POSIX::Error::Unless(waitpid(pid, &status, 0) >= 0);
  if (WIFEXITED(status)) {
    result.Status = WEXITSTATUS(status);
    result.Signal = -1;
  } else if (WIFSIGNALED(status)) {
    result.Status = -1;
    result.Signal = WTERMSIG(status);
  } else {
    // Cannot actually happen.
    result.Status = -1;
    result.Signal = -1;
  }
}

void
POSIX::Run(const char *program, char *const args[], char *const env[],
                 ProcessResult &result)
{
  Sigset block_chld;
  block_chld.Add(SIGCHLD);
  SaveSigProcMask saved(block_chld);

  POSIX::Pipe output, error;
  output.Read().SetCloseOnExec(true);
  output.Read().SetNonBlocking(true);
  error.Read().SetCloseOnExec(true);
  error.Read().SetNonBlocking(true);

  SpawnFileActions actions;
  actions.AddOpen(0, "/dev/null", O_RDONLY, 0);
  actions.AddDup2(output.Write().GetRaw(), 1);
  actions.AddDup2(error.Write().GetRaw(), 2);

  SpawnAttr attr;
  // Ideally, we would want to restore all signals to their default
  // handlers, but if we specify the full set, the spawn fails in the
  // child (silently!).
  attr.SetSigDefault(Sigset().Add(SIGINT).Add(SIGQUIT));
  attr.SetFlags(POSIX_SPAWN_SETSIGDEF);

  pid_t pid;
  POSIX::Error::Unless
    (posix_spawnp(&pid, program,
                  actions.GetRaw(), attr.GetRaw(),
                  args, env) == 0);
  output.Write().Close();
  error.Write().Close();

  run(pid, output.Read(), error.Read(), result);
}

STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

