/*******************************************************************************
  
  Copyright(c) 2004 Intel Corporation. All rights reserved.

  Portions of this file are based on the WEP enablement code provided by the
  Host AP project hostap-drivers v0.1.3
  Copyright (c) 2001-2002, SSH Communications Security Corp and Jouni Malinen
  <jkmaline@cc.hut.fi>
  Copyright (c) 2002-2003, Jouni Malinen <jkmaline@cc.hut.fi>
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/
#include <linux/wireless.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/kmod.h> /* request_module */
#endif

#include "ieee80211.h"

static void ieee80211_crypt_delayed_deinit(struct ieee80211_device *ieee,
					   struct ieee80211_crypt_data **crypt)
{
	struct ieee80211_crypt_data *tmp;
	unsigned long flags;

	if (*crypt == NULL)
		return;
	
	tmp = *crypt;
	*crypt = NULL;

	/* must not run ops->deinit() while there may be pending encrypt or
	 * decrypt operations. Use a list of delayed deinits to avoid needing
	 * locking. */

	spin_lock_irqsave(&ieee->lock, flags);
	list_add(&tmp->list, &ieee->crypt_deinit_list);
	if (!timer_pending(&ieee->crypt_deinit_timer)) {
		ieee->crypt_deinit_timer.expires = jiffies + HZ;
		add_timer(&ieee->crypt_deinit_timer);
	}
	spin_unlock_irqrestore(&ieee->lock, flags);
}


int ieee80211_wx_set_encode(struct ieee80211_device *ieee,
			   struct iw_request_info *info, 
			   union iwreq_data *wrqu, char *key)
{
	struct iw_point *erq = &(wrqu->encoding);
#ifdef CONFIG_IPW2100_NOWEP
	if (erq->flags & IW_ENCODE_DISABLED)
		return 0;
	return -EOPNOTSUPP;
#else
	struct net_device *dev = ieee->dev;
	struct ieee80211_security sec = {
		.flags = 0
	};
	int first = 0;
	int i;

	if (erq->flags & IW_ENCODE_DISABLED) {
		sec.enabled = 0;
		sec.flags |= SEC_ENABLED;
		ieee80211_crypt_delayed_deinit(ieee, &ieee->crypt);
		goto done;
	}
	
	sec.enabled = 1;
	sec.flags |= SEC_ENABLED;

	if (ieee->crypt != NULL && ieee->crypt->ops != NULL &&
	    strcmp(ieee->crypt->ops->name, "WEP") != 0) {
		/* changing to use WEP; deinit previously used algorithm */
		ieee80211_crypt_delayed_deinit(ieee, &ieee->crypt);
	}
	
	if (ieee->crypt == NULL) {
		struct ieee80211_crypt_data *new_crypt;
		
		/* take WEP into use */
		new_crypt = (struct ieee80211_crypt_data *)
			kmalloc(sizeof(struct ieee80211_crypt_data), 
				GFP_KERNEL);
		if (new_crypt == NULL)
			return -ENOMEM;
		memset(new_crypt, 0, sizeof(struct ieee80211_crypt_data));
		new_crypt->ops = hostap_get_crypto_ops("WEP");
		if (!new_crypt->ops) {
			request_module("hostap_crypt_wep");
			new_crypt->ops = hostap_get_crypto_ops("WEP");
		}
		if (new_crypt->ops)
			new_crypt->priv = new_crypt->ops->init();
		if (!new_crypt->ops || !new_crypt->priv) {
			kfree(new_crypt);
			new_crypt = NULL;

			printk(KERN_WARNING "%s: could not initialize WEP: "
			       "load module hostap_crypt_wep.o\n",
			       dev->name);
			return -EOPNOTSUPP;
		}
		first = 1;
		ieee->crypt = new_crypt;
	}

	i = erq->flags & IW_ENCODE_INDEX;
	if (i < 1 || i > 4)
		i = ieee->crypt->ops->get_key_idx(ieee->crypt->priv);
	else
		i--;
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	if (erq->length > 0) {
		int len = erq->length <= 5 ? 5 : 13;
		if (len > erq->length)
			memset(key + erq->length, 0, len - erq->length);

		ieee->crypt->ops->set_key(i, key, len, ieee->crypt->priv);
		memcpy(sec.keys[i], key, len);
		sec.key_sizes[i] = len;
		sec.flags |= (1 << i);

		if (first) {
			ieee->crypt->ops->set_key_idx(i, ieee->crypt->priv);
			sec.active_key = i;
			sec.flags |= SEC_ACTIVE_KEY;
		}

	} else if (ieee->crypt->ops->set_key_idx(i, ieee->crypt->priv) < 0)
		return -EINVAL; /* keyidx not valid */
	else {
		sec.active_key = i;
		sec.flags |= SEC_ACTIVE_KEY;
	}
	

 done:
	ieee->open_wep = erq->flags & IW_ENCODE_OPEN;
	sec.auth_mode = ieee->open_wep ? WLAN_AUTH_OPEN : WLAN_AUTH_SHARED_KEY;
	sec.flags |= SEC_AUTH_MODE;

	if (ieee->func && ieee->func->set_security)
		ieee->func->set_security(ieee, &sec);

	/* Do not reset port if card is in Managed mode since resetting will
	 * generate new IEEE 802.11 authentication which may end up in looping
	 * with IEEE 802.1X.  If your hardware requires a reset after WEP
	 * configuration (for example... Prism2), implement the reset_port in 
	 * the callbacks structures used to initialize the 802.11 stack. */
	if (ieee->reset_on_keychange &&
	    ieee->iw_mode != IW_MODE_INFRA && 
	    ieee->func->reset_port &&
	    ieee->func->reset_port(dev)) {
		printk(KERN_DEBUG "%s: reset_port failed\n", dev->name);
		return -EINVAL;
	}
	return 0;
#endif
}

int ieee80211_wx_get_encode(struct ieee80211_device *ieee,
			   struct iw_request_info *info, 
			   union iwreq_data *wrqu, char *key)
{
	struct iw_point *erq = &(wrqu->encoding);
#ifdef CONFIG_IPW2100_NOWEP
	erq->length = 0;
	erq->flags = IW_ENCODE_DISABLED;
	return 0;
#else
	int i, len;

	if (ieee->crypt == NULL || ieee->crypt->ops == NULL) {
		erq->length = 0;
		erq->flags = IW_ENCODE_DISABLED;
		return 0;
	}

	if (strcmp(ieee->crypt->ops->name, "WEP") != 0) {
		/* only WEP is supported with wireless extensions, so just
		 * report that encryption is used */
		erq->length = 0;
		erq->flags = IW_ENCODE_ENABLED;
		return 0;
	}

	i = erq->flags & IW_ENCODE_INDEX;
	if (i < 1 || i > 4)
		i = ieee->crypt->ops->get_key_idx(ieee->crypt->priv);
	else
		i--;
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	erq->flags = i + 1;

	len = ieee->crypt->ops->get_key(i, key, WEP_KEY_LEN,
					ieee->crypt->priv);
	erq->length = (len >= 0 ? len : 0);

	if (ieee->host_encrypt)
		erq->flags |= IW_ENCODE_ENABLED;
	else
		erq->flags |= IW_ENCODE_DISABLED;
	if (ieee->open_wep)
		erq->flags |= IW_ENCODE_OPEN;
	else
		erq->flags |= IW_ENCODE_RESTRICTED;

	return 0;
#endif
}
