/*
 * High performance packet classification - optimized btree
 *
 * Authors: Michael Bellion and Thomas Heinz
 * (c) 2002-2003 by the hipac core team <nf@hipac.org>:
 *      +-----------------------+----------------------+
 *      |   Michael Bellion     |     Thomas Heinz     |
 *      | <mbellion@hipac.org>  |  <creatix@hipac.org> |
 *      +-----------------------+----------------------+
 * Licenced under the GNU General Public Licence, version >= 2.
 */


#ifndef _BTREE_H
#define _BTREE_H

#include "global.h"

/* wildcard pointer to the next btree spec */
#define WILDCARD(b) ((struct gen_spec **) ((__u8 *) (b) + sizeof(*(b))))
#define IS_BTREE(b) (((struct gen_spec *) (b))->bittype != BIT_RULE)


/* btree header */
struct btree_spec
{
	unsigned bittype   :  2; // {BIT_U8, BIT_U16, BIT_U32}
	unsigned btreetype :  2; // number to distinguish different btrees
	                         // of the same bit type
	unsigned blocks    :  4; // number of 64 byte blocks occupied by root
	unsigned dimid     :  4; // dimension id
	unsigned newspec   :  1; // indicates whether the btree is contained
                              	 // in newspec
	unsigned num       : 19; // number of elements in the btree
};

/* key and nextspec pointer found by btree_locate */
struct locate_inf
{
	__u32 key;
	struct gen_spec **nextspec;
};


/* return address of termrule pointer */
struct ptrblock **
termrule(const struct btree_spec *spec);

/* return new btree with ins_num (1 or 2) elements inserted; the elements
   are (key[i], nextspec[i]) where 0 <= i < ins_num; if ins_num == 2 then
   key[1] > key[0] */
struct btree_spec *
btree_new(__u8 bittype, __u8 dimid, __u8 ins_num, const __u32 key[],
	  struct gen_spec *nextspec[]);

/* return the size of the btree */
__u32
btree_size(const struct btree_spec *spec);

static inline void
btree_free(struct btree_spec *spec)
{
	struct ptrblock *term;
	
	if (spec == NULL) {
		ARG_MSG;
		return;
	}
	term = *termrule(spec);
	if (term != NULL) {
		hp_free(term);
	}
	hp_free(spec);
}

static inline int
btree_spec_eq(const struct btree_spec *spec1, const struct btree_spec *spec2)
{
	if (spec1 == NULL || spec2 == NULL || !IS_BTREE(spec1) ||
	    !IS_BTREE(spec2)) {
		ARG_MSG;
		return 0;
	}
	return spec1->bittype == spec2->bittype &&
		spec1->btreetype == spec2->btreetype &&
		spec1->blocks == spec2->blocks &&
		spec1->dimid == spec2->dimid &&
		spec1->num == spec2->num;
}

/* clone btree (not recursively);
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
btree_clone(const struct btree_spec *spec, struct btree_spec **clone);

/* insert (key[i], nextspec[i]) where 0 <= i < ins_num into the btree
   and store the new btree in result; if ins_num == 2 then key[1] must
   be > key[0];
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
btree_insert(const struct btree_spec *spec, __u8 ins_num, const __u32 key[],
	     struct gen_spec *nextspec[], struct btree_spec **result);

/* delete (key[i], nextspec[i]) from the btree where 0 <= i < del_num
   and nextspec[i] is associated with key[i] and store the new btree in
   result; if ins_num == 2 then key[1] must be > key[0];
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
btree_delete(const struct btree_spec *spec, __u8 del_num, const __u32 key[],
	     struct btree_spec **result);

/* return (key', nextspec) where key' = min{k : k >= key} and nextspec
   is associated with key';
   possible errors: HE_LOW_MEMORY, HE_IMPOSSIBLE_CONDITION */
hipac_error
btree_locate(const struct btree_spec *spec, struct locate_inf *inf,
	     __u32 key);

static inline void
btree_set_iteratevals(const struct btree_spec *spec, __u16 *leaves,
		      __u16 *num, __u16 *jump, __u16 *last_leaf,
		      struct gen_spec ***nextspec);



/***************************************************************************
 **      **********************************************************       **
 ***    ********                                                 ****    ***
 *************      B T R E E   A P I   E N D S   H E R E      *************
 ***    ****                                                 ********    ***
 **      ***********************************************************      **
 ***************************************************************************/



/* btree nodes always consume k * BLOCKSIZE bytes where k >= 1 */
#define BLOCKSIZE 64

/* btree type encoding the shape of the btree depending on the bit type */
#define BT0 0
#define BT1 1
#define BT2 2
#define BT3 3


/*
 * x_y_ROOT        - maximum number of elements in root
 * x_y_LEAF        - maximum number of elements in a leaf
 * x_y_NODE_BLOCKS - number of blocks a full internal node occupies
 * x_y_LEAF_BLOCKS - number of blocks a full leaf occupies
 *
 * all values depend on x (btree type) and y (bit type)
 */

#define BT1_U8_ROOT           56
#define BT1_U8_LEAF           12

#define BT1_U16_ROOT          28
#define BT1_U16_LEAF          10
#define BT2_U16_LEAF          32
#define BT3_U16_NODE          32
#define BT3_U16_NODE_BLOCKS    1
#define BT3_U16_LEAF          32

#define BT1_U32_LEAF          16
#define BT2_U32_NODE          16
#define BT2_U32_NODE_BLOCKS    1
#define BT2_U32_LEAF          16
#define BT3_U32_NODE          32
#define BT3_U32_NODE_BLOCKS    2
#define BT3_U32_LEAF          32

#ifdef BIT32_ARCH
#  define BT1_U8_LEAF_BLOCKS   1
#  define BT1_U16_LEAF_BLOCKS  1
#  define BT2_U16_LEAF_BLOCKS  3
#  define BT3_U16_LEAF_BLOCKS  3
#  define BT1_U32_LEAF_BLOCKS  2
#  define BT2_U32_LEAF_BLOCKS  2
#  define BT3_U32_LEAF_BLOCKS  4
#else
#  define BT1_U8_LEAF_BLOCKS   2
#  define BT1_U16_LEAF_BLOCKS  2
#  define BT2_U16_LEAF_BLOCKS  5
#  define BT3_U16_LEAF_BLOCKS  5
#  define BT1_U32_LEAF_BLOCKS  3
#  define BT2_U32_LEAF_BLOCKS  3
#  define BT3_U32_LEAF_BLOCKS  6
#endif


/*
 * x_y_MAX - maximum number of elements allowed for a btree of type x
 *           and bit type y
 */

#define BT0_U8_MAX  62

#define BT0_U16_MAX 41
#define BT1_U16_MAX 150
#define BT2_U16_MAX 2944

#define BT0_U32_MAX 41
#define BT1_U32_MAX 1232
#define BT2_U32_MAX 6656


/* number of bytes missing to let a node of bit type bittype with kpn keys
   occupy a multiple of BLOCKSIZE bytes */
#define OFFSET(kpn, bittype)                                                 \
((BLOCKSIZE - ((kpn) * ((1 << (bittype)) + PTRSIZE)) % BLOCKSIZE) % BLOCKSIZE)

/* number of blocks that root occupies;
   num is the number of elements in root */
#define BLOCKS_U16(num) (DIV(((num) + 2 + (PTRSIZE >> 1)), 32))
#define BLOCKS_U32(num) (DIV(((num) + 1 + (PTRSIZE >> 1)), 16))

/* DIV(num, div) == ceil(num / div) */
#define DIV(num, div)     \
((((num) % (div)) == 0) ? \
 ((num) / (div)) :        \
 (((num) / (div)) + 1))

/* pointer to the beginning of the btree-keys */
#define ROOT(spec) ((__u32 *) ((__u8 *) (spec) + sizeof(__u32) + \
		     ((spec)->bittype >> 1) * PTRSIZE))

/* pointer to the leftmost leaf in a btree of depth 1 */
#define LEAF1(spec) ((__u8 *) (spec) + (spec)->blocks * BLOCKSIZE)

/* pointer to the leftmost leaf in a btree of depth 2 */
#define LEAF2(spec, node_size, node_blocks, leaf_size)               \
((__u8 *) (spec) + BLOCKSIZE *                                       \
 ((spec)->blocks + (DIV((spec)->num, ((node_size) * (leaf_size)))) * \
  (node_blocks)))


static inline void
btree_set_iteratevals(const struct btree_spec *spec, __u16 *leaves,
		      __u16 *num, __u16 *jump, __u16 *last_leaf,
		      struct gen_spec ***nextspec)
{
	assert(spec != NULL);
	assert(IS_BTREE(spec));
	assert(leaves != NULL);
	assert(num != NULL);
	assert(jump != NULL);
	assert(last_leaf != NULL);
	assert(nextspec != NULL);

	switch (spec->btreetype) {
	    case BT0:
		    *leaves = 0;
		    *nextspec = (struct gen_spec **)
			    ((__u8 *) ROOT(spec) + 
			     (spec->num * (1 << spec->bittype)));
		    *num = 0;
		    *jump = 0;
		    *last_leaf = spec->num;
		    break;
		    
	    case BT1:
		    switch (spec->bittype) {
			case BIT_U32:
				*leaves = spec->num / BT1_U32_LEAF;
				*nextspec = (struct gen_spec **)
					((__u32 *) LEAF1(spec) + BT1_U32_LEAF);
				*num = BT1_U32_LEAF;
				*jump = BT1_U32_LEAF_BLOCKS * BLOCKSIZE;
				*last_leaf = ((spec->num % BT1_U32_LEAF == 0) ?
					      0 : spec->num % BT1_U32_LEAF);
				break;

			case BIT_U16:
				*leaves = spec->num / BT1_U16_LEAF;
				*nextspec = (struct gen_spec **)
					((__u16 *) LEAF1(spec) + BT1_U16_LEAF);
				*num = BT1_U16_LEAF;
				*jump = BT1_U16_LEAF_BLOCKS * BLOCKSIZE;
				*last_leaf = ((spec->num % BT1_U16_LEAF == 0) ?
					      0 : spec->num % BT1_U16_LEAF);
				break;

			default:
				assert(spec->bittype == BIT_U8);
				*leaves = spec->num / BT1_U8_LEAF;
				*nextspec = (struct gen_spec **)
					((__u8 *) LEAF1(spec) + BT1_U8_LEAF);
				*num = BT1_U8_LEAF;
				*jump = BT1_U8_LEAF_BLOCKS * BLOCKSIZE;
				*last_leaf = ((spec->num % BT1_U8_LEAF == 0) ?
					      0 : spec->num % BT1_U8_LEAF);
				break;
		    }
		    break;

	    case BT2: 
		    if (spec->bittype == BIT_U32) {
			    *leaves = spec->num / BT2_U32_LEAF;
			    *nextspec = (struct gen_spec **)
				    ((__u32 *) LEAF2(spec, BT2_U32_NODE,
						     BT2_U32_NODE_BLOCKS,
						     BT2_U32_LEAF)
				     + BT2_U32_LEAF);
			    *num = BT2_U32_LEAF;
			    *jump = BT2_U32_LEAF_BLOCKS * BLOCKSIZE;
			    *last_leaf = ((spec->num % BT2_U32_LEAF == 0) ?
					  0 : spec->num % BT2_U32_LEAF);
		    } else {
			    *leaves = spec->num / BT2_U16_LEAF;
			    *nextspec = (struct gen_spec **)
				    ((__u16 *) LEAF1(spec) + BT2_U16_LEAF);
			    *num = BT2_U16_LEAF;
			    *jump = BT2_U16_LEAF_BLOCKS * BLOCKSIZE;
			    *last_leaf = ((spec->num % BT2_U16_LEAF == 0) ?
					  0 : spec->num % BT2_U16_LEAF);
		    }
		    break;

	    default:
		    assert(spec->btreetype == BT3);
		    if (spec->bittype == BIT_U32) {
			    *leaves = spec->num / BT3_U32_LEAF;
			    *nextspec = (struct gen_spec **)
				    ((__u32 *) LEAF2(spec, BT3_U32_NODE,
						     BT3_U32_NODE_BLOCKS,
						     BT3_U32_LEAF)
				     + BT3_U32_LEAF);
			    *num = BT3_U32_LEAF;
			    *jump = BT3_U32_LEAF_BLOCKS * BLOCKSIZE;
			    *last_leaf = ((spec->num % BT3_U32_LEAF == 0) ?
					  0 : spec->num % BT3_U32_LEAF);
		    } else {
			    *leaves = spec->num / BT3_U16_LEAF;
			    *nextspec = (struct gen_spec **)
				    ((__u16 *) LEAF2(spec, BT3_U16_NODE,
						     BT3_U16_NODE_BLOCKS,
						     BT3_U16_LEAF)
				     + BT3_U16_LEAF);
			    *num = BT3_U16_LEAF;
			    *jump = BT3_U16_LEAF_BLOCKS * BLOCKSIZE;
			    *last_leaf = ((spec->num % BT3_U16_LEAF == 0) ?
					  0 : spec->num % BT3_U16_LEAF);
		    }
		    break;
	}
}

static inline __u32 *
findkey_32(__u32 key, const __u32 *start)
{
	__u32 *p = (__u32 *) start;
	while (*p < key) {
		p++;
	}
	return p;
}

static inline __u16 *
findkey_16(__u16 key, const __u16 *start)
{
	__u16 *p = (__u16 *) start;
	while (*p < key) {
		p++;
	}
	return p;
}

static inline __u8 *
findkey_8(__u8 key, const __u8 *start)
{
	__u8 *p = (__u8 *) start;
	while (*p < key) {
		p++;
	}
	return p;
}

static inline struct gen_spec *
next_spec_32(__u32 key, const __u32 *start, __u32 num)
{
	return (struct gen_spec *)
		*(void **) (start + num + (PTRSIZE >> 2) *
			    (findkey_32(key, start) - start));
}

static inline struct gen_spec *
next_spec_16(__u32 key, const __u16 *start, __u32 num)
{
	return (struct gen_spec *)
		*(void **) (start + num +(PTRSIZE >> 1) * 
			    (findkey_16(key, start) - start));
}

static inline struct gen_spec *
next_spec_8(__u32 key, const __u8 *start, __u32 num)
{
	return (struct gen_spec *)
		*(void **) (start + num + PTRSIZE * 
			    (findkey_8(key, start) - start));
}

static inline __u32 *
next_node1_32(__u32 key, const struct btree_spec *spec, const __u32 *start,
	      __u32 blocks)
{
	return (__u32 *)
		((__u8 *) spec + BLOCKSIZE *
		 (spec->blocks + (findkey_32(key, start) - start) * blocks));
}

static inline __u16 *
next_node1_16(__u32 key, const struct btree_spec *spec, const __u16 *start,
	      __u32 blocks)
{
	return (__u16 *)
		((__u8 *) spec + BLOCKSIZE * 
		 (spec->blocks + (findkey_16(key, start) - start) * blocks));
}

static inline __u8 *
next_node1_8(__u32 key, const struct btree_spec *spec, const __u8 *start,
	     __u32 blocks)
{
	return (__u8 *)
		((__u8 *) spec + BLOCKSIZE * 
		 (spec->blocks + (findkey_8(key, start) - start) * blocks));
}

static inline __u32 *
next_node2_32(__u32 key, const __u8 *leaf1_start, __u32 root_num,
	      const __u32 *node_start, __u32 node_blocks, __u32 leaf_blocks)
{
	return (__u32 *)
		(leaf1_start + BLOCKSIZE *
		 (root_num * node_blocks +
		  (findkey_32(key, node_start) -
		   (__u32 *) leaf1_start) * leaf_blocks));
}

static inline __u16 *
next_node2_16(__u32 key, const __u8 *leaf1_start, __u32 root_num,
	      const __u16 *node_start, __u32 node_blocks, __u32 leaf_blocks)
{
	return (__u16 *)
		(leaf1_start + BLOCKSIZE *
		 (root_num * node_blocks +
		  (findkey_16(key, node_start) -
		   (__u16 *) leaf1_start) * leaf_blocks));
}

static inline struct gen_spec *
btree_match_32_next(const struct btree_spec *spec, const __u32 key)
{
	__u32 *start = (__u32 *) ((__u8 *) spec + sizeof(__u32) + PTRSIZE);

	switch (spec->btreetype) {
		
	    case BT0:
		    return next_spec_32(key, start, spec->num);
		    
	    case BT1:
		    return next_spec_32(
			    key, next_node1_32(key, spec, start,
					       BT1_U32_LEAF_BLOCKS),
			    BT1_U32_LEAF);
	
	    case BT2: {
		    __u32 root_num = DIV(spec->num, (BT2_U32_NODE *
						     BT2_U32_LEAF));
		    return next_spec_32(
			    key, next_node2_32(
				    key, LEAF1(spec), root_num,
				    next_node1_32(key, spec, start,
						  BT2_U32_NODE_BLOCKS),
				    BT2_U32_NODE_BLOCKS, BT2_U32_LEAF_BLOCKS),
			    BT2_U32_LEAF);
	    }
	
	    default: {
		    __u32 root_num = DIV(spec->num, (BT3_U32_NODE *
						     BT3_U32_LEAF));
		    assert(spec->btreetype == BT3);
		    return next_spec_32(
			    key, next_node2_32(
				    key, LEAF1(spec), root_num, 
				    next_node1_32(key, spec, start,
						  BT3_U32_NODE_BLOCKS),
				    BT3_U32_NODE_BLOCKS, BT3_U32_LEAF_BLOCKS),
			    BT3_U32_LEAF);
	    }
	}
}

#endif
