/*
 * Linux Event Logging for the Enterprise
 * Copyright (c) International Business Machines Corp., 2001
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Please send e-mail to lkessler@users.sourceforge.net if you have
 *  questions or comments.
 *
 *  Project Website:  http://evlog.sourceforge.net/
 *
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <linux/smp.h>
#include <linux/ptrace.h>
#include <linux/string.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <asm/semaphore.h>

#ifdef __i386__
#include <asm/fixmap.h>
#include <asm/mpspec.h>
#include <asm/io_apic.h>
#include <asm/apic.h>
#endif

#include <asm/bitops.h>
#include <asm/smp.h>

#include <linux/evl_log.h>

#define EVL_BUF_SIZE (CONFIG_EVLOG_BUFSIZE * 1024)   /* EVL buffer size */
#define EVL_BUF_FREESPACE (64*1024) /* max free space reqd after buff */
				    /* overrun to start writing events again. */
/*
 * This data structure describes the circular buffer that is written into
 * by evl_kwrite_buf() and drained by evl_kbufread().
 *
 * bf_buf, bf_len, and bf_end are the start, length, and end of the buffer,
 * and in the current implementation these remain constant.
 *
 * bf_tail advances as event records are logged to the buffer, and bf_head
 * advances as records are drained from the buffer.  bf_curr is used
 * internally by certain functions.  bf_dropped maintains a count of
 * records that have been dropped due to buffer overrun.
 */
struct cbuf {
	unsigned char	*bf_buf;	/* base buffer address */
	unsigned int	bf_len;		/* buffer length */
	unsigned int	bf_dropped;	/* (internal) dropped count */
	unsigned char	*bf_head;	/* head-pointer for circ. buf */
	unsigned char	*bf_tail;	/* tail-pointer for circ. buf */
	unsigned char	*bf_curr;	/* (internal) current write ptr */
	unsigned char	*bf_end;	/* end buffer address */
};
 
static unsigned char evl_buffer[EVL_BUF_SIZE + sizeof(long)];

static struct cbuf evl_ebuf = {
	evl_buffer,
	EVL_BUF_SIZE,
	0,
	evl_buffer,
	evl_buffer, 
	evl_buffer,
	evl_buffer + EVL_BUF_SIZE
};

#define min(a,b) (((a)<(b))?(a):(b))

/*
 * This is for the serialisation of reads from the kernel buffer.
 */
static DECLARE_MUTEX(evl_read_sem);
DECLARE_WAIT_QUEUE_HEAD(readq);
spinlock_t ebuf_lock = SPIN_LOCK_UNLOCKED;

extern void mk_rec_header(struct posix_log_entry *rechdr,
	posix_log_facility_t facility, int event_type,
	posix_log_severity_t severity, size_t recsize,
	uint recflags, int format);

extern char *copy_attr_data(va_list, char *, char *, char *, int *, int *);

extern void evl_console_print(posix_log_facility_t facility, int event_type,
	posix_log_severity_t severity, int format, uint rec_len, 
	const char * databuf);
extern int console_loglevel;
extern struct console *console_drivers;


/*
 * FUNCTION     : cbufwrap
 * Caller wants to write a chunk of size len bytes into the buffer starting at
 * bf_curr.  If bf_curr is near the end of the buffer, then we may have to
 * split the chunk so that some of it appears at the end of the buffer and
 * the rest appears at the beginning.  dont_split_hdr=1 means the chunk is
 * an entire record, and we're not allowed to split the record's header --
 * we may have to leave unused space at the end of the buffer and start the
 * record at the beginning.
 *
 * Return the location where the chunk should start, or NULL if there's
 * no room for it (i.e., it would overrun the buffer's head pointer).
 *
 * ARGUMENTS    : bf_curr - where caller wants to put record
 *              : len - total record size
 *              : dont_split_hdr - 0 if it's OK to split the header, else 1
 *
 * RETURN       : where the record should be put, or NULL if no room
 */
unsigned char *
cbufwrap(unsigned char *bf_curr, size_t len, int dont_split_hdr)
{
	unsigned char *head, *end;
	unsigned char *wrapbuf = bf_curr;

	end = bf_curr + len;
	head = evl_ebuf.bf_head;

	if (bf_curr < head && end >= head) {
		/* Insufficient free space in buffer; drop record. */
		return NULL;
	}
	if (end > evl_ebuf.bf_end) {
		/* end would be of end of buffer */
		if (dont_split_hdr
		    && bf_curr + REC_HDR_SIZE > evl_ebuf.bf_end) {
			/* Start record at start of buffer. */
			wrapbuf = evl_ebuf.bf_buf;
		} else {
			/* Split record */
			len -= evl_ebuf.bf_end - bf_curr;
		}

		end = evl_ebuf.bf_buf + len;
		if (end >= head) {
			/* Insufficient free space in buffer; drop record. */
			return NULL;
		}
	}
	return wrapbuf;
}

/*
 * FUNCTION     : cbufwrite
 * Copy the len bytes starting at s into the buffer starting at bufp,
 * splitting the data if we run off the end of the buffer.
 *
 * ARGUMENTS    : The pointer in cbuf to write at,
 *                the buffer to write,
 *                the number of bytes to write
 * RETURN       : The pointer in cbuf at the end of the copied data
 * NOTE
 *      bf_head is owned by the drainer. bf_curr is allowed to be equal to
 *      bf_head only when buffer is empty indicated by drainer. It is never
 *      set to be equal by writer.
 */
unsigned char *
cbufwrite(unsigned char *bufp, unsigned char *s, size_t len)
{
	register unsigned char *e;
	register size_t n, lw;

	e = bufp + len;
	n = len;

	if (e > evl_ebuf.bf_end) {
		lw = evl_ebuf.bf_end - bufp;
		n -= lw;
		e = evl_ebuf.bf_buf + n;
		memcpy(bufp, s, lw);
		memcpy(evl_ebuf.bf_buf, s + lw, n);
		return e;
	} else {
		memcpy(bufp, s, n);
		return (e == evl_ebuf.bf_end) ? evl_ebuf.bf_buf : e;
	}
}

/*
 * FUNCTION     : cbufptr
 * Return cbuf_ptr + offset, taking possible wraparound into account.
 *
 * ARGUMENTS    : starting pointer in cbuf,
 *                offset in bytes from pointer
 * RETURN       : The pointer in circular buf that is offset bytes from
 *                starting pointer.
 */
unsigned char *
cbufptr(unsigned char *cbuf_ptr, size_t offset)
{
	register unsigned char *e;
	register size_t lw;

	e = cbuf_ptr + offset;
	if (e >= evl_ebuf.bf_end) {
		lw = evl_ebuf.bf_end - cbuf_ptr;
		e = evl_ebuf.bf_buf + (offset - lw);
	}
	return e;
}

/*
 * FUNCTION     : copy_data_to_cbuf
 * Copies the indicated record into the buffer at evl_ebuf.bf_curr.
 * Assumes we've previously verified that the header won't be split.
 * Updates evl_ebuf.bf_curr to point past the record just copied in.
 *
 * ARGUMENTS    : Record header
 *                pointer to variable data
 */
int
copy_data_to_cbuf(struct posix_log_entry *rhdr, unsigned char *vbuf)
{
	memcpy(evl_ebuf.bf_curr, rhdr, REC_HDR_SIZE);
	evl_ebuf.bf_curr += REC_HDR_SIZE;

	if (rhdr->log_size > 0) {
		evl_ebuf.bf_curr = 
			cbufwrite(evl_ebuf.bf_curr, vbuf, rhdr->log_size);
	}
	
	return 0;
}

/*
 * EVL circular buffer had been full and caused later messages to be
 * dropped. Now the buffer has space.  (Still it is better to identify the
 * cause so it doesn't repeat. The buffer gets full if the rate of incoming
 * messages is much higher than can be drained. This could happen if messages
 * are repeated at an excessively high rate or the daemon in user-space is
 * not running.)
 *
 * Now that we can log events again, log one giving the number of events
 * dropped.
 */
static int
log_dropped_recs_event(void)
{
	unsigned char sbuf[255];
	struct posix_log_entry drechdr;
	size_t vbuflen;
	unsigned char *oldcur = evl_ebuf.bf_curr;

	snprintf(sbuf, sizeof(sbuf), "%d event records dropped due to EVL buffer overflow.", 
		evl_ebuf.bf_dropped);
	evl_ebuf.bf_dropped = 0;
	vbuflen = strlen(sbuf) + 1;
	mk_rec_header(&drechdr, LOG_KERN, EVL_BUFFER_OVERRUN, LOG_INFO,
		vbuflen, 0, POSIX_LOG_STRING);
	evl_ebuf.bf_curr = cbufwrap(evl_ebuf.bf_curr, REC_HDR_SIZE+vbuflen, 1);
	if (evl_ebuf.bf_curr != (unsigned char *)NULL) {
		copy_data_to_cbuf(&drechdr, sbuf);
		return 0;
	} else {
		/*
		 * This shouldn't happen, since EVL_BUF_FREESPACE is much
		 * bigger than the event we're logging here.
		 */
		evl_ebuf.bf_curr = oldcur;
		return -1;
	}
}

/*
 * FUNCTION     : evl_check_buf
 * ARGUMENTS    : NONE
 * RETURN       : -1 for failure, ie. insufficient buffer space
 *                 0 for success
 * If buffer free space is greater than the applicable water-mark,
 * returns 0.  If not, return -1.  Sets evl_buf.bf_curr to the location
 * where the next record should go.
 *
 * Once the high water mark is hit and failure is returned (discarded
 * messages) it sets a substantial low water mark before permitting
 * messages to be buffered again.  It counts the number of discards
 * in the meantime and reports them when restarted.  If the water
 * marks were equivalent, then there could be a thrashing of stops
 * and starts, making the discarded message reporting annoying.
 *
 */
int
evl_check_buf(void)
{
	unsigned char *head, *tail;
	size_t water_mark, avail;

	head    = evl_ebuf.bf_head;
	tail	= evl_ebuf.bf_tail;
	avail   = (head <= tail) ?
	      (evl_ebuf.bf_len - (tail - head)) :
	      (head - tail);

	if (evl_ebuf.bf_dropped != 0) {
		/*
		 * Still recovering from buffer overflow.
		 * Apply the low water mark.
		 */
		water_mark = min(EVL_BUF_FREESPACE, evl_ebuf.bf_len / 2);
	} else {
		water_mark = REC_HDR_SIZE;
	}

	if (avail < water_mark) {
		return -1;
	}

	/* There's enough free buffer space.  Return success. */
	evl_ebuf.bf_curr = tail;
	if (evl_ebuf.bf_dropped != 0) {
		return log_dropped_recs_event();
	}
	return 0;
}

/*
 * FUNCTION     : evl_getnext_rec
 * ARGUMENTS    : rec is a pointer to the log event record.
 *                The next record pointer mustn't be beyond tail.
 * RETURN       : This function returns a pointer to the place to start the
 *                next record in the circular buffer.  As a special case,
 *                if rec is NULL, then the location of the first record at
 *                or after bf_head is returned; else the record following
 *                rec is returned.
 */
unsigned char *
evl_getnext_rec(unsigned char *rec, size_t recsize, unsigned char *tail)
{
	if (rec == NULL) {
		rec = evl_ebuf.bf_head;
	} else {
		rec = cbufptr(rec, recsize);
	}

	if (rec == tail) {
		return rec;
	}

	/* Check for wrap. */
	if ((rec + REC_HDR_SIZE) > evl_ebuf.bf_end) {
		rec = evl_ebuf.bf_buf;
	}
	return rec;
}

#if 0
/*
 * FUNCTION     : evl_uwrite_buf - Used to write user space messages.
 * ARGUMENTS    : buf is a pointer to buffer that needs to be written into the
 *                        EVL kernel buffer.
 *                      : buflen is length of the buffer to be written.
 * RETURN       : 0 if writing to buffer is successful, else -ve value.
 * USAGE        : This function acquires the circular buffer lock and copies the
 *                event record and data to buffer.
 * 
 * NOTE:                This function is left over from earlier event logging releases
 *                      when user events were passed into the kernel buffer (instead of 
 *                      being passed directly to the evlogd daemon via socket in the
 *                      later release).  For some environments, using this function 
 *                      could be the preferred method (and thus it was left in for now).
 */

int
evl_uwrite_buf(char *buf, uint buflen)
{
	int error = 0, recsize;
	unsigned char *old_cur = NULL;
	evl_buf_rec_t *rec;
	char *kbuf;
	char *oldtail = evl_ebuf.bf_tail; /* Used to wake the read call if it sleeps */
	long iflags;

	if (buflen < REC_HDR_SIZE) {
		return -EINVAL;
	}
	kbuf = kmalloc(buflen, GFP_KERNEL);
	if (kbuf == (char *)NULL) {
		return -ENOMEM;
	} 
	error = copy_from_user(kbuf, buf, buflen);
	if (error) {
		kfree(kbuf); 
		return -EINVAL;
	} 
	rec = (evl_buf_rec_t *)kbuf;
	if (buflen != (REC_HDR_SIZE + rec->rechdr.log_size)) {
		kfree(kbuf); 
		return -EINVAL;
	}
	/*
	 * An event came from user space but has pretended to be
	 * from kernel space. We just return with EPERM
	 */
	if (rec->rechdr.log_flags & EVL_KERNEL_EVENT) {
		kfree(kbuf);
		return -EPERM;
	}
	rec->rechdr.log_processor = smp_processor_id();

	spin_lock_irqsave(&ebuf_lock, iflags);
	if (evl_check_buf() < 0) {
		evl_ebuf.bf_dropped++;
		spin_unlock_irqrestore(&ebuf_lock, iflags);
		kfree(kbuf);
		return -ENOSPC;
	}

   	/* store off buf_curr to restore if necessary */
   	old_cur = evl_ebuf.bf_curr;

	/*
	 * The buffer must have enough space to store the core structure
	 */
	recsize = REC_HDR_SIZE + rec->rechdr.log_size;

	/* 
	 *If insufficient space in buffer after wrap around, drop record
	 */
	if ((evl_ebuf.bf_curr = cbufwrap(evl_ebuf.bf_curr, recsize, 1)) 
		!= (char *)NULL) {
		copy_data_to_cbuf(&(rec->rechdr),(char *)(kbuf + REC_HDR_SIZE));
	}
	if (evl_ebuf.bf_curr == (char *)NULL){
		error = -ENOSPC;
		evl_ebuf.bf_dropped++;
		evl_ebuf.bf_curr = old_cur;
	} else {
		evl_ebuf.bf_tail = evl_ebuf.bf_curr;
		if ((evl_ebuf.bf_head == oldtail) &&
			(evl_ebuf.bf_head  != evl_ebuf.bf_tail)) {
			wake_up_interruptible(&readq);
		}
	}

	spin_unlock_irqrestore(&ebuf_lock, iflags);
	kfree(kbuf); 
	return error;
}
#endif


/*
 * FUNCTION     : evl_kbufread - Used to read event records from the EVL
 *                circular buffer.
 * ARGUMENTS    : retbuf is a pointer to the buffer to be filled with the
 *                event records.
 *              : bufsize is length of the buffer allocated by the user.
 * RETURN       : Number of bytes copied if read is successful, else -ve value.
 *                event record and data to buffer.
 */

int
evl_kbufread(unsigned char *retbuf, size_t bufsize)
{
	unsigned char *rec;
	register size_t rec_size;
	int error = 0;
	int retbuflen = 0;
	unsigned char *tail, *buf = retbuf;

	/*
	 * the read request size must be at least rec_hdr_t size
	 */
	if (bufsize < REC_HDR_SIZE) {
		return -EINVAL;
	}
	/* 
	 * Serialize all reads, just in case someone got sneaky 
	 */
	error = down_interruptible(&evl_read_sem);
	if (error == -EINTR) {
		return -EINTR;
	}
	/*
	 * Go to sleep if the buffer is empty.
	 */
	error = wait_event_interruptible(readq, 
		(evl_ebuf.bf_head != evl_ebuf.bf_tail));
	if (error) {
		up(&evl_read_sem);
		return error;
	}
	/*
	 * Assemble message(s) into the user buffer, as many as will
	 * fit.  On running out of space in the buffer, try to copy
	 * the header for the overflowing message.  This means that
	 * there will always be at least a header returned.  The caller
	 * must compare the numbers of bytes returned (remaining) with
	 * the length of the message to see if the entire message is
	 * present.  A subsequent read will get the entire message,
	 * including the header (again).
	 */
	tail = evl_ebuf.bf_tail;
	rec = evl_getnext_rec(NULL, 0, tail);	/* typically evl_ebuf.bf_head */
	if (rec == NULL) { 
		/* Should not happen. Buffer must have atleast one record. */
		error = -EFAULT;
		goto out;
	}

	do {
#if defined(__ia64__)
		evl_buf_rec_t record;
		memcpy(&record, rec, sizeof(evl_buf_rec_t));
		rec_size = REC_HDR_SIZE + record.rechdr.log_size;
#else
		evl_buf_rec_t *p_rec;
		p_rec = (evl_buf_rec_t *) rec;
		rec_size = REC_HDR_SIZE + p_rec->rechdr.log_size;
#endif

		if (bufsize < REC_HDR_SIZE) {
			/* user buffer is smaller than header */
			break;
		}
		if (bufsize < rec_size) {
			/* 
			 * Copyout only the header 'cause user buffer can't
			 * hold full record.
			 */
			error = copy_to_user(buf, rec, REC_HDR_SIZE);
			if (error) {
				error = -EFAULT;
				break;
			}
			bufsize -= REC_HDR_SIZE;
			retbuflen += REC_HDR_SIZE;
			break;
		}
		if ((rec + rec_size) > evl_ebuf.bf_end) {
			size_t lw = evl_ebuf.bf_end - rec;
			error = copy_to_user(buf, rec, lw);
			if (!error) {
				error = copy_to_user(buf + lw, evl_ebuf.bf_buf, 
					rec_size - lw); 
			}
		} else {
			error = copy_to_user(buf, rec, rec_size);
		}
		if (error) {
			error = -EFAULT;
			break;
		}
		rec = evl_getnext_rec(rec, rec_size, tail);
		buf += rec_size;
		bufsize -= rec_size;
		retbuflen += rec_size;
	} while (rec != tail);

	if (error == 0) {
		evl_ebuf.bf_head = rec;
		error = retbuflen;
	}

out:
	up(&evl_read_sem);
	return(error);
}

/*
 * FUNCTION	: kwrite_args
 * Called by evl_kwrite_buf() to write to the buffer an event that was
 * created by evl_writek() (or some other function with args that must
 * be interpreted by copy_attr_data()).
 *
 * RETURN	: 0 on success, -ENOSPC (or an error code from copy_attr_data())
 *		on failure
 *		On success evl_ebuf.bf_curr is updated to point just past
 *		the event we just wrote to the buffer, and *pvardata points
 *		to the variable-data portion of the record.
 */
static int
kwrite_args(struct posix_log_entry *rec_hdr, va_list args,
	unsigned char **pvardata)
{
	unsigned char *hdr, *vardata, *rec_end;
	int error = 0;
	uint var_rec_len = 0;

	hdr = cbufwrap(evl_ebuf.bf_curr, REC_HDR_SIZE, 1);
	if (hdr == (unsigned char*) NULL) {
		return -ENOSPC;
	}
	vardata = hdr + REC_HDR_SIZE;
	rec_end = copy_attr_data(args, evl_ebuf.bf_buf, evl_ebuf.bf_end,
		vardata, &var_rec_len, &error);
	if (rec_end == (unsigned char*) NULL) {
		if (error == 0) {
			error = -ENOSPC;
		}
		return error;
	}
	rec_hdr->log_size = var_rec_len;
	if (var_rec_len == 0) {
		rec_hdr->log_format = POSIX_LOG_NODATA;
	} else if (var_rec_len > POSIX_LOG_ENTRY_MAXLEN) {
		rec_hdr->log_size = POSIX_LOG_ENTRY_MAXLEN;
		rec_hdr->log_flags |= POSIX_LOG_TRUNCATE;
	}
	memcpy(hdr, rec_hdr, REC_HDR_SIZE);
	*pvardata = vardata;
	evl_ebuf.bf_curr = rec_end;
	return 0;
}

/*
 * FUNCTION	: kwrite_buf
 * Called by evl_kwrite_buf() to write to the buffer the event record
 * consisting of rec_hdr and vardata.
 *
 * RETURN	: 0 on success, -ENOSPC on failure
 *		On success evl_ebuf.bf_curr is updated to point just past
 *		the event we just wrote to the buffer.
 */
static int
kwrite_buf(struct posix_log_entry *rec_hdr, unsigned char *vardata)
{
	size_t recsize;
	unsigned char *rec;
	if (rec_hdr->log_size > POSIX_LOG_ENTRY_MAXLEN) {
		rec_hdr->log_size = POSIX_LOG_ENTRY_MAXLEN;
		rec_hdr->log_flags |= POSIX_LOG_TRUNCATE;
	}
	recsize = REC_HDR_SIZE + rec_hdr->log_size;
	rec = cbufwrap(evl_ebuf.bf_curr, recsize, 1);
	if (rec == (unsigned char *)NULL) {
		return -ENOSPC;
	}

	evl_ebuf.bf_curr = rec;
	copy_data_to_cbuf(rec_hdr, vardata);

	/*
	 * If the variable data is a truncated string, make sure it
	 * ends with a null character.
	 */
	if ((rec_hdr->log_flags & POSIX_LOG_TRUNCATE) &&
	    rec_hdr->log_format == POSIX_LOG_STRING) {
		if (evl_ebuf.bf_curr == evl_ebuf.bf_buf) {
			*(evl_ebuf.bf_end - 1) = '\0';
		} else {
			*(evl_ebuf.bf_curr - 1) = '\0';
		}
	}
	return 0;
}

/*
 * FUNCTION:	log_event_to_console
 * Called by evl_kwrite_buf() to log to the console the event record
 * consisting of rec_hdr and vardata.
 */
static void
log_event_to_console(const struct posix_log_entry *rec_hdr,
	const unsigned char *vardata)
{
	unsigned char consolebuf[16];

	if (rec_hdr->log_format == POSIX_LOG_BINARY
	    && (evl_ebuf.bf_buf <= vardata && vardata < evl_ebuf.bf_end)) {
		/*
		 * vardata points into the circular buffer.  This means
		 * that vardata was pieced together by copy_attr_data(),
		 * and the circular buffer is the closest thing we have
		 * to a contiguous copy.  Since the format is BINARY, we
		 * "know" that evl_console_print() cares only about the
		 * first 16 bytes.
		 */
		int nimportant = min(16, rec_hdr->log_size);
		if (vardata + nimportant > evl_ebuf.bf_end) {
			/*
			 * The section of interest wraps back to the start
			 * of the buffer.  Copy it into consolebuf to make
			 * it contiguous.
			 */
			int first_part =  evl_ebuf.bf_end - vardata;
			memcpy(consolebuf, vardata, first_part);
			memcpy(consolebuf+first_part, evl_ebuf.bf_buf,
				nimportant - first_part);
			vardata = consolebuf;
		}
	}
	evl_console_print(rec_hdr->log_facility, rec_hdr->log_event_type,
		rec_hdr->log_severity, rec_hdr->log_format, rec_hdr->log_size,
		vardata);
}

/*
 * FUNCTION     : evl_kwrite_buf - Used to write kernel level messages.
 * RETURN       : 0 if writing to buffer is successful, else -errno.
 */

int
evl_kwrite_buf(posix_log_facility_t    fac,
		int                    ev_type,
		posix_log_severity_t   sev,
		int                    format,
		unsigned char          *recbuf,
		uint                   var_rec_len,
		uint                   flags,
		va_list                args)    /* Used only for evl_writek */
{
	uint recflags = flags;
	struct posix_log_entry rec_hdr;
	int error = 0;
	unsigned char *oldtail = evl_ebuf.bf_tail;
		/* Used to wake the read call if it sleeps */ 
	long iflags;	/* for spin_lock_irqsave() */

	if (sev > LOG_DEBUG) {
		return -EINVAL;
	}
	
	recflags |= EVL_KERNEL_EVENT;   /* kernel mesagge */    
	if (in_interrupt()) {
		recflags |= EVL_INTERRUPT;
	}
	mk_rec_header(&rec_hdr, fac, ev_type, sev, var_rec_len, recflags,
		format);
	
	spin_lock_irqsave(&ebuf_lock, iflags);
	if (evl_check_buf() < 0) {
		evl_ebuf.bf_dropped++;
		spin_unlock_irqrestore(&ebuf_lock, iflags);
		return -ENOSPC;
	}

	if (args) {
		/*
		 * Copy the header and format the args into the circular
		 * buffer, and make recbuf point to the variable data.
		 */
		error = kwrite_args(&rec_hdr, args, &recbuf);
		if (error == 0
		    && (rec_hdr.log_flags & EVL_PRINTK) == 0
		    && sev < console_loglevel
		    && console_drivers) {
			log_event_to_console(&rec_hdr, recbuf);
		}
	} else {
		error = kwrite_buf(&rec_hdr, recbuf);
		/* Caller will call evl_console_print() as needed. */
	}

	if (error == 0) {
		evl_ebuf.bf_tail = evl_ebuf.bf_curr;
		if ((evl_ebuf.bf_head == oldtail) &&
		    (evl_ebuf.bf_head != evl_ebuf.bf_tail)) {
			wake_up_interruptible(&readq);
		}
	} else if (error == -ENOSPC) {        
		evl_ebuf.bf_dropped++;
	}
	spin_unlock_irqrestore(&ebuf_lock, iflags);
	return error;
}
