#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/file.h>
#include <linux/smp_lock.h>

#include "shfs.h"
#include "shfs_proc.h"
#include "shfs_proto.h"

MODULE_AUTHOR("Zemljanka core team");
MODULE_DESCRIPTION("Shell File System");

extern struct inode_operations shfs_dir_inode_operations;
extern struct file_operations shfs_dir_operations;
extern struct inode_operations shfs_file_inode_operations;
extern struct file_operations shfs_file_operations;
extern struct address_space_operations shfs_file_aops;
extern struct inode_operations shfs_symlink_inode_operations;

static void shfs_delete_inode(struct inode*);
static void shfs_put_super(struct super_block*);
static int  shfs_statfs(struct super_block*, struct statfs*);
static void shfs_set_inode_attr(struct inode*, struct shfs_fattr*);

kmem_cache_t	*dir_head_cache = NULL;
kmem_cache_t	*dir_entry_cache = NULL;
kmem_cache_t	*dir_name_cache = NULL;
kmem_cache_t	*file_cache = NULL;

static struct super_operations shfs_sops = {
	put_inode:	force_delete,
	delete_inode:	shfs_delete_inode,
	put_super:	shfs_put_super,
	statfs:		shfs_statfs,
};

#ifdef SHFS_ALLOC_DEBUG
	unsigned long alloc;
#endif

struct inode*
shfs_iget(struct super_block *sb, struct shfs_fattr *fattr)
{
	struct inode *res;

	DEBUG("\n");

	res = new_inode(sb);
	if(!res)
		return NULL;
	res->i_ino = fattr->f_ino;
	shfs_set_inode_attr(res, fattr);

	if (S_ISDIR(res->i_mode)) {
		DEBUG(" It's a directory\n");
		res->i_op = &shfs_dir_inode_operations;
		res->i_fop = &shfs_dir_operations;
	} else if (S_ISLNK(res->i_mode)) { 
		DEBUG(" It's a link\n");
		res->i_op = &shfs_symlink_inode_operations;
	} else if (S_ISREG(res->i_mode) || S_ISBLK(res->i_mode) || S_ISCHR(res->i_mode)) {
		DEBUG(" It's a file/block/char\n");
		res->i_op = &shfs_file_inode_operations;
		res->i_fop = &shfs_file_operations;
		res->i_data.a_ops = &shfs_file_aops;
	}
	
	insert_inode_hash(res);
	return res;
}

static void 
shfs_set_inode_attr(struct inode *inode, struct shfs_fattr *fattr)
{
	inode->i_mode 	= fattr->f_mode;
	inode->i_nlink	= fattr->f_nlink;
	inode->i_uid	= fattr->f_uid;
	inode->i_gid	= fattr->f_gid;
	inode->i_rdev	= fattr->f_rdev;
	inode->i_ctime	= fattr->f_ctime;
	inode->i_atime	= fattr->f_atime;
	inode->i_mtime	= fattr->f_mtime;
	inode->i_blksize= fattr->f_blksize;
	inode->i_blocks	= fattr->f_blocks;
	inode->i_size	= fattr->f_size;
}

static void 
shfs_delete_inode(struct inode *inode)
{
	DEBUG("\n");
	shfs_dcache_clear_inode(inode);
	clear_inode(inode);
}

int
shfs_revalidate_inode(struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	struct super_block *sb;
	struct shfs_sb_info *info;
	int valid;

        DEBUG("%s\n", dentry->d_name.name);
	lock_kernel();
	if (!inode) {
		valid = 1;
		goto out;
	}
	if (is_bad_inode(inode)) {
		valid = 0;
		goto out;
	}
	sb = inode->i_sb;
	if (sb->s_magic != SHFS_SUPER_MAGIC) {
		valid = 1;
		goto out;
	}
	info = (struct shfs_sb_info*)sb->u.generic_sbp;
	if (!shfs_lock(info)) {
		valid = 0;
		goto out;
	}
	valid = shfs_dcache_revalidate_entry(dentry);
	shfs_unlock(info);
out:
	unlock_kernel();
	DEBUG("%d\n", valid);
	shfs_remove_sigpipe(valid);
	if (valid < 0)
		valid = 0;
	return !valid;
}

static void
shfs_put_super(struct super_block *sb)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)sb->u.generic_sbp;
	int res;

	res = shfs_finish(info);
	shfs_remove_sigpipe(res);
	fput(info->mnt.pin);
	fput(info->mnt.pout);
	if (!shfs_lock(info)) {
		VERBOSE("Cannot free caches! Memory leak will occur!\n");
		return;
	}
	kfree(info->mnt.printf_buffer);
	kfree(info->mnt.readln_buffer);
	kfree(info);
	VERBOSE("Super block discarded!\n");
}

static int
shfs_statfs(struct super_block *sb, struct statfs *attr)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)sb->u.generic_sbp;
	char buf[SHFS_LINE_MAX];
	char *s, *p;
	int res;

	DEBUG("\n");

	attr->f_type = SHFS_SUPER_MAGIC;
	attr->f_bsize = 1024;
	attr->f_blocks = 0;
	attr->f_bfree = 0;
	attr->f_bavail = 0;
	attr->f_files = 1;
	attr->f_bavail = 1;
	attr->f_namelen = SHFS_PATH_MAX;

	if (!shfs_lock(info))
		return -EINTR;
	if ((res = do_statfs(info)) < 0) {
		VERBOSE("Couldn't send statfs command!\n");
		shfs_unlock(info);
		return res;
	}
	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);

	s = buf;
	if ((p = strsep(&s, " ")))
		attr->f_blocks = simple_strtoul(p, NULL, 10);
	if ((p = strsep(&s, " ")))
		attr->f_bfree = attr->f_blocks - simple_strtoul(p, NULL, 10);
	if ((p = strsep(&s, " ")))
		attr->f_bavail = simple_strtoul(p, NULL, 10);

	shfs_pipe_readln(info, buf, SHFS_LINE_MAX);
	shfs_unlock(info);
	/* ignore return value here */
	
	return 0;
}

struct super_block*
shfs_read_super(struct super_block *sb, void *opts, int silent)
{
	struct shfs_sb_info *info;
	struct shfs_fattr root;
	struct inode *root_inode;
	
	info = (struct shfs_sb_info*)kmalloc(sizeof(struct shfs_sb_info), GFP_KERNEL);
	if (!info) {
		VERBOSE("Not enough kmem to allocate info!!\n");
		goto out;
	}

	memset(info, 0, sizeof(struct shfs_sb_info));

	sb->u.generic_sbp = info;
	sb->s_blocksize = 4096;
	sb->s_blocksize_bits = 12;
	sb->s_magic = SHFS_SUPER_MAGIC;
	sb->s_op = &shfs_sops;
	sb->s_flags = 0;
	
	/* fill-in default values */
	info->mnt.version = 0;
	info->mnt.ttl = SHFS_DEFAULT_TTL;
	info->mnt.root_mode = (S_IRUSR | S_IWUSR | S_IXUSR | S_IFDIR);
	info->mnt.uid = current->uid;
	info->mnt.gid = current->gid;
	info->mnt.garbage_read = 0;
	info->mnt.garbage_write = 0;
	info->mnt.garbage = 0;
	info->mnt.readonly = 0;
	info->mnt.preserve_own = 0;
	info->mnt.disable_fcache = 0;
	info->mnt.generic_host = 0;
	info->mnt.stable_symlinks = 0;
	info->mnt.printf_buffer = (char *)kmalloc(PRINTF_BUFFER_SIZE, GFP_KERNEL);
	if (!info->mnt.printf_buffer) {
		VERBOSE("Not enough kmem to allocate printf buffer!\n");
		goto out_no_mem;
	}
	info->mnt.readln_buffer_len = 0;
	info->mnt.readln_buffer = (char *)kmalloc(READLN_BUFFER_SIZE, GFP_KERNEL);
	if (!info->mnt.readln_buffer) {
		VERBOSE("Not enough kmem to allocate readln buffer!\n");
		kfree(info->mnt.printf_buffer);
		goto out_no_mem;
	}
	init_MUTEX(&info->sem);
	
	info->mnt.mount_point[0] = 0;
	
	if (shfs_parse_options(info, (char *)opts) < 0) {
		VERBOSE("Invalid options!\n");
		goto out_no_opts;
	}
	
	DEBUG("uid %d, gid %d, root=%s, ro: %d\n", 
		info->mnt.uid, info->mnt.gid, info->mnt.root, info->mnt.readonly);

	if (info->mnt.version != SHFS_VERSION) {
		printk(KERN_NOTICE "shfs: version mismatch (module: %d, mount: %d)\n", SHFS_VERSION, info->mnt.version);
		goto out_no_opts;
	}

	if (shfs_init(info) < 0) {
		VERBOSE("Broken pipe, cannot connect!\n");
		goto out_no_opts;
	}
	
	shfs_init_root_dirent(info, &root);
	root_inode = shfs_iget(sb, &root);
	if (!root_inode) 
		goto out_no_root;
	sb->s_root = d_alloc_root(root_inode);
	if (!sb->s_root) 
		goto out_no_root;

	DEBUG("Mount succeded!\n");
	return sb;

out_no_root:
	iput(root_inode);
out_no_opts:
	kfree(info->mnt.printf_buffer);
	kfree(info->mnt.readln_buffer);
out_no_mem:
	kfree(info);
out:
	DEBUG("Mount failed!\n");
	return NULL;
}

static DECLARE_FSTYPE (sh_fs_type, "shfs", shfs_read_super, 0);

static int __init 
init_sh_fs(void)
{
	printk(KERN_NOTICE "SHell File System, Version: 0.32-pre2, (c) 2002, 2003 Miroslav Spousta\n");
	dir_head_cache = kmem_cache_create("shfs_dir_head", sizeof(struct shfs_dir_head), 0, 0, NULL, NULL);
	dir_entry_cache = kmem_cache_create("shfs_dir_entry", sizeof(struct shfs_dir_entry), 0, 0, NULL, NULL);
	dir_name_cache = kmem_cache_create("shfs_name_cache", SHFS_PATH_MAX, 0, 0, NULL, NULL);
	file_cache = kmem_cache_create("shfs_file", sizeof(struct shfs_file), 0, 0, NULL, NULL);
	DEBUG("dir_head_cache: %p\n", dir_head_cache);
	DEBUG("dir_entry_cache: %p\n", dir_entry_cache);
	DEBUG("dir_name_cache: %p\n", dir_name_cache);
	DEBUG("file_cache: %p\n", file_cache);
#ifdef SHFS_ALLOC_DEBUG
	alloc = 0;
#endif
	return register_filesystem(&sh_fs_type);
}

static void __exit 
exit_sh_fs(void)
{
	VERBOSE("Unregistering shfs.\n");
#ifdef SHFS_ALLOC_DEBUG
	if (alloc)
		VERBOSE("Memory leak (%lu)!\n", alloc);
#endif
	unregister_filesystem(&sh_fs_type);
	kmem_cache_destroy(file_cache);
	kmem_cache_destroy(dir_name_cache);
	kmem_cache_destroy(dir_entry_cache);
	kmem_cache_destroy(dir_head_cache);
}

EXPORT_NO_SYMBOLS;

module_init(init_sh_fs);
module_exit(exit_sh_fs);

#include <linux/version.h>

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,4,10))
	MODULE_LICENSE("GPL");
#endif

