/*======================================================================*/
/*									*/
/*  Immunix CryptoMark Module 						*/
/*  Copyright 1998, 1999, 2000 Wirex Communications & 			*/
/*			Oregon Graduate Institute 			*/
/*									*/
/*	Written by Greg Kroah-Hartman <greg@wirex.com>			*/
/*									*/
/*======================================================================*/

/* Module versioning code  */
#include <linux/autoconf.h> 		/* Retrieve the CONFIG_* macros */
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
	#define MODVERSIONS 		/* Force it on */
#endif

#ifdef MODVERSIONS
	#include <linux/modversions.h>
#endif

#include <linux/string.h>
#include <linux/dirent.h>

#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/limits.h>
#include <linux/elf.h>

#include <linux/version.h>
#include <linux/module.h>

#include <asm/uaccess.h>

#include <linux/immunix.h>

#include "cryptomark.h"
#include "cryptomark_log.h"
#include "cryptomark_version.h"

#include "gnupg/errors.h"

static struct cryptomark_operations ops = {
	version:		CRYPTOMARK_INTERFACE_VERSION,
	compute_elf_sig:	crypto_compute_elf_sig,
	compare_elf_sig:	crypto_compare_elf_sig,
};


const char *cryptomark_title =		"WireX/Immunix CryptoMark module";
const char *cryptomark_copyright =	"Copyright 1998, 1999, 2000 WireX Communications Inc.";

extern unsigned char *cryptomark_key;

/*======================================================================
*
*  init_module:
*
*======================================================================*/
int init_module (void)
{
	int error ;

	dbg (__FUNCTION__);
	
	error = register_cryptomark (&ops);
	if (error) {
		warn(__FUNCTION__": Unable to register module, error = %d", error);
		return error;
	}

	proc_init();

	printk (KERN_INFO "%s v" CRYPTOMARK_VERSION" initialized\n", cryptomark_title);
	
	return 0;
}


/*======================================================================
*
*  cleanup_module:
*
*======================================================================*/
int cleanup_module ( void )
{
	dbg (__FUNCTION__);

	proc_done();
	
	unregister_cryptomark(&ops);

	printk (KERN_INFO "%s v" CRYPTOMARK_VERSION" removed\n", cryptomark_title);

	return 0;
}



/*======================================================================
*
*  crypto_compare_elf_sig:
*
*======================================================================*/
extern int crypto_compare_elf_sig (struct dentry *elf_dentry, struct elfhdr *elf_ex, struct immunix_digital_sig *sig, char * name)
{
	struct elf_shdr *elf_shdata = NULL;
	struct elf_shdr *section_header;
	int elf_section_size;
	int i;
	int retval ;
	int error = -ENOENT; 	/* Return value */
	unsigned char *stordigest = NULL;

	if (elf_ex->e_shoff == 0) {
		dbg (__FUNCTION__ ": %s has no section header in it's ELF header.", name);
		return -ENOENT;
	}
	if (elf_ex->e_shnum == 0) {
		dbg (__FUNCTION__ ": %s has no section header entries in it's ELF header.", name);
		return -ENOENT;
	}
	if (elf_ex->e_shentsize == 0) {
		dbg (__FUNCTION__ ": %s has no section header size in it's ELF header.", name);
		return -ENOENT;
	}
	if (sig == NULL) {
		dbg (__FUNCTION__ ": %s has no md5 sig?", name);
		return -ENOENT;
	}
	
	elf_section_size = elf_ex->e_shentsize * elf_ex->e_shnum;
	elf_shdata = (struct elf_shdr *)kmalloc (elf_section_size , GFP_KERNEL);
	if (!elf_shdata) {
		dbg (__FUNCTION__ ": Unable to allocate memory for section header!");
		return -ENOMEM;
	}

	/* read the section header into our structure */
	retval = read_exec (elf_dentry, elf_ex->e_shoff, (char *)elf_shdata, elf_section_size, 1);
	if (retval != (elf_section_size)) {
		dbg (__FUNCTION__": Error reading elf section %s!", name );
		kfree (elf_shdata);
		return -1;
	}

	/* loop through the sections, looking for the one we are interested in */
	for (i = 0, section_header = elf_shdata; i < elf_ex->e_shnum; i++, section_header++) {
		if (section_header->sh_type == SHT_NOTE) {
			unsigned char *buf = NULL;
			struct elf_note en;
			unsigned char * encdigest = NULL;
			
			buf = (unsigned char *)kmalloc (section_header->sh_size - sizeof (struct elf_note), GFP_KERNEL);
			if (!buf) {
				warn (__FUNCTION__ ": Unable to allocate memory!" );
				kfree (elf_shdata);
				return -ENOMEM ; 
			}

			retval = read_exec (elf_dentry, section_header->sh_offset, (char *)&en, sizeof(struct elf_note), GFP_KERNEL);

			retval = read_exec (elf_dentry, section_header->sh_offset + sizeof(struct elf_note),(unsigned char *)buf, section_header->sh_size - sizeof (struct elf_note), GFP_KERNEL);

			if (strncmp (buf, CRYPTOMARK_NAME, strlen(CRYPTOMARK_NAME)) == 0) {
				encdigest = (unsigned char *)kmalloc (en.n_descsz + 1, GFP_KERNEL);
				if (!encdigest) {
					warn (__FUNCTION__ ": Unable to allocate memory!");
					kfree (buf);
					kfree (elf_shdata);
					return -ENOMEM ; 
				}

				memcpy (encdigest, &buf[(en.n_namesz + ((4 - (en.n_namesz % 4)) % 4))], en.n_descsz);
				encdigest[en.n_descsz] = '\0' ;
				
				dbg(__FUNCTION__": stored signature is:\n%s", encdigest);

				retval = crypto_verify_data (encdigest, en.n_descsz, &stordigest);
				dbg(__FUNCTION__": crypto_verify_data return = %d", retval);
				switch (retval) {
					case 0:
						dbg (__FUNCTION__": Good Signature in file %s", name);
						error = 0;
						break;

					case G10ERR_NO_PUBKEY:
						dbg (__FUNCTION__": Good Sig, but couldn't verify public key in file %s", name);
						error = EPERM;
						break;
					
					default:
						dbg (__FUNCTION__": Bad Signature in file %s", name);
						error = EPERM;
						break;
				}
			}
			kfree (buf);
			kfree (encdigest);

			if (stordigest) {
				dbg (__FUNCTION__": md5 in file is: %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x"
					"%02x%02x%02x%02x%02x",
					stordigest[0],  stordigest[1],
					stordigest[2],  stordigest[3],
					stordigest[4],  stordigest[5],
					stordigest[6],  stordigest[7],
					stordigest[8],  stordigest[9],
					stordigest[10], stordigest[11],
					stordigest[12], stordigest[13],
					stordigest[14], stordigest[15]);
			}
		} /* if */
	} /* for */

	if (stordigest) {
		if (memcmp (sig->md5, stordigest, 16) != 0 ) {
			warn (__FUNCTION__": Digests are different :(");
			error = EPERM;
		}
		else {
//			error = 0;
			dbg (__FUNCTION__": Digests are the same! md5 is: "
				"%02x%02x%02x%02x%02x%02x%02x%02x"
				"%02x%02x%02x%02x%02x%02x%02x%02x",
				sig->md5[0],  sig->md5[1],
				sig->md5[2],  sig->md5[3],
				sig->md5[4],  sig->md5[5],
				sig->md5[6],  sig->md5[7],
				sig->md5[8],  sig->md5[9],
				sig->md5[10], sig->md5[11],
				sig->md5[12], sig->md5[13],
				sig->md5[14], sig->md5[15]);
		}
	} else {
		dbg (__FUNCTION__": No Signature found in file %s", name);
		error = -ENOENT;
	}

	if (stordigest)
		kfree (stordigest);
	if (elf_shdata) 
		kfree (elf_shdata);

	/* let's impose a bit of policy here, if there is any kind of error, 
	 * and we are root, then we force the error to be a fatal one.
	 * Need to add better policy management in the future.
	 */
	if ((capable(CAP_SYS_ADMIN)) &&
	     (error < 0)) {
		dbg (__FUNCTION__ ": since we have privilidges, and an error occured, make it a fatal error.");
		if (cryptomark_key == NULL) {
			dbg (__FUNCTION__ ": our key is null, so don't do anything for now.");
			error = 0;
		} else {
			error = -error;
		}
	}
	
	return error ;
}

