// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/metrics/public/cpp/ukm_source_id.h"

#include "base/atomic_sequence_num.h"
#include "base/logging.h"
#include "base/rand_util.h"

namespace ukm {

namespace {

const int64_t kLowBitsMask = (INT64_C(1) << 32) - 1;
const int64_t kNumTypeBits = 2;
const int64_t kTypeMask = (INT64_C(1) << kNumTypeBits) - 1;

}  // namespace

SourceId AssignNewSourceId() {
  // Generate some bits which are unique to this process, so we can generate
  // IDs independently in different processes. IDs generated by this method may
  // collide, but it should be sufficiently rare enough to not impact data
  // quality.
  const static int64_t process_id_bits =
      static_cast<int64_t>(base::RandUint64()) & ~kLowBitsMask;
  // Generate some bits which are unique within the process, using a counter.
  static base::AtomicSequenceNumber seq;
  SourceId local_id = ConvertToSourceId(seq.GetNext() + 1, SourceIdType::UKM);
  // Combine the local and process bits to generate a unique ID.
  return (local_id & kLowBitsMask) | process_id_bits;
}

SourceId ConvertToSourceId(int64_t other_id, SourceIdType id_type) {
  const int64_t type_bits = static_cast<int64_t>(id_type);
  DCHECK_EQ(type_bits, type_bits & kTypeMask);
  // Stores the the type ID in the low bits of the source id, and shift the rest
  // of the ID to make room.  This could cause the original ID to overflow, but
  // that should be rare enough that it won't matter for UKM's purposes.
  return (other_id << kNumTypeBits) | type_bits;
}

SourceIdType GetSourceIdType(SourceId source_id) {
  return static_cast<SourceIdType>(source_id & kTypeMask);
}

}  // namespace ukm
