/* 
 *  Copyright (C) 1997 Cullen Jennings
 *
 *  Some brief changes by A. Kicelew and M. Zakaluzhsky
 *
 *  Based on the skeleton.c: A network driver outline for linux.
 *	Written 1993-94 by Donald Becker.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful, but
 *   *WITHOUT ANY WARRANTY; without even the implied warranty of
 *   *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General *Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * This module provides support for the Arlan 655 card made by Aironet
 * Wireless Communication Inc. in Fairlawn, Ohio, USA This is a spread
 * spectrum radio modem with speeds up to 2Mb/s. They can be reached at
 * sales@aironet.com or 1-800-3-WIRELESS
 * 
 * Running FTP, I get a transfer rate of about 800kb/s
 *
 */

static const char* version = "$(@) $Revision: 100.1 $ C. Jennings Jun'97";

#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <linux/errno.h>
#include <linux/delay.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_ether.h>	/* For the statistics structure. */
#include <linux/if_arp.h>	/* For ARPHRD_ETHER */


extern int probeEverywhere(struct device *dev);

static int probeArlanHere(struct device *dev, int ioaddr);
static int openArlan(struct device *dev);
static int transmitPacket(struct sk_buff *skb, struct device *dev);
static void arlanInterupt(int irq, void *dev_id, struct pt_regs *regs);
static void receivePacket(struct device *dev);
static int closeArlan(struct device *dev);
static struct enet_statistics *getStatistics(struct device *dev);
static void setMulticast(struct device *dev);
static void hardwareTransmitPacket(struct device *dev, char *buf, int length);
static int initializeHardware(struct device *dev, int startp);
static void enableReceive(struct device* dev);
static int waitReset(struct device* dev);
static int waitCard(struct device* dev, int line );

static char devicename[9] = { 'a','r','l','a','n','0',0, };
static struct device this_device = {
   devicename, /* will be inserted by linux/drivers/net/net_init.c */
   0, 0, 0, 0,
   0, 0,  /* I/O address, IRQ */
   0, 0, 0, NULL, probeEverywhere };

int irq = 0;
int mem = 0;
int debug = 0;

#define IFDEBUG( L ) if ( (L) <= debug ) 

/*
 * The name of the card. Is used for messages and in the requests for
 * io regions, irqs and dma channels
 */
static const char* cardname = "arlan";


/* Information that need to be kept for each board. */
struct net_local
{
      struct enet_statistics stats;
      long open_time;			/* Useless example local info. */
};

struct arlan_private
{
      /* Header Signature */
      char textRegion[48];
      u_char resetFlag;
      u_char  diagnosticInfo;
      u_short diagnosticOffset;
      u_char _1[12];
      u_char lanCardNodeId[6];
      u_char broadcastAddress[6];
      u_char hardwareType;
      u_char majorHardwareVersion;
      u_char minorHardwareVersion;
      u_char radioModule;
      u_char defaultChannelSet;
      u_char _2[47];
      
      /* Control/Status Block - 0x0080 */
      u_char interruptInProgress;
      u_char cntrlRegImage;
      u_char _3[14];
      u_char commandByte;
      u_char commandParameter[15];

      /* Receive Status - 0x00a0 */
      u_char rxStatusVector;
      u_char rxFrmType;
      u_short rxOffset;
      u_short rxLength;
      u_char rxSrc[6];
      u_char rxBroadcastFlag;
      u_char rxQuality;
      u_char scrambled;
      u_char _4[1];
      
      /* Transmit Status - 0x00b0 */
      u_char txStatusVector;
      u_char txAckQuality;
      u_char numRetries;
      u_char _5[14];
      u_char registeredRouter[6];
      u_char backboneRouter[6];
      u_char registrationStatus;
      u_char configuredStatusFlag;
      u_char _6[1];
      u_char ultimateDestAddress[6];
      u_char immedDestAddress[6];
      u_char immedSrcAddress[6];
      u_short rxSequenceNumber;
      u_char assignedLocaltalkAddress;
      u_char _7[27];

      /* System Parameter Block */

      /* - Driver Parameters (Novell Specific) */

      u_short txTimeout;
      u_short transportTime;
      u_char _8[4];

      /* - Configuration Parameters */
      u_char irqLevel;
      u_char spreadingCode;
      u_char channelSet;
      u_char channelNumber;
      u_short radioNodeId;
      u_char _9[2];
      u_char scramblingDisable;
      u_char radioType;
      u_short routerId;
      u_char _10[9];
      u_char txAttenuation;
      u_char systemId[4]; /* on an odd address for a long !!! */
      u_short globalChecksum;
      u_char _11[4];
      u_short maxDatagramSize;
      u_short maxFrameSize;
      u_char maxRetries;
      u_char receiveMode;
      u_char priority;
      u_char rootOrRepeater;
      u_char specifiedRouter[6];
      u_short fastPollPeriod;
      u_char pollDecay;
      u_char fastPollDelay[2];
      u_char arlThreshold;
      u_char arlDecay;
      u_char _12[1];
      u_short specRouterTimeout;
      u_char _13[5];

      /* Scrambled Area */
      u_char SID[4];
      u_char encryptionKey[12];
      u_char _14[2];
      u_char waitTime[2];
      u_char lParameter[2];
      u_char _15[3];
      u_short headerSize;
      u_short sectionChecksum;

      u_char registrationMode;
      u_char registrationFill;
      u_short pollPeriod;
      u_short refreshPeriod;
      u_char name[16];
      u_char NID[6];
      u_char localTalkAddress;
      u_char codeFormat;
      u_char SSCode[64];

      u_char _16[0x140];

      /* Statistics Block - 0x0300 */
      u_char hostcpuLock;
      u_char lancpuLock;
      u_char resetTime[18];
      u_int numDatagramsTransmitted;
      u_int numReTransmissions;
      u_int numFramesDiscarded;
      u_int numDatagramsReceived;
      u_int numDuplicateReceivedFrames;
      u_int numDatagramsDiscarded;
      u_short maxNumReTransmitDatagram;
      u_short maxNumReTransmitFrames;
      u_short maxNumConsecutiveDuplicateFrames;
      /* misaligned here so we have to go to characters */
      u_char numBytesTransmitted[4];
      u_char numBytesReceived[4];
      u_char numCRCErrors[4];
      u_char numLengthErrors[4];
      u_char numAbortErrors[4];
      u_char numTXUnderruns[4];
      u_char numRXOverruns[4];
      u_char numHoldOffs[4];
      u_char numFramesTransmitted[4];
      u_char numFramesReceived[4];
      u_char numReceiveFramesLost[4];
      u_char numRXBufferOverflows[4];
      u_char numFramesDiscardedAddrMismatch[4];
      u_char numFramesDiscardedSIDMismatch[4];
      u_char numPollsTransmistted[4];
      u_char numPollAcknowledges[4];
      u_char numStatusVectorTimeouts[4];
      u_char numNACKReceived[4];

      u_char _17[0x86];

      u_char txBuffer[0x800];
      u_char rxBuffer[0x800];

      u_char _18[0x0bff];

      u_char controlRegister;
};


typedef struct
{
      short offset;
      short length;
      u_char dest[6];
      unsigned char clear;
      unsigned char retries;
      unsigned char routing;
      unsigned char scrambled;
} TxParam;


#define HARDWARE_RESET 0x01
#define CHANNEL_ATTENTION 0x02
#define INTERRUPT_ENABLE 0x04
#define CLEAR_INTERRUPT 0x08


static int
readControlRegister(struct device *dev)
{
 volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   
   return arlan->cntrlRegImage;
}


static void
writeControlRegister(struct device *dev,int v)
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;

   v = v&0xF; /* only use lower 4 bits */
   
   arlan->cntrlRegImage = v;
   arlan->controlRegister = v;

   IFDEBUG(100) printk("Arlan control register %02x\n",  arlan->cntrlRegImage);
}


static void
toggleChannelAttention(struct device *dev)
{
   int cr = readControlRegister(dev);
   
   if (cr & CHANNEL_ATTENTION)
   {
      writeControlRegister(dev,cr & ~CHANNEL_ATTENTION);
   }
   else
   {
      writeControlRegister(dev,cr | CHANNEL_ATTENTION);
   }
}


static void
clearChannelAttention(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) & ~CHANNEL_ATTENTION);
}


static void
setHardwareReset(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) | HARDWARE_RESET);
}


static void
clearHardwareReset(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) & ~HARDWARE_RESET);
}


static void
setInterruptEnable(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) | INTERRUPT_ENABLE);   
}


static void
setClearInterrupt(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) | CLEAR_INTERRUPT);   
}


static void
clearClearInterrupt(struct device *dev)
{
   writeControlRegister(dev,readControlRegister(dev) & ~CLEAR_INTERRUPT);   
}


static void
printDiag(struct device *dev)
{
   int i;
   union 
   {
	 char c[4];
	 long i;
   }
   sysId;
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   
   
   if (arlan->configuredStatusFlag == 0)
      printk("Arlan: Card NOT configured\n");
   else
      printk("Arlan: Card is configured\n");
    
   switch ( arlan->diagnosticInfo )
   {
      case 0xFF: printk("Arlan: ERROR Diagnostic info is OK\n"); break;
      case 0xF7: printk("Arlan: ERROR Missing SS Code\n"); break;
      case 0xF6: printk("Arlan: ERROR Invalid config format\n"); break;
      case 0xF4:
      printk("Arlan: ERROR Invalid spreading code/channel number\n"); break;
      default:
      printk("Arlan: Diagnostic Info ERROR %d\n",
	     (int)arlan->diagnosticInfo);
      break;
   }

   printk("arlan: LAN CODE ID = ");
   for( i=0; i<6; i++ ) printk("%03d:",arlan->lanCardNodeId[i]);
   printk("\n");

   printk("arlan: channelNumber=%d\n",(int)arlan->channelNumber);
   printk("arlan: channelSet=%d\n",(int)arlan->channelSet);
   printk("arlan: spreadingCode=%d\n",(int)arlan->spreadingCode);
   printk("arlan: radioNodeId=%d\n",(int)arlan->radioNodeId);
   
   printk("arlan: registration mode is %d\n", (int)arlan->registrationMode );

   for( i=0; i<4; i++ )sysId.c[i] =  arlan->systemId[i];
   printk("arlan: system id = %ld\n",sysId.i);
      
   printk("arlan: name=");
   for( i=0; i<16; i++ ) if (arlan->name[i]) printk("%c",arlan->name[i]);
   printk("\n");
}


static int
noopCommand(struct device *dev,int interrupt)
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;

   /* check command byte is zero - ie. no command in progress */
   if (arlan->commandByte != 0)
   {
      return -1;
   }
   
   /* then put parameters in parameter block (noop has no parameters) */
   
   /*  write command to command byte */
   if (interrupt)
   {
      arlan->commandByte = 0x87;
   }
   else
   {
      arlan->commandByte = 0x07;
   }
   IFDEBUG(100)
      printk("Arlan command byte 0x%04x\n", arlan->commandByte);
   
   /* interrupt lancpu */
   IFDEBUG(100)
      printk("Arlan control register before 0x%02x\n", arlan->controlRegister);
   toggleChannelAttention(dev);
   
   IFDEBUG(100)
      printk("Arlan control register 0x%02x\n", arlan->controlRegister);
   IFDEBUG(100)
      printk("Arlan command byte 0x%02x\n", arlan->commandByte);

   if ( waitCard( dev, __LINE__ ) ) return -1; /* wait for not to process */
   
   return 0;
}


static int
waitCard(struct device* dev, int line )
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;

   int i=0;
   while (arlan->commandByte != 0)
   {
      i++;
      udelay(100);

      if (i>200*10) /* 200 ms */
      {
	 printk(KERN_WARNING "%s: waitCard timeout on line %d \n",
		cardname,line);
	 return -1;
      }
   }
   IFDEBUG(5) printk("waitCard took time %d on line %d \n", i, line);
   return 0;
}


static int
waitReset(struct device* dev)
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   int i;
   
   /* wait for reset flag to become zero, we'll wait for two seconds */
   i = 0;
   while(1)
   {
      if (arlan->resetFlag == 0)
      {
	 IFDEBUG(10)
	    printk("Arlan reset acknowledged after %d/1000 seconds\n",
		   i);
	 break;
      }

      if ( i > 900 ) /* 900 ms */
      {
	 printk(KERN_WARNING "Arlan reset acknowledge timeout");
	 return -1;	 
      }

      i++;
      udelay(1000);
   }
   return 0;
}

       
static int
setupCard(struct device* dev)
{
   u_char* ptr;
   int i;
   int memlen = sizeof(struct arlan_private) - 0xF;/* avoid control register */
   volatile char* arlan_mem = (char*)(dev->mem_start);
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   
   char pattern;
   char tempBuf[48+1];
   
   ptr=NULL;
   
   if ( arlan->configuredStatusFlag != 0 )
   {
      IFDEBUG(10) printk("arlan: CARD IS CONFIGURED\n");
   }
   else
   {
      IFDEBUG(10) printk("arlan: card is NOT configured\n");
   }

   /* hold card in reset state */
   setHardwareReset(dev);

   /* test memory */
   pattern = 0;
   for (i = 0; i < memlen; i++)
   {
      arlan_mem[i] = pattern++;
   }

   pattern = 0;
   for (i = 0; i < memlen; i++)
   {
      if (arlan_mem[i] != pattern++)
      {
	 printk(KERN_WARNING "Arlan driver memory test 1 failed at %06x\n",
		(int)(&arlan_mem[i]) );
	 return -1;
      }
   }

   pattern = 0;
   for (i = 0; i < memlen; i++)
   {
      arlan_mem[i] = ~(pattern++);
   }

   pattern = 0;
   for (i = 0; i < memlen; i++)
   {
      if (arlan_mem[i] != ~(pattern++) )
      {
	 printk(KERN_WARNING "Arlan driver memory test 2 failed at %06x\n",
		(int)(&arlan_mem[i]));
	 return -1;
      }
   }
   
   /* zero memory */
   for (i = 0; i < memlen; i++)
   {
      arlan_mem[i] = 0;
   }
   IFDEBUG(1) printk(KERN_INFO "Arlan: memory tests ok\n");
   
   /* set reset flag and then release reset */
   arlan->resetFlag = 0xff;
   
   clearChannelAttention(dev);
   clearHardwareReset(dev);

   /* wait for reset flag to become zero, we'll wait for two seconds */
   if (waitReset(dev)) return -1;
      
   IFDEBUG(4)
      printk("arlan configuredStatus = %d \n",
	     (int)arlan->configuredStatusFlag );
   IFDEBUG(4)
      printk("arlan driver diagnostic: 0x%2x\n",
	     arlan->diagnosticInfo);

   /* issue nop command - no interupt */
   if (noopCommand(dev,0) != 0)
   {
      return -1;
   }
   IFDEBUG(50) printk("1st Noop successfully executed !!\n");

   /* try to turn on the arlan interrupts */
   clearClearInterrupt(dev);
   setClearInterrupt(dev);
   setInterruptEnable(dev);
   
   /* issue nop command - with interrupt */
   if (noopCommand(dev,1) != 0)
   {
      return -1;
   }
   IFDEBUG(50) printk("2nd Noop successfully executed !!\n");

   if ( (int)arlan->irqLevel != dev->irq )
   {
      printk(KERN_WARNING "arlan dip switches set irq to %d\n",
	     (int)arlan->irqLevel);
      printk(KERN_WARNING "device driver irq set to %d - does not match\n",
	     dev->irq);
      return -1;
   }
   else
   {
      IFDEBUG(2) printk("irq level is OK\n");
   }

   IFDEBUG(10) printDiag(dev);

   if (initializeHardware(dev, 0)) return -1;
   
   IFDEBUG(3) printDiag(dev);

   if (arlan->configuredStatusFlag == 0)
   {
      printk(KERN_WARNING "arlan configure failed\n");
      return -1;
   }

   for( i=0; i<48; i++ ) tempBuf[i]=arlan->textRegion[i];
   tempBuf[48]=0;
   IFDEBUG(2) printk("arlan text region is: %s\n",tempBuf);

   /* waitCard(dev,__LINE__); */
   
   printk(KERN_NOTICE "%s: driver version %s loaded for %s\n",
	  cardname, version, &(tempBuf[23]) );
   return 0; /* no errors */
}


static void
enableReceive(struct device* dev)
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;

   IFDEBUG(2) printk("enableReceive\n");

   /* enable receives */

   while (1)
   {
      waitCard(dev,__LINE__);
      cli();
      if (arlan->commandByte == 0)
	 break;
      sti();
   }
	
   arlan->commandByte = 0x83;
   arlan->commandParameter[0] = 1; /* broadcasts */
   toggleChannelAttention(dev);
   sti();

   waitCard(dev,__LINE__);
}


void
hardwareTransmitPacket(struct device* dev, char *buf, int length )
{
   int i;
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   TxParam txParam;

   IFDEBUG(5) printk("hardwareTransmitPacket\n");
 
   IFDEBUG(2)
   {
#ifdef MODE_640
      printk("%s: send packet length %d to ",
	     cardname, length-12 );
#else
      printk("%s: send packet length %d to ",
	     cardname, length );
#endif
      /* Retrieve and print the ethernet address. */
      for (i = 0; i < 6; i++)
      {
	 printk("%02x:", (unsigned int)( buf[i] ) );
      }
      printk("\n");
   }

   while (1)
   {
      waitCard(dev,__LINE__);
      cli();
      if (arlan->commandByte == 0)
	 break;
      sti();
   }
#ifdef MODE_640
   memcpy( (void*)arlan->txBuffer , buf+12 ,length-12 );
#else
   memcpy( (void*)arlan->txBuffer , buf ,length );
#endif
   txParam.offset= ( (int)arlan->txBuffer - (int)arlan);

#ifdef MODE_640
   txParam.length = length-12;
#else
   txParam.length = length;
#endif
   /* set the destination */
   for (i = 0; i < 6; i++)
   {
      txParam.dest[i] = buf[i];
   }
   txParam.clear = 0;
   txParam.retries = 0; /* 0 is use default */
   txParam.routing = 0x01;
   txParam.scrambled = 0;
   memcpy( (void*) arlan->commandParameter , &txParam , 14 );
  
   /* -- toggle attention */
   arlan->commandByte = 0x85;
   toggleChannelAttention(dev);

   /* waitCard(dev,__LINE__); */

   IFDEBUG(2) printk("Qued %d bytes \n",length);
}


int 
initializeHardware(struct device *dev, int startReceiving )
{
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   volatile char* arlan_mem = (char*)(dev->mem_start);

   printk(KERN_INFO "%s: Doing chip reset\n", dev->name );
   
   /* hold card in reset state */
   setHardwareReset(dev);
   /* set reset flag and then release reset */
   arlan->resetFlag = 0xff;
   
   clearChannelAttention(dev);
   clearHardwareReset(dev);
   
   /* wait for reset to complete */
   if (waitReset(dev)) return -1;
   
   /* try to turn on the arlan interrupts */
   clearClearInterrupt(dev);
   setClearInterrupt(dev);
   setInterruptEnable(dev);
   memset( (void*) arlan_mem+0x100,0,0x200);
   
   arlan->txTimeout = 0x0080;
   arlan->transportTime = 0x0040;
   arlan->irqLevel = dev->irq;
   
   arlan->channelSet = 9;
   arlan->spreadingCode = 4;
   arlan->channelNumber = 1;

#ifdef MODE_640
   arlan->registrationMode=1;
#endif   
   arlan->radioNodeId = 0;
   arlan->scramblingDisable = 0;
   arlan->radioType = 0;
   arlan->routerId = 0;
   arlan->systemId[0] = 4;
   arlan->systemId[1] = 0;
   arlan->systemId[2] = 0;
   arlan->systemId[3] = 0;
   arlan->globalChecksum = 0x0;
   arlan->fastPollPeriod = 0x0030;
   strncpy( (void*)(arlan->name) , "LinuxArlan" , 16 ); 
   
   if (arlan->commandByte != 0)
   {
      printk(KERN_WARNING "arlan error: busy after setup \n");
      return -1;
   }
   arlan->commandByte = 0x81; /* do configure */
   memset( (void*) arlan->commandParameter ,0,0xF);
   arlan->commandParameter[0] = 0;
   arlan->commandParameter[1] = 0;
   arlan->commandParameter[2] = 0;
   IFDEBUG(10) printk("Arlan: Doing configure\n");
   
   toggleChannelAttention(dev);
   udelay(100);
   
   if (waitCard(dev,__LINE__) ) return -1;
   
   if ( startReceiving ) 
   {
      enableReceive(dev);
   }

   IFDEBUG(1) printk("%s: Done full reset\n",cardname);

   return 0;
}


int
probeEverywhere(struct device *dev)
{
   int m;
   int mem_start = dev ? dev->mem_start : 0;

   if (mem_start !=0 )    /* Check a single specified location. */
   {
      return probeArlanHere(dev, mem_start);
   }
	
   for (m = 0xC0000 ; m<= 0xDE000 ; m+= 0x2000 )
   {
      if (probeArlanHere(dev, m) == 0)
	 return 0;
   }

   return -ENODEV;
}


/*
 * This is the real probe routine. Linux has a history of friendly device
 * probes on the ISA bus. A good device probes avoids doing writes, and
 * verifies that the correct device exists and functions.
 */
static int
probeArlanHere(struct device *dev, int memaddr)
{
   int i,ret;
   static char probeText[] = "TELESYSTEM SLW INC.    ARLAN 655";
   char tempBuf[49];
   volatile struct arlan_private *arlan = (struct arlan_private*) memaddr;
   int reqError;

   if (  irq < 0 )
   {
      free_irq(-irq, NULL);
      return -1;
   }
   
   /* clearHardwareReset(dev); - bad craps memory */
   for( i=0; i<48; i++ ) tempBuf[i]=arlan->textRegion[i];
   tempBuf[48]=0;

   /* check for card at this address */
   if ( 0!=strncmp( tempBuf , probeText , strlen(probeText)) )
   {
      return -ENODEV;
   }
	
   /* Allocate a new 'dev' if needed. */
   if (dev == NULL) {
      /*
       * Don't allocate the private data here, it is done later
       * This makes it easier to free the memory when this driver
       * is used as a module.
       */
      dev = init_etherdev(0, 0);
      if (dev == NULL)
	 return -ENOMEM;
   }

   IFDEBUG(1)
      printk(KERN_INFO "%s: %s found at %#5x, ",
	     dev->name, cardname, memaddr);
   
   /* Fill in the 'dev' fields. */
   dev->base_addr = 0;
   dev->mem_start = memaddr;
   dev->mem_end = memaddr + 0x1FFF;

   dev->family = AF_INET;
      
   dev->type = ARPHRD_ETHER;
   dev->hard_header_len = 0;
   dev->flags = IFF_BROADCAST;
   
   /* Retrieve and print the ethernet address. */
   for (i = 0; i < 6; i++)
   {
      dev->dev_addr[i] = arlan->lanCardNodeId[i];
      IFDEBUG(3) printk("%02x:", dev->dev_addr[i] );
   }
   IFDEBUG(3) printk("\n");
      
   /*
    * If this board has jumpered interrupts, allocate the interrupt
    * vector now. There is no point in waiting since no other device
    * can use the interrupt, and this marks the irq as busy. Jumpered
    * interrupts are typically not reported by the boards, and we must
    * used autoIRQ to find them.
    */

   if (dev->irq < 2)
   {
      /* just ask the card what IRQ is in use */
      dev->irq = arlan->irqLevel; 
   }
   else if (dev->irq == 2)
   {
      /*
       * Fixup for users that don't know that IRQ 2 is really
       * IRQ9, or don't know which one to set.
       */
      dev->irq = 9;
   }

   reqError = request_irq(dev->irq,
			&arlanInterupt, 0, cardname, NULL);
   if (reqError)
   {
      printk(KERN_WARNING "%s: unable to get IRQ %d (reqError=%d).\n",
	     dev->name, dev->irq, reqError);
      return -EAGAIN;
   }
   irq2dev_map[dev->irq] = dev;
   
   dev->dma = 0;
	
   /* Initialize the device structure. */
   if (dev->priv == NULL)
   {
      dev->priv = kmalloc(sizeof(struct net_local), GFP_KERNEL);
      if (dev->priv == NULL)
	 return -ENOMEM;
   }

   memset(dev->priv, 0, sizeof(struct net_local));

   dev->open		= openArlan;
   dev->stop		= closeArlan;
   dev->hard_start_xmit = transmitPacket;
   dev->get_stats	= getStatistics;
   dev->set_multicast_list = &setMulticast;

   /* Fill in the fields of the device structure with ethernet values. */
   ether_setup(dev);

   ret = setupCard(dev);
   if ( ret )
   {
      /* failed config - free irq and return */
      free_irq(dev->irq, NULL);
      return ret;
   }
   
   return 0;
}


/*
 * Open/initialize the board. This is called (in the current kernel)
 * sometime after booting when the 'ifconfig' program is run.
 *
 * This routine should set everything up anew at each open, even
 * registers that "should" only need to be set once at boot, so that
 * there is non-reboot way to recover if something goes wrong.
 */
static int
openArlan(struct device *dev)
{
   struct net_local *lp = (struct net_local *)dev->priv;

   IFDEBUG(5) printk("openArlan");
  
   /* Reset the hardware here. Don't forget to set the station address. */
   if (initializeHardware(dev, 1)) return -1;
   lp->open_time = jiffies;

   dev->tbusy = 0;
   dev->interrupt = 0;
   dev->start = 1;

   MOD_INC_USE_COUNT;

   return 0;
}


static int
transmitPacket(struct sk_buff *skb, struct device *dev)
{
  IFDEBUG(5) printk("transmitPacket\n");
  
   if (dev->tbusy)
   {
      /*
       * If we get here, some higher level has decided we are broken.
       * There should really be a "kick me" function call instead.
       */
      int tickssofar = jiffies - dev->trans_start;
      if (tickssofar < 50) /* half a second */
      {
	 return 1;
      }
      
      printk(KERN_WARNING "%s: transmit timed out, %s?\n",
	     dev->name,
	     "network problem");
      /* Try to restart the adaptor. */
      initializeHardware(dev, 1);
      dev->tbusy=0;
      dev->trans_start = jiffies;
   }
   /*
    * If some higher layer thinks we've missed an tx-done interrupt
    * we are passed NULL. Caution: dev_tint() handles the cli()/sti()
    * itself.
    */
   if (skb == NULL)
   {
      dev_tint(dev);
      return 0;
   }
   /*
    * Block a timer-based transmit from overlapping. This could better be
    * done with atomic_swap(1, dev->tbusy), but set_bit() works as well.
    */
   if (set_bit(0, (void*)&dev->tbusy) != 0)
   {
      printk(KERN_WARNING "%s: Transmitter access conflict.\n",
	     dev->name);
   }
   else
   {
      short length = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN;
      unsigned char *buf = skb->data;

      hardwareTransmitPacket(dev, buf, length);
      dev->trans_start = jiffies;
   }
   dev_kfree_skb (skb, FREE_WRITE);

   return 0;
}


static void
arlanInterupt(int irq, void *dev_id, struct pt_regs * regs)
{
   struct device *dev = (struct device *)(irq2dev_map[irq]);
   struct net_local *lp;
   volatile struct arlan_private *arlan;
   int clearTbusy = 0;
   arlan = (struct arlan_private*) dev->mem_start;

   IFDEBUG(2) printk("%s: arlanInterupt got IRQ \n", cardname);
 
   if (dev == NULL)
   {
      printk(KERN_WARNING "%s: irq %d for unknown device.\n", cardname, irq);
      return;
   }
   dev->interrupt = 1;

   lp = (struct net_local *)dev->priv;
   
   if ( arlan->rxStatusVector != 0 )
   {
      /* Got a packet(s). */
      receivePacket(dev);
   }
   if ( arlan->txStatusVector != 0)
   {
      switch ( arlan->txStatusVector )
      {
	 case 1 :
	 IFDEBUG(2) printk("arlan intr: transmit OK\n");
	 break;
	 case 2 :
	 IFDEBUG(1) printk("arlan intr: transmit timed out\n");
	 break;
	 case 3 :
	 IFDEBUG(1) printk("arlan intr: transmit max retries\n");
	 break;
	 case 4 :
	 IFDEBUG(1) printk("arlan intr: transmit aborted\n");
	 break;
	 case 5 :
	 IFDEBUG(1) printk("arlan intr: transmit not registered\n");
	 break;
	 case 6 :
	 IFDEBUG(1) printk("arlan intr: transmit destination full\n");
	 break;
	 case 7 :
	 IFDEBUG(1) printk("arlan intr: transmit unknown ack\n");
	 break;
	 case 8 :
	 IFDEBUG(1) printk("arlan intr: transmit dest mail box full\n");
	 break;
	 case 9 :
	 IFDEBUG(1) printk("arlan intr: transmit root dest not reg.\n");
	 break;
	 default:
	 IFDEBUG(1) printk("arlan intr: transmit status unknown\n");
	 break;
      }
      arlan->txStatusVector = 0;
      lp->stats.tx_packets++;
      clearTbusy=1;
      mark_bh(NET_BH);	/* Inform upper layers. */
   }


   clearClearInterrupt(dev);
   setClearInterrupt(dev);
   if ( clearTbusy )
   {
      dev->tbusy = 0;
   }
   
   dev->interrupt = 0;
}


/* We have a good packet(s), get it/them out of the buffers. */
static void
receivePacket(struct device *dev)
{
   struct net_local *lp = (struct net_local *)dev->priv;
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   static char broadcastAddr[] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};
   
   if ( arlan->rxStatusVector == 0)	/* Read all the frames? */
	 return;			/* Done for now */
   
   /* got an rx vector */
   IFDEBUG(2) printk("arlan intr: received a packet\n");
   
   switch ( arlan->rxStatusVector )
   {
      case 1 :
      {
	 /* Malloc up new buffer. */
	 struct sk_buff *skb;
	 int pkt_len;
	 
	  IFDEBUG(2)
	    printk("arlan intr: recieved data frame len=%d off=%d\n",
#ifdef MODE_640
		   (int)arlan->rxLength+12,
#else
		   (int)arlan->rxLength,
#endif
		   (int)arlan->rxOffset);

	  pkt_len = arlan->rxLength;

#ifdef MODE_640	  
	 skb = dev_alloc_skb(pkt_len+14);
#else
	 skb = dev_alloc_skb(pkt_len+2);
#endif
	 if (skb == NULL)
	 {
	    printk(KERN_NOTICE "%s: Memory squeeze, dropping packet.\n",
		   dev->name);
	    lp->stats.rx_dropped++;
	    break;
	 }
	 skb->dev = dev;

	 skb_reserve(skb,2);
#ifdef MODE_640	 
	 memcpy (skb_put (skb, 6), dev->dev_addr, 6);
	 memcpy (skb_put (skb, 6), (void*) arlan->registeredRouter, 6);
#endif
	 memcpy(skb_put(skb,pkt_len),
		((char*)arlan)+arlan->rxOffset,
		pkt_len);
	 
	 if ( memcmp(skb->data,broadcastAddr,6) == 0 )
	 {
	    skb->pkt_type = PACKET_BROADCAST;
	    IFDEBUG(2) printk("%s: got BROADCAST packet.\n",
			      dev->name);
	 }
	 else if ( memcmp(skb->data,dev->dev_addr,6) == 0 )
	 {
	    skb->pkt_type = PACKET_HOST;
	    IFDEBUG(2) printk("%s: got HOST  packet.\n",
			      dev->name);
	 }
	 else
	 {
	    skb->pkt_type = PACKET_OTHERHOST;
	    IFDEBUG(2) printk("%s: got _OTHERHOST packet.\n",
			      dev->name);
	 }
	 
	 skb->protocol = eth_type_trans(skb,dev);
	 skb->mac.raw = skb->data;
	 
	 netif_rx(skb);
	 lp->stats.rx_packets++;
      }
      break;

      case 2 :
      printk(KERN_WARNING "arlan intr: recieved control frame\n");
      break;

      default:
      printk(KERN_WARNING "arlan intr: recieved unknown status\n");
      lp->stats.rx_crc_errors++;
      break;
   }
   arlan->rxStatusVector = 0;

   enableReceive(dev);
}



static int
closeArlan(struct device *dev)
{
   struct net_local *lp = (struct net_local *)dev->priv;

   printk(KERN_NOTICE "%s: Closing device\n", dev->name );
 
   lp->open_time = 0;

   dev->tbusy = 1;
   dev->start = 0;

   free_irq(dev->irq, NULL);
   irq2dev_map[dev->irq] = 0;

   MOD_DEC_USE_COUNT;

   return 0;
}


static long
alignLong( volatile char* ptr )
{
   long ret;
   memcpy( &ret , (void*)ptr , 4 );
   return ret;
}


/*
 * Get the current statistics.
 * This may be called with the card open or closed.
 */
static struct enet_statistics *
getStatistics(struct device *dev)
{
   struct net_local *lp = (struct net_local *)dev->priv;
   volatile struct arlan_private *arlan
      = (struct arlan_private*) dev->mem_start;
   
   /* Update the statistics from the device registers. */
   
   lp->stats.collisions       = (arlan->numReTransmissions);
   lp->stats.rx_crc_errors    = alignLong(arlan->numCRCErrors);
   lp->stats.rx_dropped       = (arlan->numFramesDiscarded);
   lp->stats.rx_fifo_errors   = alignLong(arlan->numRXBufferOverflows);
   lp->stats.rx_frame_errors  = alignLong(arlan->numReceiveFramesLost);
   lp->stats.rx_over_errors   = alignLong(arlan->numRXOverruns);
   lp->stats.rx_packets       = (arlan->numDatagramsReceived);
   lp->stats.tx_aborted_errors= alignLong(arlan->numAbortErrors);
   lp->stats.tx_carrier_errors= alignLong(arlan->numStatusVectorTimeouts);
   lp->stats.tx_dropped       = (arlan->numDatagramsDiscarded);
   lp->stats.tx_fifo_errors   = alignLong(arlan->numTXUnderruns);
   lp->stats.tx_packets       = (arlan->numDatagramsTransmitted);
   lp->stats.tx_window_errors = alignLong(arlan->numHoldOffs);
   
   return &lp->stats;
}


/*
 * Set or clear the multicast filter for this adaptor.
 * num_addrs == -1	Promiscuous mode, receive all packets
 * num_addrs == 0	Normal mode, clear multicast list
 * num_addrs > 0	Multicast mode, receive normal and MC packets,
 *			and do best-effort filtering.
 */
static void
setMulticast(struct device *dev)
{
   if (dev->flags&IFF_PROMISC)
   {
      /* Enable promiscuous mode */
      /*  TODO */
   }
   else 
   {
      /* turn off promiscuous mode */
      /* oTODO */
   }
}


int init_module(void)
{
   int result;

   IFDEBUG(1) printk(KERN_DEBUG "%s\n", version);

   /* Copy the parameters from insmod into the device structure. */
   this_device.base_addr = 0;
   this_device.irq       = irq;
   this_device.dma       = 0;
   this_device.mem_start = mem;
   
   sprintf( this_device.name , "arlan%d" , 0 ); /* TODO - suport multiple */

   result = register_netdev(&this_device);
      
   if ( result != 0)
   {
      return result;
   }
   
   return 0;
}


void
cleanup_module(void)
{
   IFDEBUG(1) printk(KERN_INFO "%s: unloading module\n", cardname);
   
   /* No need to check MOD_IN_USE, as sys_delete_module() checks. */
   unregister_netdev(&this_device);

   /*
    * If we don't do this, we can't re-insmod it later.
    * Release irq/dma here, when you have jumpered versions and
    * allocate them in net_probe1().
    */

   free_irq(this_device.irq, NULL);

   if (this_device.priv)
   {
      kfree_s(this_device.priv, sizeof(struct net_local));
   }
}

/*
 * Local variables:
 *  compile-command:
 *	gcc -D__KERNEL__ -Wall -Wstrict-prototypes -Wwrite-strings
 *	-Wredundant-decls -O2 -m486 -c arlan.c
 *  version-control: t
 *  kept-new-versions: 5
 *  tab-width: 4
 *  c-indent-level: 4
 * End:
 */
