/* Get preshared key for remote IP from the SHARED_SECRETS_FILE
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: preshared.c,v 1.16 1999/04/11 00:44:21 dhr Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <glob.h>
#ifndef GLOB_ABORTED
# define GLOB_ABORTED    GLOB_ABEND	/* fix for old versions */
#endif

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "connections.h"
#include "state.h"
#include "preshared.h"
#include "log.h"

const char *shared_secrets_file = SHARED_SECRETS_FILE;

struct secret_index {
    int type;	/* ID_IPV4_ADDR, ID_FQDN, or ID_USER_FQDN */
    struct in_addr ipv4;
    char *name;
    struct secret_index *next;
};

struct secret {
    struct secret_index *indices;
    size_t len;
    unsigned char *secret;
    struct secret *next;
};

struct secret *secrets = NULL;

/* find the preshared key associated with the peer.
 * Note: the result is a chunk of bytes, not a string: no '\0' termination.
 */

const u_char *
get_preshared_secret(struct state *st, size_t *length)
{
    struct secret *s;
    const u_char *winner = NULL;
    struct in_addr x = st->st_connection->this.host;
#ifdef ROAD_WARRIOR_FUDGE
    struct in_addr y = st->st_connection->rw_instance
	? mask0.sin_addr : st->st_connection->that.host;
#else
    struct in_addr y = st->st_connection->that.host;
#endif /* ROAD_WARRIOR_FUDGE */

    *length = 0;	/* in case of failure */

    for (s = secrets; s != NULL; s = s->next)
    {
	unsigned int found = 0;
	struct secret_index *i;

	for (i = s->indices; i != NULL; i = i->next)
	{
	    switch (i->type)
	    {
	    case ID_IPV4_ADDR:
		if (x.s_addr == i->ipv4.s_addr)
		    found |= 01;
		if (y.s_addr == i->ipv4.s_addr)
		    found |= 02;
		break;
	    }
	    if (found == (01 | 02))
	    {
		/* both ends matched: success */
		if (winner == NULL)
		{
		    *length = s->len;
		    winner = s->secret;
		}
		else
		{
		    if (s->len != *length
		    || memcmp(s->secret, winner, s->len) != 0)
			log("multiple preshared secrets entries match endpoints: first secret used");
		}
	    }
	}
    }
    return winner;
}

/* digest a secrets file
 *
 * The file is a sequence of entries and include directives.
 *
 * An include directive is a line starting with "include"
 * followed by whitespace, followed by a filename.
 * If the filename does not start with /, it is taken to
 * be relative to the directory containing the current file.
 *
 * An entry is a sequence of indices followed by a secret.
 * the first index must start a line, with no indentation.
 * Each index is followed by whitespace; this can include
 * a newline, but the newline must be followed by whitespace.
 * The secret must be the last thing on it line.
 *
 * A line that is empty, or consists only of whitespace,
 * or whose first non-whitespace character is '#', is treated
 * as a comment.
 *
 * An index is an ip address, as parsed by atoaddr.
 *
 * A key is a sequence of characters with a " before and after.
 * The none of the characters may be a newline or a ".
 *
 * Once "including" gets to too great a depth, all processing
 * stops to prevent infinite looping.  This is indicated by
 * process_secrets_file returning FALSE.
 */
static int
globugh(const char *epath, int eerrno)
{
    log_errno_routine(eerrno, "problem with secrets file \"%s\"", epath);
    return 1;	/* stop glob */
}

static bool
process_secrets_file(const char *file_pat, int depth)
{
    char buffer[2049];    /* note: one extra char for our use */
    int lino = 0;
    FILE *fp;
    struct secret *s = NULL;
    bool ok = TRUE;
    glob_t globbuf;
    char **fnp;
    

    if (depth > 10)
    {
	log("preshared secrets file \"%s\" nested too deeply", file_pat);
	return TRUE;
    }

    /* do globbing */
    {
	int r = glob(file_pat, GLOB_ERR, globugh, &globbuf);

	if (r != 0)
	{
	    switch (r)
	    {
	    case GLOB_NOSPACE:
		log("out of space processing secrets filename \"%s\"", file_pat);
		break;
	    case GLOB_ABORTED:
		break;	/* already logged */
	    case GLOB_NOMATCH:
		log("no secrets filename matched \"%s\"", file_pat);
		break;
	    default:
		log("unknown glob error %d", r);
		break;
	    }
	    globfree(&globbuf);
	    return TRUE;
	}
    }

    for (fnp = globbuf.gl_pathv; ok && *fnp != NULL; fnp++)
    {
	fp = fopen(*fnp, "r");
	if (fp == (FILE *) NULL)
	{
	    log_errno((e, "could not open \"%s\"", *fnp));
	    break;
	}

	log("loading secrets from \"%s\"", *fnp);

	/* We really need a co-routine structure, but we don't have it.
	 * This loops once for each line.  If s is NULL, we are between
	 * entries; if it is non-NULL, we are in the middle of one.
	 */
	while (ok && fgets(buffer, sizeof(buffer)-1, fp) != (char *) NULL)
	{
	    char *p;

	    lino++;

	    /* strip trailing whitespace, including \n */

	    for (p = buffer+strlen(buffer); p>buffer && isspace(p[-1]); p--)
		;
	    *p = '\0';

	    p = buffer;

	    if (s == NULL)
	    {
		/* between entries -- check for include */
		static const char include[] = "include";

		if (strncmp(p, include, sizeof(include)-1) == 0
		&& isspace(p[sizeof(include)-1]))
		{
		    char *q;
		    char *prefix_end = NULL;

		    /* find start of filename */
		    p += sizeof(include)-1;
		    while (isspace(*p))
			p++;

		    /* find end of filename */
		    for (q = p; *q != '\0' && !isspace(*q); q++)
			;
		    if (isspace(*q))
		    {
			/* we find whitespace: better be a comment */
			*q = '\0';	/* truncate name */
			do q++; while (isspace(*q));    /* skip whitespace */
			if (*q != '#')
			{
			    log("\"%s\" line %d: include filename malformed"
				, *fnp, lino);
			    continue;   /* ignore directive */
			}
		    }

		    if (*p != '/')
			prefix_end = strrchr(*fnp, '/');
		    if (prefix_end != NULL)
		    {
			/* jam directory prefix of old filename on front of new */
			char fn[sizeof(buffer)];
			if (prefix_end - *fnp + 1 + strlen(p) + 1 > sizeof(fn))
			{
			    log("\"%s\" line %d: include filename too long"
				, *fnp, lino);
			}
			else
			{
			    sprintf(fn, "%.*s/%s"
				, prefix_end - *fnp, *fnp, p);
			    ok = process_secrets_file(fn, depth + 1);
			}
		    }
		    else
		    {
			ok = process_secrets_file(p, depth + 1);
		    }
		    continue;	/* done with this line */
		}
	    }
	    else
	    {
		/* within an entry -- demand indentation */
		if (!isspace(*p))
		    log("\"%s\" line %d: line in middle of entry is not indented"
			, *fnp, lino);
	    }

	    for (;;)
	    {
		/* process a token: index or secret */

		/* eat intra-token whitespace */
		while (isspace(*p))
		    p++;

		if (*p == '\0' || *p == '#')
		    break;	/* end of line or start of comment -- this record is done */

		if (*p != '"')
		{
		    /* an index */
		    char *q, under;
		    const char *ugh;
		    struct in_addr host;

		    for (q = p; *q != '\0' && !isspace(*q); q++)
			;
		    under = *q;
		    *q = '\0';
		    ugh = atoaddr(p, 0, &host);

		    if (ugh != NULL)
		    {
			log("\"%s\" line %d: %s \"%s\""
			    , *fnp, lino, ugh, p);
		    }
		    else
		    {
			struct secret_index *i = alloc_thing(struct secret_index, "index");

			i->type = ID_IPV4_ADDR;
			i->ipv4 = host;
			i->name = NULL;
			if (s == NULL)
			{
			    /* start a new secret */
			    s = alloc_thing(struct secret, "secret");
			    s->indices = NULL;
			    s->len = 0;
			    s->secret = NULL;
			    s->next = NULL;
			}
			i->next = s->indices;
			s->indices = i;
		    }
		    *q = under;
		    p = q;
		}
		else
		{
		    /* a secret */
		    char *q = strchr(p+1, *p);

		    if (q == NULL)
		    {
			log("\"%s\" line %d: unterminated secret string"
			    , *fnp, lino);
			/* as a fix-up, pretend " at end */
			q = p + strlen(p);
			*q = *p;
			q[1] = '\0';
		    }

		    if (s == NULL)
		    {
			log("\"%s\" line %d: secret found without an index"
			    , *fnp, lino);
		    }
		    else
		    {
			s->len = q - (p+1);
			s->secret = clone_bytes(p+1, s->len, "secret text");
			s->next = secrets;
			secrets = s;
			s = NULL;
		    }
		    p = q + 1;
		}
	    }
	}

	if (s != NULL)
	    log("\"%s\" ended in the middle of an entry.", *fnp);
	fclose(fp);
    }

    globfree(&globbuf);
    return ok;
}

void
free_preshared_secrets(void)
{
    if (secrets != NULL)
    {
	struct secret *s, *ns;

	log("forgetting secrets");

	for (s = secrets; s != NULL; s = ns)
	{
	    struct secret_index *i, *ni;

	    ns = s->next;	/* grab before freeing s */
	    for (i = s->indices; i != NULL; i = ni)
	    {
		ni = i->next;	/* grab before freeing n */
		if (i->name != NULL)
		    pfree(i->name);
		pfree(i);
	    }
	    pfree(s->secret);
	    pfree(s);
	}
	secrets = NULL;
    }
}

void
load_preshared_secrets(void)
{
    free_preshared_secrets();
    (void) process_secrets_file(shared_secrets_file, 1);
}
