/* error logging functions
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: log.c,v 1.24 1999/04/11 00:44:20 dhr Exp $
 */

#include <stdio.h>
#include <syslog.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "log.h"
#include "state.h"
#include "connections.h"    /* needed by whack.h */
#include "whack.h"

bool
    log_to_stderr = TRUE,	/* should log go to stderr? */
    log_to_syslog = TRUE;	/* should log go to syslog? */

/* A copy of any logged message will be sent to whack via this
 * file descriptor, if it isn't NULL_FD.
 * This global variable must be carefully adjusted at transaction boundaries!
 */
int whack_log_fd = NULL_FD;

/* Global variables: must be carefully adjusted at transactionn boundaries! */
struct state *cur_state = NULL;	/* current state, for diagnostics */
struct connection *cur_connection = NULL;	/* current connection, for diagnostics */
const struct sockaddr_in *cur_from = NULL;	/* source of current current message */

void
init_log(void)
{
    if (log_to_stderr)
	setbuf(stderr, NULL);
    if (log_to_syslog)
	openlog("Pluto", LOG_CONS | LOG_NDELAY | LOG_PID, LOG_AUTHPRIV);
}

void
close_log(void)
{
    if (log_to_syslog)
	closelog();
}

#include <stdarg.h>

static void
fmt_log(char *buf, size_t buf_len, const char *fmt, va_list ap)
{
    size_t ps;

    if (cur_state != NULL)
	snprintf(buf, buf_len, "\"%s\" #%lu: "
	    , cur_state->st_connection->name, cur_state->st_serialno);
    else if (cur_connection != NULL)
	snprintf(buf, buf_len, "\"%s\": ", cur_connection->name);
    else if (cur_from != NULL)
	snprintf(buf, buf_len, "packet from %s: ", show_sa_in(cur_from));
    else
	buf[0] = '\0';
    ps = strlen(buf);
    vsnprintf(buf + ps, buf_len - ps, fmt, ap);
}

void
log(const char *message, ...)
{
    va_list args;
    char m[1024];

    va_start(args, message);
    fmt_log(m, sizeof(m), message, args);
    va_end(args);

    if (log_to_stderr)
	fprintf(stderr, "%s\n", m);
    if (log_to_syslog)
	syslog(LOG_WARNING, "%s", m);

    if (whack_log_fd != NULL_FD)
	whack_log(whack_log_fd, RC_LOG, "%s", m);
}

void
log_errno_routine(int e, const char *message, ...)
{
    va_list args;
    char m[1024];

    va_start(args, message);
    fmt_log(m, sizeof(m), message, args);
    va_end(args);

    if (log_to_stderr)
	fprintf(stderr, "ERROR: %s. Errno %d: %s\n", m, e, strerror(e));
    if (log_to_syslog)
	syslog(LOG_ERR, "ERROR: %s. Errno %d: %s", m, e, strerror(e));

    if (whack_log_fd != NULL_FD)
	whack_log(whack_log_fd, RC_LOG
	    , "ERROR: %s. Errno %d: %s", m, e, strerror(e));
}

void
exit_log(const char *message, ...)
{
    va_list args;
    char m[1024];

    va_start(args, message);
    fmt_log(m, sizeof(m), message, args);
    va_end(args);

    if (log_to_stderr)
	fprintf(stderr, "FATAL ERROR: %s\n", m);
    if (log_to_syslog)
	syslog(LOG_ERR, "FATAL ERROR: %s", m);

    if (whack_log_fd != NULL_FD)
	whack_log(whack_log_fd, RC_LOG, "FATAL ERROR: %s", m);

    exit_pluto(1);
}

void
exit_log_errno_routine(int e, const char *message, ...)
{
    va_list args;
    char m[1024];

    va_start(args, message);
    fmt_log(m, sizeof(m), message, args);
    va_end(args);

    if (log_to_stderr)
	fprintf(stderr, "FATAL ERROR: %s. Errno %d: %s\n", m, e, strerror(e));
    if (log_to_syslog)
	syslog(LOG_ERR, "FATAL ERROR: %s. Errno %d: %s", m, e, strerror(e));

    if (whack_log_fd != NULL_FD)
	whack_log(whack_log_fd, RC_LOG
	    , "FATAL ERROR: %s. Errno %d: %s", m, e, strerror(e));

    exit_pluto(1);
}

/* emit message to whack.
 * form is "ddd statename text" where
 * - ddd is a decimal status code (RC_*) as described in whack.h
 * - text is a human-readable annotation
 */
void
whack_log(int whackfd, int mess_no, const char *message, ...)
{
    if (whackfd != NULL_FD)
    {
	va_list args;
	char m[1024];
	int prelen = snprintf(m, sizeof(m), "%03d ", mess_no);
	size_t len;

	passert(prelen >= 0);

	va_start(args, message);
	fmt_log(m+prelen, sizeof(m)-prelen, message, args);
	va_end(args);

	len = strlen(m);
	m[len] = '\n';	/* don't need NUL, do need NL */

	{
	    void (*old_sigpipe)(int) = signal(SIGPIPE, SIG_IGN);

	    write(whackfd, m, len + 1);
	    signal(SIGPIPE, old_sigpipe);
	}
    }
}

/*
 * This routine returns a user readable form of an address contained in a
 * sockaddr structure. The return value CAN be a statically allocated
 * object (as is the case with inet_ntoa()).
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

char *
get_address_in(struct sockaddr_in sin)
{
    static char mess[SOCKADDR_STRING_SIZE];

    switch (sin.sin_family)
    {
	case AF_INET:
	    return inet_ntoa(sin.sin_addr);

	default:
	    snprintf(mess, sizeof(mess), "(UNEXPECTED address family %d)",
		sin.sin_family);
	    return mess;
    }
}

char *
get_address(struct sockaddr sa)
{
    static char mess[SOCKADDR_STRING_SIZE];
    struct sockaddr_in sin;

    switch (sa.sa_family)
    {
	case AF_INET:
	    memcpy(&sin, &sa, sizeof(sa));
	    return get_address_in(sin);

	default:
	    snprintf(mess, sizeof(mess), "(unknown address family %d)",
		sa.sa_family);
	    return mess;
    }
}

/*
 * Return a port (if applicable), from a struct sockaddr. If not applicable,
 * return -1.
 */

int
get_port_in(struct sockaddr_in sin)
{
    switch (sin.sin_family)
    {
	case AF_INET:
	    return ntohs(sin.sin_port);

	default:
	    return -1;
    }
}

int
get_port(struct sockaddr sa)
{
    struct sockaddr_in sin;

    switch (sa.sa_family)
    {
	case AF_INET:
	    memcpy(&sin, &sa, sizeof(sa));
	    return get_port_in(sin);

	default:
	    return -1;
    }
}

/* format address and port together in a static buffer */

const char *
show_sa(const struct sockaddr *sa)
{
    static char buf[SOCKADDR_STRING_SIZE];

    switch (sa->sa_family)
    {
	case AF_INET:
	    snprintf(buf, sizeof(buf), "%s, port %d",
		get_address(*sa), get_port(*sa));
	    break;

	default:
	    snprintf(buf, sizeof(buf), "(unexpected address family %d)",
		sa->sa_family);
	    break;
    }
    return buf;
}

const char *
show_sa_in(const struct sockaddr_in *sin)
{
    return show_sa((struct sockaddr *)sin);
}


/* Debugging message support */

#ifdef DEBUG

unsigned int debugging = DBG_NONE;	/* default to reporting nothing */

/* log a debugging message (prefixed by "| ") */

void
DBG_log(char *message, ...)
{
    va_list args;
    char m[1024];

    va_start(args, message);
    vsnprintf(m, sizeof(m), message, args);
    va_end(args);

    if (log_to_stderr)
	fprintf(stderr, "| %s\n", m);
    if (log_to_syslog)
	syslog(LOG_DEBUG, "| %s", m);
}

/* dump raw bytes in hex to stderr (for lack of any better destination) */

void
DBG_dump(const char *label, const void *p, size_t len)
{
    char buf[100];
    char *bp;
    const unsigned char *cp = p;
    
    bp = buf;

    if (label != NULL && label[0] != '\0')
    {
	size_t len = strlen(label);

	passert(len + 1 <= sizeof(buf));
	strcpy(buf, label);
	if (buf[len-1] == '\n')
	{
	    buf[len-1] = '\0';	/* get rid of newline */
	    DBG_log("%s", buf);
	}
	else
	{
	    passert(len + 4 * (1 + 4 * 3) + 1 <= sizeof(buf));
	    bp = buf + len;
	}
    }

    do {
	int i, j;

	for (i = 0; len!=0 && i!=4; i++) {
	    *bp++ = ' ';
	    for (j = 0; len!=0 && j!=4; len--, j++)
	    {
		static const char hexdig[] = "0123456789abcdef";

		*bp++ = ' ';
		*bp++ = hexdig[(*cp >> 4) & 0xF];
		*bp++ = hexdig[*cp & 0xF];
		cp++;
	    }
	}
	*bp = '\0';
	DBG_log("%s", buf);
	bp = buf;
    } while (len != 0);
}

#endif /* DEBUG */
