/*
 * All-in-one program to set Security Association parameters
 * Copyright (C) 1996  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char spi_c_version[] = "RCSID $Id: spi.c,v 1.37 1999/04/11 00:12:08 henry Exp $";

#include <asm/types.h>
#include <sys/types.h>
#include <sys/ioctl.h>
/* #include <linux/netdevice.h> */
#include <net/if.h>
/* #include <linux/types.h> */ /* new */
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

/* #include <sys/socket.h> */

#include <netinet/in.h>
#include <arpa/inet.h>
/* #include <linux/ip.h> */
#include <netdb.h>

#include <unistd.h>
#include <getopt.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include "../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ipe4.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

struct encap_msghdr *em;

char *program_name;
char *command;
extern char *optarg;
extern int optind, opterr, optopt;
char scratch[2];
char *iv = NULL, *enckey = NULL, *authkey = NULL;
size_t ivlen = 0, enckeylen = 0, authkeylen = 0;
struct in_addr edst, dst, src;
unsigned char proto = 0;
int alg = 0;
int replay_window = 0;
char *iv_opt, *akey_opt, *ekey_opt, *alg_opt, *edst_opt, *spi_opt, *proto_opt, *said_opt, *dst_opt, *src_opt;
char sa[SATOA_BUF];

#define streql(_a,_b) (!strcmp((_a),(_b)))

inline int x2i(char *s)
{
	char ss[3];
	char *endptr;
	__u32 n;
	ss[0] = s[0];
	ss[1] = s[1];
	ss[2] = 0;

	n = strtoul(ss, &endptr, 16);
	if(!(endptr == ss + strlen(ss)))
	{
		fprintf(stderr, "%s: Invalid character in numerical input stream: %s\n",
			program_name, ss);
		exit (1);
	}
	return n;
}

static char *usage_string = "\
Usage: spi\n\
	in the following, <SA> is: --edst <dstaddr> --spi <spi> --proto <proto>\n\
spi --clear\n\
spi --help\n\
spi --version\n\
spi --del\n\
spi --ip4 <SA> --src <encap-src> --dst <encap-dst>\n\
spi --ah md5 <SA> --authkey <key>\n\
spi --ah <algo> <SA> [ --replay_window <replay_window> ] --authkey <key>\n\
	where <algo> is one of:	hmac-md5-96 | hmac-sha1-96\n\
spi --esp des-old <SA> --iv <iv> --authkey <key>\n\
spi --esp <algo> <SA> [ --replay_window <replay-window> ] --enckey <ekey> --authkey <akey>\n\
	where <algo> is one of:	des-md5 | 3des-md5\n\
				des-md5-96 | 3des-md5-96\n\
				des-sha1-96 | 3des-sha1-96\n\
spi --esp <algo> <SA> [ --replay_window <replay-window> ] --authkey <akey>\n\
	where <algo> is one of:	null-md5-96 | null-sha1-96 \n\
spi --esp <algo> <SA> [ --replay_window <replay-window> ] --iv <iv> --enckey <ekey>\n\
	where <algo> is one of:	des | 3des\n\
";
/* --iv <iv> */

struct cmd
{
	char *cm_name;
	int (*cm_nxtok)(int, char *([]), struct cmd *);
	int cm_alg;			/* see ipsec_xform.h */
};

int f_ip4(void);
int f_ah(void);
int f_esp(void);
int f_md5(void);
int f_hmac(void);
int f_des(void);
int f_blkrply(void);
int f_del(void);
int f_clr(void);

void
usage(char *s, FILE *f)
{
	/* s argument is actually ignored, at present */
	fprintf(f, "%s", usage_string);
	exit(-1);
}

int
f_ip4(void)
{
	struct ipe4_xdata *xd;

	if(dst.s_addr == INADDR_ANY) {
		fprintf(stderr, "%s: tunnel destination address invalid or not specified for SA:%s.\n",
			program_name, sa);
	}
	if(src.s_addr == INADDR_ANY) {
		fprintf(stderr, "%s: tunnel source address invalid or not specified for SA:%s.\n",
			program_name, sa);
	}
	em = malloc(sizeof(struct encap_msghdr) + sizeof(struct ipe4_xdata));
	if(em == NULL) {
		fprintf(stderr, "%s: Memory allocation error.\n", program_name);
		exit(1);
	}

	xd = (struct ipe4_xdata *)(em->em_dat);

	em->em_msglen = EMT_SETSPI_FLEN + EMT_IPE4_ULEN;
	if(proto != SA_IPIP) {
		fprintf(stderr, "%s: SA protocol selector (tun) does not agree with the tunnelling algorithm requested (%s) for SA:%s.\n",
			program_name, alg_opt, sa);
	}
	em->em_alg = XF_IP4;
	
	xd->i4_src = src;
	xd->i4_dst = dst;
	return 0;
}

int
f_del(void)
{
	int i;

	if((em = malloc(sizeof(struct encap_msghdr))) == NULL) {
		fprintf(stderr, "%s: Memory allocation error.\n", program_name);
		exit(1);
	}

	em->em_msglen = EMT_SETSPI_FLEN;
	em->em_alg = alg;
	
	return 0;
}

int
f_clr(void)
{
	int i;

	if((em = malloc(sizeof(struct encap_msghdr))) == NULL) {
		fprintf(stderr, "%s: Memory allocation error.\n", program_name);
		exit(1);
	}
	em->em_msglen = EMT_SETSPI_FLEN;
	em->em_alg = alg;
	
	return 0;
}

int
f_md5(void)
{
	struct ahmd5_xdata *xd;

	if(authkeylen > AHMD5_KMAX) {
		fprintf(stderr, "%s: Failed -- (%s) requires an authentication key length < %d bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, alg_opt, AHMD5_KMAX, authkeylen, akey_opt, sa);
#if 0
		fprintf(stderr, "%s: authentication key invalid or not specified -- must be 16 bytes.\n",
			program_name);
#endif
	}
	em = (struct encap_msghdr *)malloc(sizeof(struct encap_msghdr) +
		  sizeof(struct ahmd5_xdata));
	if(em == NULL) {
		fprintf(stderr, "%s: Memory allocation error\n", program_name);
		exit(1);
	}

	xd = (struct ahmd5_xdata *)(em->em_dat);

	em->em_msglen = EMT_SETSPI_FLEN + 2 * sizeof (u_short) + authkeylen;
	if(proto != SA_AH) {
		fprintf(stderr, "%s: SA protocol selector (ah) does not agree with the Authentication algorithm requested (%s) for SA:%s.\n",
			program_name, alg_opt, sa);
	}
	em->em_alg = alg;
	
	xd->amx_klen = authkeylen;
	xd->amx_alen = 16;

	memcpy(&(xd->amx_key), authkey, authkeylen);
	return 0;
}

int
f_hmac(void)
{
	struct ahhmacsha1_edata *xd;
	size_t edatasize = (alg == XF_AHHMACMD5) ?
			      sizeof(struct ahhmacmd5_edata) :
			      sizeof(struct ahhmacsha1_edata);

	em = malloc(sizeof(struct encap_msghdr) + edatasize);
	if(em == NULL) {
		fprintf(stderr, "%s: Memory allocation error\n", program_name);
		exit(1);
	}
#if 0
	if(alg == XF_AHHMACMD5) {
		xd = (struct ahhmacmd5_edata *)(em->em_dat);
	} else {
#endif
		xd = (struct ahhmacsha1_edata *)(em->em_dat);
#if 0
	}
#endif

	if (alg == XF_AHHMACMD5) {
		xd->ame_alen = AHMD596_KLEN;
		if(authkeylen != AHMD596_KLEN) {
			fprintf(stderr, "%s: Failed -- (%s) requires an authentication key length of %d bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, alg_opt, AHMD596_KLEN, authkeylen, akey_opt, sa);
			exit(1);
		}
	} else 	{
		xd->ame_alen = AHSHA196_KLEN;
		if(authkeylen != AHSHA196_KLEN) {
			fprintf(stderr, "%s: Failed -- (%s) requires an authentication key length of %d bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, alg_opt, AHSHA196_KLEN, authkeylen, akey_opt, sa);
			exit(1);
		}
	}

	em->em_msglen = EMT_SETSPI_FLEN + 3 * sizeof (u_short) + 2 * sizeof (u_char) + authkeylen;
	if(proto != SA_AH) {
		fprintf(stderr, "%s: SA protocol selector (ah) does not agree with the Authentication algorithm requested (%s) for SA:%s.\n",
			program_name, alg_opt, sa);
	}
	em->em_alg = alg;

	xd->ame_klen = authkeylen;
	xd->ame_x0 = 0;

	if (replay_window) {
		xd->ame_replayp = 1;
		xd->ame_ooowin = replay_window;
	}
	else
		xd->ame_replayp = xd->ame_ooowin = 0;

	memcpy(&(xd->ame_key), authkey, authkeylen);
	return 0;
}

int
f_des(void)
{
	struct espdesold_xdata *xd;
	int i, dlen;
	
	if((em = malloc(sizeof(struct encap_msghdr) + sizeof(struct espdesold_xdata))) == NULL) {
		fprintf(stderr, "%s: Memory allocation error\n", program_name);
		exit(1);
	}
	em->em_msglen = EMT_SETSPI_FLEN + EMT_ESPDESCBC_ULEN;
	if(proto != SA_ESP) {
		fprintf(stderr, "%s: SA protocol selector (esp) does not agree with the Encryption algorithm requested (%s) for SA:%s.\n",
			program_name, alg_opt, sa);
	}
	em->em_alg = XF_ESPDESOLD;

	if ((ivlen != 4) && (ivlen != 8)) {
		fprintf(stderr, "%s: Failed -- requires IV lengths of 4 or 8 bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
			program_name, ivlen, iv_opt, sa);
		exit(1);
	}
	
	xd = (struct espdesold_xdata *)em->em_dat;
	xd->edx_ivlen = ivlen;

	memcpy(&(xd->edx_iv), iv, ivlen);

	if (enckeylen != 8) {
		fprintf(stderr, "%s: Failed -- requires a key length of 8 bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
			program_name, enckeylen, ekey_opt, sa);
		exit(1);
	}
	
	memcpy(&(xd->edx_rk), enckey, enckeylen);
	return 0;
}

int
f_blkrply(void)
{
	int i, dlen, eklen, wl=1, dir;
	struct espblkrply_edata *ed;
	char *endptr;

	if((em = malloc(sizeof(struct encap_msghdr) +
		       sizeof(struct espblkrply_edata))) == NULL) {
		fprintf(stderr, "%s: Memory allocation error\n", program_name);
		exit(1);
	}
	ed = (struct espblkrply_edata *)(em->em_dat);

	ed->eme_flags = 0;
        if ((alg != XF_ESPNULLMD596) && (alg != XF_ESPNULLSHA196)) {
#if 0
		if ((ivlen != 8) ) {
			fprintf(stderr, "%s: Failed -- expects an IV length of exactly 8 bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, ivlen, iv_opt, sa);
			exit(1);
		}
		ed->eme_ivlen = ivlen;

		memcpy(&(ed->eme_iv), iv, ivlen);
#endif	
		switch(alg) {
		case XF_ESPDESMD5:
		case XF_ESPDESMD596:
		case XF_ESPDESSHA196:
		case XF_ESPDES:
			eklen = EMT_ESPDES_KEY_SZ;
			break;
		case XF_ESP3DESMD5:
		case XF_ESP3DESMD596:
		case XF_ESP3DESSHA196:
		case XF_ESP3DES:
			eklen = EMT_ESP3DES_KEY_SZ;
			break;
		}
		if ( (enckeylen != eklen ) ) {
			fprintf(stderr, "%s: Failed -- expects an encryption key of exactly %d bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, eklen, enckeylen, ekey_opt, sa);
			exit(1);
		}
		ed->eme_klen = enckeylen;
		memcpy(&(ed->eme_key), enckey, enckeylen);
	}
	
	switch(alg) {
	case XF_ESPDESMD5:
	case XF_ESP3DESMD5:

	case XF_ESPDESMD596:
	case XF_ESP3DESMD596:
	case XF_ESPNULLMD596:
		if ((authkeylen != AHMD596_KLEN) ) {
			fprintf(stderr, "%s: Failed -- MD5 expects an authentication key of exactly %d bytes, found %d in '%s' for SA:%s (1 byte = 2 hexadecimal digits).\n",
				program_name, AHMD596_KLEN, authkeylen, akey_opt, sa);
			exit(1);
		}
		break;
	case XF_ESPDESSHA196:
	case XF_ESP3DESSHA196:
	case XF_ESPNULLSHA196:
		if ((authkeylen != AHSHA196_KLEN) ) {
			fprintf(stderr, "%s: Failed -- SHA1 expects an authentication key of exactly %d bytes, found %d in '%s' (1 byte = 2 hexadecimal digits).\n",
				program_name, AHSHA196_KLEN, authkeylen, akey_opt);
			exit(1);
		}
		break;
	case XF_ESPDES:
	case XF_ESP3DES:
		authkeylen = 0;
		break;
	}
	ed->ame_klen = authkeylen;
	if(authkeylen) {
		memcpy(&(ed->ame_key), authkey, authkeylen);
	}
	if(proto != SA_ESP) {
		fprintf(stderr, "%s: SA protocol selector (esp) does not agree with the Encryption algorithm requested (%s) for SA:%s.\n",
			program_name, alg_opt, sa);
	}
	em->em_alg = alg;
	em->em_msglen = EMT_SETSPI_FLEN + sizeof (struct espblkrply_edata);
	ed->eme_ooowin = replay_window;

	return 0;
}

int
resolve_ip(char *arg, struct in_addr *ip)
{
	struct hostent *hostp;
	struct netent *netp;
	
	if (!strcmp(arg, "default")) {/* default is really 0.0.0.0 */
		ip->s_addr = htonl(INADDR_ANY);
		return(1);
	}
	if (!(strcmp(arg, "255.255.255.255"))) {
		/* inet_addr() returns -1 for this, as it should, but that means an error */
		ip->s_addr = htonl(INADDR_BROADCAST);
		return (1);
	}
	if ((netp = getnetbyname(arg)) != (struct netent *)NULL) {
		ip->s_addr = htonl(netp->n_net);
		return(1);
	}
	if ((hostp = gethostbyname(arg)) == (struct hostent *)NULL) {
		errno = h_errno;
		return(-1);
	}
	memcpy((char *) ip, (char *) hostp->h_addr_list[0], hostp->h_length);
	return(0);
}

static struct option const longopts[] =
{
	{"ah", 1, 0, 'H'},
	{"esp", 1, 0, 'P'},
	{"ip4", 0, 0, '4'},
	{"del", 0, 0, 'd'},

	{"authkey", 1, 0, 'A'},
	{"enckey", 1, 0, 'E'},
	{"edst", 1, 0, 'e'},
	{"spi", 1, 0, 's'},
	{"proto", 1, 0, 'p'},
	{"replay_window", 1, 0, 'w'},
	{"iv", 1, 0, 'i'},
	{"dst", 1, 0, 'D'},
	{"src", 1, 0, 'S'},
	{"said", 1, 0, 'I'},

	{"help", 0, 0, 'h'},
	{"version", 0, 0, 'v'},
	{"clear", 0, 0, 'c'},
	{"label", 1, 0, 'l'},
	{"optionsfrom", 1, 0, '+'},
	{0, 0, 0, 0}
};

int
main(int argc, char *argv[])
{
	int fd = 0;
	char *endptr;
	__u32 spi = 0;
	int c, previous = -1;
	int ret;
	struct sa_id said;
	size_t sa_len;
	const char* error_s;

	edst.s_addr = 0, dst.s_addr = 0, src.s_addr = 0;

	program_name = argv[0];

	while((c = getopt_long(argc, argv, "H:P:4dcA:E:e:s:w:i:D:S:hvl:+:", longopts, 0)) != EOF)
	{
		switch(c) {
		case 'H':
			if(alg) {
				fprintf(stderr, "%s: Only one of '--ah', '--esp', '--ip4', '--del' or '--clear'  options permitted.\n",
					program_name);
				exit(1);
			}
			if(!strcmp(optarg, "md5")) {
				alg = XF_AHMD5;
			} else if(!strcmp(optarg, "hmac-md5")) {
				alg = XF_AHHMACMD5;
			} else if(!strcmp(optarg, "hmac-md5-96")) {
				alg = XF_AHHMACMD5;
			} else if(!strcmp(optarg, "hmac-sha1")) {
				alg = XF_AHHMACSHA1;
			} else if(!strcmp(optarg, "hmac-sha1-96")) {
				alg = XF_AHHMACSHA1;
			} else {
				fprintf(stderr, "%s: Unknown authentication algorithm '%s' follows '--ah' option.\n",
					program_name, optarg);
				exit(1);
			}
#ifdef DEBUG
			fprintf(stdout, "Algorithm %d selected.\n", alg);
#endif
			alg_opt = optarg;
			break;
		case 'P':
			if(alg) {
				fprintf(stderr, "%s: Only one of '--ah', '--esp', '--ip4', '--del' or '--clear'  options permitted.\n",
					program_name);
				exit(1);
			}
			if       (!strcmp(optarg, "des-old")) {
				alg = XF_ESPDESOLD;
			} else if(!strcmp(optarg, "des-md5")) {
				alg = XF_ESPDESMD5;
			} else if(!strcmp(optarg, "3des-md5")) {
				alg = XF_ESP3DESMD5;
			} else if(!strcmp(optarg, "des-md5-96")) {
				alg = XF_ESPDESMD596;
			} else if(!strcmp(optarg, "3des-md5-96")) {
				alg = XF_ESP3DESMD596;
			} else if(!strcmp(optarg, "null-md5-96")) {
				alg = XF_ESPNULLMD596;
			} else if(!strcmp(optarg, "null-sha1-96")) {
				alg = XF_ESPNULLSHA196;
			} else if(!strcmp(optarg, "des-sha1-96")) {
				alg = XF_ESPDESSHA196;
			} else if(!strcmp(optarg, "3des-sha1-96")) {
				alg = XF_ESP3DESSHA196;
			} else if(!strcmp(optarg, "des")) {
				alg = XF_ESPDES;
			} else if(!strcmp(optarg, "3des")) {
				alg = XF_ESP3DES;
			} else {
				fprintf(stderr, "%s: Invalid encryption algorithm '%s' follows '--esp' option.\n",
					program_name, optarg);
				exit(1);
			}
#ifdef DEBUG
			fprintf(stdout, "Algorithm %d selected.\n", alg);
#endif
			alg_opt = optarg;
			break;
		case '4':
			if(alg) {
				fprintf(stderr, "%s: Only one of '--ah', '--esp', '--ip4', '--del' or '--clear' options permitted.\n",
					program_name);
				exit(1);
			}
		       	alg = XF_IP4;
#ifdef DEBUG
			fprintf(stdout, "Algorithm %d selected.\n", alg);
#endif
			alg_opt = optarg;
			break;
		case 'd':
			if(alg) {
				fprintf(stderr, "%s: Only one of '--ah', '--esp', '--ip4', '--del' or '--clear'  options permitted.\n",
					program_name);
				exit(1);
			}
			alg = XF_DEL;
#ifdef DEBUG
			fprintf(stdout, "Algorithm %d selected.\n", alg);
#endif
			alg_opt = optarg;
			break;
		case 'c':
			if(alg) {
				fprintf(stderr, "%s: Only one of '--ah', '--esp', '--ip4', '--del' or '--clear'  options permitted.\n",
					program_name);
				exit(1);
			}
			alg = XF_CLR;
#ifdef DEBUG
			fprintf(stdout, "Algorithm %d selected.\n", alg);
#endif
			alg_opt = optarg;
			break;
		case 'e':
			if(said_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(edst_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, edst_opt);
				exit (1);
			}				
			if(ret = resolve_ip(optarg, &edst) != 0) {
				if(ret == 1) {
					fprintf(stderr, "%s: Destination address cannot be a network: %s\n",
						program_name, optarg);
					exit (1);
				} else {
					fprintf(stderr, "%s: Invalid formation of destination address: %s\n",
						program_name, optarg);
					exit (1);
				}
			}
			edst_opt = optarg;
			break;
		case 's':
			if(said_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(spi_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, spi_opt);
				exit (1);
			}				
			spi = strtoul(optarg, &endptr, 0);
			if(!(endptr == optarg + strlen(optarg))) {
				fprintf(stderr, "%s: Invalid character in SPI parameter: %s\n",
					program_name, optarg);
				exit (1);
			}
			if(spi < 0x100) {
				fprintf(stderr, "%s: Illegal reserved spi: %s => 0x%x Must be larger than 0x100.\n",
					program_name, optarg, spi);
				exit(1);
			}
			spi_opt = optarg;
			break;
		case 'p':
			if(said_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(proto_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, proto_opt);
				exit (1);
			}
			if(!strcmp(optarg, "ah"))
				proto = SA_AH;
			if(!strcmp(optarg, "esp"))
				proto = SA_ESP;
			if(!strcmp(optarg, "tun"))
				proto = SA_IPIP;
			if(proto == 0) {
				fprintf(stderr, "%s: Invalid PROTO parameter: %s\n",
					program_name, optarg);
				exit (1);
			}
			proto_opt = optarg;
			break;
		case 'I':
			if(said_opt) {
				fprintf(stderr, "%s: Error, SAID parameter redefined:%s, already defined in SA:%s\n",
					program_name, optarg, said_opt);
				exit (1);
			}				
			if(proto_opt) {
				fprintf(stderr, "%s: Error, PROTO parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, proto_opt);
				exit (1);
			}
			if(edst_opt) {
				fprintf(stderr, "%s: Error, EDST parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, edst_opt);
				exit (1);
			}
			if(proto_opt) {
				fprintf(stderr, "%s: Error, SPI parameter redefined in SA:%s, already defined as:%s\n",
					program_name, optarg, spi_opt);
				exit (1);
			}
			if((error_s = atosa(optarg, 0, &said)) != NULL) {
				fprintf(stderr, "%s: Error, %s converting --sa argument:%s\n",
					program_name, error_s, optarg);
				exit (1);
			}
			said_opt = optarg;
			break;
		case 'A':
			if(optarg[0] == '0') {
				switch(optarg[1]) {
				case 't':
				case 'x':
				case 's':
					break;
				default:
					fprintf(stderr, "%s: Authentication key must have a '0x', '0t' or '0s' prefix to select the format: %s\n",
						program_name, optarg);
					exit(1);
				}
			}
			authkeylen = atodata(optarg, 0, NULL, 0);
			if(!authkeylen) {
				fprintf(stderr, "%s: unknown format or syntax error in authentication key: %s\n",
					program_name, optarg);
				exit (1);
			}
			authkey = malloc(authkeylen);
			if(authkey == NULL) {
				fprintf(stderr, "%s: Memory allocation error.\n", program_name);
				exit(1);
			}
			memset(authkey, 0, authkeylen);
			authkeylen = atodata(optarg, 0, authkey, authkeylen);
			akey_opt = optarg;
			break;
		case 'E':
			if(optarg[0] == '0') {
				switch(optarg[1]) {
				case 't':
				case 'x':
				case 's':
					break;
				default:
					fprintf(stderr, "%s: Encryption key must have a '0x', '0t' or '0s' prefix to select the format: %s\n",
						program_name, optarg);
					exit(1);
				}
			}
			enckeylen = atodata(optarg, 0, NULL, 0);
			if(!enckeylen) {
				fprintf(stderr, "%s: unknown format or syntax error in encryption key: %s\n",
					program_name, optarg);
				exit (1);
			}
			enckey = malloc(enckeylen);
			if(enckey == NULL) {
				fprintf(stderr, "%s: Memory allocation error.\n", program_name);
				exit(1);
			}
			memset(enckey, 0, enckeylen);
			enckeylen = atodata(optarg, 0, enckey, enckeylen);
			ekey_opt = optarg;
			break;
		case 'w':
			replay_window = strtoul(optarg, &endptr, 0);
			if(!(endptr == optarg + strlen(optarg))) {
				fprintf(stderr, "%s: Invalid character in replay_window parameter: %s\n",
					program_name, optarg);
				exit (1);
			}
			if((replay_window < 0x1) || (replay_window > 64)) {
				fprintf(stderr, "%s: Failed -- Illegal window size: %s  Must be 1 <= size <= 64.\n",
					program_name, optarg);
				exit(1);
			}
			break;
		case 'i':
			if(optarg[0] == '0') {
				switch(optarg[1]) {
				case 't':
				case 'x':
				case 's':
					break;
				default:
					fprintf(stderr, "%s: IV must have a '0x', '0t' or '0s' prefix to select the format.\n",
						program_name, optarg);
					exit(1);
				}
			}
			ivlen = atodata(optarg, 0, NULL, 0);
			if(!ivlen) {
				fprintf(stderr, "%s: unknown format or syntax error in IV: %s\n",
					program_name, optarg);
				exit (1);
			}
			iv = malloc(ivlen);
			if(iv == NULL) {
				fprintf(stderr, "%s: Memory allocation error.\n", program_name);
				exit(1);
			}
			memset(iv, 0, ivlen);
			ivlen = atodata(optarg, 0, iv, ivlen);
			iv_opt = optarg;
			break;
		case 'D':
			if(dst_opt) {
				fprintf(stderr, "%s: Error, DST parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, dst_opt);
				exit (1);
			}				
			if(ret = resolve_ip(optarg, &dst) != 0) {
				if(ret == 1) {
					fprintf(stderr, "%s: Tunnel destination address cannot be a network: %s\n",
						program_name, optarg);
					exit (1);
				} else {
					fprintf(stderr, "%s: Invalid formation of tunnel destination address: %s\n",
						program_name, optarg);
					exit (1);
				}
			}
			dst_opt = optarg;
			break;
		case 'S':
			if(src_opt) {
				fprintf(stderr, "%s: Error, SRC parameter redefined:%s, already defined as:%s\n",
					program_name, optarg, src_opt);
				exit (1);
			}				
			if(ret = resolve_ip(optarg, &src) != 0) {
				if(ret == 1) {
					fprintf(stderr, "%s: Tunnel source address cannot be a network: %s\n",
						program_name, optarg);
					exit (1);
				} else {
					fprintf(stderr, "%s: Invalid formation of tunnel source address: %s\n",
						program_name, optarg);
					exit (1);
				}
			}
			src_opt = optarg;
			break;
		case 'h':
			usage(program_name, stdout);
			exit(0);
		case '?':
			usage(program_name, stderr);
			exit(1);
		case 'v':
			printf("%s, %s\n", program_name, spi_c_version);
			exit(1);
		case 'l':
			program_name = optarg;
			break;
		case '+': /* optionsfrom */
			optionsfrom(optarg, &argc, &argv, optind, stderr);
			/* no return on error */
			break;
		default:
		}
		previous = c;
	}
#ifdef DEBUG
	printf("All options processed.\n");
#endif
	switch(alg) {
	case XF_IP4:
	case XF_DEL:
	case XF_AHMD5:
	case XF_AHHMACMD5:
	case XF_AHHMACSHA1:
	case XF_ESPDESOLD:
	case XF_ESPDESMD5:
	case XF_ESP3DESMD5:
	case XF_ESPDESMD596:
	case XF_ESP3DESMD596:
	case XF_ESPDESSHA196:
	case XF_ESP3DESSHA196:
	case XF_ESPNULLMD596:
	case XF_ESPNULLSHA196:
	case XF_ESPDES:
	case XF_ESP3DES:
		if(!said_opt) {
			if(!edst.s_addr) {
				fprintf(stderr, "%s: SA destination not specified.\n",
					program_name);
				exit(1);
			}
			if(!spi) {
				fprintf(stderr, "%s: SA SPI not specified.\n",
					program_name);
				exit(1);
			}
			if(!proto) {
				fprintf(stderr, "%s: SA PROTO not specified.\n",
					program_name);
				exit(1);
			}
			said.dst = edst;
			said.spi = htonl(spi);
			said.proto = proto;
		} else {
			proto = said.proto;
		}
		sa_len = satoa(said, 0, sa, SATOA_BUF);

#ifdef DEBUG
		printf("SA valid.\n");
#endif
		break;
	case XF_CLR:
		break;
	default:
		fprintf(stderr, "%s: No action chosen.  See '%s --help' for usage.\n",
			program_name, program_name);
		exit(1);
	}

	switch(alg) {
	case XF_CLR:
		f_clr();
		break;
	case XF_DEL:
		f_del();
		break;
	case XF_IP4:
		f_ip4();
		break;
	case XF_AHMD5:
		f_md5();
		break;
	case XF_AHHMACMD5:
	case XF_AHHMACSHA1:
		f_hmac();
		break;
	case XF_ESPDESOLD:
		f_des();
		break;
	case XF_ESPDESMD5:
	case XF_ESP3DESMD5:
	case XF_ESPDESMD596:
	case XF_ESP3DESMD596:
	case XF_ESPDESSHA196:
	case XF_ESP3DESSHA196:
	case XF_ESPNULLMD596:
	case XF_ESPNULLSHA196:
	case XF_ESPDES:
	case XF_ESP3DES:
		f_blkrply();
		break;
	default:
		fprintf(stderr, "%s: No action chosen.  See '%s --help' for usage.\n",
			program_name, program_name);
		exit(1);
	}
#ifdef DEBUG
	printf("Algorithm ok.\n");
#endif
	if(alg != XF_CLR) {
		em->em_said = said;
	}
	em->em_magic = EM_MAGIC;
	em->em_version = 0;
	em->em_if = 0;
	em->em_flags = 0;

	if(alg == XF_DEL) {
		em->em_type = EMT_DELSPI;
	} else if(alg == XF_CLR) {
		em->em_type = EMT_CLRSPIS;
	} else {
		em->em_type = EMT_SETSPI;
#ifdef DEBUG
		printf("type set to EMT_SETSPI\n");
#endif
	}
#if 1 /* def SA_DIRECTION */
	{
		int inet_sock = -1;                     /* INET socket                  */
		char buff[1024];
		struct ifconf ifc;
		struct ifreq ifr;
		struct ifreq *ifrp;
		int i;

		/* Create a channel to the NET kernel. */
		if ((inet_sock = socket(AF_INET, SOCK_DGRAM, 0) ) < 0) {
			fprintf(stderr, "%s: Trouble openning channel to NET.\n", program_name);
			exit(-1);
		}
		
		ifc.ifc_len = sizeof(buff);
		ifc.ifc_buf = buff;
		if (ioctl(inet_sock, SIOCGIFCONF, &ifc) < 0) {
			fprintf(stderr, "SIOCGIFCONF: %s\n", strerror(errno));
			return;
		}
		ifrp = ifc.ifc_req;
		for (i = ifc.ifc_len / sizeof(struct ifreq); --i >= 0; ifrp++) {
			strcpy(ifr.ifr_name, ifrp->ifr_name);
			if (ioctl(inet_sock, SIOCGIFFLAGS, &ifr) < 0) continue;
			if ((ifr.ifr_flags & IFF_UP) == 0) continue;
			if (ioctl(inet_sock, SIOCGIFADDR, &ifr) < 0) continue;
			if(ifr.ifr_addr.sa_family == AF_INET) {
				struct sockaddr_in *sinp = (struct sockaddr_in*)&(ifr.ifr_addr);
				if(sinp->sin_addr.s_addr == edst.s_addr) {
					em->em_flags |= EMT_INBOUND;
				}
			}
		}
		(void) close(inet_sock);
	}
#endif

#ifdef XDUMP
	xdump((caddr_t)em, em->em_msglen, "SETSPI:");
	exit(1);
#else
	fd = open("/dev/ipsec", O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "%s: Could not open /dev/ipsec -- ", argv[0]);
		switch(errno) {
		case ENOENT:
			fprintf(stderr, "device does not exist.  See FreeS/WAN installation procedure.\n");
		case EACCES:
			fprintf(stderr, "access denied.  ");
			if(getuid() == 0) {
				fprintf(stderr, "Check permissions.  Should be 600.\n");
			} else {
				fprintf(stderr, "You must be root to open this file.\n");
			}
			break;
		case EUNATCH:
			fprintf(stderr, "Netlink not enabled OR KLIPS not loaded.\n");
			break;
		default:
			fprintf(stderr, "Unknown file open error %d.  Please report to development team.\n", errno);
		}
		exit(1);
	}
#ifdef DEBUG
	fprintf(stdout, "%s: message size is %d.\n", program_name, em->em_msglen);
#endif
	if (!(write(fd, (caddr_t)em, em->em_msglen) == em->em_msglen)) {
		fprintf(stderr, "%s: Had trouble writing to /dev/ipsec SA:%s -- ",
			program_name, (alg != XF_CLR ? sa : "none"));
		switch(errno) {
		case EINVAL:
			fprintf(stderr, "Invalid argument, check kernel log messages for specifics.\n");
			break;
		case ENODEV:
			fprintf(stderr, "No device?!?\n");
			break;
		case ENOBUFS:
			fprintf(stderr, "No kernel memory to allocate SA.\n");
			break;
		case ESOCKTNOSUPPORT:
			fprintf(stderr, "Algorithm support not available in the kernel.  Please compile in support.\n");
			break;
		case EEXIST:
			fprintf(stderr, "SA already in use.  Delete old one first.\n");
			break;
		case ENXIO:
			fprintf(stderr, "SA does not exist.  Cannot delete.\n");
			break;
		default:
			fprintf(stderr, "Unknown file write error %d.  Please report to development team\n", errno);
		}
		close(fd);
		exit(1);
	}
#endif
	close(fd);
	if(em) {
		memset((caddr_t)em, 0, sizeof(*em));
		free(em);
	}
	if(authkey) {
		memset((caddr_t)authkey, 0, authkeylen);
		free(authkey);
	}
	if(enckey) {
		memset((caddr_t)enckey, 0, enckeylen);
		free(enckey);
	}
	if(iv) {
		memset((caddr_t)iv, 0, ivlen);
		free(iv);
	}
	exit(0);
}



#define isdot(_x)  (((_x)&0x80)|!(isprint(_x)))

/*
 * NAME
 *	xdump -- make a hex dump of a region in memory
 *
 * SYNOPSIS
 *	xdump(base, length, title)
 *	caddr_t base;
 *	int length;
 *	char *title;
 */
	
static char line[80];
static char hd[17]="0123456789ABCDEF";

#define HI(_x)  (hd[((_x)>>4)&0xF])
#define LO(_x)  (hd[(_x)&0xF])

xdump(base, length, title)
caddr_t base;
int length;
char *title;
{
	register char *bp, *hp, *cp;
	register int cnt;

	printf("%s\n", title);
	
	bp = base;

	hp = line;
	cp = line+50;
	
	for (cnt=80; cnt; line[--cnt]=' ')
	  ;

	line[49] = line[66] = '*';
	line[67] = '\0';

	while(length-- > 0)
	{
		*hp++ = HI(*bp);
		*hp++ = LO(*bp);
		hp++;
		*cp++ = isdot(*bp)?'.':(*bp);
		bp++;

		if (++cnt == 16)
		{
			cnt = 0;
			hp = line;
			cp = line+50;
			puts(line);
		}
	}

	if (cnt)
	{
		while (cnt++ < 16)
		{
			*hp++=' ';
			*hp++=' ';
			*hp++=' ';
			*cp++=' ';
		}
		puts(line);
	}

	return 0;
}

/*
 * $Log: spi.c,v $
 * Revision 1.37  1999/04/11 00:12:08  henry
 * GPL boilerplate
 *
 * Revision 1.36  1999/04/06 04:54:38  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.35  1999/03/17 15:40:07  rgb
 * Make explicit main() return type of int.
 * Fix memory clear bug in spi.c.
 *
 * Revision 1.34  1999/02/16 05:20:49  rgb
 * Fix memory clear bugs just prior to normal exit that were causing ipsec
 * manual scripts to fail and potentially leaving large core files.
 *
 * Revision 1.33  1999/02/09 00:13:16  rgb
 * Fix replay window htonl bug.
 *
 * Revision 1.32  1999/01/22 06:35:54  rgb
 * 64-bit clean-up.
 * Added algorithm switch code.
 * Removed IV requirement, now an option (kept code for back-compat).
 * Cruft clean-out.
 * Add error-checking.
 * Removed PFKEY code, will re-add later.
 *
 * Revision 1.31  1998/11/12 21:08:04  rgb
 * Add --label option to identify caller from scripts.
 *
 * Revision 1.30  1998/11/11 18:34:12  rgb
 * Fixed #includes for RH5.1.
 *
 * Revision 1.29  1998/11/11 07:14:18  rgb
 * #include cleanup to hopefully compile under RH5.1.
 *
 * Revision 1.28  1998/11/10 05:34:11  rgb
 * Add support for SA direction flag.
 * Add more specific error output messages.
 *
 * Revision 1.27  1998/10/27 00:31:12  rgb
 * Set replay structure flag to 0 (not used).
 *
 * Revision 1.26  1998/10/26 01:28:38  henry
 * use SA_* protocol names, not IPPROTO_*, to avoid compile problems
 *
 * Revision 1.25  1998/10/25 02:45:39  rgb
 * Change program to program_name to bring in line with other utils.
 * Added debugging code to find null proto bug, premature exit on hex info bug.
 * Fixed premature exit on hex info bug.
 *
 * Revision 1.24  1998/10/22 06:34:16  rgb
 * Fixed bad stucture pointer.
 * Fixed unknown var (cut and paste error).
 *
 * Revision 1.23  1998/10/19 18:56:24  rgb
 * Added inclusion of freeswan.h.
 * sa_id structure implemented and used: now includes protocol.
 * Start to add some inactive pfkey2 code.
 *
 * Revision 1.22  1998/10/09 18:47:30  rgb
 * Add 'optionfrom' to get more options from a named file.
 *
 * Revision 1.21  1998/10/09 04:36:03  rgb
 * Standardise on '-96' notation for AH transforms.
 *
 * Revision 1.20  1998/09/03 01:29:32  henry
 * improve atodata()-failed error messages a bit
 *
 * Revision 1.19  1998/09/02 03:14:33  henry
 * no point in printing zero lengths used as error returns
 *
 * Revision 1.18  1998/09/02 03:12:08  henry
 * --help output goes on stdout, not stderr
 *
 * Revision 1.17  1998/09/01 19:50:50  henry
 * fix operator-precedence bug that often messed up --ah SPI creation
 * minor cleanup
 *
 * Revision 1.16  1998/08/28 03:14:12  rgb
 * Simplify/Clarify usage text.
 *
 * Revision 1.15  1998/08/12 00:16:46  rgb
 * Removed a lot of old cruft that was commented out.
 * Updated usage text.
 * Added config options for new xforms.
 *
 * Revision 1.14  1998/08/05 22:24:45  rgb
 * Change includes to accomodate RH5.x
 *
 * Revision 1.13  1998/07/29 21:41:17  rgb
 * Fix spi bug, add hexadecimal value entry debugging.
 *
 * Revision 1.12  1998/07/28 00:14:24  rgb
 * Convert from positional parameters to long options.
 * Add --clean option.
 * Add hostname lookup support.
 *
 * Revision 1.11  1998/07/14 18:15:55  rgb
 * Fix undetected bug using AH-SHA1 with manual keying:  The key was
 * truncated by the data structure used to get it to the kernel.
 *
 * Revision 1.10  1998/07/09 18:14:11  rgb
 * Added error checking to IP's and keys.
 * Made most error messages more specific rather than spamming usage text.
 * Added more descriptive kernel error return codes and messages.
 * Converted all spi translations to unsigned.
 * Removed all invocations of perror.
 *
 * Revision 1.9  1998/06/30 18:04:31  rgb
 * Fix compiler warning: couldn't find 'struct option' prototype.
 *
 * Revision 1.8  1998/06/11 05:40:04  rgb
 * Make usage text more concise WRT replay window sizes and defaults.
 * Make error reporting more concise WRT exact IV and key lengths supported
 * and their units.
 *
 * Revision 1.7  1998/06/08 17:54:58  rgb
 * Fixed string escape code in usage.
 *
 * Revision 1.6  1998/06/05 02:22:49  rgb
 * Clarify usage text and update for key splitting and i/r removal.
 * Require keys of exact length.
 *
 * Revision 1.5  1998/05/27 20:54:11  rgb
 * Added --help and --version directives.  Separated auth and encr keys.
 *
 * Revision 1.4  1998/05/18 21:12:13  rgb
 * Clean up debugging code, clean up after keys, cleaner options setting.
 *
 * Revision 1.3  1998/05/06 03:37:11  rgb
 * Fixed incorrect signed interpretation of command line spi to unsigned long.
 * It prevented deletion of ~spi values generated by pluto.
 *
 * Revision 1.2  1998/05/01 23:34:01  rgb
 * Clarified the usage text.
 *
 * Revision 1.1.1.1  1998/04/08 05:35:10  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.5  1997/06/03 04:31:55  ji
 * Added esp 3des-md5-96
 *
 * Revision 0.4  1997/01/15 01:37:54  ji
 * New program in this release, replaces set* programs.
 *
 *
 */
