/*
 * Common routines for IPSEC transformations.
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: ipsec_xform.c,v 1.15 1999/04/11 00:29:01 henry Exp $
 */

#include <linux/config.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <linux/random.h>	/* get_random_bytes() */
#include <asm/checksum.h>
#include <net/ip.h>

#include "../../../lib/freeswan.h"
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ipe4.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#ifdef DEBUG_IPSEC
int debug_xform = 0;
#endif

extern int des_set_key(caddr_t, caddr_t);

struct xformsw xformsw[] = {
{ XF_IP4,		0,		"IPv4_Encapsulation"},
{ XF_AHHMACMD5,		XFT_AUTH,	"HMAC_MD5_Authentication"},
{ XF_AHHMACSHA1,	XFT_AUTH,	"HMAC_SHA-1_Authentication"},
{ XF_ESPDES,		XFT_CONF,	"DES_Encryption"},
{ XF_ESPDESMD596,	XFT_CONF,	"DES-MD5-96_Encryption"},
{ XF_ESPDESSHA196,	XFT_CONF,	"DES-SHA1-96_Encryption"},
{ XF_ESP3DES,		XFT_CONF,	"3DES_Encryption"},
{ XF_ESP3DESMD596,	XFT_CONF,	"3DES-MD5-96_Encryption"},
{ XF_ESP3DESSHA196,	XFT_CONF,	"3DES-SHA1-96_Encryption"},
{ XF_ESPNULLMD596,	XFT_CONF,	"NULL-MD5-96_ESP_*Plaintext*"},
{ XF_ESPNULLSHA196,	XFT_CONF,	"NULL-SHA1-96_ESP_*Plaintext*"},
};

struct xformsw *xformswNXFORMSW = &xformsw[sizeof(xformsw)/sizeof(xformsw[0])];

struct tdb *
gettdb(struct sa_id said)
{
	int hashval;
	struct tdb *tdbp;

	hashval = (said.spi+said.dst.s_addr+said.proto) % TDB_HASHMOD;
	
	for (tdbp = tdbh[hashval]; tdbp; tdbp = tdbp->tdb_hnext)
		if ((tdbp->tdb_said.spi == said.spi) &&
		    (tdbp->tdb_said.dst.s_addr == said.dst.s_addr) &&
		    (tdbp->tdb_said.proto == said.proto))
			break;
	
	return tdbp;
}

void
puttdb(struct tdb *tdbp)
{
	int hashval;
	hashval = ((tdbp->tdb_spi + tdbp->tdb_dst.s_addr + tdbp->tdb_proto) % TDB_HASHMOD);
	tdbp->tdb_hnext = tdbh[hashval];
	tdbh[hashval] = tdbp;
}

void
deltdb(struct tdb *tdbp)
{
	int hashval;
	struct tdb *tdbtp;

	hashval = ((tdbp->tdb_spi + tdbp->tdb_dst.s_addr + tdbp->tdb_proto) % TDB_HASHMOD);
	if (tdbp == tdbh[hashval]) {
		tdbh[hashval] = tdbp->tdb_hnext;
		tdbp->tdb_hnext = NULL;
	} else
		for (tdbtp = tdbh[hashval]; tdbtp; tdbtp = tdbtp->tdb_hnext)
			if (tdbtp->tdb_hnext == tdbp) {
				tdbtp->tdb_hnext = tdbp->tdb_hnext;
				tdbp->tdb_hnext = NULL;
				break;
			}
}

int
tdb_init(struct tdb *tdbp, struct encap_msghdr *em)
{
	int alg;
	struct xformsw *xsp;

        int i;
#if defined(CONFIG_IPSEC_ENC_DES) || defined(CONFIG_IPSEC_ENC_3DES)
        int error;
#endif /* CONFIG_IPSEC_ENC_{,3}DES */

        char sa[SATOA_BUF];
        size_t sa_len;

	sa_len = satoa(em->em_said, 0, sa, SATOA_BUF);

        KLIPS_PRINT(debug_esp,
		    "klips_debug:tdb_init: (algo_switch defined) called for SA:%s\n", sa);
	alg = em->em_alg;
	
	for (xsp = xformsw; xsp < xformswNXFORMSW; xsp++)
		if (xsp->xf_type == alg) {
			KLIPS_PRINT(debug_netlink,
				    "klips_debug:tdb_init: called with tdbp=0x%p, xsp=0x%p, em=0x%p\n",
				    tdbp, xsp, em);
			KLIPS_PRINT(debug_netlink,
				    "klips_debug:tdb_init: calling init routine of %s\n",
				    xsp->xf_name);
			tdbp->tdb_xform = xsp;
			tdbp->tdb_replaywin_lastseq = 0;
			tdbp->tdb_replaywin_bitmap = 0;
			/* check size of message here XXX */
			switch(alg) {
#ifdef CONFIG_IPSEC_IPIP
			case XF_IP4: {
				struct ipe4_xdata *xd;
				xd = (struct ipe4_xdata *)(em->em_dat);

				tdbp->tdb_authalg = AH_NONE;
				tdbp->tdb_encalg = ESP_NONE;
				
 				if((tdbp->tdb_addr_s = (struct sockaddr*)
				   kmalloc((tdbp->tdb_addr_s_size = sizeof(struct sockaddr_in)),
					   GFP_ATOMIC)) == NULL) {
					return ENOMEM;
				}
				if((tdbp->tdb_addr_d = (struct sockaddr*)
				   kmalloc((tdbp->tdb_addr_d_size = sizeof(struct sockaddr_in)),
					   GFP_ATOMIC)) == NULL) {
					return ENOMEM;
				}
				
				/* might want to use a different structure here, or set sin_family and sin_port */
				((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr = xd->i4_src;
				((struct sockaddr_in*)(tdbp->tdb_addr_d))->sin_addr = xd->i4_dst;
			}
				break;
#endif /* !CONFIG_IPSEC_IPIP */
#ifdef CONFIG_IPSEC_AH
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
			case XF_AHHMACMD5: {
				struct ahhmacmd5_edata *ed;
				unsigned char kb[AHMD596_BLKLEN];

				ed = (struct ahhmacmd5_edata *)em->em_dat;

				tdbp->tdb_authalg = AH_MD5;
				tdbp->tdb_encalg = ESP_NONE;
				
				tdbp->tdb_flags = em->em_flags;

				if (em->em_msglen - EMT_SETSPI_FLEN > sizeof (struct ahhmacmd5_edata))
					return EINVAL;
				
				if (ed->ame_klen != AHMD596_KLEN) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: incorrect key size: %d"
						    "-- must be %d octets (bytes)\n",
						    ed->ame_klen, AHMD596_KLEN);
					return EINVAL;
				}
				
				if (ed->ame_alen != AHMD596_ALEN) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: authenticator size: %d"
						    " -- must be %d octets (bytes)\n",
						    ed->ame_alen, AHMD596_ALEN);
					return EINVAL;
				}
				
				KLIPS_PRINT(debug_ah,
					    "klips_debug:tdb_init: hmac md5-96 key is 0x%08lx %08lx %08lx %08lx\n",
					    ntohl(*(((__u32 *)ed->ame_key)+0)),
					    ntohl(*(((__u32 *)ed->ame_key)+1)),
					    ntohl(*(((__u32 *)ed->ame_key)+2)),
					    ntohl(*(((__u32 *)ed->ame_key)+3)));
				
				tdbp->tdb_auth_bits = ed->ame_alen * 8;
				
				if(ed->ame_ooowin > 64) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: replay window size: %d"
						    " -- must be 0 <= size <= 64\n",
						    ed->ame_ooowin);
					return EINVAL;
				}
				tdbp->tdb_replaywin = ed->ame_ooowin;
				tdbp->tdb_replaywin_lastseq = tdbp->tdb_replaywin_bitmap = 0;
				
				if((tdbp->tdb_key_a = (caddr_t)
				    kmalloc((tdbp->tdb_key_a_size = sizeof(struct md5_ctx)),
					    GFP_ATOMIC)) == NULL) {
					return ENOMEM;
				}
				MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx);
				
				for (i = 0; i < ed->ame_klen; i++)
					kb[i] = ed->ame_key[i] ^ 0x36;
				for (; i < AHMD596_BLKLEN; i++)
					kb[i] = 0x36;
				MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx, kb, AHMD596_BLKLEN);
				
				MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx);
				for (i = 0; i < AHMD596_BLKLEN; i++)
					kb[i] ^= (0x36 ^ 0x5c);
				MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx, kb, AHMD596_BLKLEN);
				
				KLIPS_PRINT(debug_ah,
					    "klips_debug:tdb_init: MD5 ictx=0x%08x %08x %08x %08x"
					    " octx=0x%08x %08x %08x %08x\n",
					    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
					    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 1),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 2),
					    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 3));
				
				/* zero key buffer -- paranoid */
				memset(kb, 0, sizeof(kb));
				memset((caddr_t)&(ed->ame_key), 0, ed->ame_klen);
			}
				break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
			case XF_AHHMACSHA1: {
				struct ahhmacsha1_edata *ed;
				unsigned char kb[AHSHA196_BLKLEN];

				ed = (struct ahhmacsha1_edata *)em->em_dat;
				
				tdbp->tdb_authalg = AH_SHA;
				tdbp->tdb_encalg = ESP_NONE;
				
				tdbp->tdb_flags = em->em_flags;

				if (em->em_msglen - EMT_SETSPI_FLEN > sizeof (struct ahhmacsha1_edata))
					return EINVAL;
				
				if (ed->ame_klen != AHSHA196_KLEN) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: incorrect key size: %d"
						    "-- must be %d octets (bytes)\n",
						    ed->ame_klen, AHSHA196_KLEN);
					return EINVAL;
				}
				
				if (ed->ame_alen != AHSHA196_ALEN) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: authenticator size: %d"
						    " -- must be %d octets (bytes)\n",
						    ed->ame_alen, AHSHA196_ALEN);
					return EINVAL;
				}
				
				KLIPS_PRINT(debug_ah,
					    "klips_debug:tdb_init: hmac sha1-96 key is 0x%08lx %08lx %08lx %08lx\n",
					    ntohl(*(((__u32 *)ed->ame_key)+0)),
					    ntohl(*(((__u32 *)ed->ame_key)+1)),
					    ntohl(*(((__u32 *)ed->ame_key)+2)),
					    ntohl(*(((__u32 *)ed->ame_key)+3)));
				
				tdbp->tdb_auth_bits = ed->ame_alen * 8;
				
				if(ed->ame_ooowin > 64) {
					KLIPS_PRINT(debug_ah,
						    "klips_debug:tdb_init: replay window size: %d"
						    " -- must be 0 <= size <= 64\n",
						    ed->ame_ooowin);
					return EINVAL;
				}
				tdbp->tdb_replaywin = ed->ame_ooowin;
				tdbp->tdb_replaywin_lastseq = tdbp->tdb_replaywin_bitmap = 0;
				
				if((tdbp->tdb_key_a = (caddr_t)
				    kmalloc((tdbp->tdb_key_a_size = (__u16)sizeof(struct sha1_ctx)),
					    GFP_ATOMIC)) == NULL) {
					return ENOMEM;
				}
				SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx);
				
				for (i = 0; i < ed->ame_klen; i++)
					kb[i] = ed->ame_key[i] ^ 0x36;
				for (; i < AHSHA196_BLKLEN; i++)
					kb[i] = 0x36;
				SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx, kb, AHSHA196_BLKLEN);
				
				SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx);
				for (i = 0; i < AHSHA196_BLKLEN; i++)
					kb[i] ^= (0x36 ^ 0x5c);
				SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx, kb, AHSHA196_BLKLEN);
				
				KLIPS_PRINT(debug_ah,
					    "klips_debug:tdb_init: SHA1 ictx=0x%08x %08x %08x %08x"
					    " octx=0x%08x %08x %08x %08x\n", 
					    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
					    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 1),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 2),
					    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 3));
				
				/* zero key buffer -- paranoid */
				memset(kb, 0, sizeof(kb));
				memset((caddr_t)&(ed->ame_key), 0, ed->ame_klen);
			}
				break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
#endif /* CONFIG_IPSEC_AH */
#ifdef CONFIG_IPSEC_ESP
#ifdef CONFIG_IPSEC_ENC_DES
			case XF_ESPDES:
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
			case XF_ESPDESMD596:
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
			case XF_ESPDESSHA196:
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
#endif /* CONFIG_IPSEC_ENC_DES */
#ifdef CONFIG_IPSEC_ENC_3DES
			case XF_ESP3DES:
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
			case XF_ESP3DESMD596:
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
			case XF_ESP3DESSHA196:
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
#endif /* CONFIG_IPSEC_ENC_3DES */
#ifdef CONFIG_IPSEC_ENC_NULL
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
			case XF_ESPNULLMD596:
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
			case XF_ESPNULLSHA196:
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
#endif /* CONFIG_IPSEC_ENC_NULL */
			{
				struct espblkrply_edata *ed;
				unsigned char kb[AHMD596_BLKLEN];
				ed = (struct espblkrply_edata *)em->em_dat;

				tdbp->tdb_flags = em->em_flags;

				if(ed->eme_ooowin > 64) {
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: replay window size: %d"
						    "-- must be 0 <= size <= 64\n",
						    ed->eme_ooowin);
					return EINVAL;
				}
				tdbp->tdb_replaywin = ed->eme_ooowin;

				switch(alg) {
				case XF_ESPDES:
				case XF_ESP3DES:
				case XF_ESPDESMD596:
				case XF_ESP3DESMD596:
				case XF_ESPDESSHA196:
				case XF_ESP3DESSHA196:
					if((tdbp->tdb_iv = (caddr_t)
					   kmalloc((tdbp->tdb_iv_size = EMT_ESPDES_IV_SZ), GFP_ATOMIC)) == NULL) {
						return ENOMEM;
					}
					get_random_bytes((void *)tdbp->tdb_iv, EMT_ESPDES_IV_SZ);
					break;
				default:
				}

				switch(alg) {
#ifdef CONFIG_IPSEC_ENC_DES
				case XF_ESPDES:
				case XF_ESPDESMD596:
				case XF_ESPDESSHA196:
					tdbp->tdb_encalg = ESP_DES;
				
					if (ed->eme_klen != EMT_ESPDES_KEY_SZ) {
						KLIPS_PRINT(debug_esp,
							    "klips_debug:tdb_init: incorrect encryption"
							    "key size: %d -- must be %d octets (bytes)\n",
							    ed->eme_klen, EMT_ESPDES_KEY_SZ);
						return EINVAL;
					}

					if((tdbp->tdb_key_e = (caddr_t)
					   kmalloc((tdbp->tdb_key_e_size = sizeof(struct des_eks)),
						   GFP_ATOMIC)) == NULL) {
						return ENOMEM;
					}
#if 0
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: des key 1 is 0x%08lx%08lx\n",
						    ntohl(*((__u32 *)ed->eme_key)),
						    ntohl(*((__u32 *)ed->eme_key + 1)));
#endif
					error = des_set_key((caddr_t)(ed->eme_key), (caddr_t)(tdbp->tdb_key_e));
					if (error == -1)
						printk("klips_debug:tdb_init: parity error in des key\n");
					else if (error == -2)
						printk("klips_debug:tdb_init: illegal weak des key\n");
					if (error) {
						memset(tdbp->tdb_key_e, 0, sizeof (struct des_eks));
						kfree_s(tdbp->tdb_key_e, sizeof(struct des_eks));
						return (EINVAL);
					}
					break;
#endif /* CONFIG_IPSEC_ENC_DES */
#ifdef CONFIG_IPSEC_ENC_3DES
				case XF_ESP3DES:
				case XF_ESP3DESMD596:
				case XF_ESP3DESSHA196:
					tdbp->tdb_encalg = ESP_3DES;
				
					if (ed->eme_klen != EMT_ESP3DES_KEY_SZ) {
						KLIPS_PRINT(debug_esp,
							    "klips_debug:tdb_init: incorrect encryption"
							    "key size: %d -- must be %d octets (bytes)\n",
							    ed->eme_klen, EMT_ESP3DES_KEY_SZ);
						return EINVAL;
					}

					if((tdbp->tdb_key_e = (caddr_t)
					   kmalloc((tdbp->tdb_key_e_size = 3 * sizeof(struct des_eks)),
						   GFP_ATOMIC)) == NULL) {
						return ENOMEM;
					}

					for(i = 0; i < 3; i++) {
#if 0
						KLIPS_PRINT(debug_esp,
							    "klips_debug:tdb_init: 3des key %d/3 is 0x%08lx%08lx\n",
							    i + 1,
							    ntohl(*((__u32 *)ed->eme_key + i * 2)),
							    ntohl(*((__u32 *)ed->eme_key + i * 2 + 1)));
#endif
						error = des_set_key((caddr_t)(ed->eme_key) + EMT_ESPDES_KEY_SZ * i,
								    (caddr_t)&((struct des_eks*)(tdbp->tdb_key_e))[i]);
						if (error == -1)
							printk("klips_debug:tdb_init: parity error in des key %d/3\n", i + 1);
						else if (error == -2)
							printk("klips_debug:tdb_init: illegal weak des key %d/3\n", i + 1);
						if (error) {
							memset(tdbp->tdb_key_e, 0, 3 * sizeof(struct des_eks));
							kfree_s(tdbp->tdb_key_e, 3 * sizeof(struct des_eks));
							return (EINVAL);
						}
					}

					break;
#endif /* CONFIG_IPSEC_ENC_3DES */
				default:
					tdbp->tdb_encalg = ESP_NULL;
				}

				switch(alg) {
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
				case XF_ESPDESMD596:
				case XF_ESP3DESMD596:
				case XF_ESPNULLMD596:
					tdbp->tdb_authalg = AH_MD5;
				
					if (ed->ame_klen != AHMD596_KLEN) {
						KLIPS_PRINT(debug_esp,
							    "klips_debug:tdb_init: incorrect authorisation"
							    " key size: %d -- must be %d octets (bytes)\n",
							    ed->ame_klen, AHMD596_KLEN);
						return EINVAL;
					}

					if((tdbp->tdb_key_a = (caddr_t)
					   kmalloc((tdbp->tdb_key_a_size = sizeof(struct md5_ctx)),
						   GFP_ATOMIC)) == NULL) {
						return ENOMEM;
					}
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: hmac md5-96 key is 0x%08lx %08lx %08lx %08lx\n",
						    ntohl(*(((__u32 *)ed->ame_key)+0)),
						    ntohl(*(((__u32 *)ed->ame_key)+1)),
						    ntohl(*(((__u32 *)ed->ame_key)+2)),
						    ntohl(*(((__u32 *)ed->ame_key)+3)));
					for (i=0; i< AHMD596_KLEN; i++)
						kb[i] = (*(((unsigned char *)(ed->ame_key)) + i)) ^ 0x36;
					/*
					 * HMAC_key is now contained in the first 128 bits of kb.
					 * Pad with zeroes and XOR with 0x36 to create the inner context
					 */
					for (; i<AHMD596_BLKLEN; i++)
						kb[i] = 0x36;
					
					MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx);
					MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx, kb, sizeof kb);
					
					for (i=0; i<AHMD596_BLKLEN; i++)
						kb[i] ^= (0x36 ^ 0x5c);
					
					MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx);
					MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx, kb, sizeof kb);
					
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: MD5 ictx=0x%08x %08x %08x %08x"
						    " octx=0x%08x %08x %08x %08x\n",
						    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
						    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 1),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 2),
						    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 3));
					memset(kb, 0, sizeof(kb)); /* paranoid */
					memset((caddr_t)&(ed->eme_key), 0, ed->eme_klen);
					memset((caddr_t)&(ed->ame_key), 0, ed->ame_klen);
					break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
				case XF_ESPNULLSHA196:
				case XF_ESPDESSHA196:
				case XF_ESP3DESSHA196:
					tdbp->tdb_authalg = AH_SHA;
				
					if (ed->ame_klen != AHSHA196_KLEN) {
						KLIPS_PRINT(debug_esp,
							    "klips_debug:tdb_init: incorrect authorisation"
							    " key size: %d -- must be %d octets (bytes)\n",
							    ed->ame_klen, AHSHA196_KLEN);
						return EINVAL;
					}

					if((tdbp->tdb_key_a = (caddr_t)
					   kmalloc((tdbp->tdb_key_a_size = sizeof(struct sha1_ctx)),
						   GFP_ATOMIC)) == NULL) {
						return ENOMEM;
					}
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: hmac sha1-96 key is 0x%08lx %08lx %08lx %08lx\n",
						    ntohl(*(((__u32 *)ed->ame_key)+0)),
						    ntohl(*(((__u32 *)ed->ame_key)+1)),
						    ntohl(*(((__u32 *)ed->ame_key)+2)),
						    ntohl(*(((__u32 *)ed->ame_key)+3)));
					for (i=0; i< AHSHA196_KLEN; i++)
						kb[i] = (*(((unsigned char *)(ed->ame_key)) + i)) ^ 0x36;
					/*
					 * HMAC_key is now contained in the first 128 bits of kb.
					 * Pad with zeroes and XOR with 0x36 to create the inner context
					 */
					for (; i<AHSHA196_BLKLEN; i++)
						kb[i] = 0x36;
					
					SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx);
					SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx, kb, sizeof kb);
					
					for (i=0; i<AHSHA196_BLKLEN; i++)
						kb[i] ^= (0x36 ^ 0x5c);
					
					SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx);
					SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx, kb, sizeof kb);
					
					KLIPS_PRINT(debug_esp,
						    "klips_debug:tdb_init: SHA1 ictx=0x%08x %08x %08x %08x"
						    " octx=0x%08x %08x %08x %08x\n",
						    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
						    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 1),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 2),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 3));
					memset(kb, 0, sizeof(kb)); /* paranoid */
					memset((caddr_t)&(ed->eme_key), 0, ed->eme_klen);
					memset((caddr_t)&(ed->ame_key), 0, ed->ame_klen);
					break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
				case XF_ESPDES:
				case XF_ESP3DES:
					tdbp->tdb_authalg = AH_NONE;
					break;
				default:
				}
			}
				break;
#endif /* !CONFIG_IPSEC_ESP */
			default:
				return EINVAL;
			}
			return 0;
		}
	
	KLIPS_PRINT(debug_xform & DB_XF_INIT,
		    "klips_debug:tdb_init: unregistered algorithm %d requested\n"
		    "klips_debug:   trying to setup SA:%s\n", alg, sa);
	return EINVAL;
}

int 
ipsec_tdbcleanup()
{
	int i;
	struct tdb *tdbp, *tdbp2;

	for (i = 0; i < TDB_HASHMOD; i++)
	{
		tdbp = tdbh[i];
		tdbh[i] = NULL;
		for (; tdbp;)
		{
			tdbp2 = tdbp;
			tdbp = tdbp->tdb_hnext;
			ipsec_tdbwipe(tdbp2);
			memset((caddr_t)tdbp2, 0, sizeof(*tdbp2));
			kfree_s(tdbp2, sizeof(*tdbp2));
		}
	}
	return 0;
}

int
ipsec_tdbwipe(struct tdb *tdbp)
{
	if(tdbp->tdb_addr_s) {
		memset((caddr_t)tdbp->tdb_addr_s, 0, tdbp->tdb_addr_s_size);
		kfree_s(tdbp->tdb_addr_s, tdbp->tdb_addr_s_size);
	}
	tdbp->tdb_addr_s = NULL;

	if(tdbp->tdb_addr_d) {
		memset((caddr_t)tdbp->tdb_addr_d, 0, tdbp->tdb_addr_d_size);
		kfree_s(tdbp->tdb_addr_s, tdbp->tdb_addr_d_size);
	}
	tdbp->tdb_addr_d = NULL;

	if(tdbp->tdb_addr_p) {
		memset((caddr_t)tdbp->tdb_addr_p, 0, tdbp->tdb_addr_p_size);
		kfree_s(tdbp->tdb_addr_p, tdbp->tdb_addr_p_size);
	}
	tdbp->tdb_addr_p = NULL;

	if(tdbp->tdb_key_a) {
		memset((caddr_t)tdbp->tdb_key_a, 0, tdbp->tdb_key_a_size);
		kfree_s(tdbp->tdb_key_a, tdbp->tdb_key_a_size);
	}
	tdbp->tdb_key_a = NULL;

	if(tdbp->tdb_key_e) {
		memset((caddr_t)tdbp->tdb_key_e, 0, tdbp->tdb_key_e_size);
		kfree_s(tdbp->tdb_key_e, tdbp->tdb_key_e_size);
	}
	tdbp->tdb_key_e = NULL;

	if(tdbp->tdb_iv) {
		memset((caddr_t)tdbp->tdb_iv, 0, tdbp->tdb_iv_size);
		kfree_s(tdbp->tdb_iv, tdbp->tdb_iv_size);
	}
	tdbp->tdb_iv = NULL;

	return 0;
}

/*
 * $Log: ipsec_xform.c,v $
 * Revision 1.15  1999/04/11 00:29:01  henry
 * GPL boilerplate
 *
 * Revision 1.14  1999/04/06 04:54:28  rgb
 * Fix/Add RCSID Id: and Log: bits to make PHMDs happy.  This includes
 * patch shell fixes.
 *
 * Revision 1.13  1999/02/19 18:23:01  rgb
 * Nix debug off compile warning.
 *
 * Revision 1.12  1999/02/17 16:52:16  rgb
 * Consolidate satoa()s for space and speed efficiency.
 * Convert DEBUG_IPSEC to KLIPS_PRINT
 * Clean out unused cruft.
 * Ditch NET_IPIP dependancy.
 * Loop for 3des key setting.
 *
 * Revision 1.11  1999/01/26 02:09:05  rgb
 * Remove ah/esp/IPIP switching on include files.
 * Removed CONFIG_IPSEC_ALGO_SWITCH macro.
 * Removed dead code.
 * Clean up debug code when switched off.
 * Remove references to INET_GET_PROTOCOL.
 * Added code exclusion macros to reduce code from unused algorithms.
 *
 * Revision 1.10  1999/01/22 06:28:55  rgb
 * Cruft clean-out.
 * Put random IV generation in kernel.
 * Added algorithm switch code.
 * Enhanced debugging.
 * 64-bit clean-up.
 *
 * Revision 1.9  1998/11/30 13:22:55  rgb
 * Rationalised all the klips kernel file headers.  They are much shorter
 * now and won't conflict under RH5.2.
 *
 * Revision 1.8  1998/11/25 04:59:06  rgb
 * Add conditionals for no IPIP tunnel code.
 * Delete commented out code.
 *
 * Revision 1.7  1998/10/31 06:50:41  rgb
 * Convert xform ASCII names to no spaces.
 * Fixed up comments in #endif directives.
 *
 * Revision 1.6  1998/10/19 14:44:28  rgb
 * Added inclusion of freeswan.h.
 * sa_id structure implemented and used: now includes protocol.
 *
 * Revision 1.5  1998/10/09 04:32:19  rgb
 * Added 'klips_debug' prefix to all klips printk debug statements.
 *
 * Revision 1.4  1998/08/12 00:11:31  rgb
 * Added new xform functions to the xform table.
 * Fixed minor debug output spelling error.
 *
 * Revision 1.3  1998/07/09 17:45:31  rgb
 * Clarify algorithm not available message.
 *
 * Revision 1.2  1998/06/23 03:00:51  rgb
 * Check for presence of IPIP protocol if it is setup one way (we don't
 * know what has been set up the other way and can only assume it will be
 * symmetrical with the exception of keys).
 *
 * Revision 1.1  1998/06/18 21:27:51  henry
 * move sources from klips/src to klips/net/ipsec, to keep stupid
 * kernel-build scripts happier in the presence of symlinks
 *
 * Revision 1.3  1998/06/11 05:54:59  rgb
 * Added transform version string pointer to xformsw initialisations.
 *
 * Revision 1.2  1998/04/21 21:28:57  rgb
 * Rearrange debug switches to change on the fly debug output from user
 * space.  Only kernel changes checked in at this time.  radij.c was also
 * changed to temporarily remove buggy debugging code in rj_delete causing
 * an OOPS and hence, netlink device open errors.
 *
 * Revision 1.1  1998/04/09 03:06:13  henry
 * sources moved up from linux/net/ipsec
 *
 * Revision 1.1.1.1  1998/04/08 05:35:02  henry
 * RGB's ipsec-0.8pre2.tar.gz ipsec-0.8
 *
 * Revision 0.5  1997/06/03 04:24:48  ji
 * Added ESP-3DES-MD5-96
 *
 * Revision 0.4  1997/01/15 01:28:15  ji
 * Added new transforms.
 *
 * Revision 0.3  1996/11/20 14:39:04  ji
 * Minor cleanups.
 * Rationalized debugging code.
 *
 * Revision 0.2  1996/11/02 00:18:33  ji
 * First limited release.
 *
 *
 */
