/*
 *	AX.25 release 029
 *
 *	This is ALPHA test software. This code may break your machine, randomly fail to work with new 
 *	releases, misbehave and/or generally screw up. It might even work. 
 *
 *	This code REQUIRES 1.2.1 or higher/ NET3.029
 *
 *	This module:
 *		This module is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 *	Most of this code is based on the SDL diagrams published in the 7th
 *	ARRL Computer Networking Conference papers. The diagrams have mistakes
 *	in them, but are mostly correct. Before you modify the code could you
 *	read the SDL diagrams as the code is not obvious and probably very
 *	easy to break;
 *
 *	History
 *	AX.25 028a	Jonathan(G4KLX)	New state machine based on SDL diagrams.
 *	AX.25 029	Alan(GW4PTS) Switched to KA9Q constant names.
 */
 
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include "ax25.h"
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include "sock.h"
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/interrupt.h>

#ifdef CONFIG_AX25

int ax25_output(ax25_socket *sk, struct sk_buff *skb)
{
	skb_queue_tail(&sk->write_queue, skb);	/* Throw it on the queue */
	ax25_kick(sk);				/* See if we can transmit anything */
	return 0;
}

/* 
 *	This procedure is passed a buffer descriptor for an iframe. It builds
 *	the rest of the control part of the frame and then writes it out.
 */
 
static void ax25_send_iframe(ax25_socket *sk, struct sk_buff *skb, int poll_bit)
{
	unsigned char *frame;

	if (skb == NULL)
		return;
	
	frame = skb->h.raw;

	frame[1] = AX25_P_TEXT;
	frame[0] = I;
	frame[0] |= (sk->ax25_vr<<5);	/* NR */
	frame[0] |= poll_bit;		/* PF */
	frame[0] |= (sk->ax25_vs<<1);	/* NS */

	if (sk->debug)
		printk("S%d: send I frame vr=%d vs=%d\n", sk->ax25_state, sk->ax25_vr, sk->ax25_vs);

	ax25_transmit_buffer(sk, skb, C_COMMAND);	
}

void ax25_kick(ax25_socket *sk)
{
	struct sk_buff *skb, *skbn;
	int last = 1;
	unsigned char start, end, next;

	del_timer(&sk->timer);

	start = (skb_peek(&sk->ax25_ack_queue) == NULL) ? sk->ax25_va : sk->ax25_vs;
	end   = (sk->ax25_va + sk->window) % MODULUS;

	if (!(sk->ax25_condition & PEER_RX_BUSY_CONDITION) &&
	    start != end                                   &&
	    skb_peek(&sk->write_queue) != NULL) {

		sk->ax25_vs = start;

		/*
		 * Transmit data until either we're out of data to send or
		 * the window is full. Send a poll on the final I frame if
		 * the window is filled.
		 */
		do 
		{
			/*
			 * Dequeue the frame and copy it.
			 */
			skb  = skb_dequeue(&sk->write_queue);

			if ((skbn = skb_clone(skb, GFP_ATOMIC)) == NULL) {
				skb_queue_head(&sk->write_queue, skb);
				return;
			}

			next = (sk->ax25_vs + 1) % MODULUS;
			last = (next == end);

			/*
			 * Transmit the frame copy.
			 */
			ax25_send_iframe(sk, skbn, (last) ? PF : 0);

			sk->ax25_vs = next;

			/*
			 * Requeue the original data frame.
			 */
			skb_queue_tail(&sk->ax25_ack_queue, skb);

		} while (!last && skb_peek(&sk->write_queue) != NULL);

		sk->ax25_condition &= ~ACK_PENDING_CONDITION;

		if (sk->ax25_t1timer == 0) 
		{
			sk->ax25_t3timer = 0;
			sk->ax25_t1timer = sk->ax25_t1;
		}
	}

	ax25_set_timer(sk);
}

void ax25_transmit_buffer(ax25_socket *sk, struct sk_buff *skb, int type)
{
	struct device *dev = sk->ax25_device;
	unsigned char *ptr = skb->data;

	if (dev == NULL) 
	{
		sk->err = ENETUNREACH;
		sk->error_report(sk);
		return;
	}

	*ptr++ = 0;	/* KISS data */
	ptr   += build_ax25_addr(ptr, &sk->ax25_source_addr, &sk->ax25_dest_addr, sk->ax25_digipeat, type);

	skb->arp = 1;

	dev_queue_xmit(skb, dev, sk->priority);
}

/*
 *	The following routines are taken from page 170 of the 7th ARRL Computer
 *	Networking Conference paper, as is the whole state machine.
 */

void ax25_nr_error_recovery(ax25_socket *sk)
{
	ax25_establish_data_link(sk);
}

void ax25_establish_data_link(ax25_socket *sk)
{
	sk->ax25_condition = 0x00;
	sk->ax25_n2count   = 0;

	if (sk->debug)
		printk("S%d: sending SABM(P) N2=0\n", sk->ax25_state);

	ax25_send_control(sk, SABM|PF, C_COMMAND);

	sk->ax25_t3timer = 0;
	sk->ax25_t1timer = sk->ax25_t1;
}

void ax25_transmit_enquiry(ax25_socket *sk)
{
	if (sk->ax25_condition & OWN_RX_BUSY_CONDITION) 
	{
		if (sk->debug)
			printk("S%d: sending RNR(P) vr=%d n2=%d\n", sk->ax25_state, sk->ax25_vr, sk->ax25_n2count);
		ax25_send_control(sk, RNR|PF, C_COMMAND);
	} 
	else 
	{
		if (sk->debug)
			printk("S%d: sending RR(P) vr=%d n2=%d\n", sk->ax25_state, sk->ax25_vr, sk->ax25_n2count);
		ax25_send_control(sk, RR|PF, C_COMMAND);
	}

	sk->ax25_condition &= ~ACK_PENDING_CONDITION;

	sk->ax25_t1timer = sk->ax25_t1;
}
 	
void ax25_enquiry_response(ax25_socket *sk)
{
	if (sk->ax25_condition & OWN_RX_BUSY_CONDITION) 
	{
		if (sk->debug)
			printk("S%d: sending RNR(F) vr=%d\n", sk->ax25_state, sk->ax25_vr);
		ax25_send_control(sk, RNR|PF, C_RESPONSE);
	} else {
		if (sk->debug)
			printk("S%d: sending RR(F) vr=%d\n", sk->ax25_state, sk->ax25_vr);
		ax25_send_control(sk, RR|PF, C_RESPONSE);
	}

	sk->ax25_condition &= ~ACK_PENDING_CONDITION;
}

void ax25_invoke_retransmission(ax25_socket *sk, unsigned char nr)
{
}

void ax25_check_iframes_acked(ax25_socket *sk, unsigned char nr)
{
	if (sk->ax25_vs == nr) {
		ax25_frames_acked(sk, nr);
		sk->ax25_t1timer = 0;
		sk->ax25_t3timer = sk->ax25_t3;
	}
	else 
	{
		if (sk->ax25_va != nr) 
		{
			ax25_frames_acked(sk, nr);
			sk->ax25_t1timer = sk->ax25_t1;
		}
	}
}

void ax25_check_need_response(ax25_socket *sk, int type, int pf)
{
	if (type == C_COMMAND && pf) 
	{
		ax25_enquiry_response(sk);
	}
}

#endif
