static char g_ident[] = "@(#) MultiSound ioctl utility $Revision: 2.0 $ M.Mummert";
/*
 *	$Id: msndctl.c,v 2.0 1994/11/13 03:59:11 mummert Exp mummert $
 *
 *  	MSNDCTL.C
 *      =========
 *
 *	Utility to set ioctls of MultiSound device driver. If name of
 *	executable starts with letter `r' or `p' a specified audio file
 *	can be recorded or played back respectively.
 *
 *	Copyright (c) 1994 by Markus Mummert
 *
 *	Redistribution and use of this software, modifcation and inclusion
 *	into other forms of software are permitted provided that the following
 *	conditions are met:
 *
 *	1. Redistributions of this software must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	2. If this software is redistributed in a modified condition
 *	   it must reveal clearly that it has been modified.
 *	
 *	THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 *	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 *	TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *	PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR
 *	CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *	EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *	PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *	PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 *	OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *	USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 *	DAMAGE.
 *
 * 	extern: 	msndio.h 	ioctl-defines
 *
 *      history:	29.8.92		begin coding
 *			4.9.92/V1.0	begin RCS-logging
 *			(...)		see RCS history or file HISTORY
 */

#include<stdio.h>
#include<fcntl.h>
#include<errno.h>
#include"msndio.h"

#define BYTE			char
#define UBYTE			unsigned BYTE
#define WORD			int
#define UWORD			unsigned WORD

#define BUFFSIZE_IN_BYTES	0x4000

#ifndef	MULTISOUNDDEVICE
#define MULTISOUNDDEVICE	"/dev/msnd"
#endif

#define	MSNDCTL			0
#define PLAY			1
#define	REC			2


int
	g_fileidx,
	g_fstfileidx,
	g_runmode = PLAY,
	g_rate = M_CLKDIV4,
	g_mode = M_MONO,
	g_smpfmt = M_SWAP,
	g_monitor = M_SILENT,
	g_autorewind_flag = 0,
	g_stdinout = 0,
	g_needreopen = 0,
	g_needreinit = 0,
	g_playlim = 0,
	g_reclim = 0x100000,
	g_inpt = 0x4040,
	g_auxpt = 0x0,
	g_soundfile_handle,
	g_msnd_handle
;
BYTE
	g_buff[BUFFSIZE_IN_BYTES]
;

void show_config_and_usage(name)
char *name;
{
	/* the following is just for my personal sense of beauty */
	char version[160];
	sscanf(g_ident, "%[^:]: %[^$]", version, version);

printf("\n");

	if (g_runmode == PLAY) {

printf("    MultiSound audio playback V%s M.Mummert\n", version);
printf("    Usage: %s <options> <file(s)>\n", name);
printf("    Options:\n");
printf("      -         use stdin instead of <file(s)>\n");
printf("      -a        autorewind\n");
printf("\n");
printf("    Options affecting MultiSound device configuration:\n");
printf("      -p<bytes> set playback limit in bytes, 0->none (%d)\n",g_playlim);

	} else if (g_runmode == REC) {

printf("    MultiSound audio recording V%s M.Mummert\n", version);
printf("    Usage: %s <options> <file>\n", name);
printf("    Options:\n");
printf("      -         use stdout instead of <file>\n");
printf("\n");
printf("    Options affecting MultiSound device configuration:\n");
printf("      -r<bytes> set recording limit in bytes, 0->none (%d)\n",g_reclim);

	} else {

printf("    MultiSound device configuration control V%s M.Mummert\n",version);
printf("    Usage: %s <options>\n", name);
printf("    Options:\n");
printf("      -z        force initialization of MultiSound hardware\n");
printf("      -illrr    set input pot to hex digits llrr (%04x)\n", g_inpt);
printf("      -xllrr    set aux pot to hex digits llrr (%04x)\n", g_auxpt);
printf("      -p<bytes> set playback limit in bytes, 0->none (%d)\n",g_playlim);
printf("      -r<bytes> set recording limit in bytes, 0->none (%d)\n",g_reclim);

	}

	if (g_rate == M_CLKDIV4) {
printf("      -1 -2 -4  sampling rate 11.025, 22.05 or 44.1kHz (11.025)\n");
	} else if (g_rate == M_CLKDIV2) {
printf("      -1 -2 -4  sampling rate 11.025, 22.05 or 44.1kHz (22.05)\n");
	} else {
printf("      -1 -2 -4  sampling rate 11.025, 22.05 or 44.1kHz (44.1)\n");
	}

	if (g_mode == M_MONO) {
printf("      -m -s     mono or stereo (mono)\n");
	} else {
printf("      -m -s     mono or stereo (stereo)\n");
	}

	if (g_smpfmt == M_SWAP) {
printf("      -h -l     sample format HB,LB or LB,HB (HB,LB)\n");
	} else {
printf("      -h -l     sample format HB,LB or LB,HB (LB,HB)\n");
	}

	if (g_monitor == M_AUTO) {
printf("      -f -q     feed in-to-out when closed or quiet (feed)\n");
	} else {
printf("      -f -q     feed in-to-out when closed or quiet (quiet)\n");
	}

printf("    Current device configuration shown in brackets\n");
printf("\n");

}

int scan_and_check_args(argc, argv)
int argc;
char *argv[];
{
	int i, k, l;

	if (argc < 2) 
		return(-1);
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-')
			break;
		for (k = 1; argv[i][k] != 0; k++) {
			switch (argv[i][k]) {
			case 'p': 	if (g_runmode == REC)
						return(-1);
					for (l = k+1; argv[i][l] != 0 && 
						argv[i][l] >= '0' &&
						argv[i][l] <= '9'; l++) ;
					sscanf(argv[i]+k+1,"%d",&g_playlim);
					k = l - 1;
					break;
			case 'r': 	if (g_runmode == PLAY)
						return(-1);
					for (l = k+1; argv[i][l] != 0 && 
						argv[i][l] >= '0' &&
						argv[i][l] <= '9'; l++) ;
					sscanf(argv[i]+k+1,"%d",&g_reclim);
					k = l - 1;
					break;
			case 'i': 	if (g_runmode != MSNDCTL)
						return(-1);
					sscanf(argv[i]+k+1,"%x",&g_inpt);
					k += 4;
					break;
			case 'x': 	if (g_runmode != MSNDCTL)
						return(-1);
					sscanf(argv[i]+k+1,"%x",&g_auxpt);
					k += 4;
					break;
			case 'z': 	if (g_runmode != MSNDCTL)
						return(-1);
					g_needreinit = -1; break;
			case 'f':	g_monitor = M_AUTO; break;
			case 'q':	g_monitor = M_SILENT; break;
			case '1': 	g_rate = M_CLKDIV4; break;
			case '2': 	g_rate = M_CLKDIV2; break;
			case '4': 	g_rate = M_CLKDIV1; break;
			case 'h': 	g_smpfmt = M_SWAP; break;
			case 'l': 	g_smpfmt = M_NOSWAP; break;
			case 'm': 	g_mode = M_MONO; break;
			case 's': 	g_mode = M_STEREO; break;
			case 'a': 	if (g_runmode != PLAY)
						return(-1);
					g_autorewind_flag = -1; break;
			default: 	return(-1);
			}
		}
		if (k == 1)
			g_stdinout = -1;
	}
	g_fstfileidx = i;
	if (g_stdinout) {
		g_fstfileidx = i-1;
		g_autorewind_flag = 0;
	}
	if (g_fstfileidx >= argc && g_runmode != MSNDCTL)
		return(-1);
	return(0);
}

int get_msnd(cmd, argp)
int cmd, *argp;
{
	if (ioctl(g_msnd_handle, cmd, argp)) {
		perror ("MultiSound get ioctl");
		return(-1);
	}
	return(0);
}

int set_msnd_if_needed(cmd, arg)
int cmd, arg;
{
	int argp[1];

	/* change `SET' command cmd in to a `GET' command and read setting */
	if (ioctl(g_msnd_handle, cmd - M_SETRATE + M_GETRATE, argp)) {
		perror ("MultiSound get ioctl");
		return(-1);
	}
	if (*argp != arg) {
		if (ioctl(g_msnd_handle, cmd, arg)) {
			perror ("MultiSound set ioctl");
			return(-1);
		}
		g_needreopen = 1;
	}
	return(0);
}

int read_config()
{
	int unexpected = 0;

	if (get_msnd(M_GETRATE, &g_rate))
		return(-1);
	if (g_rate != M_CLKDIV1 &&
	    g_rate != M_CLKDIV2 &&
	    g_rate != M_CLKDIV4) 
		unexpected = -1;

	if (get_msnd(M_GETMODE, &g_mode))
		return(-1);
	if (g_mode != M_MONO &&
	    g_mode != M_STEREO)
		unexpected = -1;

	if (get_msnd(M_GETSMPFMT, &g_smpfmt))
		return(-1);
	if (g_smpfmt != M_SWAP &&
	    g_smpfmt != M_NOSWAP)
		unexpected = -1;

	if (get_msnd(M_GETMONITOR, &g_monitor))
		return(-1);
	if (g_monitor != M_AUTO &&
	    g_monitor != M_SILENT)
		unexpected = -1;

	if (get_msnd(M_GETINPTVAL, &g_inpt))
		return(-1);
	if (get_msnd(M_GETAUXPTVAL, &g_auxpt))
		return(-1);
	if (get_msnd(M_GETRECBYTES, &g_reclim))
		return(-1);
	if (get_msnd(M_GETPLAYBYTES, &g_playlim))
		return(-1);

	/* we've waited to complain about this */ 
	if (unexpected) {
		perror ("Unexpected result from MultiSound get ioctl");
		return(-1);
	}
	return(0);
}

int config_device_if_needed()
{
	/*
	 * The idea of the set_msnd_if_needed-business
	 * here is that we need to close the device
         * after setting driver paramters via ioctl, because
	 * new paramters only become effective after next
	 * open. Of course we could close and reopen even if
	 * no paramters have been changed, but closing and reopening
         * induces considerable delay. So we only close
         * and reopen if something's changed with regard to
         * last invocation of this program.
	 *
	 * For the change of input and/or aux pots to take effect,
	 * reopening won't be enough. Then the driver has to 
	 * reinitialize MultiSound, which takes about 1 second
	 * of delay. So we reinitialize only if we have to.
	 *
         */
	if (set_msnd_if_needed(M_SETINPTVAL, g_inpt))
		return(-1);
	if (set_msnd_if_needed(M_SETAUXPTVAL, g_auxpt))
		return(-1);
	if (g_needreopen || g_needreinit) {
		if (ioctl(g_msnd_handle, M_REINIT, 0)) {
			perror ("MultiSound reinit via ioctl");
			return(-1);
		}
	}
	if (set_msnd_if_needed(M_SETRECBYTES, g_reclim))
		return(-1);
	if (set_msnd_if_needed(M_SETPLAYBYTES, g_playlim))
		return(-1);
	if (set_msnd_if_needed(M_SETMONITOR, g_monitor))
		return(-1);
	if (set_msnd_if_needed(M_SETRATE, g_rate)) 
		return(-1);
	if (set_msnd_if_needed(M_SETSMPFMT, g_smpfmt))
		return(-1);
	if (set_msnd_if_needed(M_SETMODE, g_mode))
		return(-1);

	return(0);
}


int play(argc, argv)
int argc;
char *argv[];
{
	int i, status = 0, count;

	do {
	    for (i = g_fstfileidx; !status && i < argc; i++) {
		if (g_stdinout) {
			g_soundfile_handle = 0;
		} else {
			if ((g_soundfile_handle = 
			    open(argv[i], O_RDONLY)) == -1) {
				perror(argv[i]);
			    	return(-1);
			}
		}
		count = BUFFSIZE_IN_BYTES;
		count = read(g_soundfile_handle, g_buff, count);
		while (count > 0 && count <= BUFFSIZE_IN_BYTES) {	
		    if (count != write(g_msnd_handle, g_buff, count)) {
			status = 1;
			break;
		    }
		    count = read(g_soundfile_handle, g_buff, count);
		}
		if (close(g_soundfile_handle)) {
			return(-1);
		}
	    }
	    if (g_autorewind_flag)
		if (!lseek(g_soundfile_handle, 0, 0))
			return(-1);
	} while (!status && g_autorewind_flag);
	return(0);
}


int rec(argc, argv)
int argc;
char *argv[];
{
	int count;

	if (g_stdinout) {
		g_soundfile_handle = 1;
	} else {
		if ((g_soundfile_handle = open(argv[g_fstfileidx],
		    O_WRONLY|O_CREAT|O_TRUNC,0x1b6)) == -1) {
			perror(argv[g_fstfileidx]);
		    	return(-1);
		}
	}
	count = BUFFSIZE_IN_BYTES;
	count = read(g_msnd_handle, g_buff, count);
	while (count > 0 && count <= BUFFSIZE_IN_BYTES) {	
		if (count != write(g_soundfile_handle, g_buff, count)) {
			perror(MULTISOUNDDEVICE);
			return(-1);
	    	}
		count = read(g_msnd_handle, g_buff, count);
	}
	if (close(g_soundfile_handle)) {
		perror("output");
		return(-1);
	}
	return(0);
}

void main(argc, argv)
int argc;
char *argv[];
{
	int access, errorcode = 0;

	/* our invocation name determines our job */
	if (argv[0][0] == 'p') {
		g_runmode = PLAY;
		access = O_WRONLY;
	} else if (argv[0][0] == 'r') {
		g_runmode = REC;
		access = O_RDONLY;
	} else {
		g_runmode = MSNDCTL;
		access = O_RDONLY;
	}

	/* open and read config of driver */
	if ((g_msnd_handle =
	    open(MULTISOUNDDEVICE, access)) == -1) {
		perror(MULTISOUNDDEVICE);
		exit(-1);
	}
	if (read_config())
		exit(-1);

	/* at the worst we can at least show driver config */
	if (scan_and_check_args(argc, argv)) {
		show_config_and_usage(argv[0]);
		errorcode = -1;
	}

	/* maybe we have to re-config driver ... */ 
	if (config_device_if_needed())
		exit(-1);

	/* ... if so we ihave to reopen */
	if (g_needreopen) {
		if (close(g_msnd_handle)) {
			perror(MULTISOUNDDEVICE);
			exit(-1);
		}
		if ((g_msnd_handle =
		    open(MULTISOUNDDEVICE, access)) == -1) {
			perror(MULTISOUNDDEVICE);
			exit(-1);
		}
	}

	/* do the job if it wasn't only looking at config and/or reconfig */
	if (!errorcode) {
		switch(g_runmode) {
		    default:
		    case MSNDCTL:                                   
			break;
		    case PLAY:
			errorcode = play(argc, argv);
			break;
		    case REC:
			errorcode = rec(argc, argv);
			break;
		}
	}

	if (close(g_msnd_handle)) {
		perror(MULTISOUNDDEVICE);
		if (!errorcode)
			errorcode = -1;
	}
	exit(errorcode);
}	
/*
 *	EOT
 */
