#define DEBUG 0
static char g_ident[] = "@(#)Driver MultiSound $Revision: 2.1.0.1 $ M.Mummert";
/*
 *	$Id: msnd.c,v 2.1.0.1 1995/04/11 14:46:40 mummert Exp mummert $
 *
 *  	msnd.c 
 *	======
 *
 *  	UN*X V386/R3.2 character driver for audio recording and playback
 *	with MultiSound soundcard made by Turtle Beach Systems, Inc. 
 *
 *	Copyright (c) 1994-1995 by Markus Mummert
 *
 *	Redistribution and use of this software, modifcation and inclusion
 *	into other forms of software are permitted provided that the following
 *	conditions are met:
 *
 *	1. Redistributions of this software must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	2. If this software is redistributed in a modified condition
 *	   it must reveal clearly that it has been modified.
 *	
 *	THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 *	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 *	TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *	PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR
 *	CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *	EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *	PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *	PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 *	OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *	USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 *	DAMAGE.
 *
 *  	Interfacing procedures originate solely from disassembly of
 *	the DOS-support programs msndtrec.com, msndtply.com and msdiag.exe
 *	being part of Turtle Beach Systems MultiSound software V1.0.
 *      The DSP-programs mdiag.bin msndpp11.bin and msndpr11.bin - as they
 *	are bundled with the MultiSound software V1.0 - have to be
 *	incorporated as binary image for the driver to become functional.
 *	They are NOT included in this file for copyright reasons.
 *	The MultiSound software V1.0 is publicly available.
 *
 * 	extern: 	msndio.h 	ioctl-defines
 *			Space.c		kernel build-time presets
 *
 *      history:        21.8.92/1   	begin coding
 *		 	29.8.92/V1.5	begin RCS-logging
 *			(...)		see RCS history or file HISTORY
 *			27.1.95/V2.1	sysdeps moved to osdep.h; g_state
 *				        becomes m_driver_state in Space.c;
 *					check_MSNDseg_unused() modfied;
 *					copyout arg-types fixed; design
 *					now allows buffersizes smaller
 *					than banksizes, mem alloc scheme
 *					changed to allocate buffs one by
 *					one, ENOMEM on error; renamed
 *					several symbols; secure driver
 *					states to allow kill during
 *					sleep ->EINTR.
 */

#define _IN_SOURCE_MSND_C_
#include "osdep.h"				/* System dependecies	*/
#include "msndio.h"				/* ioctl-defines	*/

/*
 *
 *	local defines
 *	-------------
 *
 */
						/* delay times in msec	*/	
#define SHORT_DELAY_MSEC	10             	/*   when system is	*/
#define MID_DELAY_MSEC		50             	/*   multitasking, tics */
#define LONG_DELAY_MSEC		100  		/*   for m_DELAY() - or	*/
#define HUGE_DELAY_MSEC		500 		/*   spinloop cycles	*/
						/*   as in msndinit()	*/
#define SHORT_DELAY_TICS	((HZ*SHORT_DELAY_MSEC)/1000)
#define MID_DELAY_TICS		((HZ*MID_DELAY_MSEC)/1000)
#define LONG_DELAY_TICS		((HZ*LONG_DELAY_MSEC)/1000)
#define HUGE_DELAY_TICS		((HZ*HUGE_DELAY_MSEC)/1000)

#define SHORT_DELAY_CYCLES	(m_lpspms * SHORT_DELAY_MSEC)
#define MID_DELAY_CYCLES	(m_lpspms * MID_DELAY_MSEC)      
#define LONG_DELAY_CYCLES	(m_lpspms * LONG_DELAY_MSEC)
#define HUGE_DELAY_CYCLES	(m_lpspms * HUGE_DELAY_MSEC)
						/* max. number of wait	*/
#define MAX_BITWAIT_CYCLES	200             /*   cycles for bit	*/
						/*   wait on MSNDregs	*/
#define MAX_BITWAIT_RETRIES     5              	/*   before some short	*/
						/*   delayed retries	*/
						/* max. number of read	*/
#define MAX_FLUSH_CYCLES	200		/*   cycles to flush	*/
						/*   MSNDreg5_7	before	*/
#define MAX_FLUSH_RETRIES       10             	/*   some mid delayed	*/
						/*   retries to flush	*/
						/* DSP program types	*/
#define MSNDUNDEF               0               /*   loaded up to	*/
#define	MSNDPLAY                1               /*   MultiSound	via	*/
#define MSNDREC                 2               /*   install_MSNDpgm()	*/
#define MSNDDIAG		3
						/* bits m_driver_state 	*/
						/* (from Space.c)	*/
#define INSTALLED		0x01		/* describing state	*/
#define OPENED			0x02		/*   driver HAS ... 	*/
#define WRITTEN			0x04
#define STARTEDMSND            	0x08
#define STOPPEDMSND		0x10
#define PREPAREDBUFF		0x20
#define SLEEPER			0x40
						/* makros to handle	*/
						/* m_driver_state:	*/
#define is(x)			(m_driver_state & (x))
#define isnot(x)		(!(m_driver_state & (x)))
#define set(x)			(m_driver_state |= (x))
#define clr(x)			(m_driver_state &= ~(x))
						/* other makros:	*/
#if DEBUG
#define	DEB(action)		(action)
#else
#define	DEB(action)
#endif

#define rate_to_index(r)	(((r)==M_CLKDIV2)*1+((r)==M_CLKDIV4)*2)

/*
 *
 *	configurable external globals (see Space.c)
 *	-------------------------------------------
 *
 */

extern int			/* MultiSound on board parameters ...	*/
    m_MSNDiobase,		/*   I/O-adress				*/
    m_MSNDseg,			/*   shared memory segment		*/
    m_MSNDint,			/*   interrupt line			*/
    m_MSNDinptval,		/*   input pot preset			*/
    m_MSNDauxptval,		/*   aux. input pot preset		*/

    m_driver_state,		/* Driver state (used as bit array) 	*/

				/* Driver default presets:		*/ 
    m_monitor,   		/*   monitor function			*/
    m_mode,			/*   mono or stereo			*/
    m_rate,			/*   sample rate clock divider		*/
    m_smpfmt,			/*   swap bytes in word sample or don't	*/
    m_recbytes,			/*   limit of bytes to be recorded	*/
    m_playbytes,		/*   or to by played (M_NOLIMIT=0)	*/
    m_lpspms,			/*   empty for loops per msec (sys.dep.)*/
    m_buffed_banks_rec[], 	/*   number of buffered banks rec/play	*/
    m_buffed_banks_play[]	/*     for m_rate = M_CLKDIV1/2/4	*/
;				/*     (>= MINBUFFS and <= MAXBUFFS) 	*/

/*
 *
 *	fixed and state dependend globals
 *	---------------------------------
 *
 */
static int
	g_buffs,		/* no. of buffers allocated in open	*/
	g_buff_state[MAXBUFFS],	/* state for each buffer, FULL/EMPTY	*/
	g_installedpgm,		/* currently running dsp program on M.	*/
	g_installedrate,	/* sampling rate hereof			*/
	g_intserv_buff_idx,	/* index of cur. served buffer by intr	*/
	g_rwclient_buff_idx,	/* as above, but by read/write ops.	*/
	g_rwclient_buff_pos,	/* cur. byte pos. in cur. serv. buffer	*/
	g_recbytes_ctr,		/* counter to track recorded bytes	*/
	g_playbytes_ctr,	/* same for played bytes		*/
	g_buff_err_ctr,		/* counts out-of-syncs events between	*/
				/*   intserv and read/writeserv		*/
	g_MSNDbank_err_ctr,	/* counts out-of-sync detected by dsp	*/
	g_MSNDseg_code,		/* shared mem. segment code on M.	*/
	g_MSNDint_code,		/* interrupt line code on M.		*/
	g_MSNDreg1,		/* MultiSound register values ...	*/
	g_MSNDreg5,
	g_MSNDreg6,
	g_MSNDreg7,

	g_MSNDreg0_adr,		/* MultiSound register adresses ...	*/
	g_MSNDreg1_adr,
	g_MSNDreg2_adr,
	g_MSNDreg3_adr,
	g_MSNDreg4_adr,
	g_MSNDreg5_adr,
	g_MSNDreg6_adr,
	g_MSNDreg7_adr,
	g_MSNDreg8_adr,
	g_MSNDreg9_adr,
	g_MSNDreg10_adr,
	g_MSNDreg11_adr,
	g_MSNDreg12_adr,
	g_MSNDreg13_adr,
				/* allowed values for configuration	*/
				/* validation ...			*/
	g_valid_MSNDiobase_tab[] = {
		0x290, 0x260, 0x250, 0x240,
		0x230, 0x220, 0x210, 0x3E0
	},
	g_valid_MSNDint_tab[] = {
		0, 5, 7, 9, 10, 11, 12, 15
	},
	g_valid_MSNDseg_tab[] = {
		0x00000, 0xB0000, 0xC8000, 0xD0000,
		0xD4000, 0xD8000, 0xE0000, 0xE8000
	},
				/* interrupt service calltable		*/
	bank_err_on_MSND(), buff_queue_to_bank1(), bank1_to_buff_queue(),
	buff_queue_to_bank2(), bank2_to_buff_queue(), no_service(),

	(*g_intservicetab[])() = {
		bank_err_on_MSND,
		buff_queue_to_bank1,
		bank1_to_buff_queue,
		buff_queue_to_bank2,
		bank2_to_buff_queue,
		no_service
	}
;
static caddr_t			/* bank and buffer adresses		*/
	g_MSNDbank1,
	g_MSNDbank2,
	g_buff_adr[MAXBUFFS]
;

/*
 *
 *	MultiSound DSP-programs
 *      -----------------------
 *
 *	The program code msndprXX.bin to be loaded into the dsp for
 *	recording at 11.025, 22.05 and 44.1kHz differs only at two
 *	positions which will be patched later as required.
 *	Therefore we only need to provide one array of code.
 *	The same procedure WORKS for all 3 rates of playback code
 *	as well, at least if the code for 11.025 or 22.05kHz is
 *	being patched. However, the code provided by
 *	msndpp44.bin for 44.1kHz differs widely from
 *	the code of the other 2 rates. I have no idea what this
 *	difference is about. Of course we could provide an
 *	extra array for 44.1kHz playback ... (TO BE INVESTIGATED)
 *	The code provided by mdiag.bin is needed to reset the board
 *	(kill PROTEUS demo ...) and to adjust the input gain.
 *
 */

static BYTE g_MSNDpgm_PLAY[0x600] = {
### msndpp11.bin to be patched in by make as init code ###
};        
static BYTE g_MSNDpgm_REC[0x600] = {
### msndpr11.bin to be patched in by make as init code ###
};        
static BYTE g_MSNDpgm_DIAG[0x600] = {
### mdiag.bin to be patched in by make as init code ###
};        

/*
 *
 *	MultiSound low level routines
 *	-----------------------------
 *
 */

static void set_bit0_MSNDreg0()
{
	m_OUTB(g_MSNDreg0_adr, m_INB(g_MSNDreg0_adr) | 0x01); 
}

static void clr_bit0_MSNDreg0()
{
	m_OUTB(g_MSNDreg0_adr, m_INB(g_MSNDreg0_adr) & 0xFE);
}

static void set_bit1_MSNDreg0()
{
	m_OUTB(g_MSNDreg0_adr, m_INB(g_MSNDreg0_adr) | 0x02); 
}

static void clr_bit1_MSNDreg0()
{
	m_OUTB(g_MSNDreg0_adr, m_INB(g_MSNDreg0_adr) & 0xFD);
}

static void set_seg_on_MSND(code)
int code;
{
	m_OUTB(g_MSNDreg8_adr, code);
}

static void read_MSNDreg5_7()
{
	g_MSNDreg5 = m_INB(g_MSNDreg5_adr);
	g_MSNDreg6 = m_INB(g_MSNDreg6_adr);
	g_MSNDreg7 = m_INB(g_MSNDreg7_adr);
}

static void write_MSNDreg5_7()
{
	m_OUTB(g_MSNDreg5_adr, g_MSNDreg5);
	m_OUTB(g_MSNDreg6_adr, g_MSNDreg6);
	m_OUTB(g_MSNDreg7_adr, g_MSNDreg7);
}

static void set_intvector_on_MSND(code)
int code;
{
	set_bit1_MSNDreg0();
	m_OUTB(g_MSNDreg9_adr, code);
	clr_bit1_MSNDreg0();
}

static int wait_bit7_MSNDreg1_clear()
{
	int ctr =  MAX_BITWAIT_CYCLES, i;

	while ((m_INB(g_MSNDreg1_adr) & 0x80) !=0) {
		if (--ctr < 0) {
			if (ctr < -MAX_BITWAIT_RETRIES)
				return(-1);
			DEB(printf(" waiting_reg1_bit7 "));
			/*
			 * During msndinit we're not multitasking yet,
			 * so delay() cannot be used, use spin-loop instead
			 */
			if (is(INSTALLED)) {
				m_DELAY(SHORT_DELAY_TICS);
			} else {
				for (i = SHORT_DELAY_CYCLES; i > 0; i--);
			}
		}
	}
	return(0);
}


static int wait_bit1_MSNDreg2_set()
{
	int ctr =  MAX_BITWAIT_CYCLES, i;

	while ((m_INB(g_MSNDreg2_adr) & 0x02) == 0) {
		if (--ctr < 0) {
			if (ctr < -MAX_BITWAIT_RETRIES)
				return(-1);
			DEB(printf(" waiting_reg2_bit1 "));
			/* delay() cannot be used during msndinit() ...	*/
			if (is(INSTALLED)) {
				m_DELAY(SHORT_DELAY_TICS);
			} else {
				for (i = SHORT_DELAY_CYCLES; i > 0; i--);
			}
		}
	}
	return(0);
}


static int waitready_write_MSNDreg1()
{
	if (wait_bit7_MSNDreg1_clear()) 
		return(-1);
	m_OUTB(g_MSNDreg1_adr, g_MSNDreg1);
	return(0);
}

static int waitready_write_MSNDreg5_7()
{
	if (wait_bit1_MSNDreg2_set()) 
		return(-1);
	write_MSNDreg5_7();
	return(0);
}

static int flush_MSNDreg5_7()
{
	int ctr = MAX_FLUSH_CYCLES, i;

	while ((m_INB(g_MSNDreg2_adr) & 0x01) != 0) {
	    read_MSNDreg5_7();
	    if (--ctr < 0) {
		if (ctr < -MAX_FLUSH_RETRIES) {
		    DEB(printf("\nERROR: cannot flush MultiSound reg5-7\n"));
		    return(-1);
		}
		DEB(printf(" try_flush_reg5-7 "));
		/* delay() cannot be used during msndinit() ...	*/
		if (is(INSTALLED)) {
		    m_DELAY(MID_DELAY_TICS);
		} else {
		    for (i = MID_DELAY_CYCLES; i > 0; i--);
		}
	    }
	}
	return(0);
}

/* I don't now what these crazy turtles live on ... */
static int feed_byte_to_MSND(b)
int b;
{
	g_MSNDreg1 = 0x9C;
	g_MSNDreg5 =  0;
	g_MSNDreg6 =  b;
	g_MSNDreg7 =  0;
	if (!waitready_write_MSNDreg5_7())
	    if (!waitready_write_MSNDreg1())
		return(0);
	return(-1);
}

/*
 *
 *	MultiSound mid level routines
 *	-----------------------------
 *
 */

/* this gives us a c(l)ick on the DAC: */
static void full_reset_MSND()
{
	clr_bit0_MSNDreg0();
	set_intvector_on_MSND(0);
	set_seg_on_MSND(0);
	m_OUTB(g_MSNDreg10_adr, 1);
	m_OUTB(g_MSNDreg11_adr, 1);
	m_OUTB(g_MSNDreg12_adr, 0);
	m_OUTB(g_MSNDreg13_adr, 0);
	m_OUTB(g_MSNDreg10_adr, 0);
	m_OUTB(g_MSNDreg11_adr, 0);
	clr_bit0_MSNDreg0();
	set_intvector_on_MSND(0);
	set_seg_on_MSND(0);
}


/* looks like this doesn't influence the ADC/DAC section */
static void small_reset_MSND()
{
	clr_bit0_MSNDreg0();
	set_intvector_on_MSND(0);
	set_seg_on_MSND(0);
	m_OUTB(g_MSNDreg10_adr, 1);
	m_OUTB(g_MSNDreg12_adr, 0);
	m_OUTB(g_MSNDreg13_adr, 0);
	m_OUTB(g_MSNDreg10_adr, 0);
	clr_bit0_MSNDreg0();
	set_intvector_on_MSND(0);
	set_seg_on_MSND(0);
}

static int upload_MSND(buff)
BYTE *buff;
{
	int i;

	for (i = 0; i < 0x600;   ) {
		g_MSNDreg5 = buff[i++];
		g_MSNDreg6 = buff[i++];
		g_MSNDreg7 = buff[i++];
		if (waitready_write_MSNDreg5_7()) {
			DEB(printf("\nERROR: cannot upload Multisound\n"));
			return(-1);
		}
	}
	return(0);
}

static int install_MSNDpgm(pgm, rate)
int pgm, rate;
{
	BYTE *pgmcode, ratepatch;

	switch (pgm) {
	default:
	case MSNDPLAY:	pgmcode = g_MSNDpgm_PLAY; break;
	case MSNDREC:	pgmcode = g_MSNDpgm_REC; break;
	case MSNDDIAG:	pgmcode = g_MSNDpgm_DIAG; break;
	}
	switch (rate) {
	default:
	case M_CLKDIV4:	ratepatch = 2; break;
	case M_CLKDIV2:	ratepatch = 1; break;
	case M_CLKDIV1:	ratepatch = 0; break;
	}
	if (pgm == MSNDPLAY || pgm == MSNDREC)
		pgmcode[0x110] = pgmcode[0x116] = ratepatch;
	if (upload_MSND(pgmcode))
		return(-1);
	return(0);
}

/* is maybe unnecessary: */
static int put_MSNDadda_out_of_reset()
{
	g_MSNDreg5 = 1;
	g_MSNDreg6 = 0;
	g_MSNDreg7 = 0;
	g_MSNDreg1 = 0x9C;
	if (!waitready_write_MSNDreg5_7()) 
	    if (!waitready_write_MSNDreg1())
		return(0);
	DEB(printf("\nERROR: cannot put MultiSound AD/DA out of reset\n"));
	return(-1);
}

static int set_MSNDpot(potcode, value)
int potcode;
int value;
{
	int i, bitmask = 0x8000;

	if (!feed_byte_to_MSND(potcode)) {
	    if (!feed_byte_to_MSND(potcode + 0x08)) {
		if (!feed_byte_to_MSND(potcode)) {
		    for (i = 16;   ; i--) {
			if (bitmask & value) {
			    if (feed_byte_to_MSND(potcode + 0x10)) break;
			    if (feed_byte_to_MSND(potcode + 0x18)) break;
			    if (feed_byte_to_MSND(potcode)) break;
			} else {
			    if (feed_byte_to_MSND(potcode + 0x08)) break;
			    if (feed_byte_to_MSND(potcode)) break;
			}
			if (i <= 1)
			    if (!feed_byte_to_MSND(0))
				return(0);
			bitmask >>= 1;
		    }
		}
	    }
	}
	return(-1);
}

static int set_all_MSNDpots()
{
	if (!set_MSNDpot(0x02, m_MSNDinptval))
	    if (!set_MSNDpot(0x04, m_MSNDauxptval))
		if (!feed_byte_to_MSND(0x80))
		    return(0);
	DEB(printf("\nERROR: cannot set pot value\n"));
	return(-1);
}

/*
 *
 *	MultiSound configurable hardware parameter check
 *	------------------------------------------------
 *
 */

static int check_MSNDseg_unused()
{
/*
 * FIXME:
 * 	on a demand page system we obviously can't decide if
 *	we have RAM or foreign ADAPTOR already residing at
 *	to-be-MultiSound mem location by just poking in there:
 *	no RAM/ADPATOR -> page fault -> page in -> RAM
 *	(sniff)
 */
#if CHECK_MSND_SEG_UNUSED
	UBYTE *p, *e;
	UBYTE b;

	set_seg_on_MSND(0);
	if (m_MSNDseg != 0) {
		p = (UBYTE*)g_MSNDbank1;
		e = p + 2*BANKSIZE_IN_BYTES;
		do {
			b = *p;
			*p = 0;
			if (*p == 0) {
				*p = 0xFF;
				if (*p == 0xFF) {
					*p = b;
					return(-1);
				}
			}
			*p = b;
		} while( ++p != e);
	}
#endif
	return(0);
}

static int set_MSNDioadr()
{
	int i, adr = m_MSNDiobase;

	for (i = 0; i < 8; i++)
		if (adr == g_valid_MSNDiobase_tab[i])
			break;
	if (i > 7)
		return(-1);
	g_MSNDreg0_adr = adr++;
	g_MSNDreg1_adr = adr++;
	g_MSNDreg2_adr = adr++;
	g_MSNDreg3_adr = adr++;
	g_MSNDreg4_adr = adr++;
	g_MSNDreg5_adr = adr++;
	g_MSNDreg6_adr = adr++;
	g_MSNDreg7_adr = adr++;
	g_MSNDreg8_adr = adr++;
	g_MSNDreg9_adr = adr++;
	g_MSNDreg10_adr = adr++;
	g_MSNDreg11_adr = adr++;
	g_MSNDreg12_adr = adr++;
	g_MSNDreg13_adr = adr++;
	return(0);
}

static int MSNDseg_to_code()
{
	int i;

	for (i = 0; i < 8; i++)
		if (m_MSNDseg == g_valid_MSNDseg_tab[i])
			break;
	g_MSNDseg_code = i;
	if (i >= 8) {
		g_MSNDseg_code = 0;
		return(-1);
	}
	return(0);
}

static int MSNDint_to_code()
{
	int i;

	for (i = 0; i < 8; i++)
		if (m_MSNDint == g_valid_MSNDint_tab[i])
			break;
	g_MSNDint_code = i;
	if (i >= 8) {
		g_MSNDint_code = 0;
		return(-1);
	}
	return(0);
}

/*
 *
 *	MultiSound high level control interface
 * 	---------------------------------------
 *
 *	Note: Only call these routines to control
 *	MultiSound from higher level. They handle
 *	the lower stages appropriatly.
 *
 */

static int read_and_set_cfg()
{
	if (set_MSNDioadr()) {
		printf("ERROR: undefined MultiSound i/o-adress");
		return(ENODEV);
	}
	if (MSNDseg_to_code() || !g_MSNDseg_code) {
		printf("ERROR: undefined MultiSound memory segment");
		return(ENODEV);
	}
	g_MSNDbank1 = (caddr_t)phystokv((paddr_t)(m_MSNDseg));
	g_MSNDbank2 = g_MSNDbank1 + BANKSIZE_IN_BYTES;
	if (check_MSNDseg_unused()) {
		printf("ERROR: MultiSound memory segment used by other");
		return(ENODEV);
	}
	if (MSNDint_to_code() || !g_MSNDint_code) {
		printf("ERROR: undefined Multisound interrupt number");
		return(ENODEV);
	}
	return(0);
}

static int initialize_MSND()
{
	int i;

	full_reset_MSND();
	clr(STARTEDMSND);
	g_installedpgm = MSNDUNDEF;
	if (install_MSNDpgm(MSNDDIAG, 0))
		return(ENODEV);
	if (flush_MSNDreg5_7())
		return(ENODEV);
	if (put_MSNDadda_out_of_reset())
		return(ENODEV);
	if (is(INSTALLED)) {
		m_DELAY(HUGE_DELAY_TICS);
	} else {
		for (i = HUGE_DELAY_CYCLES; i > 0; i--);
	}
	if (set_all_MSNDpots())
		return(ENODEV);
	g_installedpgm = MSNDDIAG;
	g_installedrate = 0;
	return(0);
}

static int config_MSND(pgm, rate)
int pgm, rate;
{
	int i;

	small_reset_MSND();
	clr(STARTEDMSND);
	set_seg_on_MSND(g_MSNDseg_code);
	g_installedpgm = MSNDUNDEF;
	if (install_MSNDpgm(pgm, rate))
		return(ENODEV);
	if (is(INSTALLED)) {
		m_DELAY(LONG_DELAY_TICS);
	} else {
		for (i = LONG_DELAY_CYCLES; i > 0; i--);
	}
	g_installedpgm = pgm;
	g_installedrate = rate;
	return(0);
}

static int reconfig_MSND(pgm, rate)
int pgm, rate;
{
	int i;

	/* if dsp is running the right code, let him live in peace */
	if (g_installedpgm == pgm && g_installedrate == rate)
		return(0);
	if (config_MSND(pgm, rate))
		return(ENODEV);
	return(0);
}

static void startplay_MSND()
{
	if (isnot(STARTEDMSND)) {
		DEB(printf(" play "));
		set_intvector_on_MSND(g_MSNDint_code);
		set_bit0_MSNDreg0();
		read_MSNDreg5_7();
		g_intserv_buff_idx = 0;
		m_OUTB(g_MSNDreg1_adr, buff_queue_to_bank1());
		m_OUTB(g_MSNDreg1_adr, buff_queue_to_bank2());
		m_OUTB(g_MSNDreg1_adr, 0x93);
		set(STARTEDMSND);
	}
}

static void startrec_MSND()
{
	if (isnot(STARTEDMSND)) {
		DEB(printf(" record "));
		set_intvector_on_MSND(g_MSNDint_code);
		set_bit0_MSNDreg0();
		read_MSNDreg5_7();
		m_OUTB(g_MSNDreg1_adr, 0x9B);
		m_OUTB(g_MSNDreg1_adr, 0x9D);
		m_OUTB(g_MSNDreg1_adr, 0x94);
		set(STARTEDMSND);
	}
}

static void stop_MSND()
{
	int x;

	m_DISABLE_INTR(x); 
	DEB(printf(" stop "));
	m_OUTB(g_MSNDreg1_adr, 0x92);
	read_MSNDreg5_7();
	clr_bit0_MSNDreg0();
	set_intvector_on_MSND(0);
	clr(STARTEDMSND);
	set(STOPPEDMSND);
	m_RESTORE_INTR(x); 
}

/*
 *
 *	buffer-queue handling on MultiSound side invoked by interrupt 
 *	-------------------------------------------------------------
 *
 */

static void int_rout()
{
	int code;
	
	/*
	 * this is a protection against spurious interrupts
	 * (we don't want the bank_to_buff_queue transfer to unload into
	 *  where unintialized buffer adresses might point, do we?)
	 */ 
	if (is(STARTEDMSND)) {
	    read_MSNDreg5_7();
	    if (g_MSNDreg5 <=5 && g_MSNDreg5 >=0){
		/*
		 * MultiSound codes via register 5 how it expects
		 * to be served. Use register 5 as index to calltable,
		 * thus calling one of the routines below.
		 * (except for buff_err_on_pc()). They return
		 * an acknoledge code that we push back to register 1.
		 */
		if ((code = (g_intservicetab[g_MSNDreg5])()) != 0)
			m_OUTB(g_MSNDreg1_adr, code);
	    }
	}
	if (is(SLEEPER))
		m_WAKEUP;
}

static int buff_err_on_pc()
{
	/* overrun in buffer queue detected */
	g_buff_err_ctr++;
	return(0);
	/*
	 * return value finally will be written in MSNDreg1
	 * by int_rout, 0->stop, other codes switch banks (s.b.)
	 */
}

static int buff_queue_to_bank1()
{
	int i;

	for (i = 0; i < BUFFS_PER_BANK; i++) {
		if (g_buff_state[g_intserv_buff_idx] == EMPTY)
			return (buff_err_on_pc());
		bcopy(g_buff_adr[g_intserv_buff_idx],
			g_MSNDbank1 + i*BUFFSIZE_IN_BYTES,
			BUFFSIZE_IN_BYTES);
		g_buff_state[g_intserv_buff_idx++] = EMPTY;
		g_intserv_buff_idx %= g_buffs;
	}
	return(0x9C);
}

static int buff_queue_to_bank2()
{
	int i;

	for (i = 0; i < BUFFS_PER_BANK; i++) {
		if (g_buff_state[g_intserv_buff_idx] == EMPTY)
			return (buff_err_on_pc());
		bcopy(g_buff_adr[g_intserv_buff_idx],
			g_MSNDbank2 + i*BUFFSIZE_IN_BYTES,
			BUFFSIZE_IN_BYTES);
		g_buff_state[g_intserv_buff_idx++] = EMPTY;
		g_intserv_buff_idx %= g_buffs;
	}
	return(0x9E);
}

static int bank1_to_buff_queue()
{
	int i;

	for (i = 0; i < BUFFS_PER_BANK; i++) {
		if (g_buff_state[g_intserv_buff_idx] == FULL)
			return (buff_err_on_pc());
		bcopy(g_MSNDbank1 + i*BUFFSIZE_IN_BYTES,
			g_buff_adr[g_intserv_buff_idx],
			BUFFSIZE_IN_BYTES);
		g_buff_state[g_intserv_buff_idx++] = FULL;
		g_intserv_buff_idx %= g_buffs;
	}
	return(0x9B);
}

static int bank2_to_buff_queue()
{
	int i;

	for (i = 0; i < BUFFS_PER_BANK; i++) {
		if (g_buff_state[g_intserv_buff_idx] == FULL)
			return (buff_err_on_pc());
		bcopy(g_MSNDbank2 + i*BUFFSIZE_IN_BYTES,
			g_buff_adr[g_intserv_buff_idx],
			BUFFSIZE_IN_BYTES);
		g_buff_state[g_intserv_buff_idx++] = FULL;
		g_intserv_buff_idx %= g_buffs;
	}
	return(0x9D);
}

static int bank_err_on_MSND()
{
	/* overrun detected by MultiSound */
	g_MSNDbank_err_ctr++;
	return(0);
}

static int no_service()
{
	return(g_MSNDreg7);
}

/*
 *
 *	buffer-queue handling on driver entry side
 *	------------------------------------------
 *
 */

static void mono_to_leftright_expand(buff)
WORD *buff;
{
	WORD *s, *e, *d;

	s = buff + BUFFSIZE_IN_BYTES/4;
	d = buff + BUFFSIZE_IN_BYTES/2;
	e = buff;
	/* this is slow, but disk I/O may relax on mono mode */
	do {				
		*--d = *--s;
		*--d = *s; 
	} while (s != e);
}

static void leftright_to_mono_shrink(buff)
WORD *buff;
{
	WORD *s, *e, *d;

	s = buff;
	d = buff;
	e = buff + BUFFSIZE_IN_BYTES/2;
	/* this is slow, but disk I/O may relax on mono mode */
	do {
		*d++ = (WORD)(((int)(*s++) + (int)(*s++))/2); 
	} while (s != e);
}

static void byteswap_samples(buff, size)
BYTE *buff;
int size;
{
	BYTE *s, *d, *e;
	BYTE b;

	s = buff;
	d = buff;
	e = buff + size;
	/* this IS slow at worst case recording max. rate stereo */
	do {
		b = *s++;
		*d++ = *s++;
		*d++ = b;
	} while (s != e);
}
		
static int wait_for_buff_to_be(state, index)
int state, index;
{
	int x, err = 0;

	if (g_buff_err_ctr || g_MSNDbank_err_ctr || is(STOPPEDMSND)) {
		stop_MSND();
		return(EIO);
	}	
	while (g_buff_state[index] != state) {
		if (state == FULL)
			startrec_MSND();
		if (state == EMPTY)
			startplay_MSND();
		/*
		 * enter CRITICAL/ATOMIC section ....
		 * (sleep only if we can be sure we'll be raised)
		 *
		 * UGLY:  We had this problem that we wanted to leave
		 *	  with EINTR if we got killed. On the other
		 *	  hand we were not able to flush the buff queue
		 *	  completely on close because we received strange
		 *	  wakeup during sleeping on to-be-flushed buffers.
		 *	  So we have this makro m_PROCESS_ABORTING which
		 *	  tells us when we are really killed.
		 *	  The m_SLEEP macro might help us do ugly things
		 *	  in osdep.h ... (to be replaced by DO_SLEEP)
		 */
		m_DISABLE_INTR(x); 
		if (g_buff_state[index] != state) {
			DEB(printf("("));
			set(SLEEPER);
			m_SLEEP;
			clr(SLEEPER);
			DEB(printf(")"));
			if (m_PROCESS_ABORTING)
 				err = EINTR;
			if (g_buff_err_ctr || g_MSNDbank_err_ctr) 
				err = EIO;
		}
		m_RESTORE_INTR(x); 
		/*
	 	 * ... leave CRITICAL/ATOMIC section 
	 	 */
		if (err) {
			stop_MSND();
			return(err);
		}
	}
	return(0);
}

static int read_buff_queue(dest, count)
BYTE *dest;
int count;
{
	BYTE *buff;
	int err, space, size, offset = 0;

	if (count < 0)
		return(EFAULT);
	while (count > 0) {
		g_rwclient_buff_idx %= g_buffs;
		if (err = wait_for_buff_to_be(FULL, g_rwclient_buff_idx))
			return(err);
		buff = g_buff_adr[g_rwclient_buff_idx];
		size = BUFFSIZE_IN_BYTES/(m_mode == M_MONO ? 2 : 1);
		space = size - g_rwclient_buff_pos;
		if (isnot(PREPAREDBUFF)) {
			if (m_mode == M_MONO)
				leftright_to_mono_shrink(buff);
			if (m_smpfmt == M_SWAP) 
				byteswap_samples(buff, size);
			set(PREPAREDBUFF);
		}
		if (count < space) {
			if (copyout(buff + g_rwclient_buff_pos,
			    dest + offset, count)) 
				return(EFAULT);
			g_rwclient_buff_pos += count;
			count = 0;
		} else {
			if (copyout(buff + g_rwclient_buff_pos,
			    dest + offset, space))
				 return(EFAULT);
			g_buff_state[g_rwclient_buff_idx++] = EMPTY;
			g_rwclient_buff_pos = 0;
			clr(PREPAREDBUFF);
			count -= space;
			offset += space;
		}
	}
	return(0);
}

static int write_buff_queue(source, count)
BYTE *source;
int count;
{
	char *buff;
	int err, space, size, offset = 0;

	if (count < 0)
		return(EFAULT);
	while (count > 0) {
		g_rwclient_buff_idx %= g_buffs;
		if (err = wait_for_buff_to_be(EMPTY, g_rwclient_buff_idx))
			return(err);
		buff = g_buff_adr[g_rwclient_buff_idx];
		size = BUFFSIZE_IN_BYTES/(m_mode == M_MONO ? 2 : 1);
		space = size - g_rwclient_buff_pos;
		if (count < space) {
			if (copyin(source + offset,
			    buff + g_rwclient_buff_pos, count)) 
				return(EFAULT);
			g_rwclient_buff_pos += count;
			count = 0;
		} else {
			if (copyin(source + offset, 
			    buff + g_rwclient_buff_pos, space))
				return(EFAULT);
			if (m_smpfmt == M_SWAP) 
				byteswap_samples(buff, size);
			if (m_mode == M_MONO)
				mono_to_leftright_expand(buff);
			g_buff_state[g_rwclient_buff_idx++] = FULL;
			g_rwclient_buff_pos = 0;
			count -= space;
			offset += space;
		}
		set(WRITTEN);
	}
	return(0);
}

static int write_zero_completed_buff_to_queue()
{
	char *buff, *p, *e;
	int space, size, err;

	g_rwclient_buff_idx %= g_buffs;
	if (err = wait_for_buff_to_be(EMPTY, g_rwclient_buff_idx))
		return(err);
	buff = g_buff_adr[g_rwclient_buff_idx];
	size = BUFFSIZE_IN_BYTES/(m_mode == M_MONO ? 2 : 1);
	if ((size - g_rwclient_buff_pos) > 0) {
		p = buff + g_rwclient_buff_pos;
		e = buff + size;
		do {
			*p++ = 0;
		} while (p != e);
	}
	if (m_smpfmt == M_SWAP) 
		byteswap_samples(buff, size);
	if (m_mode == M_MONO)
		mono_to_leftright_expand(buff);
	g_buff_state[g_rwclient_buff_idx++] = FULL;
	g_rwclient_buff_pos = 0;
	set(WRITTEN);
	return(0);
}

static int flush_buff_queue()
{
	int err, i, watch;
	/*
	 * check if we were not written or MultiSound
	 * isn't running anyway
	 */
	if (isnot(WRITTEN) || is(STOPPEDMSND)) {
		stop_MSND();
		return(0);
	}
	watch = (g_rwclient_buff_idx + 2*BUFFS_PER_BANK) % g_buffs;
	/*
	 * (MultiSound is still running and was written)
	 * The current buff might be empty or left partly filled
	 * with relevant data (see write_buff_queue()). We consider
	 * this buff the last that MUST be played. To be sure to have
	 * it played we have to stuff some zeroed buffs into the queue. 
	 * Then we'll watch a buffer TWO BANKS away from current
	 * buff to become empty. We'll be notified by wakeup when it got
	 * emptied via int_rout. At this time MultiSound has finished playing
	 * the bank containing our last relevant buffer and has just
	 * started playing a first irrelevant bank (containing zeros).
	 * We are only notified of the watch buffer being emptied when
	 * int_rout() has already moved data onto a second irrelevant bank.
	 * Now we can safely halt MultiSound. In order keep not to detect
	 * an overrun condition in int_rout() we have to have another 
	 * three zeroed buffers waiting behind.
	 *
	 * Summary:	Assume we have 4 buffs fitting on a bank, than
	 *		stuff current buff padded with zeros and another 11
	 *		zero buffs onto the que. Thats 12 buffs to be pushed.
	 *		Wait until the 9th got emptied, then stop MultiSound.
	 *		Easy, eh?
	 *
	 */
	DEB(printf(" flushing "));
	for (i = 3*BUFFS_PER_BANK; i > 0; i--)
		if (err = write_zero_completed_buff_to_queue())
			return(err);
	if (err = wait_for_buff_to_be(EMPTY, watch)) {
		return(err);
	}
	stop_MSND();
	return(0);
}

static int check_transm_err()
{
	int err = 0;

	if (g_MSNDbank_err_ctr) {
		DEB(printf("\nERROR: bank error(s) on Multisound\n"));
		err = EIO;	/* overrun detected by MultiSound	*/	
	}
	if (g_buff_err_ctr) {
		DEB(printf("\nERROR: buffer error(s) on host\n"));
		err = EIO;	/* overrun detected by us		*/	
	}
	return(err);
}

static void free_buffs_until(badindex) 
int badindex;
{
	int i;

	for (i = 0; i < badindex; i++) 
		sptfree(g_buff_adr[i], btoc(BUFFSIZE_IN_BYTES), 1);
	 
}

static int allocate_buffs(n) 
int n;
{
	int i;
	/*
	 * Allocate buffers one be one.
	 * Should we fail once return already allocated buffs
	 * to avoid mem leakage
	 */
 	for (i = 0; i < n; i++) {
	    if ((g_buff_adr[i] = sptalloc(btoc(BUFFSIZE_IN_BYTES),
	 	PG_P|PG_RW, 0, 0)) == NULL) {
		    DEB(printf("\nERROR: not enough memory for MultiSound\n"));
		    free_buffs_until(i);
		    return(ENOMEM);
	    }
	}
	DEB(printf(" %d-buffers allocated ", n));
	return(0);
}

/*
 *
 * SUMMARY: calls (as defined before) to implement buffered MultiSound driver
 * --------------------------------------------------------------------------
 *
 *	static int reconfig_MSND(int pgm, int rate)
 *	static int config_MSND(int pgm, int rate)
 *	static int initialize_MSND()
 *	static int read_and_set_cfg()
 *	static void stop_MSND()
 *	static void int_rout()
 *
 *	static int allocate_buffers(int n) 
 *	static void free_buffers_until(int badindex) 
 *	static int read_buff_queue(int dest, int count)
 *	static int write_buff_queue(int source, int count)
 *	static int flush_buff_queue()
 *	static int check_transm_err()
 *
 */
	
/*
 *
 *	entry points for SysV386 specific driver
 *	----------------------------------------
 *
 */

void msndinit()
{
	int pgm;

	m_driver_state = 0;
	/* this is a last warning for hackers on the external Space.c: */
	if (read_and_set_cfg()) {
		printf(" - DRIVER NOT INSTALLED\n");
		return;
	}
	pgm = (m_monitor == M_SILENT ? MSNDPLAY : MSNDREC);
	if (!initialize_MSND()) {
		if (!config_MSND(pgm, m_rate)) {
			set(INSTALLED);
			printf("%s I/O:0x%x IRQ:%d MEM:0x%x\n",
				&g_ident[4],
				m_MSNDiobase,
				m_MSNDint,
				m_MSNDseg);
			return;
		}
	}
	/* do you have a MultiSound card at all? */
	printf("ERROR: no MultiSound at I/O-adress");
	printf(" - DRIVER NOT INSTALLED\n");
}

void msndintr()
{
	if (isnot(INSTALLED))
		return;
	/* show buffer que length */
	DEB(printf("<%d>",(g_intserv_buff_idx+g_buffs-g_rwclient_buff_idx) %
		g_buffs));
	int_rout();
}

void msndopen(dev, flags)
dev_t dev;
int flags;
{
	int i;

	if (isnot(INSTALLED)) {
		u.u_error = ENODEV;
		return;
	}
	if (is(OPENED)) {
		/* yes, we're busy */
		u.u_error = EBUSY;
		return;
	}
	DEB(printf("\n\nOPENING: "));
	clr(WRITTEN|PREPAREDBUFF|SLEEPER|STARTEDMSND|STOPPEDMSND);

	if ((flags & FREAD) && !(flags & FWRITE)) {
		if (u.u_error = reconfig_MSND(MSNDREC, m_rate)) 
			return;
		g_buffs = BUFFS_PER_BANK *
			m_buffed_banks_rec[rate_to_index(m_rate)];
	} else if (!(flags & FREAD) && (flags & FWRITE)) {
		if (u.u_error = reconfig_MSND(MSNDPLAY, m_rate)) 
			return;
		g_buffs = BUFFS_PER_BANK *
			m_buffed_banks_play[rate_to_index(m_rate)];
	} else {
		u.u_error = ENXIO;
		return;
	}
	if (g_buffs < MINBUFFS)
		g_buffs = MINBUFFS;
	if (g_buffs > MAXBUFFS)
		g_buffs = MAXBUFFS;

 	for (i = 0; i < g_buffs; i++) {
		g_buff_state[i] = EMPTY;
		g_buff_adr[i] = NULL;
	}
	g_buff_err_ctr = 0;
	g_MSNDbank_err_ctr = 0;
	g_intserv_buff_idx = 0;
	g_rwclient_buff_idx = 0;
	g_rwclient_buff_pos = 0;
	g_recbytes_ctr = 0;
	g_playbytes_ctr = 0;

	if (u.u_error = allocate_buffs(g_buffs)) 
	    return;
	u.u_offset = 0;
	set(OPENED);
	DEB(printf(" opened\n"));
}

void msndclose()
{
	int i, pgm, rate;

	DEB(printf("\nCLOSING: "));
	if (isnot(INSTALLED))
		DEB(printf(" was_not_installed " ));
	if (is(STOPPEDMSND))
		DEB(printf(" was_stopped " ));
	if (is(STARTEDMSND))
		DEB(printf(" is_recording/playing "));
	if (isnot(OPENED))
		DEB(printf(" wasnt_opened "));
	if (is(WRITTEN))
		DEB(printf(" was_written "));
	if (is(SLEEPER))
		DEB(printf(" is_sleeper "));
	if (is(PREPAREDBUFF))
		DEB(printf(" buff_was_prepared "));

	if (isnot(INSTALLED)) {		/* this can't happen ...	*/
		u.u_error = ENODEV;
		return;
	}				/* not open can't happen ...	*/
	if (is(OPENED)) {
		flush_buff_queue();
		free_buffs_until(g_buffs);
	}
	stop_MSND();
	pgm = (m_monitor == M_SILENT ? MSNDPLAY : MSNDREC);
	rate = (m_monitor == M_SILENT ? m_rate : M_CLKDIV1);
	if (u.u_error = config_MSND(pgm, rate)) {
	    clr(OPENED|WRITTEN|PREPAREDBUFF|SLEEPER|STARTEDMSND|STOPPEDMSND);
	    DEB(printf(" closed\n"));
	    return;
	}
	u.u_error = check_transm_err();
	clr(OPENED|WRITTEN|PREPAREDBUFF|SLEEPER|STARTEDMSND|STOPPEDMSND);
	DEB(printf(" closed\n"));
}

void msndread()
{
	int left, read = u.u_count;

	if (isnot(INSTALLED) && isnot(OPENED)) {
		u.u_error = ENODEV;
		return;
	}
	if (m_recbytes) {
		left = m_recbytes - g_recbytes_ctr;
		if (left < read) {
			read = left;
		}
	}
	if (!(u.u_error = read_buff_queue(u.u_base, read))) {
		g_recbytes_ctr += read;
	} else {
		stop_MSND();
 		read = 0;
	}
	u.u_count -= read;
	u.u_offset += read;
}

void msndwrite()
{
	int left, write = u.u_count;

	if (isnot(INSTALLED) && isnot(OPENED)) {
		u.u_error = ENODEV;
		return;
	}
	if (m_playbytes) {
		left = m_playbytes - g_playbytes_ctr;
		if (left < write) {
			write = left;
			u.u_error = ENOSPC;
		}
	}
	if (!(u.u_error = write_buff_queue(u.u_base, write))) {
		g_playbytes_ctr += write;
	} else {
		write = 0;
		stop_MSND();
	}
	u.u_count -= write;
	u.u_offset += write;
}

void msndioctl(dev, cmd, argp)
dev_t dev;
int cmd;
char *argp;
{
	int pgm, rate, arg = (int)argp, einval = -1, efault = 0;

	if (isnot(INSTALLED) && isnot(OPENED)) {
		u.u_error = ENODEV;
		return;
	}
	if (is(STARTEDMSND|WRITTEN) && cmd < M_GETRATE) {
		u.u_error = EBUSY;
		return;
	}
	switch (cmd) {
	case M_REINIT:
		einval = 0;
		break;
	case M_SETRATE:
		if (arg!=M_CLKDIV1 && arg!=M_CLKDIV2 && arg!=M_CLKDIV4)
			break;
		m_rate = arg;
		einval = 0;
		break;
	case M_GETRATE:
		efault = copyout((char*)&m_rate, argp, sizeof(int));
		einval = 0;
		break;
	case M_SETMODE:
		if (arg != M_MONO && arg != M_STEREO) 
			break;
		m_mode = arg;
		einval = 0;
		break;
	case M_GETMODE:
		efault = copyout((char*)&m_mode, argp, sizeof(int));
		einval = 0; 
		break;
	case M_SETSMPFMT:
		if (arg != M_SWAP && arg != M_NOSWAP) 
			break;
		m_smpfmt = arg;
		einval = 0; 
		break;
	case M_GETSMPFMT:
		efault = copyout((char*)&m_smpfmt, argp, sizeof(int));
		einval = 0; 
		break;
	case M_SETINPTVAL:
		m_MSNDinptval = arg;
		einval = 0; 
		break;
	case M_GETINPTVAL:
		efault = copyout((char*)&m_MSNDinptval,argp,sizeof(int));
		einval = 0; 
		break;
	case M_SETAUXPTVAL:
		m_MSNDauxptval = arg;
		einval = 0; 
		break;
	case M_GETAUXPTVAL:
		efault = copyout((char*)&m_MSNDauxptval,argp,sizeof(int));
		einval = 0; 
		break;
	case M_SETRECBYTES:
		/* roundoff to pairs of samples */
		arg >>= 2; arg <<= 2;
		if (arg < 0)	
			break;
		m_recbytes = arg;
		einval = 0; 
		break;
	case M_GETRECBYTES:
		efault = copyout((char*)&m_recbytes, argp, sizeof(int));
		einval = 0; 
		break;
	case M_SETPLAYBYTES:
		arg <<= 2; arg >>=2;
		if (arg < 0)
			break;
		m_playbytes = arg;
		einval = 0; 
		break;
	case M_GETPLAYBYTES:
		efault = copyout((char*)&m_playbytes, argp, sizeof(int));
		einval = 0; 
		break;
	case M_SETMONITOR:
		if (arg != M_AUTO && arg != M_SILENT)
			break;
		m_monitor = arg;
		einval = 0; 
		break;
	case M_GETMONITOR:
		efault = copyout((char*)&m_monitor, argp, sizeof(int));
		einval = 0; 
		break;
	}
	if (!einval && !efault) {
	    if (cmd < M_GETRATE) {
		if (cmd == M_SETINPTVAL || cmd == M_SETAUXPTVAL ||
			cmd == M_REINIT) {
		    if (u.u_error = initialize_MSND())
			return;
		}
		pgm = (m_monitor == M_SILENT ? MSNDPLAY : MSNDREC);
		rate = (m_monitor == M_SILENT ? m_rate : M_CLKDIV1);
		if (u.u_error = config_MSND(pgm, rate)) 
		    return;
	    }
	} else {
		if (efault)
			u.u_error = EFAULT;
		else
			u.u_error = EINVAL;
	}
}

/*
 *	EOT
 */

