#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <netinet6/in6.h>
#include <netinet6/icmpv6.h>
#include <netinet6/ipv6.h>
#include <net/ipv6.h>
#include <net/ndisc.h>
#include <net/addrconf.h>

#include "timer.h"
#include "debug.h"
#include "radv.h"


struct Interface *IfaceList;

#define DFLT_CONF_FILE "/etc/radv.conf"

char usage[] = "[-d] [-C config_file]";

struct option prog_opt[] = {
	{"debug",  0, 0, 'd'},
	{"config", 1, 0, 'C'}
};

int debug_level = 4;
extern FILE *yyin;
char *conf_file = NULL;

int iface_num = 2;
int sock = -1;

extern int yyparse(void);

void setup_ll_addr(struct Interface *iface);

void sig_handler(int sig);

void process_msg(char *msg, int len, struct sockaddr_in6 *addr,
		 struct in6_pktinfo *pkt_info);

void timer_handler(void *data);

void send_ra(struct Interface *iface, struct in6_addr *dest);

int main(int argc, char *argv[])
{
	unsigned char msg[2048];
	struct Interface *iface;
	struct icmp6_filter filter;
	int err, val;
	int opt_idx;
	int c, sk;

	/* parse args */

	while ((c = getopt_long(argc, argv, "dC:", prog_opt, &opt_idx)) > 0)
	{
		switch (c) {
		case 'd':
			debug_level = 1;
			break;
		case 'C':
			conf_file = optarg;
			break;
		case ':':
			fprintf(stderr, "option %s: parameter expected\n",
				prog_opt[opt_idx].name);
			exit(1);
		case '?':
			fprintf(stderr, "unknown option %s\n", argv[optind]);
			exit(1);
		}
	}

	/* parse config file */

	if (conf_file == NULL)
	{
		conf_file = DFLT_CONF_FILE;
	}

	if ((yyin = fopen(conf_file, "r")) == NULL)
	{
		perror(conf_file);
		exit(1);
	}

	yyparse();

	/* get raw socket */

	sk = socket(AF_INET6, SOCK_RAW, IPPROTO_ICMPV6);

	if (sk < 0)
	{
		perror("sys_socket");
		exit(1);
	}

	sock = sk;

	/*
	 *	get link local addr for ifaces and set timers
	 */

	for(iface=IfaceList; iface; iface=iface->next)
	{
		setup_ll_addr(iface);

		/* send an initial advertisement */
		send_ra(iface, NULL);
		
		memset(&iface->tm, 0, sizeof(struct timer_lst));
		iface->tm.handler = timer_handler;
		iface->tm.data = (void *) iface;
		set_timer(&iface->tm, iface->MaxRtrAdvInterval);
	}

	/*
	 *	config sig_handlers
	 */

	signal(SIGHUP, sig_handler);
	signal(SIGTERM, sig_handler);

	val = 1;

	err = setsockopt(sk, SOL_IPV6, IPV6_RXINFO, &val, sizeof(int));
	
	if (err < 0)
	{
		perror("sockopt RXINFO");
	}

	val = 2;
	err = setsockopt(sk, SOL_RAW, RAW_CHECKSUM, &val, sizeof(int));

	if (err < 0)
	{
		perror("sockopt RAW_CHECKSUM");
	}

	val = 255;

	err = setsockopt(sk, SOL_IPV6, IPV6_UNICAST_HOPS, &val, sizeof(int));

	if (err < 0)
	{
		perror("sockopt UNICAST_HOPS");
	}

	ICMPV6_FILTER_SETBLOCKALL(&filter);

	ICMPV6_FILTER_SETPASS(NDISC_ROUTER_SOLICITATION, &filter);
	ICMPV6_FILTER_SETPASS(NDISC_ROUTER_ADVERTISEMENT, &filter);

	err = setsockopt(sk, SOL_ICMPV6, ICMPV6_FILTER, &filter,
			 sizeof(struct icmp6_filter));

	if (err < 0)
	{
		perror("sockopt ICMPV6_FILTER");
	}

	/* enter loop */

	for (;;)
	{
		struct sockaddr_in6 rcv_addr;
		struct in6_pktinfo *pkt_info;
		struct msghdr mhdr;
		struct cmsghdr *cmsg;
		struct iovec iov;
		char chdr[sizeof(struct cmsghdr) + sizeof(struct in6_pktinfo)];
		int len;
		
		iov.iov_len = 2048;
		iov.iov_base = msg;

		cmsg = (struct cmsghdr *) chdr;

		cmsg->cmsg_len   = (sizeof(struct cmsghdr) +
				    sizeof(struct in6_pktinfo));
		cmsg->cmsg_level = SOL_IPV6;
		cmsg->cmsg_type  = IPV6_RXINFO;

		pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;

		mhdr.msg_name = &rcv_addr;
		mhdr.msg_namelen = sizeof(struct sockaddr_in6);
		mhdr.msg_iov = &iov;
		mhdr.msg_iovlen = 1;
		mhdr.msg_control = (void *) cmsg;
		mhdr.msg_controllen = (sizeof(struct cmsghdr) +
				       sizeof(struct in6_pktinfo));

		len = recvmsg(sk, &mhdr, 0);

		if (len < 0)
		{
			perror("recvmsg ");
			continue;
		}

		dprintf(4, "recvmsg len=%d\n", len);

		process_msg(msg, len, &rcv_addr, pkt_info);
	}
	
}

void sig_handler(int sig)
{
}

void process_msg(char *msg, int len, struct sockaddr_in6 *addr,
		 struct in6_pktinfo *pkt_info)
{
	struct Interface *iface;
	struct icmpv6hdr *icmph;

	icmph = (struct icmpv6hdr *) msg;

	if (icmph->type != NDISC_ROUTER_SOLICITATION &&
	    icmph->type != NDISC_ROUTER_ADVERTISEMENT)
	{
		/*
		 *	We just want to listen to RSs and RAs
		 */
		
		dprintf(2, "icmp filter failed\n");
		return;
	}

	dprintf(4, "if_index %d\n", pkt_info->ipi6_ifindex);

	/* get iface by receive if_index */

	for (iface = IfaceList; iface; iface=iface->next)
	{
		if (iface->if_index == pkt_info->ipi6_ifindex)
		{
			break;
		}
	}

	if (iface == NULL)
	{
		fprintf(stderr, "recv packet for unkown iface\n");
		return;
	}

	dprintf(4, "found Interface: %s\n", iface->Name);

	if (icmph->type == NDISC_ROUTER_SOLICITATION)
	{
		struct timeval tv;

		gettimeofday(&tv, NULL);
		
		if (tv.tv_sec - iface->tstamp < iface->MinRtrAdvInterval)
		{
			/*
			 *	unicast reply
			 */
			send_ra(iface, &addr->sin6_addr);
		}
		else
		{
			/*
			 *	multicast reply
			 */

			clear_timer(&iface->tm);
			send_ra(iface, NULL);
			set_timer(&iface->tm, iface->MaxRtrAdvInterval);
		}
	}
}

#define PROC_PATH "/proc/net/if_inet6"

void setup_ll_addr(struct Interface *iface)
{
	FILE *fp;
	char str_addr[40];
	int plen, scope, dad_status, if_idx;
	char devname[10];

	fp = fopen(PROC_PATH, "r");

	if (fp == NULL)
	{
		perror("error opening /proc file");
		exit(1);
	}
	
	while (fscanf(fp, "%32s %02x %02x %02x %02x %s\n",
		      str_addr, &if_idx, &plen, &scope, &dad_status,
		      devname) != EOF)
	{
		if (scope == IPV6_ADDR_LINKLOCAL &&
		    strcmp(devname, iface->Name) == 0)
		{
			struct in6_addr addr;
			int i;
			
			for (i=0; i<16; i++)
			{
				sscanf(str_addr + i * 2, "%02x",
				       &addr.s6_addr[i]);
			}
			memcpy(&iface->if_addr, &addr,
			       sizeof(struct in6_addr));

			iface->if_index = if_idx;
			fclose(fp);
			return;
		}
	}

	fprintf(stderr, "%s: no linklocal addr configured\n", iface->Name);
	fclose(fp);
}

void timer_handler(void *data)
{
	struct Interface *iface = (struct Interface *) data;

	dprintf(4, "timer_handler called for %s\n", iface->Name);

	send_ra(iface, NULL);
	set_timer(&iface->tm, iface->MaxRtrAdvInterval);
}

void send_ra(struct Interface *iface, struct in6_addr *dest)
{
	const struct in6_addr all_hosts_addr = {{{
		0xFF, 0x02, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 1
	}}};
	struct sockaddr_in6 addr;
	struct in6_pktinfo *pkt_info;
	struct msghdr mhdr;
	struct cmsghdr *cmsg;
	struct iovec iov;
	char chdr[sizeof(struct cmsghdr) + sizeof(struct in6_pktinfo)];

	struct icmpv6hdr *icmph;
	struct AdvPrefix *prefix;
	char buff[2048];
	int len = 0;
	int err;
	__u32 *wp;


	dprintf(3, "sending RA on %s\n", iface->Name);

	if (dest == NULL)
	{
		struct timeval tv;

		dest = &all_hosts_addr;
		gettimeofday(&tv, NULL);
		iface->tstamp = tv.tv_sec;
	}
	
	addr.sin6_family = AF_INET6;
	addr.sin6_port = htons(IPPROTO_ICMPV6);
	memcpy(&addr.sin6_addr, dest, sizeof(struct in6_addr));


	icmph = (struct icmpv6hdr *) buff;

	icmph->type = NDISC_ROUTER_ADVERTISEMENT;
	icmph->code = 0;
	icmph->checksum = 0;

	icmph->icmp6_hop_limit		= iface->AdvCurHopLimit;
	icmph->icmp6_addrconf_managed	= iface->AdvManagedFlag;
	icmph->icmp6_addrconf_other	= iface->AdvOtherConfigFlag;
	icmph->icmp6_rt_lifetime	= iface->AdvDefaultLifetime;

	len = sizeof(struct icmpv6hdr);

	wp = (__u32 *) (buff + len);

	*wp = htonl(iface->AdvReachableTime);
	len += sizeof(__u32);
	wp++;

	*wp = htonl(iface->AdvRetransTimer);
	len += sizeof(__u32);

	prefix = iface->AdvPrefixList;

	/*
	 *	now for prefix information
	 */

	while(prefix)
	{
		struct prefix_info *pinfo;
		
		pinfo = (struct prefix_info *) (buff + len);

		pinfo->type		= ND_OPT_PREFIX_INFO;
		pinfo->length		= 4;
		pinfo->prefix_len	= prefix->PrefixLen;
		pinfo->onlink		= prefix->AdvOnLinkFlag;
		pinfo->autoconf		= prefix->AdvAutonomousFlag;
		pinfo->reserved		= 0;
		pinfo->valid		= htonl(prefix->AdvValidLifetime);
		pinfo->prefered		= htonl(prefix->AdvPreferedLifetime);
		pinfo->reserved2	= 0;
		
		memcpy(&pinfo->prefix, &prefix->Prefix,
		       sizeof(struct in6_addr));

		len += sizeof(struct prefix_info);

		prefix = prefix->next;
	}


	iov.iov_len  = len;
	iov.iov_base = buff;
	
	cmsg = (struct cmsghdr *) chdr;
	
	cmsg->cmsg_len   = (sizeof(struct cmsghdr) +
			    sizeof(struct in6_pktinfo));
	cmsg->cmsg_level = SOL_IPV6;
	cmsg->cmsg_type  = IPV6_TXINFO;
	
	pkt_info = (struct in6_pktinfo *) cmsg->cmsg_data;
	memset(pkt_info, 0, sizeof(struct in6_pktinfo));
	pkt_info->ipi6_ifindex = iface->if_index;
	memcpy(&pkt_info->ipi6_addr, &iface->if_addr, sizeof(struct in6_addr));

	mhdr.msg_name = &addr;
	mhdr.msg_namelen = sizeof(struct sockaddr_in6);
	mhdr.msg_iov = &iov;
	mhdr.msg_iovlen = 1;
	mhdr.msg_control = (void *) cmsg;
	mhdr.msg_controllen = (sizeof(struct cmsghdr) +
			       sizeof(struct in6_pktinfo));

	err = sendmsg(sock, &mhdr, 0);

	if (err < 0)
	{
		perror("sendmsg");
	}

}

void dprintf(int level, const char* fmt, ...)
{
        va_list args;
 
        if (level > debug_level)
                return;
 
        va_start(args, fmt);
        vfprintf(stderr, fmt, args);
        va_end(args);
	fflush(stderr);
}
