/* ------------------------------------------------------------------------- */
/* 									     */
/* tcl_i2c.c - a new command for Tcl/Tk supporting the \iic-bus interface    */
/* 									     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995 Simon G. Vogl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */
#include <tcl.h>
#include <tk.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "i2c.h"

#define I2C_DEVICE "/dev/i2c0"

typedef struct {
	Tcl_Interp *interp;         /* Interpreter associated with widget. */
	int adr;
	int udelay;
	int mdelay;
	int file;
	char *buf;
	unsigned int buflen;
} I2C;


void I2CInit(Tcl_Interp *interp);
void I2CDestroy(ClientData clientData);
int I2CCmd(ClientData clientData, Tcl_Interp *interp, 
			int argc, char**argv);
int I2CGadgetCmd(ClientData data, Tcl_Interp *interp,int argc, char **argv);

/* ----------------------------------------------------------------------- */
int I2CGadgetCmd(ClientData data, Tcl_Interp *interp,int argc, char **argv){
	int i,invalid=0;
	I2C *ip=(I2C*)data;

	if ( argc<2 ){
	        Tcl_AppendResult(interp, "wrong # args: should be \"",
                        argv[0], " option ?arg arg ...?\"", (char *) NULL);
                return TCL_ERROR;
	}
	if  (strncmp("write",argv[1],strlen(argv[1]))==0){
		int ret;
		if (!ip->buflen){
			ip->buf=(char*)malloc(128);
			if (ip->buf==NULL){
				interp->result="i2c write: Out of memory :-<<";
				return TCL_ERROR;
			}else{
				ip->buflen=128;
			}
		}
		if (argc<3) {
			Tcl_AppendResult(interp, "wrong # args: should be \"",
		        	argv[0], " write byte1 ?... byteN?\"",
		        	"   bytei as decimal", (char *) NULL);
			return TCL_ERROR;
		};
		ip->buf[0]=ip->adr;
		ioctl(ip->file,I2C_UDELAY,ip->udelay);
		ioctl(ip->file,I2C_MDELAY,ip->mdelay);
		argc-=2;
		i=1;
		while (argc){
			if (TCL_OK != Tcl_GetInt(interp,argv[i+1],&ret))
				return TCL_ERROR;
			ip->buf[i]=ret;
/*printf("buf[%d]=%d\n",i,ret);*/
			argc--;
			i++;
			if (i>=ip->buflen){ /* buffer overflow -> realloc */
				ip->buf=(char*)realloc(ip->buf,ip->buflen+128);
				if (ip->buf==NULL){
					interp->result="i2c write: Out of memory :-<<";
					ip->buflen=0;
					return TCL_ERROR;
				}else{
					ip->buflen+=128;
				}
				printf("overflow prevented - writing first 127 bytes!");
				break;
			}
		};
/*printf ("i=%d\n",i);*/
		ret=write(ip->file,ip->buf,i);
		if (ret==-1){
			interp->result=strerror(errno);
			return TCL_ERROR;
		}
	} else if  (strncmp("read",argv[1],strlen(argv[1]))==0){
		unsigned char num[8],buf[4096];
		int val;
                if (argc>4) {
                        Tcl_AppendResult(interp, "wrong # args: should be \"",
                                argv[0], " read ?offset? numBytes\"",
                                (char *) NULL);
                        return TCL_ERROR;
                };
                ioctl(ip->file,I2C_UDELAY,ip->udelay);
                ioctl(ip->file,I2C_MDELAY,ip->mdelay);
                buf[0]=ip->adr;
                i=0;
		if (argc==4 && TCL_OK != Tcl_GetInt(interp,argv[2],&i))
			return TCL_ERROR;
		buf[1]=i;
		write(ip->file,buf,(argc==4)?2:1);
		buf[0]=ip->adr|1;
		if (TCL_OK != Tcl_GetInt(interp,argv[(argc==4)?3:2],&val))
			return TCL_ERROR;
		i=read (ip->file,buf,val);
                if (i==-1){
                	interp->result=strerror(errno);
                	return TCL_ERROR;
                }
 		for (i=0;i<val;i++){
			sprintf(num,"%d",(unsigned int)buf[i]);
			Tcl_AppendResult(interp,num,(char*)NULL);
			if (i<val-1){
				Tcl_AppendResult(interp," ",(char*)NULL);
			}
		}
	} else if (argc==2){
		if (strncmp("udelay",argv[1],strlen(argv[1]))==0){
			sprintf(interp->result,"%d",ip->udelay);
		} else 	if (strncmp("mdelay",argv[1],strlen(argv[1]))==0){
			sprintf(interp->result,"%d",ip->mdelay);
		} else 	if (strncmp("adr",argv[1],strlen(argv[1]))==0){
			sprintf(interp->result,"%d",ip->adr);
		} else 	if (strncmp("reset",argv[1],strlen(argv[1]))==0){
			ioctl(ip->file,I2C_RESET);
		} else
			invalid=1;
	} else if (argc==3){
		if (strncmp("udelay",argv[1],strlen(argv[1]))==0){
			if (TCL_OK != Tcl_GetInt(interp,argv[2],&ip->udelay))
				return TCL_ERROR;
		} else 	if (strncmp("mdelay",argv[1],strlen(argv[1]))==0){
			if (TCL_OK != Tcl_GetInt(interp,argv[2],&ip->mdelay))
				return TCL_ERROR;
		} else 	if (strncmp("adr",argv[1],strlen(argv[1]))==0){
			if (TCL_OK != Tcl_GetInt(interp,argv[2],&i))
				return TCL_ERROR;
			if ( i & 1 || i<0 || i>=256){
				interp->result="invalid adr! (<256, least bit zero)";
			};
                        ip->adr=i;
       		} else
			invalid=1;

	}	if (invalid){
		interp->result="invalid argument! should be read,write,udelay,mdelay,adr";
		return TCL_ERROR;
	}
	return TCL_OK;
}



/*
 *  I2CCommand
 *      This procedure is invoked to process the "i2c" Tcl
 *      command.  It creates a new "i2c" gadget.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      A new gadget is created and configured.
 */
int I2CCmd(ClientData clientData, Tcl_Interp *interp, 
			int argc, char**argv)
{
    I2C *inew, *ip =(I2C*)clientData;

    if (argc != 2) {
        Tcl_AppendResult(interp, "wrong # args:  should be \"",
                argv[0], " pathName \"", (char *) NULL);
        return TCL_ERROR;
    }
    inew = (I2C *) ckalloc(sizeof(I2C));
    inew->udelay=ip->udelay;
    inew->mdelay=ip->mdelay;
    inew->file=ip->file;
    inew->buf=(char*)malloc(128);
    inew->buflen=(inew->buf==NULL)?0:128; /* if we haven't got memory we'll try later */
    Tcl_CreateCommand(interp,argv[1], I2CGadgetCmd,
                (ClientData) inew, I2CDestroy);
    return TCL_OK;
}


/*
 *
 * I2CDestroy 
 *      Everything associated with the i2c bus gadget is freed up.
 *
 */
void I2CDestroy(ClientData clientData) {
    I2C *ip = (I2C *) clientData;
    close(ip->file);
	free(ip->buf);
    ckfree((char *) ip);
}


/*   
 *
 * I2CInit 
 *      Main init procedure. To be called in Tcl_AppInit.
 *
 */
void I2CInit(Tcl_Interp *interp){
    I2C *ip;
    
    ip=(I2C*)ckalloc(sizeof(I2C));
    ip->file=open(I2C_DEVICE,O_RDWR);
    if (ip->file==-1){
    	printf("i2c_init: error opening device: %s\n",strerror(errno));
    }
    ip->udelay=ip->mdelay=0;
    ip->adr=0;
    Tcl_CreateCommand(interp,"i2c", I2CCmd,
                (ClientData) ip, (void (*)()) NULL);
}
