/* diag.c - Diagnostic messages */

/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */


#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "atmd.h"


typedef struct _component {
    const char *name;
    int verbosity;
    struct _component *next;
} COMPONENT;


static const char *app_name = NULL;
static COMPONENT *components = NULL;
static int default_verbosity = DIAG_INFO;


void set_application(const char *name)
{
    app_name = name;
}


void set_verbosity(const char *component,int level)
{
    COMPONENT *walk;

    if (!component) {
	default_verbosity = level;
	return;
    }
    for (walk = components; walk; walk = walk->next)
	if (!strcmp(walk->name,component)) break;
    if (!walk) {
	walk = alloc_t(COMPONENT);
	walk->name = component;
	walk->next = components;
	components = walk;
    }
    walk->verbosity = level;
}


void vdiag(const char *component,int severity,const char *fmt,va_list ap)
{
    COMPONENT *walk;

    for (walk = components; walk; walk = walk->next)
	if (!strcmp(walk->name,component)) break;
    if (severity > (walk ? walk->verbosity : default_verbosity)) return;
    fflush(stdout);
    if (app_name) fprintf(stderr,"%s:%s: ",app_name,component);
    else fprintf(stderr,"%s: ",component);
    vfprintf(stderr,fmt,ap);
    fputc('\n',stderr);
    if (severity == DIAG_FATAL) {
	fprintf(stderr,"Fatal error - Terminating\n");
	exit(1);
    }
}


void diag(const char *component,int severity,const char *fmt,...)
{
    va_list ap;

    va_start(ap,fmt);
    vdiag(component,severity,fmt,ap);
    va_end(ap);
}
