/*
 * loopback: program to control the loopback mode on a ForeRunner LE25
 *           ATM NIC. Sadly, due to immaturity of the Linux ATM API,
 *           this needs to be done via device-specific ioctl()s.
 *
 * Written by Greg Banks, NEC Australia <gnb@linuxfan.com>
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <atm.h>
#include <linux/atmdev.h>
#include <linux/atm_idt77105.h>


static void
usage(void)
{
    static const char usage_string[] = "\
Usage: loop25 interface\n\
       loop25 interface mode\n\
   where `mode' is one of:\n\
        none        normal operation.\n\
        local       Tx looped to Rx. Transmitted cells are immediately\n\
                    received as if they had bounced.\n\
        diag        Same as local.\n\
        remote      Rx teed to Tx. Received cells are received (e.g. for\n\
                    analysis) and also retransmitted.\n\
        loop        Same as remote.\n\
   `interface' is an ATM interface number, e.g. 0 for first NIC.\n\
";

    fputs(usage_string, stderr);
    exit(1);
}

/* Sadly, an equivalent define is in the nicstar.h private header */
#define PCR_ATM25   59111

/* Check for correct ATM device driver */
static int
check_is_idt77105_phy(int sock, int interface)
{
    int link_rate = -1;
    char dev_type[256];
    struct atmif_sioc sioc;

    dev_type[0] = '\0';

    memset(&sioc, 0, sizeof(sioc));
    sioc.arg = dev_type;
    sioc.length = sizeof(dev_type);
    sioc.number = interface;
    if (ioctl(sock, ATM_GETTYPE, &sioc) < 0)
    {
        perror("ioctl(ATM_GETTYPE)");
        return -1;
    }
    if (strcmp(dev_type, "nicstar"))
    {
        fprintf(stderr, "Interface %d is not a NICStAR (type is \"%s\")\n",
            interface, dev_type);
        return -1;
    }


    memset(&sioc, 0, sizeof(sioc));
    sioc.arg = &link_rate;
    sioc.length = sizeof(link_rate);
    sioc.number = interface;
    if (ioctl(sock, ATM_GETLINKRATE, &sioc) < 0)
    {
        perror("ioctl(ATM_GETLINKRATE)");
        return -1;
    }
    if (link_rate != PCR_ATM25)
    {
        fprintf(stderr, "Interface %d is not 25 Mbps (PCR is %d not %d)\n",
            interface, link_rate, PCR_ATM25);
        return -1;
    }
    
    return 0;
}


int
main(int argc, char **argv)
{
    int sock;
    int mode;
    int interface;
    const char *modestr = 0;
    struct atmif_sioc sioc;
    
    if (argc <2 || argc > 3)
        usage();
    interface = atoi(argv[1]);
    modestr = (argc > 2 ? argv[2] : 0);

    sock = socket(PF_ATMPVC, SOCK_DGRAM, 0);
    if (sock < 0)
    {
        perror("socket(PF_ATMPVC)");
        exit(3);
    }
    
    if (check_is_idt77105_phy(sock, interface) < 0)
        exit(1);
        
    memset(&sioc, 0, sizeof(sioc));
    sioc.number = interface;

    if (modestr == 0)
    {
        /* get and print the current loopback mode */
        mode = -1;
        sioc.arg = &mode;
        if (ioctl(sock, IDT77105_GETLOOP, &sioc) < 0)
        {
            perror("ioctl(IDT77105_GETLOOP)");
            exit(1);
        }
        switch (mode)
        {
        case IDT77105_LM_NONE: printf("none\n"); break;
        case IDT77105_LM_DIAG: printf("local\n"); break;
        case IDT77105_LM_LOOP: printf("remote\n"); break;
        default: printf("unknown (%d)\n", mode); break;
        }
    }
    else
    {
        /* set a new loopback mode */
        if (!strcmp(modestr, "none"))
            mode = IDT77105_LM_NONE;
        else if (!strcmp(modestr, "local"))
            mode = IDT77105_LM_DIAG;
        else if (!strcmp(modestr, "diag"))
            mode = IDT77105_LM_DIAG;
        else if (!strcmp(modestr, "remote"))
            mode = IDT77105_LM_LOOP;
        else if (!strcmp(modestr, "line"))
            mode = IDT77105_LM_LOOP;
        else
            usage();

        sioc.arg = (void*)mode;
        if (ioctl(sock, IDT77105_SETLOOP, &sioc) < 0)
        {
            perror("ioctl(IDT77105_SETLOOP)");
            exit(1);
        }
    }
        
    return 0;
}
