/* atm2text.c - Converts binary encoding of ATM address to textual
		representation */

/* Written 1995 by Werner Almesberger, EPFL-LRC */


#include <stdio.h>
#include <string.h>
#include <linux/atm.h>

#include "atm.h"


static int put_item(char **buffer,int *length,int value)
{
    char *walk,*scan;
    char tmp;

    if (!*length) return -1;
    if (!value) {
	*(*buffer)++ = '0';
	(*length)--;
	return 0;
    }
    for (walk = *buffer; value; value /= 10) {
	if (!(*length)--) return -1;
	*walk++ = '0'+(value % 10);
    }
    for (scan = walk-1; scan > *buffer; (*buffer)++) {
	tmp = *scan;
	*scan-- = **buffer;
	**buffer = tmp;
    }
    *buffer = walk;
    return 0;
}


static int do_pvc(char *buffer,int length,const struct sockaddr_atmpvc *addr,
  int flags)
{
    int orig_len;

    orig_len = length;
    if (put_item(&buffer,&length,addr->sap_addr.itf)) return -1;
    if (!length--) return -1;
    *buffer++ = '.';
    if (put_item(&buffer,&length,addr->sap_addr.vpi)) return -1;
    if (!length--) return -1;
    *buffer++ = '.';
    if (put_item(&buffer,&length,addr->sap_addr.vci)) return -1;
    if (!length) return -1;
    *buffer = 0;
    return orig_len-length;
}


static int do_svc(char *buffer,int length,const struct sockaddr_atmsvc *addr,
  int flags)
{
    static int pure[] = { 20 };
    static int bin[] = { 1,2,10,6,1 };
    static int e164[] = { 4,6,1 };

    int orig_len,len,i,left,value;
    int *fmt;

    orig_len = length;
    if (*addr->sas_addr.pub) {
	len = strlen(addr->sas_addr.pub);
	if (length < len+1) return -1;
	strcpy(buffer,addr->sas_addr.pub);
	buffer += len;
	length -= len;
	if (*addr->sas_addr.prv) {
	    if (!length--) return -1;
	    *buffer++ = '+';
	}
    }
    if (*addr->sas_addr.prv) {
	fmt = pure;
	i = 0;
	if (flags & A2T_PRETTY)
	    switch (*addr->sas_addr.prv) {
		case ATM_AFI_DCC:
		case ATM_AFI_ICD:
		    fmt = bin;
		    break;
		case ATM_AFI_E164:
		    for (i = 2; i < 17; i++)
			if (addr->sas_addr.prv[i >> 1] & (0xf0 >> 4*(i & 1)))
			    break;
		    while (i < 17) {
			value = (addr->sas_addr.prv[i >> 1] >> 4*(1-(i & 1))) &
			  0xf;
			if (value > 9) return -1;
			if (!length--) return -1;
			*buffer++ = '0'+value;
			i++;
		    }
		    if (!length--) return -1;
		    *buffer++ = ':';
		    i = 9;
		    fmt = e164;
		    break;
		default:
		    break;
	    }
	for (left = *fmt++; i < ATM_ESA_LEN; i++) {
	    if (!left--) {
		if (!length--) return -1;
		*buffer++ = '.';
		left = *fmt++-1;
	    }
	    if (length < 2) return -1;
	    sprintf(buffer,"%02X",addr->sas_addr.prv[i]);
	    length -= 2;
	    buffer += 2;
	}
    }
    if (!length) return -1;
    *buffer = 0;
    return orig_len-length;
}


int atm2text(char *buffer,int length,const struct sockaddr *addr,int flags)
{
    if (!length) return -1;
    if (flags & A2T_NAME) { /* attempt name lookup */ };
    if (addr->sa_family == AF_ATMPVC)
	return do_pvc(buffer,length,(const struct sockaddr_atmpvc *) addr,
	  flags);
    if (addr->sa_family == AF_ATMSVC)
	return do_svc(buffer,length,(const struct sockaddr_atmsvc *) addr,
	  flags);
    return -1;
}
