/* switch.c - Handles signaling an ATM switch */

/* Written 1997-1998 by Roman Pletka, EPFL SSC */
/* Modified 1998 by Werner Almesberger, EPFL ICA */


#include <stdio.h>           /* perror */
#include <stdlib.h>
#include <sys/types.h>       /* select,open */
#include <sys/socket.h>
#include <sys/stat.h>        /* open */
#include <fcntl.h>           /* open */
#include <sys/time.h>        /* select */
#include <unistd.h>          /* select */

#include "io.h"
#include "proto.h"

/*#include "atmsvc.h"*/



#define MAX_ITF 10           /* limit max number of interfaces */

static int in,out[MAX_ITF],rm;



static void from_sigd(struct atmsvc_msg *msg)
{
  CALL *call;
  CALL *old_call;
  unsigned long source;

  /* Choose right call and source (caller/called/rm)*/
  call = demux_in(&source, msg);
  print_msg(msg,call,source);
  print_call(call);

  switch (call->state) {
  case cs_null:
    if (source == CALLER) {
      switch(msg->type) {
      case as_okay:     /* enter cs_listening state */
	new_state(call,cs_listening);
	break;
      case as_error:    /* sigd says "no" */
	printf("sigd not ready for listening\n");
	/* kill the listening "call" ??!! */
	break;
      default:
	printf("Error: Socket\n");
	abort();
	break;
      }
    }
    break;
  case cs_listening:
    if (source == CALLER && msg->type == as_indicate) { 
      /* now work starts... */
      old_call = call;
      call = new_call(old_call->in.id); /* fd to out pipe */
      /* save data in call: pvc,local,qos,sap */
      call->in.pvc = msg->pvc; /* maybe it's in the msg */
      call->in.qos = msg->qos;
      call->out.svc = msg->local;
      call->in.svc = msg->svc;
      call->sap = msg->sap;
      call->listen = old_call;
      
      msg->vcc = msg->listen_vcc;
      /* ask for vci,vpi... */
      msg->vcc = (unsigned long) call | RM;  /* some kind of magic... */
      to_rm(rm,msg); /* forwards indicate as it is */
      new_state(call,cs_indicated);
    }
    else {
      if (msg->type == as_close) {
	/* close the listening "call" */
	send_close(call, CALLER);
	new_state(call, cs_will_close);
      }
    } 
    break;
  case cs_indicated:
    if (source == RM) {
      switch(msg->type) {
      case as_okay: 
      	/* find fd/itf->out.id for called from msg */
	call->out.id = out[msg->pvc.sap_addr.itf];
	call->out.pvc = call->in.pvc = msg->pvc; /* @@@@ hmm, that looks wrong*/
	/* send connect to  called and enter state cs_rm_accepted */
	send_identify(call);
	send_connect(call);
	new_state(call,cs_rm_accepted);
	break;
      case as_error:
      	/* send as_reject to caller 
	   and enter state cs_invalid */
	send_reject_not_id(call,msg->reply);
	new_state(call,cs_invalid);
	free_call(call);
	break;
      default:
	abort();
	break;
      }
    }
    else {
      printf("Error: in state cs_indicated. Msg from %ld != RM expected.\n", 
	     source);
      abort();
    }
    break;
  case cs_rm_accepted:
    if (source == CALLED) {
      switch(msg->type) {
      case as_okay: 
	/* send claim */
	msg->vcc = (unsigned long) call | RM; /* some kind of magic... */
	/* save msg content in call */
	call->out.qos = msg->qos;
	to_rm(rm,msg);  /* claim ??!! */
	new_state(call,cs_called_accepted);
	break;
      case as_error:
      	send_reject(call,msg->reply);
	new_state(call,cs_invalid);
	free_call(call);
	break;
      default:
	abort();
	break;
      }
    }
    else {
      printf("Error: in state cs_rm_accepted. "
	     "Msg from %ld != CALLED expected.\n",
	     source);
      abort();
    }
    break;
  case cs_rm_accepted2:
    /* msg from: caller(error,okay) or called(close) */
    switch (source) {
    case CALLER: 
      switch(msg->type) {
      case as_okay: /* complete the call */
	new_state(call, cs_connected);
	break;
      case as_error:
	/* send close called */
	send_close(call, CALLED);
	new_state(call, cs_caller_error);
	break;
      default:
	abort();
	break;
      }
      break;
    case CALLED:
      if (msg->type == as_close) {
	/* send close to called */
	send_close(call, CALLED);
	new_state(call, cs_called_closed);
      }
      else abort();
      break;
    default:
	abort();
	 break;  
    }
    break;
  case cs_rejected:
    /* wait for close msg from called */
    if (source == CALLED) {
      if (msg->type == as_close){
	new_state(call,cs_invalid);
	free_call(call);
      }
    }
    else abort();
    /* else error */
    break;
  case cs_called_accepted:
    if (source == CALLED) {
      if (msg->type == as_close){
	/* send reject to caller and send close to called */
	send_reject(call, msg->reply);
	send_close(call,CALLED);
	new_state(call,cs_rejecting);
      }
    }
    if (source == RM) {
      switch(msg->type) {
      case as_okay:
	/* send accept to caller */
	send_accept(call);
	new_state(call,cs_rm_accepted2);
	break;
      case as_error:
	/* send reject to caller and close to called */
	send_reject(call,msg->reply);
	send_close(call,CALLED);
	new_state(call, cs_rejected);
	break;
      default:
	abort();
	break;
      }
    }
    break;
  case cs_connected:
    if (msg->type == as_close) {
      if (source == CALLER) {
	send_close(call, CALLER);
	send_close(call, CALLED);
	new_state(call, cs_caller_closed);
      }
      else if (source == CALLED) {
	send_close(call, CALLER);
	send_close(call, CALLED);
	new_state(call, cs_called_closed2);
      }
    }
    /* else error */
    else abort();
    break;
  case cs_rejecting:
    if (source == RM ) {
      switch(msg->type) {
      case as_error: 
	free_call(call);
	break;
      case as_okay:
	/* free resources */
	send_free_rm(rm,call);
	new_state(call, cs_free_rm);
	break;
      default:
	abort();
	break;
      }
    }
    break;
  case cs_called_closed:
    if (source == CALLER) {
      switch(msg->type) {
      case as_error: 
	send_free_rm(rm, call);
	new_state(call, cs_free_rm);
	break;
      case as_okay:
	send_close(call, CALLER);
	new_state(call, cs_caller_closing);
	break;
      default:
	abort();
	break;
      }
    }
  case cs_called_closed2:
    if (source == CALLER && msg->type == as_close) {
      send_free_rm(rm, call);
	new_state(call, cs_free_rm);
    }
    /* else error */
    break;
  case cs_caller_error:
    if (source == CALLED && msg->type == as_close) {
      send_free_rm(rm, call);
      new_state(call, cs_free_rm);
      break;
    }
  case cs_caller_closing:
    if (source == CALLER && msg->type == as_close) {
      send_free_rm(rm, call);
      new_state(call, cs_free_rm);
    }
    else abort();
    break;
  case cs_caller_closed:
    if (source == CALLED && msg->type == as_close) {
      send_free_rm(rm, call);
      new_state(call, cs_free_rm);
    }
    else abort();
    /* else error */
    break;
  case cs_free_rm:
    if (source == RM) {
      switch(msg->type) {
      case as_error: 
	free_call(call);
	printf("Error: RM couldn't free resources\n");
	break;
      case as_okay:
	free_call(call);
	break;
      default:
	abort();
	break;
      }
    }
    else abort();
    break;
  case cs_will_close:
    if (source == CALLER) {
      switch (msg->type) {
      case as_close:
	free_call(call);
	break;
      case as_indicate:
	send_reject_not_id(call, 0);
	new_state(call, cs_call_indicated);
	free_call(call);
	break;
      default:
	abort();
	break;
      }
    }
    break;
  default:
    abort();
    break;
  }  
}

/*****************************************************************************/
/*   M A I N                                                                 */
/*****************************************************************************/

int main(int argc, char *argv[])
{
  static unsigned char buffer[sizeof(struct atmsvc_msg)];
  int nb_itf;
  int i;

  if (argc < 5) {
    printf("Usage: %s in rm [out1 out_2 ...]\n",argv[0]);
    exit(1);
  }
  if ((argc+1) > MAX_ITF) {
    printf("%s: Too many interfaces (%d > %d)",argv[0],argc-1,MAX_ITF);
    exit(1);
  }
  
  nb_itf = argc-3;
  
  /* open all pipes and send listen */
  if ((in = open(argv[1],O_RDWR)) < 0) { /* actually read-only */
      perror(argv[1]);
      return 1;
    }
  if ((rm = open(argv[2],O_RDWR)) < 0) { /* actually read-only */
    perror(argv[2]);
    return 1;
  }
    
  for(i=0;i<nb_itf;i++) {
    if ((out[i] = open(argv[i+3],O_RDWR)) < 0) { /* actually read-only */
      perror(argv[i+3]);
      return 1;
    }
    /* send listen */
    if (!new_call(out[i])) {
      printf("%s: could not create new call\n",argv[0]);
      exit(1);
    }
    
  } /* end for */

  send_listen();

  /* wait for messages from any sigd and handle it */
  while(1) {
  
    /* read msg */
    read(in,buffer,sizeof(buffer));
    from_sigd((struct atmsvc_msg *)buffer);
    
    
  } /* end while(1) */
 
}
