/* q2931.c - Processing of incoming Q.2931 messages */
 
/* Written 1995-1997 by Werner Almesberger, EPFL-LRC */
 

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "atm.h"
#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "proto.h"
#include "sap.h"
#include "io.h"
#include "timeout.h"
#include "trace.h"

#define __KERNEL__ /* since that's what we effectively are */
#include <linux/errno.h>


#define COMPONENT "Q2931"


extern const char *cause_text[]; /* from mess.c */


static Q_DSC in_dsc;
static TIMER *t309 = NULL;


static int send_call_proceeding(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CALL_PROC);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if (net) {
	int vci;

	vci = get_vci(0);
	if (vci < 0) {
	    (void) q_close(&dsc);
	    return vci;
	}
	q_assign(&dsc,QF_vpi,0); /* @@@ */
	q_assign(&dsc,QF_vci,vci);
	sock->pvc.sap_addr.itf = signaling_pvc.sap_addr.itf;
	sock->pvc.sap_addr.vpi = 0;
	sock->pvc.sap_addr.vci = vci;
    }
    if (sock->ep_ref >= 0) q_assign(&dsc,QF_ep_ref,sock->ep_ref);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
    return 0;
}


static void setup_call(unsigned long call_ref)
{
    SOCKET *sock,*this,**walk;
    struct sockaddr_atmsvc *sap,*new_sap;
    struct atm_qos qos,new_qos;
    unsigned char buf[ATM_ESA_LEN];
    int len,i;

    sap = sap_decode(&in_dsc,&qos);
    new_sap = NULL;
    sock = lookup_sap(sap,&qos,&new_sap,&new_qos);
    if (!sock || new_sap) free(sap);
    else new_sap = sap; /* @@@ should strip extra BLLIs */
    new_qos.aal = ATM_AAL5; /* hack @@@ */
    if (!sock) {
	send_release_complete(call_ref,ATM_CV_INCOMP_DEST); /* @@@ dunno */
	return;
    }
    this = new_sock(0);
    this->state = ss_indicated;
    this->q2931_state = qs_in_proc;
    this->call_ref = call_ref;
    if (q_present(&in_dsc,QF_ep_ref))
	this->ep_ref = q_fetch(&in_dsc,QF_ep_ref);
#ifdef CISCO
    else
#endif
    {
	int error;

	error = send_call_proceeding(this);
	if (error) {
	    free_sock(this);
	    send_release_complete(call_ref,ATM_CV_NO_CI);
	    return;
	}
    }
    this->local = new_sap;
    this->qos = new_qos;
    /* if (sock->local) *this->local->sas_addr = sock->local->sas_addr; ??? */
    diag(COMPONENT,DIAG_DEBUG,"AAL type %ld",q_fetch(&in_dsc,QF_aal_type));
    len = q_read(&in_dsc,QF_cdpn_esa,(void *) &buf,sizeof(buf));
    if (debug) {
	printf("Addr len %d:",len);
	for (i = 0; i < (len < 30 ? len : 30); i++) printf(" %02X",buf[i]);
	putchar('\n');
    }
    if (!net) { /* already set by send_call_proceeding */
	int vpci;

	vpci = q_fetch(&in_dsc,QF_vpi);
	this->pvc.sap_addr.itf = get_itf(&vpci);
	this->pvc.sap_addr.vpi = vpci;
	this->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
    }
    diag(COMPONENT,DIAG_DEBUG,"ITF.VPI.VCI: %d.%d.%d",this->pvc.sap_addr.itf,
      this->pvc.sap_addr.vpi,this->pvc.sap_addr.vci);
    this->remote = alloc_t(struct sockaddr_atmsvc);
    memset(this->remote,0,sizeof(struct sockaddr_atmsvc));
    if (q_present(&in_dsc,QF_cgpn)) { /* should handle E.164 too */
	char buffer[MAX_ATM_ADDR_LEN+1];

	this->remote->sas_family = AF_ATMSVC;
	i = q_read(&in_dsc,QF_cgpn,(void *) this->remote->sas_addr.prv,
	  ATM_ESA_LEN);
	/*printf("addr_len = %d\n",i);*/
	if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,(struct sockaddr *) this->remote,
	  pretty) < 0) strcpy(buffer,"<invalid address>");
	diag(COMPONENT,DIAG_DEBUG,"Incoming call from %s",buffer);
    }
    send_kernel(0,sock->id,as_indicate,0,&this->pvc,this->remote,NULL,&new_qos);
    for (walk = &sock->listen; *walk; walk = &(*walk)->listen);
    *walk = this;
    diag(COMPONENT,DIAG_DEBUG,"SE vpi.vci=%d.%d",this->pvc.sap_addr.vpi,
      this->pvc.sap_addr.vci);
}


static void send_status(SOCKET *sock,unsigned long call_ref,
  unsigned char cause,...)
{
    va_list ap;
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_STATUS);
    if (sock) {
	q_assign(&dsc,QF_call_ref,sock->call_ref);
	q_assign(&dsc,QF_call_state,(int) sock->q2931_state);
	if (sock->ep_ref >= 0) {
	    q_assign(&dsc,QF_ep_ref,sock->ep_ref);
	    q_assign(&dsc,QF_ep_state,eps_map[sock->q2931_state]);
	}
    }
    else {
	q_assign(&dsc,QF_call_ref,call_ref);
	q_assign(&dsc,QF_call_state,0); /* U0 - Null / REST 0 - Null */
    }
    q_assign(&dsc,QF_cause,cause);
    va_start(ap,cause);
    switch (cause) {
	case ATM_CV_UNKNOWN_MSG_TYPE:
	case ATM_CV_INCOMP_MSG:
	    q_assign(&dsc,QF_bad_msg_type,va_arg(ap,unsigned char));
	    break;
	case ATM_CV_MAND_IE_MISSING:
	case ATM_CV_INVALID_IE:
	    {
		unsigned char ie;

		ie = va_arg(ap,unsigned char);
		q_write(&dsc,QF_ie_id6,&ie,1);
		break;
	    }
	default:
	    ;
    }
    va_end(ap);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


static void send_status_enq(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_STATUS_ENQ);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if (sock->ep_ref >= 0) q_assign(&dsc,QF_ep_ref,sock->ep_ref);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
    /* @@@ should start T322 */
}


static void send_connect_ack(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CONN_ACK);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


static void send_restart_ack(int vpi,int vci)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_REST_ACK);
    q_assign(&dsc,QF_call_ref,0);
    if (!vpi && !vci) q_assign(&dsc,QF_rst_class,ATM_RST_ALL_VC);
    else {
	q_assign(&dsc,QF_rst_class,ATM_RST_ALL_VC);
	q_assign(&dsc,QF_vpi,vpi);
	q_assign(&dsc,QF_vci,vci);
    }
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


static void send_drop_party_ack(unsigned long call_ref,short ep_ref,
  unsigned char cause)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_DROP_PARTY_ACK);
    q_assign(&dsc,QF_call_ref,call_ref);
    q_assign(&dsc,QF_ep_ref,ep_ref);
    q_assign(&dsc,QF_cause,cause);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


static void q2931_call(SOCKET *sock,unsigned char mid)
{
    char buffer[MAX_ATM_ADDR_LEN+1];
    int error;

    switch (mid) {
	case QMSG_STATUS: /* 5.5.6.12 */
	    {
		Q2931_STATE state;

		/*
		 * NOTE: T322 isn't implemented yet, but when it is, make sure
		 *	 to only stop it on STATUS iff the cause is
		 *	 ATM_CV_RESP_STAT_ENQ. Supplementary services break if
		 *	 you stop on any STATUS.
		 */
		if (sock->q2931_state == qs_rel_req || sock->q2931_state ==
		  qs_rel_ind) return;
		state = q_fetch(&in_dsc,QF_call_state);
		if (state == qs_null) break; /* clear call */
		if (state != sock->q2931_state)
		    diag(COMPONENT,DIAG_WARN,"STATUS %s received in state %s",
		      qs_name[state],qs_name[sock->q2931_state]);
	    }
	    return;
	default:
	    ;
    }
    switch (mid) {
	case QMSG_CALL_PROC: /* CONNECTING, WAIT_REL, REL_REQ */
	    if (sock->state == ss_wait_rel || sock->state == ss_rel_req) {
		send_status(sock,0,ATM_CV_INCOMP_MSG,QMSG_CALL_PROC);
		return;
	    }
	    if (sock->state != ss_connecting) break;
	    /* check for 2nd CALL_PROC @@@ */
	    STOP_TIMER(sock);
	    if (q_present(&in_dsc,QG_conn_id)) {
		int vpci;

		vpci = q_fetch(&in_dsc,QF_vpi);
		sock->pvc.sap_addr.itf = get_itf(&vpci);
		sock->pvc.sap_addr.vpi = vpci;
		sock->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
		diag(COMPONENT,DIAG_DEBUG,"ITF.VPI.VCI: %d.%d.%d",
		  sock->pvc.sap_addr.itf,sock->pvc.sap_addr.vpi,
		  sock->pvc.sap_addr.vci);
	    }
	    START_TIMER(sock,T310);
	    sock->q2931_state = qs_out_proc;
	    return;
	case QMSG_CONNECT: /* CONNECTING, REL_REQ */
	    if (sock->state == ss_rel_req) {
		send_status(sock,0,ATM_CV_INCOMP_MSG,QMSG_CONNECT);
		return;
	    }
	    if (sock->state != ss_connecting) break;
	    STOP_TIMER(sock);
	    if (q_present(&in_dsc,QG_conn_id)) {
		int vpci;

		vpci = q_fetch(&in_dsc,QF_vpi);
		sock->pvc.sap_addr.itf = get_itf(&vpci);
		sock->pvc.sap_addr.vpi = vpci;
		sock->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
		diag(COMPONENT,DIAG_DEBUG,"ITF.VPI.VCI: %d/%d.%d",
		  sock->pvc.sap_addr.itf,sock->pvc.sap_addr.vpi,
		  sock->pvc.sap_addr.vci);
	    }
	    error = 0;
	    if (!sock->pvc.sap_addr.vpi && !sock->pvc.sap_addr.vci)
		error = -EPROTO;
	    /* more problems */
	    if (error) {
		set_error(sock,error);
		send_release(sock,0); /* @@@ cause follows reason ??? */
		START_TIMER(sock,T308_1);
		new_state(sock,ss_rel_req);
		return;
	    }
	    send_connect_ack(sock);
	    /* @@@ fill in sock->remote */
	    /* @@@ fill in traffic parameters */
	    send_kernel(sock->id,0,as_okay,0,&sock->pvc,NULL,sock->local,
	      &sock->qos);
	    new_state(sock,ss_connected);
	    if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,(struct sockaddr *)
	      sock->local,0) < 0) strcpy(buffer,"<invalid>");
	    diag(COMPONENT,DIAG_INFO,"Active open succeeded (CR 0x%06X, "
	      "ID 0x%08x, addr %s)",sock->call_ref,sock->id,buffer);
	    return;
	case QMSG_CONN_ACK: /* ACCEPTING, WAIT_REL, REL_REQ */
	    diag(COMPONENT,DIAG_DEBUG,"CA vpi.vci=%d.%d",
	      sock->pvc.sap_addr.vpi,sock->pvc.sap_addr.vci);
	    if (sock->state == ss_wait_rel || sock->state == ss_rel_req) {
		send_status(sock,0,ATM_CV_INCOMP_MSG,QMSG_CONN_ACK);
		return;
	    }
	    if (sock->state != ss_accepting) break;
	    STOP_TIMER(sock);
	    send_kernel(sock->id,0,as_okay,0,NULL,NULL,sock->local,NULL);
	    new_state(sock,ss_connected);
	    if (atm2text(buffer,MAX_ATM_ADDR_LEN+1, (struct sockaddr *)
	      sock->local,0) < 0) strcpy(buffer,"<invalid>");
	    diag(COMPONENT,DIAG_INFO,"Passive open succeeded (CR 0x%06X, "
	      "ID 0x%08x, addr %s)",sock->call_ref,sock->id,buffer);
	    return;
	case QMSG_RELEASE: /* all states */
	    {
		unsigned char cause;

		cause = q_fetch(&in_dsc,QF_cause);
		diag(COMPONENT,DIAG_DEBUG,"Cause %d (%s)",cause,cause > 127 ?
		  "invalid cause" : cause_text[cause]);
	    }
	    switch (sock->state) {
		case ss_connecting:
		    set_error(sock,-ECONNREFUSED);
		    /* fall through */
		case ss_accepting:
		    set_error(sock,-ECONNRESET); /* ERESTARTSYS ? */
		    send_release_complete(sock->call_ref,0);
		    /* fall through */
		case ss_rel_req:
		    send_close(sock);
		    /* fall through */
		case ss_wait_rel:
		    STOP_TIMER(sock);
		    free_sock(sock);
		    return;
		case ss_connected:
		    diag(COMPONENT,DIAG_INFO,"Passive close (CR 0x%06X)",
		      sock->call_ref);
		    send_close(sock);
		    /* fall through */
		case ss_hold:
		    new_state(sock,ss_rel_ind);
		    return;
		case ss_indicated:
		    send_release_complete(sock->call_ref,0);
		    new_state(sock,ss_zombie);
		    /* fall through */
		case ss_rel_ind:
		    /* send_release_complete(sock->call_ref,0); */
		    return;
		default:
		    send_release_complete(sock->call_ref,0); /* @@@ should
			be ATM_CV_INCOMP_MSG */
		    break;
	    }
	    break;
	case QMSG_RESTART:
		set_error(sock,-ENETRESET);
		/* fall through */
	case QMSG_STATUS: /* fall through when clearing */
	case QMSG_REL_COMP: /* basically any state (except LISTENING,
				  ZOMBIE, REL_IND) */
	    {
		unsigned char cause;

		if (mid != QMSG_REL_COMP || !q_present(&in_dsc,QF_cause))
		    cause = 0;
		else {
		    cause = q_fetch(&in_dsc,QF_cause);
		    diag(COMPONENT,DIAG_DEBUG,"Cause %d (%s)",cause,
		      cause > 127 ? "invalid cause" : cause_text[cause]);
		}
		switch (sock->state) {
		    case ss_connecting:
			set_error(sock,cause == ATM_CV_UNALLOC ?
			  -EADDRNOTAVAIL : cause == ATM_CV_RES_UNAVAIL ||
#ifdef UNI31
			  cause == ATM_CV_UCR_UNAVAIL ||
#endif
			  cause == ATM_CV_NO_ROUTE_DEST ? -EHOSTUNREACH :
			  cause == ATM_CV_NUM_CHANGED ? -EREMCHG :
			  cause == ATM_CV_DEST_OOO ? -EHOSTDOWN :
			  -ECONNREFUSED);
			/* fall through */
		    case ss_accepting:
			set_error(sock,-ECONNRESET); /* ERESTARTSYS ? */
			/* fall through */
		    case ss_rel_req:
			send_close(sock);
			/* fall through */
		    case ss_wait_rel:
			STOP_TIMER(sock);
			free_sock(sock);
			return;
		    case ss_connected:
			diag(COMPONENT,DIAG_INFO,"Passive close (CR 0x%06X)",
			  sock->call_ref);
			send_close(sock);
			/* fall through */
		    case ss_hold:
			new_state(sock,ss_wait_close);
			return;
		    case ss_indicated:
			new_state(sock,ss_zombie);
			return;
		    default:
			break;
		}
		break; /* fail */
	    }
	default:
	    diag(COMPONENT,DIAG_WARN,"Bad Q.2931 message %d",mid);
	    send_status(sock,0,ATM_CV_UNKNOWN_MSG_TYPE,mid);
	    return;
    }
    diag(COMPONENT,DIAG_WARN,
      "Q.2931 message %s is incompatible with state %s/%s (%d?%d)",
      mid2name(mid),state_name[sock->state],qs_name[sock->q2931_state],
      (int) sock->state,(int) sock->q2931_state);
    send_status(sock,0,ATM_CV_INCOMP_MSG,mid);
}


static void clear_all_calls(void)
{
    SOCKET *curr,*next;

    for (curr = sockets; curr; curr = next) {
	next = curr->next;
	if (curr->q2931_state != qs_null) q2931_call(curr,QMSG_RESTART);
    }
}


void clear_all_calls_on_T309(void)
{
    clear_all_calls();
    t309 = NULL;
}


void saal_failure(void)
{
    SOCKET *curr,*next;

    trace_msg("SAAL went down");
    for (curr = sockets; curr; curr = next) {
	next = curr->next;
	if (curr->q2931_state != qs_null)
		if (curr->q2931_state != qs_active)
			q2931_call(curr,QMSG_RESTART);
		else if (!t309) t309 = start_timer(T309_TIME,on_T309,NULL);
    }
}


void saal_okay(void)
{
    SOCKET *curr;

    trace_msg("SAAL came up");
    if (!t309) return;
    stop_timer(t309);
    t309 = NULL;
    for (curr = sockets; curr; curr = curr->next)
	if (curr->q2931_state != qs_null) send_status_enq(curr);
}


static void process_q2931(void *msg)
{
    SOCKET *curr;
    unsigned long call_ref;
    unsigned char mid;

    call_ref = q_fetch(&in_dsc,QF_call_ref)^0x800000;
    mid = q_fetch(&in_dsc,QF_msg_type);
    if (mid == QMSG_REST_ACK) return;
    if (mid == QMSG_RESTART) { /* 5.5.5.2 */
	int rst_class;

	rst_class = q_fetch(&in_dsc,QF_rst_class);
	switch (rst_class) {
	    case ATM_RST_IND_VC:
		{
		    int vpi,vci;

		    if (!q_present(&in_dsc,QG_conn_id)) {
			send_status(NULL,0L,ATM_CV_MAND_IE_MISSING,
			  ATM_IE_CONN_ID);
			return;
		    }
		    vpi = q_fetch(&in_dsc,QF_vpi);
		    vci = q_fetch(&in_dsc,QF_vci);
		    for (curr = sockets; curr; curr = curr->next)
			if (curr->pvc.sap_addr.vpi == vpi &&
			  curr->pvc.sap_addr.vci == vci) break;
		    if (!curr) {
			send_status(NULL,0L,ATM_CV_INVALID_IE,
			  ATM_IE_CONN_ID);
			return;
		    }
		    q2931_call(curr,mid);
		    send_restart_ack(vpi,vci);
		}
		break;
	    case ATM_RST_ALL_VC:
		clear_all_calls();
		send_restart_ack(0,0);
		break;
	    default:
		send_status(NULL,0L,ATM_CV_INVALID_IE,ATM_IE_RESTART);
	}
	return;
    }
    if (!(call_ref & 0x7fffff)) {
	return; /* bad things happen ... @@@ */
    }
    for (curr = sockets; curr; curr = curr->next)
	if (curr->call_ref == call_ref) break;
    diag(COMPONENT,DIAG_DEBUG,"FROM NET: %s (0x%02X) CR 0x%06lx for 0x%lx",
      mid2name(((unsigned char *) msg)[5]),((unsigned char *)msg)[5],call_ref,
      curr ? curr->id : 0);
    if (mid == QMSG_SETUP) {
	if (!curr) setup_call(call_ref);
	return;
    }
    if (mid == QMSG_STATUS_ENQ) {
	send_status(curr,call_ref,ATM_CV_RESP_STAT_ENQ);
	return;
    }
    if (curr && q_present(&in_dsc,QF_ep_ref) && mid != QMSG_ADD_PARTY &&
      mid != QMSG_DROP_PARTY_ACK)
	if (curr->ep_ref != q_fetch(&in_dsc,QF_ep_ref)) {
	    send_drop_party_ack(call_ref,q_fetch(&in_dsc,QF_ep_ref),
	      ATM_CV_INV_EPR);
	    return;
	}
    if (!curr || curr->q2931_state == qs_null) {
	if (mid != QMSG_REL_COMP)
	    if (mid != QMSG_STATUS)
		send_release_complete(call_ref,ATM_CV_INV_CR);
	    else if (q_fetch(&in_dsc,QF_call_state) != (int) qs_null)
		    send_release_complete(call_ref,ATM_CV_INCOMP_MSG);
	return;
    }
    q2931_call(curr,mid);
}


static void abort_call(void)
{
    SOCKET *curr;
    unsigned long call_ref;

    diag(COMPONENT,DIAG_ERROR,"can't parse message - aborting the call");
    call_ref = q_fetch(&in_dsc,QF_call_ref)^0x800000;
	/* hope that at least the call ref was okay ... */
    for (curr = sockets; curr; curr = curr->next)
        if (curr->call_ref == call_ref) {
	    q2931_call(curr,QMSG_RESTART);
	    break;
	}
    send_release_complete(call_ref,ATM_CV_PROTOCOL_ERROR);
}


void to_q2931(void *msg,int size)
{
    if (q_open(&in_dsc,msg,size) < 0) {
	abort_call();
	return;
    }
    process_q2931(msg);
    if (q_close(&in_dsc) < 0)
	diag(COMPONENT,DIAG_ERROR,"q_close returned <0 in to_q2931");
}
