/* proto.c - Common protocol functions and structures */
 
/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */
 

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "io.h"
#include "proto.h"
#include "sap.h"


#define COMPONENT "SIGD"


const char *state_name[] = { /* formatting aligned with STATE */
	"<invalid>",	"ss_null",	"ss_listening",	"ss_connecting",
	"ss_connected",	"ss_indicated",	"ss_accepting",	"ss_zombie",
	"ss_hold",	"ss_wait_rel",	"ss_wait_close","ss_rel_req",
	"ss_rel_ind" };

const char *qs_name[] = {
	"NULL",		"CALL_INIT",	"<invalid>",	"OUT_PROC",
	"<invalid>",	"<invalid>",	"CALL_PRES",	"<invalid>",
	"CONN_REQ",	"IN_PROC",	"ACTIVE",	"REL_REQ",
	"REL_IND" };

const char *as_name[] = { "<invalid>","as_bind","as_connect","as_accept",
  "as_listen","as_okay","as_error","as_indicate","as_close" };

const Q2931_STATE state_map[] = { /* formatting aligned with STATE */
	qs_null,	qs_null,	qs_null,	qs_call_init,
	qs_active,	qs_in_proc,	qs_conn_req,	qs_null,
	qs_active,	qs_rel_req,	qs_null,	qs_rel_req,
	qs_rel_ind };

const PARTY_STATE eps_map[] = {
	ps_null,	ps_add_init,	ps_null,	ps_add_init,	/* 0 */
	ps_null,	ps_null,	ps_add_recv,	ps_null,	/* 4 */
	ps_active,	ps_add_recv,	ps_active,	ps_active,	/* 8 */
	ps_active };							/*12 */

SOCKET *sockets = NULL;
unsigned char q_buffer[MAX_Q_MSG];


SOCKET *new_sock(unsigned long id)
{
    SOCKET *sock;
 
    sock = alloc_t(SOCKET);
    sock->state = ss_invalid;
    sock->pvc.sap_addr.vpi = sock->pvc.sap_addr.vci = 0;
    sock->qos.txtp.traffic_class = sock->qos.rxtp.traffic_class = ATM_UBR;
    sock->id = id;
    sock->local = sock->remote = NULL;
    sock->error = 0;
    sock->q2931_state = qs_null;
    sock->ep_ref = -1;
    sock->conn_timer = NULL;
    sock->listen = NULL;
    sock->next = sockets;
    sockets = sock;
    return sock;
}


void free_sock(SOCKET *sock)
{
    SOCKET **walk;

    diag(COMPONENT,DIAG_DEBUG,"freeing socket 0x%lx@0x%lx",sock->id,
      (unsigned long) sock);
    for (walk = &sockets; *walk != sock; walk = &(*walk)->next);
    if (!*walk)
	diag(COMPONENT,DIAG_FATAL,
	  "INTERNAL ERROR: freeing non-existing socket 0x%lx",sock->id);
    *walk = sock->next;
    if (sock->conn_timer) {
	diag(COMPONENT,DIAG_ERROR,"socket 0x%lx has timer (0x%lx) running",
	  sock->id,(unsigned long) sock->conn_timer->callback);
	stop_timer(sock->conn_timer);
    }
    if (sock->listen)
        diag(COMPONENT,DIAG_ERROR,"socket 0x%lx has non-empty listen queue",
	  sock->id);
    if (sock->local) free(sock->local);
    if (sock->remote) free(sock->remote);
    sock->state = ss_invalid;
    free(sock);
}


void new_state(SOCKET *sock,STATE state)
{
    diag(COMPONENT,DIAG_DEBUG,"socket 0x%lx enters state %s (Q.2931 %s)",
      sock->id,state_name[state],qs_name[state_map[state]]);
    sock->state = state;
    sock->q2931_state = state_map[state];
}


SOCKET *lookup_sap(const struct sockaddr_atmsvc *sap,const struct atm_qos *qos,
    struct sockaddr_atmsvc **res_sap,struct atm_qos *res_qos)
{
    SOCKET *walk,*wildcard;

    wildcard = NULL;
    for (walk = sockets; walk; walk = walk->next)
	if (walk->state == ss_listening) {
	    if (walk->local && sap_compat(walk->local,sap,res_sap,&walk->qos,
	      qos,res_qos)) return walk;
	    if (!walk->local) wildcard = walk;
	}
    return wildcard;
}


const char *mid2name(unsigned char mid)
{
    switch (mid) {
	case QMSG_CALL_PROC:
	    return "CALL_PROCEEDING";
	case QMSG_CONNECT:
	    return "CONNECT";
	case QMSG_CONN_ACK:
	    return "CONNECT_ACK";
	case QMSG_SETUP:
	    return "SETUP";
	case QMSG_RELEASE:
	    return "RELEASE";
	case QMSG_REL_COMP:
	    return "REL_COMP";
	case QMSG_RESTART:
	    return "RESTART";
	case QMSG_REST_ACK:
	    return "REST_ACK";
	case QMSG_STATUS:
	    return "STATUS";
	case QMSG_STATUS_ENQ:
	    return "STATUS_ENQ";
	case QMSG_ADD_PARTY:
	    return "ADD_PARTY";
	case QMSG_ADD_PARTY_ACK:
	    return "ADD_PARTY_ACK";
	case QMSG_ADD_PARTY_REJ:
	    return "ADD_PARTY_REJECT";
	case QMSG_DROP_PARTY:
	    return "DROP_PARTY";
	case QMSG_DROP_PARTY_ACK:
	    return "DROP_PARTY_ACK";
	default:
	    return "???";
    }
}


void send_kernel(unsigned long vcc,unsigned long listen_vcc,
  enum atmsvc_msg_type type,int reply,const struct sockaddr_atmpvc *pvc,
  const struct sockaddr_atmsvc *svc,const struct sockaddr_atmsvc *local,
  const struct atm_qos *qos)
{
    struct atmsvc_msg *msg;
    struct atm_blli *walk;
    int blli,i;

    blli = 0;
    for (walk = svc ? svc->sas_addr.blli : NULL; walk; walk = walk->next)
	blli++;
    msg = alloc(bllis2msg(blli));
    msg->vcc = vcc;
    msg->listen_vcc = listen_vcc;
    msg->type = type;
    msg->reply = reply;
    if (pvc) msg->pvc = *pvc;
    else memset(&msg->pvc,0,sizeof(msg->pvc));
    if (qos) msg->qos = *qos;
    else memset(&msg->qos,0,sizeof(msg->qos));
    if (!local) memset(&msg->local,0,sizeof(msg->local));
    else {
	msg->local = *local;
	msg->local.sas_addr.blli = NULL;
    }
    if (!svc) memset(&msg->svc,0,sizeof(msg->svc));
    else {
	msg->svc = *svc;
	i = 0;
	for (walk = svc->sas_addr.blli; walk; walk = walk->next)
	    msg->blli[i++] = *walk;
    }
    to_kernel(msg);
    free(msg);
}


void send_release(SOCKET *sock,unsigned char reason,...)
{
    va_list ap;
    Q_DSC dsc;
    int size;
 
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_RELEASE);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    q_assign(&dsc,QF_cause,reason);
    va_start(ap,reason);
    switch (reason) {
	case ATM_CV_TIMER_EXP:
	    {
		char buf[4];

		sprintf(buf,"%d",va_arg(ap,int));
		q_write(&dsc,QF_timer,buf,3);
		break;
	    }
	default:
    }
    va_end(ap);
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


void send_release_complete(unsigned long call_ref,unsigned char cause)
{
    Q_DSC dsc;
    int size;
 
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_REL_COMP);
    q_assign(&dsc,QF_call_ref,call_ref);
    if (cause) q_assign(&dsc,QF_cause,cause); /* @@@ more data */
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
}


void set_error(SOCKET *sock,int code)
{
    if (!sock->error) sock->error = code;
}


void send_close(SOCKET *sock)
{
    if (sock->error == 1234) diag(COMPONENT,DIAG_ERROR,"BUG! BUG! BUG!");
    send_kernel(sock->id,0L,as_close,sock->error,NULL,NULL,NULL,NULL);
    sock->error = 1234;
}


void q_report(int severity,const char *msg,...)
{
    va_list ap;

    va_start(ap,msg);
    vdiag("QMSG",severity,msg,ap);
    va_end(ap);
}


typedef struct _vpci {
    int vpci;
    int itf;
    struct _vpci *next;
} VPCI;


static VPCI *vpcis = NULL;


void enter_vpci(int vpci,int itf)
{
    VPCI *entry;

    for (entry = vpcis; entry; entry = entry->next)
	if (entry->vpci == vpci) {
	    diag(COMPONENT,DIAG_ERROR,"ignoring duplicate VPCI %d (itf %d)",
	      vpci,itf);
	    return;
	}
    entry = alloc_t(VPCI);
    entry->vpci = vpci;
    entry->itf = itf;
    entry->next = vpcis;
    vpcis = entry;
}


int get_itf(int *vpci)
{
    VPCI *best,*walk;

    best = NULL;
    for (walk = vpcis; walk; walk = walk->next)
	if (walk->vpci <= *vpci && (!best || best->vpci < walk->vpci))
	    best = walk;
    if (!best) return signaling_pvc.sap_addr.itf;
    *vpci -= best->vpci;
    return best->itf;
}


void init_addr(void)
{
    VPCI *walk;

    itf_load(signaling_pvc.sap_addr.itf);
    for (walk = vpcis; walk; walk = walk->next) itf_load(walk->itf);
}
