/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   cm_sap.h
*   
* Overview:
*   This describes the Service Access Point (SAP) interface to the
*   Connection Manager.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   28-Sep-94  TLR   Created.
*   10-Oct-94  TLR   Updated based on Team Review.
*   12-Oct-94  TLR   Added the user_data field to the XMT and RCV calls.
*   17-Oct-94  TLR   Added Support for Multi-CM Instances.
*   24-Oct-94  TLR   Small updates while writing implementation.
*    4-Nov-94  TLR   Added Link Status Indication
*    9-Nov-94  TLR   Removed Binding and SAP_ID Concept
*   16-Dec-94  TLR   Added conn_context and conn_handle, point-to-multipoint
*                    support, transmit done callback, and miscellaneous
*                    clean-up.
*   19-Jan-95  TLR   Added length parameters to the transmit and receive calls.
*   15-Feb-95  TLR   Updated SVC Information Block Types.  Added AAL Data.
*
* Dependencies:
*   types.h    - Generally used type definitions.
*   codes.h    - Completion codes.
*   atm.h      - ATM specific definitions and types.
*   svc_info.h - SVC Information Block Definitions.
*
* Description:
*
*   The Connection Manager (CM) exports this interface to software entities
*   called Service Access Points (SAPs) which communicate over an ATM
*   network.  The SAP is a registration interface to which at any time an
*   arbitrary number of SAP entities may register.
*
*   A SAP may open and communicate on any number of VCs.  A SAP may also use a
*   mix of Permanent Virtual Channels (PVCs) and Switched Virtual Channels
*   (SVCs).
*
*   The SAP interface provides the following basic services:
*
*       - Setup and Teardown of PVCs,
*       - Setup and Teardown of SVCs,
*       - Packet Transmission based on Connection or ATM Address,
*       - Demultiplexing of Received Packets to SAPs.
*
*   This module assumes that any synchronization issues are handled externally.
*   It is assumed that there is at most one process thread running in each
*   instance of the module at any time.
*
*   The CM handles transmit and receive packets.  It is not concerned with the
*   format of the packets and does not look into the contents of packets.  The
*   CM simply treats each packet as a pointer to void.  The CM's neighboring
*   components must agree on the format of data packets
*
* Concepts:
*
*   Connection Handle:
*
*     Connection handles are used to uniquely identify individual PVCs and
*     SVCs.  Connection handles are more abstract than VCIs because the VCI
*     is not always known during the life of an SVC.  At circuit setup time,
*     the Connection Manager provides a connection handle that the SAP must
*     subsequently use.
*
*   Connection Context:
*
*     Connection contexts are defined by SAP clients and are opaque to the
*     Connection Manager.  These contexts are provided by the SAP client during
*     VC setup and are stored by the Connection Manager with the other VC
*     information.  The connection context is passed back to the SAP in each
*     callback that is specific to a particular VC.  For example, each packet
*     received is passed to the SAP along with the connection context for the
*     VC on which that packet was received.
*
*   User Data:
*
*
*   AAL Data:
*
*     The connection manager is not specific to any particular ATM Adaptation
*     Layer (AAL).  An AAL_Data field is passed along with each transmit and
*     receive packet.  Refer to the definition of AAL_DATA in the "atm.h"
*     header file.
*
*/
#ifndef CM_SAP_H
#define CM_SAP_H
#include "atm.h"
#include "svc_info.h"
#include "system.h"
/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* NOTIFY_EVENT
 *
 *   This type enumerates the possible events that may be reported to a
 *   registered SAP client by the Connection Manager.
 *
 *   CONN_VC_READY......This event marks the completion of a VC setup.
 *                      The PVC or SVC is now ready for use.
 *
 *   CONN_RELEASE.......This event occurs when an existing SVC is released.
 *                      This may happen because the remote station released
 *                      the connection, or because a local event caused the
 *                      circuit to be released (i.e. loss of link).
 *
 *   CONN_PARTY_ADD.....An endpoint was successfully added to a point-to-
 *                      multipoint SVC.
 *
 *   CONN_PARTY_DROP....An endpoint was dropped from a point-to-multipoint
 *                      SVC.  This is also used to inform the SAP client that
 *                      a requested add party operation failed.
 *
 *   CONN_STALL.........This event occurs when a VC is stalled for any reason.
 *                      VCs become stalled as a result of flow-control credit
 *                      resynchronization or by network management action.
 *                      The stalled event will not occur as a result of
 *                      congestion experienced on the network, rather only
 *                      when some local entity explicitly stalls a VC.
 *
 *   CONN_UNSTALL.......This event marks the end of stalled status of a VC.
 */

typedef enum
   {
   CONN_VC_READY,
   CONN_RELEASE,
   CONN_PARTY_ADD,
   CONN_PARTY_DROP,
   CONN_STALL,
   CONN_UNSTALL
   } NOTIFY_EVENT;

/*************************************************************************
 * SAP Interface Callbacks                                               *
 *************************************************************************/

/*++
* ======================
* = (SAP_RCV_CALLBACK) =
* ======================
*
* Overview:
*   Callback for received packets.  A pointer to this routine is supplied
*   at registration time by the SAP.
*
* Arguments:
*   conn_context - (IN) Connection Context for the incoming VC.
*   conn_handle  - (IN) Handle of Connection on which the packet was received.
*   length       - (IN) Length of the packet payload in bytes.
*   user_data    - (IN) User data that was transferred from the layers below
*                       the CM to the SAP.
*   p_aal_data   - (IN) Data specific to the AAL being used is passed in this
*                       structure.
*   pp_packet    - (IO) Pointer to a pointer to a packet.  On input, the packet
*                       is the received packet.  On output, the pointer is a
*                       reference to a buffer that is ready to be reused.  If
*                       this function does not copy the packet, it must either
*                       return a pointer to a buffer that is free or return
*                       a NULL pointer.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   The packet referred to by *pp_packet may be safely overwritten or
*   destroyed.
--*/
typedef void (*SAP_RCV_CALLBACK) (HANDLE     conn_context,
                                  HANDLE     conn_handle,
                                  UINT32     length,
                                  UINT32     user_data,
                                  AAL_DATA  *p_aal_data,
                                  void     **pp_packet);

/*++
* ===========================
* = (SAP_XMT_DONE_CALLBACK) =
* ===========================
*
* Overview:
*   Callback for transmit-done indications.  This callback notifies the
*   SAP that the transmitted buffer has been DMA'd out of memory and may
*   be freed.
*
* Arguments:
*   conn_context - (IN  Connection Context of Transmit VC.
*   p_packet     - (IN) Pointer to the data packet.
*   user_data    - (IN) User data that was provided by this SAP in the
*                       transmit request.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef void (*SAP_XMT_DONE_CALLBACK) (HANDLE  conn_context,
                                       void   *p_packet,
                                       UINT32  user_data);


/*++
* ============================
* = (SAP_VC_NOTIFY_CALLBACK) =
* ============================
*
* Overview:
*   Callback to notify the SAP of the results of a VC setup or to indicate
*   that an SVC was closed by the remote party.
*
* Arguments:
*   conn_context  - (IN) Connection Context for this VC.
*   conn_handle   - (IN) Connection Handle which identifies the VC.
*   event         - (IN) Type of event the SAP is being notified of.
*   reason        - (IN) Reason value for release and drop party events.
*   endpoint_ref  - (IN) Endpoint reference for party adds and drops.
*   calling_party - (IN) Boolean flag:  Set iff this SAP is the calling party
*                        of the new connection.
*   
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef void (*SAP_VC_NOTIFY_CALLBACK) (HANDLE         conn_context,
                                        HANDLE         conn_handle,
                                        NOTIFY_EVENT   event,
                                        UINT16         reason,
                                        UINT32         endpoint_ref,
                                        BOOLEAN        calling_party);


/*++
* ==========================
* = (SAP_CONNECT_CALLBACK) =
* ==========================
*
* Overview:
*   This is the callback that the CM invokes when it wishes to offer a new
*   connection to a SAP.  The SAP must decide whether or not it wants the
*   connection and return a boolean value indicating its decision.
*
* Arguments:
*   sap_context      - (IN)  SAP context supplied at registration time.
*   p_conn_info      - (IN)  Connection information block for the connection.
*   conn_handle      - (IN)  Handle which identifies the VC.
*   p_conn_context   - (OUT) Connection context if accepted.
*   p_age_limit      - (OUT) The age limit for this SVC in seconds.  If this
*                            SVC is idle (no traffic) for longer than the
*                            age limit, it will be torn down.  If the age limit
*                            is specified as zero, the SVC will never be
*                            aged out.
*   p_reject_reason  - (OUT) The reason this connection was rejected if the
*                            return value is FALSE.  If this connection is
*                            being rejected because the SAP is not interested
*                            in it (i.e. the called party address or BLLI
*                            does not match), this value MUST be
*                            REJECT_SAP_MISMATCH.
*              
* Returns:
*   TRUE  - Connection Accepted.
*   FALSE - Connection Rejected.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef BOOLEAN (*SAP_CONNECT_CALLBACK) (HANDLE       sap_context,
                                         CONN_INFO   *p_conn_info,
                                         HANDLE       conn_handle,
                                         HANDLE      *p_conn_context,
                                         UINT32      *p_age_limit,
                                         UINT32      *p_reject_reason);

/*++
* ==============================
* = (SAP_LINK_STATUS_CALLBACK) =
* ==============================
*
* Overview:
*   The Connection Manager invokes this callback when the link status changes.
*   This serves to notify the SAP client when the link becomes available or
*   unavailable.
*
*   When the link becomes unavailable, all open SVCs are automatically closed.
*   Therefore once a SAP receives an indication that the link is unavailable,
*   it will subsequently receive individual release notifications for each of
*   its open SVCs.
*
* Arguments:
*   sap_context    - (IN) Context of registered SAP.
*   link_status    - (IN) New value of link status.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
typedef void (*SAP_LINK_STATUS_CALLBACK) (HANDLE           sap_context,
                                          ATM_LINK_STATE   link_state);


/*************************************************************************
 * SAP Interface Function Prototypes                                     *
 *************************************************************************/
  

/*++
* ===================
* = cm_sap_register =
* ===================
*
* Overview:
*   Registers a new SAP with the Connection Manager.
*
* Arguments:
*   cm_handle               - (IN)  Handle returned by cm_create.
*
*   sap_context             - (IN)  Context handle for this SAP client.  This
*                                   handle is provided by the client and is
*                                   passed back to the client during connect
*                                   callbacks.
*
*   p_rcv_callback          - (IN)  Pointer to the receive callback.
*
*   p_vc_notify_callback    - (IN)  Pointer to the notify callback.
*
*   p_svc_connect_callback  - (IN)  Pointer to the connect callback.  If this
*                                   pointer is NULL, the Connection Manager
*                                   shall not offer any incoming connections
*                                   to this SAP.
*
*   p_link_status_callback  - (IN)  Pointer to the link status callback.
*
*   p_xmt_done_callback     - (IN)  Pointer to the transmit done callback.
*
*   p_sap_text              - (IN)  Pointer to a descriptive string.
*
*   p_sap_handle            - (OUT) Handle to be used in subsequent cm_sap_*
*                                   calls.
*              
* Returns:
*   STATUS_K_SUCCESS    - Success: See Postconditions.
*   STATUS_K_RESOURCES  - Insufficient resources to complete registration.
*
* Preconditions:
*   None
*
* Postconditions:
*   The handle is valid.
--*/
STATUS cm_sap_register (
                        HANDLE                     sap_context,
                        SAP_RCV_CALLBACK           rcv_callback,
                        SAP_VC_NOTIFY_CALLBACK     vc_notify_callback,
                        SAP_CONNECT_CALLBACK       svc_connect_callback,
                        SAP_LINK_STATUS_CALLBACK   link_status_callback,
                        SAP_XMT_DONE_CALLBACK      xmt_done_callback,
                        const char                *p_sap_text,
                        HANDLE                    *p_sap_handle);


/*++
* =====================
* = cm_sap_unregister =
* =====================
*
* Overview:
*   Cancel the registration of a SAP. The handle is unusable immediately upon
*   the completion of this call but if there were opened VCs, it will be some
*   time before those VC_IDs are again available for use.
*
* Arguments:
*   sap_handle    - (IN) Handle of the SAP to be unregistered
*              
* Returns:
*   None
*   
* Preconditions:
*   The handle is valid.
*
* Postconditions:
*   The handle is invalid.
--*/
void cm_sap_unregister (HANDLE sap_handle);


/*++
* ====================
* = cm_sap_svc_setup =
* ====================
*
* Overview:
*   Requests the setup of an SVC.  This is an asynchronous call.  The SVC
*   will not be available for use for some time after this call completes.
*   The (sap_vc_notify) callback is used to notify the caller that the
*   SVC has been set up and is ready for use.
*
* Arguments:
*   sap_handle       - (IN)  SAP Handle returned by cm_sap_register
*   conn_context     - (IN)  Connection Context for this VC.
*   p_conn_info      - (IN)  Requested VC signaling parameters.
*   p_age_limit      - (OUT) The age limit for this SVC in seconds.  If this
*                            SVC is idle (no traffic) for longer than the
*                            age limit, it will be torn down.  If the age limit
*                            is specified as zero, the SVC will never be
*                            aged out.
*   p_conn_handle    - (OUT) Connection handle that identifies pending VC.
*              
* Returns:
*   STATUS_K_PENDING           - Success: Async operation has begun.
*   STATUS_K_RESOURCES         - Insufficient internal resources to complete
*                                the setup.
*   STATUS_K_INSUFFICIENT_DATA - The connection information block doesn't have
*                                enough detail to complete the operation.
*   STATUS_K_ATM_RESOURCES     - Insufficient ATM resources available to meet
*                                requested parameters.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS cm_sap_svc_setup (HANDLE     sap_handle,
                         HANDLE     conn_context,
                         CONN_INFO *p_conn_info,
                         UINT32     age_limit,
                         HANDLE    *p_conn_handle);


/*++
* =======================
* = cm_sap_svc_teardown =
* =======================
*
* Overview:
*   Disconnect and Close an SVC.  This function begins the process of
*   closing the VC.  The VC_ID may not be available for reuse for some time
*   after this call completes.
*
*   When the connection is closed, the SAP will be notified via the 
*   sap_vc_notify callback.
*
* Arguments:
*   conn_handle     - (IN)  Connection handle of the SVC to be removed.
*              
* Returns:
*   STATUS_K_PENDING        - Success: Async operation has begun
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
STATUS cm_sap_svc_teardown (HANDLE        conn_handle);

/*++
* =================
* = cm_sap_xmt_vc =
* =================
*
* Overview:
*   Transmit a packet directly via a VC.  No lookup is necessary to perform
*   this transmit operation.  It will execute very quickly.
*
* Arguments:
*   conn_handle   - (IN)  Connection handle of outgoing channel.
*
*   p_data        - (IN)  Pointer to the data packet.
*
*   length        - (IN)  Length of the data packet in bytes. If this parameter
*                         is set to zero, the length is considered unspecified.
*                         In this case, the lower layers must be able to find
*                         the length by some other means.  For protocol-
*                         generated packets, the p_data pointer will point to
*                         some structure that contains the length and location
*                         of the packet's payload data.  In this case, an
*                         unspecified length value is appropriate.
*
*   user_data     - (IN)  User data to be transferred from the SAP to the
*                         layers below the CM.
*   p_aal_data    - (IN)  Data specific to the AAL being used is passed in this
*                         structure.
*              
* Returns:
*   STATUS_K_SUCCESS    - Packet was transmitted, buffer may be freed.
*   STATUS_K_PENDING    - Packet was queued, buffer may not be freed.
*   STATUS_K_CONGESTED  - Transmit queue full.  The packet was not transmitted
*                         and has not been deallocated.  The SAP may either
*                         queue or discard the packet.
*
* Preconditions:
*   It is assumed that the requested vc_id is bound to this SAP.
*
*   It is assumed that the VC is opened and ready for use.
*
* Postconditions:
*   None
--*/
STATUS cm_sap_xmt_vc (HANDLE    conn_handle,
                      void     *p_data,
                      UINT32    length,
                      UINT32    user_data,
                      AAL_DATA *p_aal_data);


/*++
* ==========================
* = cm_sap_link_status_get =
* ==========================
*
* Overview:
*   Returns the current link status of the driver attached to the CM.
*
* Arguments:
*   sap_handle    - (IN)  Handle returned by cm_sap_register.
*   p_link_status - (OUT) Current Link Status.
*              
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/
void cm_sap_link_status_get (HANDLE           sap_handle,
                             ATM_LINK_STATE  *p_link_state);
#endif /* CM_SAP_H */

