/* atmsigd.c - ATM signaling demon */

/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <linux/atm.h>

#include "atm.h"
#include "atmd.h"
#include "qlib.h"

#include "io.h"
#include "proto.h"
#include "saal.h"


#define COMPONENT "SIGD"
#define CONFIG_FILE "/etc/atmsigd.conf"


extern int yyparse(void);
extern FILE *yyin;

int net = 0;
int debug = 0;
int pretty = A2T_PRETTY | A2T_NAME;

static SAAL_DSC saal;


static void usage(const char *name)
{
    fprintf(stderr,"usage: %s [ -b ] [ -c config_file ] [ -d ] [ -l logfile ] "
      "[ -n ] [ -N ]\n",name);
    exit(1);
}


static void q_estab_conf(void *user_data,void *uu_data,int uu_length)
{
    saal_okay();
}


static void q_rel_ind(void *user_data,void *uu_data,int uu_length)
{
    saal_failure();
    saal_estab_req(&saal,NULL,0);
}


static void q_restart(void *user_data,void *uu_data,int uu_length,int ind)
{
    saal_failure();
    if (!ind) saal_okay(); /* actually, ind should probably never be zero */
}


void from_net(void *buffer,int size)
{
    saal_pdu(&saal,buffer,size);
}


void to_signaling(void *msg,int size)
{
    saal_send(&saal,msg,size);
}


static void q_data_ind(void *user_data,void *data,int length)
{
    to_q2931(data,length);
}


static void q_cpcs_send(void *user_data,void *data,int length)
{
    to_net(data,length);
}


static SAAL_USER_OPS ops = {
    NULL, /* no q_estab_ind - 5.5.6.9 says 5.5.6.11 and 5.5.6.11 says "may" */
    q_estab_conf,
    q_rel_ind,
    NULL, /* no q_rel_conf - what to do ? */
    q_restart,
    q_data_ind,
    NULL, /* no q_unitdata */
    q_cpcs_send
};


static void dump_sap(const char *label,struct sockaddr_atmsvc *sap)
{
    struct atm_blli *blli;
    int i,length;

    if (!sap) return;
    fprintf(stderr,"  %s ",label);
    if (*sap->sas_addr.pub)
	fprintf(stderr,"%s%s",sap->sas_addr.pub,*sap->sas_addr.prv ? "+":
	  "\n    ");
    if (*sap->sas_addr.prv) {
	for (i = 0; i < ATM_ESA_LEN; i++)
	    fprintf(stderr,"%02X",sap->sas_addr.prv[i]);
	fprintf(stderr,"\n    ");
    }
    if (sap->sas_addr.bhli.hl_type) {
	length =
#ifdef UNI30
	   sap->sas_addr.bhli.hl_type == ATM_HL_HLP ? 4 :
#endif
	  sap->sas_addr.bhli.hl_type == ATM_HL_VENDOR ? 7 :
	  sap->sas_addr.bhli.hl_length;
	fprintf(stderr,"BHLI %d[%d] = { ",sap->sas_addr.bhli.hl_type,length);
	for (i = 0; i < length; i++)
	    fprintf(stderr,"%s%02X",i ? "," : "",sap->sas_addr.bhli.hl_info[i]);
	fprintf(stderr," }\n    ");
    }
    for (blli = sap->sas_addr.blli; blli; blli = blli->next) {
	fprintf(stderr,"BLLI");
	if (blli->l2_proto) {
	    fprintf(stderr,"  L2 %d",blli->l2_proto);
	    switch (blli->l2_proto) {
		case ATM_L2_X25_LL:
		case ATM_L2_X25_ML:
		case ATM_L2_HDLC_ARM:
		case ATM_L2_HDLC_NRM:
		case ATM_L2_HDLC_ABM:
		case ATM_L2_Q922:
		case ATM_L2_ISO7776:
		    fprintf(stderr," mode=%d, window=%d\n    ",
		      blli->l2.itu.mode,blli->l2.itu.window);
		    break;
		case ATM_L2_USER:
		    fprintf(stderr," user=0x%x\n    ",blli->l2.user);
		    break;
		default:
		    fprintf(stderr,"\n    ");
	    }
	}
	if (blli->l3_proto) {
	    fprintf(stderr,"  L3 %d",blli->l3_proto);
	    switch (blli->l3_proto) {
		case ATM_L3_X25:
		case ATM_L3_ISO8208:
		case ATM_L3_X223:
		    fprintf(stderr," mode=%d, def=%d, pack=%d\n    ",
		      blli->l3.itu.mode,blli->l3.itu.def_size,
		      blli->l3.itu.window);
		    break;
		case ATM_L3_TR9577:
		    fprintf(stderr," ipi=0x%x",blli->l3.tr9577.ipi);
		    if (blli->l3.tr9577.ipi != NLPID_IEEE802_1_SNAP)
			fprintf(stderr,"\n    ");
		    else {
			for (i = 0; i < 5; i++)
			    fprintf(stderr,"%s%02X",i ? "," : " ",
			      blli->l3.tr9577.snap[i]);
			fprintf(stderr,"\n    ");
		    }
		    break;
		case ATM_L3_USER:
		    fprintf(stderr," user=0x%x\n    ",blli->l3.user);
		    break;
		default:
		    fprintf(stderr,"\n    ");
	    }
	}
    }
    fprintf(stderr,"\r");
}


static void dump_status(int sig)
{
    SOCKET *walk;

    fprintf(stderr,"Status dump (on SIGUSR1)\n");
    for (walk = sockets; walk; walk = walk->next) {
	fprintf(stderr,"0x%lx: %s, CR 0x%06lX, PVC %d.%d.%d\n",(unsigned long)
	  walk->id,state_name[walk->state],walk->call_ref,
	  walk->pvc.sap_addr.itf,walk->pvc.sap_addr.vpi,walk->pvc.sap_addr.vci);
        dump_sap("local ",walk->local);
        dump_sap("remote",walk->remote);
    }
}


static void setup_signals(void)
{
    struct sigaction act;

    act.sa_handler = dump_status;
    sigemptyset(&act.sa_mask);
    act.sa_flags = 0;
    if (sigaction(SIGUSR1,&act,NULL) < 0) {
	perror("sigaction");
	exit(1);
    }
}


int main(int argc,char **argv)
{
    const char *config_file;
    int c,background;

    set_application("atmsigd");
    config_file = CONFIG_FILE;
    background = 0;
    memset(&signaling_pvc,0,sizeof(signaling_pvc));
    signaling_pvc.sap_addr.vci = 5;
    while ((c = getopt(argc,argv,"bc:dl:nN")) != EOF)
	switch (c) {
	    case 'b':
		background = 1;
		break;
	    case 'c':
		config_file = optarg;
		break;
	    case 'd':
		set_verbosity(NULL,DIAG_DEBUG);
		set_verbosity("QMSG",DIAG_INFO);
		set_verbosity("SSCOP",DIAG_INFO);
		debug = 1;
		/*q_dump = 1;*/
		break;
	    case 'l':
		set_logfile(optarg);
		break;
	    case 'n':
		pretty = A2T_PRETTY;
		break;
	    case 'N':
		net = 1;
		break;
	    default:
		usage(argv[0]);
	}
    if (optind != argc) usage(argv[0]);
    diag(COMPONENT,DIAG_INFO,"Linux ATM signaling "
#ifdef UNI30
      "UNI 3.0"
#endif
#ifdef UNI31
      "UNI 3.1"
#ifdef ALLOW_UNI30
      "+3.0compat"
#endif
#endif
      "/AAL5, version " VERSION);
    if (!(yyin = fopen(config_file,"r")))
	diag(COMPONENT,DIAG_WARN,"%s not found. - Using defaults.",config_file);
    else if (yyparse()) {
	    diag(COMPONENT,DIAG_FATAL,"Error in config file. - Aborting.");
	    return 1;
	}
    diag(COMPONENT,DIAG_INFO,"Acting as %s side",net ? "NETWORK" : "USER");
    if (open_all()) return 1;
    init_addr();
    q_start();
    start_saal(&saal,&ops,NULL);
    saal_estab_req(&saal,NULL,0);
    setup_signals();
    if (background) {
	pid_t pid;

	pid = fork();
	if (pid < 0)
	    diag(COMPONENT,DIAG_FATAL,"fork: %s",strerror(errno));
	if (pid) {
	    diag(COMPONENT,DIAG_DEBUG,"Backgrounding (PID %d)",pid);
	    exit(0);
	}
    }
    poll_loop();
    close_all();
    stop_saal(&saal);
    return 0;
}
