/* kernel.c - Processing of incoming kernel messages */
 
/* Written 1995,1996 by Werner Almesberger, EPFL-LRC */
 

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <linux/atm.h>
#include <linux/atmsvc.h>
#include <linux/atmdev.h>

#include "atm.h"
#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "proto.h"
#include "sap.h"
#include "io.h"
#include "timeout.h"


#define COMPONENT "KERNEL"


extern int assign_vci;


static int send_setup(SOCKET *sock)
{
    static unsigned long call_ref = 0;
    struct sockaddr_atmsvc *local;
    SOCKET *walk;
    Q_DSC dsc;
    int error,size;

    do {
	if (++call_ref == 0x7fffff) call_ref = 1;
	for (walk = sockets; walk; walk = walk->next)
	    if (walk->call_ref == call_ref) break;
    }
    while (walk);
    sock->call_ref = call_ref;
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_call_ref,call_ref);
    q_assign(&dsc,QF_msg_type,QMSG_SETUP);
    q_assign(&dsc,QF_aal_type,5); /* AAL 5 */
#ifdef UNI30
    q_assign(&dsc,QF_aal_mode,1); /* Message mode - LANE seems to really want
				     this */
#endif
    q_assign(&dsc,QF_sscs_type,0); /* unspecified - LANE wants this */
    error = sap_encode(&dsc,sock->remote);
    q_assign(&dsc,QF_bearer_class,16); /* BCOB-X */
    q_assign(&dsc,QF_traf_type,0); /* force presence - UNI 3.0 wants this */
    q_assign(&dsc,QF_upcc,0); /* p2p */
#ifndef UNI30
    q_assign(&dsc,QF_qos_cs,Q2931_CS_ITU);
#endif
    q_assign(&dsc,QF_qos_cs,Q2931_CS_NET);
    q_assign(&dsc,QF_qos_fw,0); /* QOS 0 */
    q_assign(&dsc,QF_qos_bw,0);
    local = NULL;
    if (sock->local && *sock->local->sas_addr.prv) local = sock->local;
    else if (local_addr[0].state == ls_same && *local_addr[0].addr.sas_addr.prv)
	    local = &local_addr[0].addr;
    /* @@@ handle E.164 too */
    if (!local) {
	if (!error) error = -EADDRNOTAVAIL;
    }
    else {
	q_assign(&dsc,QF_cgpn_plan,2);
	q_assign(&dsc,QF_cgpn_type,0);
	q_write(&dsc,QF_cgpn,(void *) local->sas_addr.prv,ATM_ESA_LEN);
    }
    if (net) {
	q_assign(&dsc,QF_vpi,0);
	q_assign(&dsc,QF_vci,assign_vci); /* @@@ */
	sock->pvc.sap_addr.itf = get_itf(0);
	sock->pvc.sap_addr.vpi = 0;
	sock->pvc.sap_addr.vci = assign_vci++;
    }
    if ((size = q_close(&dsc)) < 0) error = -EINVAL;
    else if (!error) to_signaling(q_buffer,size);
    return error;
}


static int send_connect(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CONNECT);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if (sock->ep_ref >= 0) q_assign(&dsc,QF_ep_ref,sock->ep_ref);
    /* @@@ lots of data */
    if ((size = q_close(&dsc)) >= 0) to_signaling(q_buffer,size);
    return 0;
}


static int bind_check(SOCKET *sock,struct sockaddr_atmsvc *local)
{
    struct sockaddr_atmsvc *use;

    if (sock->local && (*sock->local->sas_addr.prv ||
      *sock->local->sas_addr.pub)) return 0;
    if (*local->sas_addr.prv || *local->sas_addr.pub) use = local;
    else {
	if (local_addr[0].state != ls_same) {
	    diag(COMPONENT,DIAG_ERROR,"no local address");
	    return -EADDRNOTAVAIL;
	}
	use = &local_addr[0].addr;
    }
    use->sas_addr.blli = NULL; /* paranoia */
    if (sock->local) *sock->local = *use;
    else sock->local = sap_copy(use);
    return 0;
}


static void dispatch(SOCKET *sock,struct atmsvc_msg *msg)
{
    SOCKET *next;
    int error;

    switch (msg->type) {
	case as_bind: /* only in NULL state */
	    if (sock) break;
	    if (!*msg->svc.sas_addr.pub && !*msg->svc.sas_addr.prv)
#ifdef BE_PICKY_ABOUT_BINDING_LOCAL_WILDCARD_ADDRESSES
		if (local_addr[0].state != ls_same)
		    SEND_ERROR(msg->vcc,-EADDRNOTAVAIL);
		else
#endif
		send_kernel(msg->vcc,0,as_okay,0,NULL,NULL,
		      &local_addr[0].addr);
	    else {
		LOCAL_ADDR *walk;

		for (walk = local_addr; walk->state != ls_unused; walk++)
		    if (walk->state == ls_same && atm_equal(&walk->addr,
		      &msg->svc,0,0)) break;
		if (walk->state == ls_unused)
		    SEND_ERROR(msg->vcc,-EADDRNOTAVAIL);
		else send_kernel(msg->vcc,0,as_okay,0,NULL,NULL,NULL);
	    }
	    return;
	case as_establish: /* NULL or LISTENING */
	    if (sock && sock->state != ss_listening) break;
	    if (!sock) {
		sock = new_sock(msg->vcc);
		error = bind_check(sock,&msg->local);
		if (!error) {
		    sock->remote = sap_copy(&msg->svc);
		    if (!sock->remote) error = -EINVAL;
		}
		if (error) {
		    free_sock(sock);
		    SEND_ERROR(msg->vcc,error);
		    return;
		}
		sock->pvc.sap_txtp = sock->remote->sas_txtp;
		sock->pvc.sap_rxtp = sock->remote->sas_rxtp;
		sock->state = ss_connecting;
		error = send_setup(sock);
		if (error) {
		    SEND_ERROR(msg->vcc,error);
		    free_sock(sock);
		    return;
		}
		START_TIMER(sock,T303);
		new_state(sock,ss_connecting);
		return;
	    }
	    /* becomes INDICATED or ZOMBIE */
	    if (!sock->listen) {
		diag(COMPONENT,DIAG_WARN,
		  "socket 0x%lx got accept with empty listen queue",msg->vcc);
		return;
	    }
	    next = sock->listen;
	    sock->listen = next->listen;
	    next->listen = NULL;
	    if (next->state == ss_zombie) {
		SEND_ERROR(msg->vcc,-ECONNRESET); /* -ERESTARTSYS ? */
		free_sock(next);
		return;
	    }
	    if (next->state != ss_indicated) {
		sock = next; /* for error reporting */
		break;
	    }
	    next->id = msg->vcc;
	    error = send_connect(next);
	    if (!error) {
		START_TIMER(next,T313);
		new_state(next,ss_accepting);
		return;
	    }
	    SEND_ERROR(next->id,error);
	    send_release(next,0); /* @@@ */
	    START_TIMER(next,T308_1);
	    new_state(next,ss_wait_rel);
	    return;
	case as_listen: /* NULL */
	    {
		struct sockaddr_atmsvc *sap;

		if (sock) break;
		sap = sap_copy(&msg->svc);
		if (lookup_sap(sap,NULL)) {
		    free(sap);
		    SEND_ERROR(msg->vcc,-EADDRINUSE);
		    return;
		}
		sock = new_sock(msg->vcc);
		sock->local = sap_copy(&msg->svc);
		send_kernel(sock->id,0,as_okay,0,NULL,NULL,NULL);
		sock->state = ss_listening;
		return;
	    }
	case as_close: /* all but ACCEPTING and WAIT_REL */
	    if (sock && (sock->state == ss_accepting || sock->state ==
	      ss_wait_rel)) break;
	    switch (sock ? sock->state : ss_null) {
		case ss_listening:
		case ss_zombie:
		    send_close(sock);
		    /* fall through */
		case ss_wait_close:
		    free_sock(sock);
		    /* fall through */
		case ss_null:
		case ss_rel_req:
		    return;
		case ss_connecting:
		    STOP_TIMER(sock);
		    /* fall through */
		case ss_connected:
		    if (sock->state == ss_connected)
			diag(COMPONENT,DIAG_INFO,"Active close (CR 0x%06X)",
			  sock->call_ref);
		    /* fall through */
		case ss_indicated:
		case ss_hold:
		    send_release(sock,
#ifdef UNI31
		      ATM_CV_NORMAL_CLEAR
#else
		      ATM_CV_NORMAL_UNSPEC
#endif
		      );
		    START_TIMER(sock,T308_1);
		    new_state(sock,sock->state == ss_connected ? ss_rel_req :
		      ss_wait_rel);
		    return;
		case ss_rel_ind:
		    send_release_complete(sock->call_ref,0); /* @@@ */
		    free_sock(sock);
		    return;
		default:
		    break;
	    }
	    /* fall through */
	default:
    	    diag(COMPONENT,DIAG_WARN,"invalid message %d",(int) msg->type);
	    return;
    }
    diag(COMPONENT,DIAG_WARN,"message %s is incompatible with state %s (%d)",
      as_name[msg->type],state_name[sock ? sock->state : ss_null],
      (int) (sock ? sock->state : ss_null));
}


void itf_load(int itf)
{
    char buf[MAX_ATM_ADDR_LEN+1];
    LOCAL_ADDR *walk;

    (void) get_addr(itf);
    for (walk = local_addr; walk->state != ls_unused; walk++)
	if (walk->itf == itf) {
	    if (atm2text(buf,MAX_ATM_ADDR_LEN+1,(struct sockaddr *)
	      &walk->addr,pretty) < 0) strcpy(buf,"<invalid>");
	    switch (walk->state) {
		case ls_added:
		    diag(COMPONENT,DIAG_INFO,"Added local ATM address %s at "
		      "itf %d",buf,itf);
		    walk->state = ls_same;
		    break;
		case ls_removed:
		    diag(COMPONENT,DIAG_INFO,"Removed local ATM address %s at "
		      "itf %d",buf,itf);
		    walk->state = ls_unused;
		    /* @@@ delete SVCs using that address ? */
		    break;
		default:
		    break;
	    }
	}
}


void from_kernel(struct atmsvc_msg *msg,int size)
{
    SOCKET *curr;
    struct atm_blli **this;
    int bllis,i;

    if (msg->type == as_itf_notify) {
	itf_load(msg->pvc.sap_addr.itf);
	return;
    }
    for (curr = sockets; curr; curr = curr->next)
	if (msg->vcc == curr->id) break;
    if (!curr)
	for (curr = sockets; curr; curr = curr->next)
	    if (msg->listen_vcc == curr->id && curr->state == ss_listening)
		break;
    diag(COMPONENT,DIAG_DEBUG,"FROM KERNEL: %s for socket 0x%lx (0x%lx/0x%lx) "
      "in state %s",as_name[msg->type],(unsigned long) curr,msg->vcc,
      msg->listen_vcc,state_name[curr ? curr->state : ss_null]);
    msg->svc.sas_addr.blli = msg->local.sas_addr.blli = NULL;
    bllis = msg2bllis(size);
    this = &msg->svc.sas_addr.blli;
    for (i = 0; i < bllis; i++) {
	*this = &msg->blli[i];
	this = &msg->blli[i].next;
    }
    *this = NULL;
    dispatch(curr,msg);
}
