/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   utl_os.h
*   
* Abstract:
*   This is the interface for the basic functions that are provided by the OS.
*
* Authors:
*   DMW - Douglas M. Washabaugh
*
* Modification History:
*   Date       Name  Description 
*   08-Aug-94  DMW   Created.
*   02-Nov-94  DMW   Updated to use handles, added timer routines.
*   10-Nov-94  DMW   Updated with error log and print routines.
*   01-Jan-95  DMW   Allocated memory is now zeroed.
*   10-Jan-95  DMW   Remove obsolete phys mem routines, added mem statistics.
*   03-Feb-95  PJR   Added test string input to os_create.
*   16-Mar-95  DMW   Added two new routines os_buff_hdr_copy, os_buff_hdr_get.
*
* Description:
*   There will be one instantiation of this module for each adapter in the 
*   system.
*
* Call Sequence:
*   The following shows a rough sequence of how calls can be made to this
*   module.  Because this module is a collection of miscellaneous routines,
*   a different method is used to represent the call sequence.  The method
*   says that a given routine cannot be called unless all routines with 
*   a smaller indention are called first.  For example, the following shows
*   that os_timer_cancel cannot be called until os_timer_set is called, which
*   cannot be called before os_timer_allocate, which cannot be called before
*   os_create.
*
*     - os_create
*        - os_timer_allocate
*           - os_timer_set
*              - os_timer_cancel
*           - os_timer_deallocate
*        - os_reg_get
*        - os_reg_set
*        - os_mem_alloc
*           - os_mem_dealloc
*        - os_phys_mem_alloc
*           - os_phys_mem_dealloc
*        - os_print
*        - os_err_log
*        - os_destroy   
*/

/*++
* ---------------------
* - OS_TIMER_CALLBACK -
* ---------------------
*
* Overview:
*   Routine that is called when a timer expires.
*
* Arguments:
*   context - (IN) The value that was specified when the timer was allocated.
*   
* Returns:
*   none
*
* Preconditions:
*   None
*
* Postconditions:
*   none
--*/
#ifndef UTL_OS_H
#define UTL_OS_H
#include <stdlib.h>
#include "g_types.h"

typedef void (*OS_TIMER_CALLBACK) (HANDLE context);


/*++
* ==================
* = os_timer_alloc =
* ==================
*
* Overview:
*   Allocates and initializes a timer.  Some implementations of this
*   routine may allocate memory or other resources.  This routine can be
*   called from the coded path that processes receives or "transmit dones"
*   (which may include interrupt level for some operating systems).
*
* Arguments:
*  os_handle      - (IN)  The handle returned from os_create.
*
*  callback       - (IN)  Pointer to the routine to callback when the timer
*                         expires.
*
*  context        - (IN)  Value to pass to the callback routine.
*
*  p_timer_handle - (OUT) Pointer to a timer handle.  This value must be used
*                         when setting, canceling, or deallocating a timer.
*                         It will have the os_handle associated with it so that
*                         if the timer is specified in a call, the os_handle
*                         does not also need to be specified.
* Returns:
*   STATUS_K_SUCCESS   - Timer was successfully allocated
*   STATUS_K_RESOURCES - Timer was NOT allocated.
*
* Preconditions:
*   none
*
* Postconditions:
*   A timer may have been allocated.
--*/
STATUS os_timer_alloc (OS_TIMER_CALLBACK  callback,
                       HANDLE             context,
                       HANDLE            *p_timer_handle);
/*++
* ================
* = os_timer_set =
* ================
*
* Overview:
*   Sets a timer to expire after a specified time.  When the timer expires,
*   its associated callback routine is called.
*
* Arguments:
*  timer_handle - (IN)  The value returned from os_timer_alloc.
*
*  time         - (IN)  Number of milliseconds until the callback is called.
*
* Returns:
*   STATUS_K_SUCCESS
*   STATUS_K_FAILURE
*
* Preconditions:
*   - The timer was allocated.
*
* Postconditions:
*   A timer is set.                    
--*/
STATUS os_timer_set (HANDLE timer_handle,
                     UINT32  timeset);

/*++
* ===================
* = os_timer_cancel =
* ===================
*
* Overview:
*   Cancels the specified timer.   The timer callback routine is guaranteed
*   not be called after this routine completes.  Also, it is permissible to
*   cancel a timer that has not been set.
*
* Arguments:
*  timer_handle - (IN)  The value returned from os_timer_alloc.
*
* Returns:
*   STATUS_K_SUCCESS       - Timer was successfully canceled.
*
*   STATUS_K_NO_WORK_TO_DO - Timer was not set.
*
* Preconditions:
*   - The timer was allocated.
*
* Postconditions:
*   - The timer is canceled.
*   - The timer callback will not be called.                    
--*/
STATUS os_timer_cancel (HANDLE timer_handle);


/*++
* ====================
* = os_timer_dealloc =
* ====================
*
* Overview:
*   Deallocates the specified timer.  Even though some implementations may not 
*   support this feature, it must be called.  Note that this routine can be
*   called even if the timer has been started, but not yet expired.
*
*   After this routine completes, it is guaranteed that the timer callback will
*   not be called.
*
* Arguments:
*  timer_handle - (IN)  The value returned from os_timer_alloc.
*
* Returns:
*   none
*
* Preconditions:
*   - The timer was allocated.
*
* Postconditions:
*   - A timer was deallocated
*   - The timer callback will not be called.
--*/
void os_timer_dealloc (HANDLE timer_handle);

/*++
* ================
* = os_mem_alloc =
* ================
*
* Overview:
*   Allocates the specified number of bytes from virtual memory, and returns
*   the virtual address of the memory.  The allocated memory is zeroed.
*
*   This routine can be called from the coded path that processes receives or
*   "transmit dones" (which may include interrupt level for some operating
*   systems).
*
* Arguments:
*   os_handle - (IN) The handle that was returned by os_create.
*
*   size      - (IN) The number of bytes to allocate.
*  
* Returns:
*   - Null, if call failed, else pointer to allocated memory.
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/

void *os_mem_alloc (UINT32 size);

/*++
* ==================
* = os_mem_dealloc =
* ==================
*
* Overview:
*   Deallocates the specified virtual memory.  This routine must not be called
*   from interrupt level.
*
* Arguments:
*   os_handle - (IN) The handle that was returned by os_create.
* 
*   p_mem     - (IN) Pointer to the memory to return.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   None
--*/

void os_mem_dealloc (void   *p_mem);

/*++
* ====================
* = os_buff_hdr_copy =
* ====================
*
* Overview:
*   Copies the specified packet (whose internal format is not known to the
*   common code) to the specified buffer.  If the packet is larger than the 
*   destination buffer, then only the part of the packet which fits is 
*   copied.  The actual number of bytes copied is returned.  If the packet is
*   corrupt, than the value returned for the number of bytes copied is 0.
*
*   The caller can determine that the entire 
*
* Arguments:
*   p_pkt          - (IN) A pointer to the packet to copy.
* 
*   pkt_len        - (IN)  Number of bytes in the buffer pointed to by p_pkt.
*                          This argument may be ignored for implementations
*                          whose packet structure contains the packet length.
*                          Note that callers must always provide this value.
*
*   p_dst          - (OUT) Pointer to the destination buffer.
*
*   dst_len        - (IN)  Number of bytes in the buffer pointed to by p_dst.
*
*   p_bytes_copied - (OUT) Number of bytes copied to p_dst.
*
* Returns:
*   None
*
* Preconditions:
*   None
*
* Postconditions:
*   - p_bytes_copied has the number of bytes copied
*   - p_bytes_copied is <= dst_len
*   - p_bytes_copied == 0 if the pkt length is '0' or the packet is corrupt.
*   - p_dst contains the bytes copied from the packet.
--*/

void os_buff_hdr_copy (void   *p_pkt,
                       UINT32  pkt_len,
                       void   *p_dst,
                       UINT32  dst_len,
                       UINT32 *p_bytes_copied);

void* mem_alloc(const char *unit, unsigned long nbytes);
void mem_free(const char *unit, const void *mem);

#endif
