/*
 *      Home-agent functionality
 *
 *      Authors:
 *      Sami Kivisaari           <skivisaa@cc.hut.fi>
 *      Henrik Petander          <lpetande@cc.hut.fi>
 *
 *      $Id: s.ha.c 1.62 02/12/19 13:57:09+02:00 vnuorval@dsl-hkigw1d8c.dial.inet.fi $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *   
 *      Changes: Venkata Jagana,
 *               Krishna Kumar     : Statistics fix
 *     
 */

#include <linux/autoconf.h>
#include <linux/net.h>
#include <linux/skbuff.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/in6.h>
#include <linux/init.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv6.h>
#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif

#include <net/neighbour.h>
#include <net/ipv6.h>
#include <net/ip6_fib.h>
#include <net/ip6_route.h>
#include <net/ndisc.h>
#include <net/addrconf.h>
#include <net/neighbour.h>
#include <net/ipv6_tunnel.h>

#include "bcache.h"
#include "tunnel.h"
#include "stats.h"
#include "debug.h"
#include "util.h"
#include "ha.h"

static int mipv6_ha_tunnel_sitelocal = 1;

#ifdef CONFIG_SYSCTL

static struct ctl_table_header *mipv6_ha_sysctl_header;

static struct mipv6_ha_sysctl_table
{
	struct ctl_table_header *sysctl_header;
	ctl_table mipv6_vars[3];
	ctl_table mipv6_mobility_table[2];
	ctl_table mipv6_proto_table[2];
	ctl_table mipv6_root_table[2];
} mipv6_ha_sysctl = {
	NULL,

        {{NET_IPV6_MOBILITY_TUNNEL_SITELOCAL, "tunnel_sitelocal",
	  &mipv6_ha_tunnel_sitelocal, sizeof(int), 0644, NULL, 
	  &proc_dointvec},
	 {0}},

	{{NET_IPV6_MOBILITY, "mobility", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_vars}, {0}},
	{{NET_IPV6, "ipv6", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_mobility_table}, {0}},
	{{CTL_NET, "net", NULL, 0, 0555, 
	  mipv6_ha_sysctl.mipv6_proto_table}, {0}}
};

#endif /* CONFIG_SYSCTL */

/*  this should be in some header file but it isn't  */
extern void ndisc_send_na(
	struct net_device *dev, struct neighbour *neigh,
	struct in6_addr *daddr, struct in6_addr *solicited_addr,
	int router, int solicited, int override, int inc_opt);

/*  this is defined in kernel IPv6 module (sockglue.c)  */
extern struct packet_type ipv6_packet_type;

/**
 * mipv6_ha_set_anycast_addr - Assign Home Agent Anycast Address to a interface
 * @ifindex: index of interface to which anycast address is assigned
 * @pfix: prefix for anycast address
 * @plen: length of prefix in bits
 *
 * Node must assign Mobile IPv6 Home Agents anycast address to all
 * interfaces it serves as a Home Agent.
 */
int mipv6_ha_set_anycast_addr(int ifindex, struct in6_addr *pfix, int plen)
{
	struct in6_ifreq ifreq;
	
	ipv6_addr_copy(&ifreq.ifr6_addr, pfix);

	if (homeagent_anycast(&ifreq.ifr6_addr, plen) < 0)
		return -EINVAL;

	ifreq.ifr6_prefixlen = plen;
	ifreq.ifr6_ifindex = ifindex;

	return addrconf_add_ifaddr(&ifreq);
}

/**
 * mipv6_proxy_nd_rem - stop acting as a proxy for @home_address
 * @home_addr: address to remove
 * @ha_addr: home agent's address on home link
 * @prefix_length: prefix length in bits
 * @single: single bit
 *
 * When Home Agent acts as a proxy for an address it must leave the
 * solicited node multicast group for that address and stop responding 
 * to neighbour solicitations.  
 **/
int mipv6_proxy_nd_rem(
	struct in6_addr *home_addr,
	int ifindex,
	int prefix_length,
	int single)
{
        /* When MN returns home HA leaves the solicited mcast groups
         * for MNs home addresses 
	 */
	int err;
	struct net_device *dev;
	
	DEBUG_FUNC();
	
        if ((dev = dev_get_by_index(ifindex)) == NULL) {
		DEBUG(DBG_ERROR, "couldn't get dev");
		return -ENODEV;
	}
#if 0	
	/* Remove link-local entry */
	if (!single) {
		struct in6_addr ll_addr;
		mipv6_generate_ll_addr(&ll_addr, home_addr);
		if ((err = pneigh_delete(&nd_tbl, &ll_addr, dev)) < 0) {
			DEBUG(DBG_INFO,
			      "peigh_delete failed for "
			      "%x:%x:%x:%x:%x:%x:%x:%x",
			      NIPV6ADDR(&ll_addr));	
		}
	}
#endif
	/* Remove global (or site-local) entry */
	if ((err = pneigh_delete(&nd_tbl, home_addr, dev)) < 0) {
		DEBUG(DBG_INFO,
		      "peigh_delete failed for " 
		      "%x:%x:%x:%x:%x:%x:%x:%x",
		      NIPV6ADDR(home_addr));
	}
	dev_put(dev);
	return err;
}

/**
 * mipv6_proxy_nd - join multicast group for this address
 * @home_addr: address to defend
 * @ha_addr: home agent's address on home link
 * @prefix_length: prefix length in bits
 * @single: single bit
 *
 * While Mobile Node is away from home, Home Agent acts as a proxy for
 * @home_address. HA responds to neighbour solicitations for  @home_address 
 * thus getting all packets destined to home address of MN. 
 **/
int mipv6_proxy_nd(struct in6_addr *home_addr, 
		   int ifindex,
		   int prefix_length,
		   int single)
{  
	/* The HA sends a proxy ndisc_na message to all hosts on MN's
	 * home subnet by sending a neighbor advertisement with the
	 * home address or all addresses of the mobile node if the
	 * prefix is not 0. The addresses are formed by combining the
	 * suffix or the host part of the address with each subnet
	 * prefix that exists in the home subnet 
	 */
	
        /* Since no previous entry for MN exists a proxy_nd advertisement
	 * is sent to all nodes link local multicast address
	 */	
	int err = -1;

	struct net_device *dev;
//	struct in6_addr ll_addr;
//	struct pneigh_entry *ll_pneigh;
	struct in6_addr mcdest;
//	int send_ll_na = 0;
	int inc_opt = 1;
	int solicited = 0;
	int override = 1;
	
	DEBUG_FUNC();
	
	if ((dev = dev_get_by_index(ifindex)) == NULL) {
		DEBUG(DBG_ERROR, "couldn't get dev");
		return -ENODEV;
	}
	
	if (!pneigh_lookup(&nd_tbl, home_addr, dev, 1)) {
		DEBUG(DBG_INFO,
		      "peigh_lookup failed for "
		      "%x:%x:%x:%x:%x:%x:%x:%x",
		      NIPV6ADDR(home_addr));
		goto free_dev;
	}
#if 0
	if (!single) {
		mipv6_generate_ll_addr(&ll_addr, home_addr);
		
		if ((ll_pneigh = pneigh_lookup(&nd_tbl, &ll_addr, dev, 
					       0)) != NULL) {
			ll_pneigh = pneigh_clone(ll_pneigh);
		} else if ((ll_pneigh = pneigh_lookup(&nd_tbl, &ll_addr, 
						      dev, 1)) == NULL) {
			DEBUG(DBG_INFO,
			      "peigh_lookup failed for "
			      "%x:%x:%x:%x:%x:%x:%x:%x",
			      NIPV6ADDR(&ll_addr));
			pneigh_delete(&nd_tbl, home_addr, dev);
			goto free_dev;
		} else {
			send_ll_na = 1;
		}
	} else {
		ll_pneigh = NULL;
	}
#endif	
	/* Proxy neighbor advertisement of MN's home address 
	 * to all nodes solicited multicast address 
	 */
	
	ipv6_addr_all_nodes(&mcdest); 
	ndisc_send_na(dev, NULL, &mcdest, home_addr, 0, 
		      solicited, override, inc_opt);
#if 0
	if (send_ll_na) {
		ndisc_send_na(dev, NULL, &mcdest, &ll_addr, 0, 
			      solicited, override, inc_opt);
	}
#endif
	err = 0;
free_dev:
	dev_put(dev);
	return err;
	
}

struct inet6_ifaddr *is_on_link_ipv6_address(struct in6_addr *mn_haddr,
					     struct in6_addr *ha_addr)
{
	struct inet6_ifaddr *ifp;
	struct inet6_dev *in6_dev;
	struct inet6_ifaddr *oifp = NULL;

	if ((ifp = ipv6_get_ifaddr(ha_addr, 0)) == NULL)
		return NULL;

	if ((in6_dev = ifp->idev) != NULL) {
		in6_dev_hold(in6_dev);
		oifp = in6_dev->addr_list;
		while (oifp != NULL) {
			spin_lock(&oifp->lock);
			if (mipv6_prefix_compare(&oifp->addr, mn_haddr,
						 oifp->prefix_len) &&
			    !(oifp->flags & IFA_F_TENTATIVE)) {
				spin_unlock(&oifp->lock);
				DEBUG(DBG_INFO, "Home Addr Opt: on-link");
				in6_ifa_hold(oifp);
				break;
			}
			spin_unlock(&oifp->lock);
			oifp = oifp->if_next;
		}
		in6_dev_put(in6_dev);
	}
	in6_ifa_put(ifp);
/*      DEBUG(DBG_WARNING, "Home Addr Opt NOT on-link"); */
	return oifp;

}

extern void mipv6_bu_finish(
	struct inet6_ifaddr *ifp, __u8 ba_status,
	__u32 ba_lifetime, __u32 ba_refresh,
	__u32 maxdelay, int plength, __u16 sequence,
	struct in6_addr *saddr, struct in6_addr *daddr,
	struct in6_addr *haddr, struct in6_addr *coa,
	int ifindex, int single, __u8 *k_bu);

void mipv6_bu_add_home(int ifindex, struct in6_addr *saddr,
		struct in6_addr *daddr, struct in6_addr *haddr,
		struct in6_addr *coa, __u32 lifetime, int plength,
		__u16 sequence, int single, int ack, __u32 ba_lifetime,
		__u32 ba_refresh, int dad, int lladdr,__u8 *k_bu)
{
	struct inet6_ifaddr *ifp = NULL;
	__u8 ba_status = SUCCESS;

	ifp = is_on_link_ipv6_address(haddr, daddr);

	if (ifp == NULL) {
		ba_status = NOT_HOME_SUBNET;
#if 0
	} else if ( test(haddr) ) {
		/* we currently don't do this */
		ba_status = ADMINISTRATIVELY_PROHIBITED;
#endif
	} else if (lladdr) {
		/* we currently don't do this */
		ba_status = ADMINISTRATIVELY_PROHIBITED;
	} else {
		ifindex = ifp->idev->dev->ifindex; 
		if (dad) {
			int ret;

			if ((ret =
			     mipv6_dad_start(ifp, ifindex, 
					     saddr, daddr, haddr, coa,
					     ba_lifetime, plength,
					     sequence, single)) < 0) {
				/* An error occurred */
				ba_status = -ret;
			} else if (ret) {
				/* DAD is needed to be performed. */
				return;
			}
			/* DAD is not needed */
		}
	}

	mipv6_bu_finish(ifp, ba_status, ba_lifetime, ba_refresh,
			0, plength, sequence, saddr, daddr, haddr,
			coa, ifindex, single, k_bu);
}

extern int mipv6_ra_rcv_ptr(struct sk_buff *skb, struct icmp6hdr *msg);

/**
 * mipv6_intercept - Netfilter hook to intercept packets
 * @hooknum: which hook we came from
 * @p_skb: pointer to *skb
 * @in: interface we came in
 * @out: outgoing interface
 * @okfn: next handler
 **/

static unsigned int mipv6_intercept(
        unsigned int hooknum,
	struct sk_buff **p_skb,
	const struct net_device *in,
	const struct net_device *out,
	int (*okfn)(struct sk_buff *))
{
	struct sk_buff *skb = (p_skb) ? *p_skb : NULL;
	struct ipv6hdr *ipv6h;
	struct in6_addr *daddr, *saddr;
	__u8 nexthdr;
	int nhoff;	

	if(skb == NULL) return NF_ACCEPT;
	
	ipv6h = skb->nh.ipv6h;
	daddr = &ipv6h->daddr;
	saddr = &ipv6h->saddr;
		
	nexthdr = ipv6h->nexthdr;
	nhoff = sizeof(*ipv6h);
	
	
	if (ipv6_ext_hdr(nexthdr)) 
		nhoff = ipv6_skip_exthdr(skb, nhoff, &nexthdr, 
					 skb->len - sizeof(*ipv6h));

	/*
	 * Possible ICMP packets are checked to ensure that all neighbor 
	 * solicitations to MNs home address are handled by the HA.
	 */

	if (nexthdr == IPPROTO_ICMPV6) {			
		struct icmp6hdr *icmp6h;
		int dest_type;

		if (nhoff < 0 || 
		    !pskb_may_pull(skb, nhoff + sizeof(struct icmp6hdr)))
			return NF_DROP;

		dest_type = ipv6_addr_type(daddr);
		icmp6h = (struct icmp6hdr *)&skb->nh.raw[nhoff];

		/* HA has to capture all unicast neighbour solicitations in 
		   order to check if it is acting as a proxy for the target 
		   address. */
		
		if ((dest_type & IPV6_ADDR_UNICAST) && 
		    icmp6h->icmp6_type == NDISC_NEIGHBOUR_SOLICITATION) {
			ip6_input(skb);
			return NF_STOLEN;
		} 
	}
	return NF_ACCEPT;
}

/*
 * Netfilter hook for packet interception
 */

static struct nf_hook_ops intercept_hook_ops = {
        {NULL, NULL},     // List head, no predecessor, no successor
        mipv6_intercept,
        PF_INET6,
        NF_IP6_PRE_ROUTING,
        NF_IP6_PRI_LAST
};

/**
 * mipv6_ha_tnl_xmit_drop_local_hook - drop local packets 
 * @skb: outgoing skb
 * @flags: flags set by tunnel device driver
 *
 *
 * Return:
 * %IPV6_TNL_ACCEPT if packet can be sent through tunnel,
 * %IPV6_TNL_DROP if packet is invalid,
 **/

static int
mipv6_ha_tnl_xmit_drop_local_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	int dest_type;

	DEBUG_FUNC();

       	/* If this isn't a tunnel used for Mobile IPv6 return */
	if (!is_mipv6_tnl(t))
		return IPV6_TNL_ACCEPT;

	dest_type = ipv6_addr_type(&skb->nh.ipv6h->daddr);

	if ((dest_type & IPV6_ADDR_LINKLOCAL) ||
	    ((dest_type & IPV6_ADDR_SITELOCAL) && 
	     !mipv6_ha_tunnel_sitelocal)) 
		return IPV6_TNL_DROP;
	
	return IPV6_TNL_ACCEPT;
}
	
static struct ipv6_tnl_hook_ops mipv6_ha_tnl_xmit_drop_local_ops = {
	{NULL, NULL}, 
	IPV6_TNL_PRE_ENCAP,
	IPV6_TNL_PRI_FIRST,
	mipv6_ha_tnl_xmit_drop_local_hook
};

static int
mipv6_ha_tnl_xmit_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_encapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_ha_tnl_xmit_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_ENCAP,
	IPV6_TNL_PRI_LAST,
	mipv6_ha_tnl_xmit_stats_hook
};

static int
mipv6_ha_tnl_rcv_stats_hook(struct ipv6_tnl *t, struct sk_buff *skb)
{
	DEBUG_FUNC();
	if (is_mipv6_tnl(t))
		MIPV6_INC_STATS(n_decapsulations);
	return IPV6_TNL_ACCEPT;
}

static struct ipv6_tnl_hook_ops mipv6_ha_tnl_rcv_stats_ops = {
	{NULL, NULL},
	IPV6_TNL_PRE_DECAP,
	IPV6_TNL_PRI_LAST,
	mipv6_ha_tnl_rcv_stats_hook
};

int __init mipv6_ha_init(void)
{
	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	if (!(mipv6_ha_sysctl_header = 
	      register_sysctl_table(mipv6_ha_sysctl.mipv6_root_table, 0)))
		printk(KERN_ERR "Failed to register sysctl handlers!");
#endif
	/*  register packet interception hooks  */
	nf_register_hook(&intercept_hook_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_xmit_drop_local_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_xmit_stats_ops);
	ipv6_ipv6_tnl_register_hook(&mipv6_ha_tnl_rcv_stats_ops);
	return 0;
}

void __exit mipv6_ha_exit(void)
{
	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	unregister_sysctl_table(mipv6_ha_sysctl_header);
#endif

	/*  remove packet interception hooks  */
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_rcv_stats_ops);
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_xmit_stats_ops);
	ipv6_ipv6_tnl_unregister_hook(&mipv6_ha_tnl_xmit_drop_local_ops);
	nf_unregister_hook(&intercept_hook_ops);
}

