/*
 *	Mobile IPv6 Mobility Header Common Functions
 *
 *	Authors:
 *	Antti Tuominen <ajtuomin@tml.hut.fi>
 *
 *      $Id: s.mh_recv.c 1.159 02/10/16 15:01:29+03:00 antti@traci.mipl.mediapoli.com $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/autoconf.h>
#include <linux/types.h>
#include <linux/in6.h>
#include <linux/skbuff.h>
#include <linux/ipsec.h>
#include <linux/init.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>
#include <net/mipv6.h>
#include <net/checksum.h>
#include <net/protocol.h>

#include "util.h"
#include "stats.h"
#include "debug.h"
#include "mobhdr.h"
#include "ha.h"
#include "bcache.h"
#include "bul.h"
#include "rr_crypto.h"
#include "exthdrs.h"
#include "config.h"
#include "mn.h"

#define MIPV6_MH_MAX MIPV6_MH_BE
struct mh_proto {
	int	(*func) (struct in6_addr *, struct in6_addr *, 
			 struct in6_addr *, struct in6_addr *, 
			 struct mipv6_mh *);
};

static struct mh_proto mh_rcv[MIPV6_MH_MAX];

int mipv6_mh_register(int type, int (*func)(
	struct in6_addr *, struct in6_addr *, 
	struct in6_addr *, struct in6_addr *, struct mipv6_mh *))
{
	if (mh_rcv[type].func != NULL)
		return -1;

	mh_rcv[type].func = func;

	return 0;
}

struct socket *mipv6_mh_socket = NULL;

/* TODO: Fix fragmentation */
static int dstopts_getfrag(
	const void *data, struct in6_addr *addr,
	char *buff, unsigned int offset, unsigned int len)
{
	memcpy(buff, data + offset, len);
	return 0;
}

struct mipv6_mh_opt *alloc_mh_opts(int totlen)
{
	struct mipv6_mh_opt *ops;

	ops = kmalloc(sizeof(*ops) + totlen, GFP_ATOMIC);
	if (ops == NULL)
		return NULL;

	memset(ops, 0, sizeof(*ops));
	ops->next_free = ops->data;
	ops->freelen = totlen;

	return ops;
}

int append_mh_opt(struct mipv6_mh_opt *ops, u8 type, u8 len, void *data)
{
	struct mipv6_mo *mo;

	if (ops->next_free == NULL) {
		DEBUG(DBG_ERROR, "No free room for option");
		return -ENOMEM;
	}
	if (ops->freelen < len + 2) {
		DEBUG(DBG_ERROR, "No free room for option");
		return -ENOMEM;
	}
	else {
		ops->freelen -= (len + 2);
		ops->totlen += (len + 2);
	}

	mo = (struct mipv6_mo *)ops->next_free;
	mo->type = type;
	mo->length = len;

	switch (type) {
	case MIPV6_OPT_ALTERNATE_COA:
		ops->alt_coa = (struct mipv6_mo_alt_coa *)mo;
		ipv6_addr_copy(&ops->alt_coa->addr, (struct in6_addr *)data);
		break;
	case MIPV6_OPT_NONCE_INDICES:
		DEBUG(DBG_INFO, "Added nonce indices pointer");
		ops->nonce_indices = (struct mipv6_mo_nonce_indices *)mo;
		ops->nonce_indices->home_nonce_i = *(__u16 *)data;
		ops->nonce_indices->careof_nonce_i = *((__u16 *)data + 1);
		break;
	case MIPV6_OPT_AUTH_DATA:
		DEBUG(DBG_INFO, "Added opt auth_data pointer");
		ops->auth_data = (struct mipv6_mo_bauth_data *)mo;
		break;
	case MIPV6_OPT_BIND_REFRESH_ADVICE:
		ops->br_advice = (struct mipv6_mo_br_advice *)mo;
		ops->br_advice->refresh_interval = htons(*(u16 *)data);
		break;
	default:
		DEBUG(DBG_ERROR, "Unknow option type");
		break;
	}
	memcpy(ops->next_free, mo, len);

	if (ops->freelen == 0)
		ops->next_free = NULL;
	else
		ops->next_free += (len + 2);

	return 0;
}

int mipv6_add_pad(u8 *data, int n)
{
	struct mipv6_mo_padn *padn;

	if (n <= 0) return 0;
	if (n == 1) {
		*data = MIPV6_OPT_PAD1;
		return 1;
	}
	padn = (struct mipv6_mo_padn *)data;
	padn->type = MIPV6_OPT_PADN;
	padn->length = n - 2;
	memset(padn->data, 0, n - 2);
	return n;
}

/*
 *   Calculates required padding before given mobility option.
 */
static int option_pad(__u8 type, int offset)
{
	switch (type) {
	case MIPV6_OPT_PAD1:			/* no pad */
	case MIPV6_OPT_PADN:
		return 0;

	case MIPV6_OPT_ALTERNATE_COA:		/* 8n + 6 */
		return (6 - (offset)) & 7;

	case MIPV6_OPT_AUTH_DATA:		/* 8n + 2 */
		return (2 - (offset)) & 7;

	case MIPV6_OPT_BIND_REFRESH_ADVICE:	/* 2n     */
	case MIPV6_OPT_NONCE_INDICES:
		return (offset) & 1;
	default:
		DEBUG(DBG_ERROR, "invalid option type 0x%x", type);
		return 0;
	}
}

int prepare_mh_opts(struct mipv6_mh *mh, int msg_len, struct mipv6_mh_opt *ops)
{
	u8 *nextopt = (u8 *)(mh + 1) + msg_len;
	u8 *first = nextopt;
	int offset = msg_len, pad = 0;

	if (ops == NULL) {
		first = nextopt = NULL;
		return -1;
	}

	if (ops->alt_coa) {
		pad = option_pad(MIPV6_OPT_ALTERNATE_COA, offset);
		nextopt += mipv6_add_pad(nextopt, pad);
		memcpy(nextopt, ops->alt_coa, sizeof(struct mipv6_mo_alt_coa));
		nextopt += sizeof(struct mipv6_mo_alt_coa);
		offset = pad + sizeof(struct mipv6_mo_alt_coa);
	}

	if (ops->br_advice) {
		pad = option_pad(MIPV6_OPT_BIND_REFRESH_ADVICE, offset);
		nextopt += mipv6_add_pad(nextopt, pad);
		memcpy(nextopt, ops->br_advice, sizeof(struct mipv6_mo_br_advice));
		nextopt += sizeof(struct mipv6_mo_br_advice);
		offset = pad + sizeof(struct mipv6_mo_br_advice);
	}

	if (ops->nonce_indices) {
		pad = option_pad(MIPV6_OPT_NONCE_INDICES, offset);
		nextopt += mipv6_add_pad(nextopt, pad);
		memcpy(nextopt, ops->nonce_indices, sizeof(struct mipv6_mo_nonce_indices));
		nextopt += sizeof(struct mipv6_mo_nonce_indices);
		offset = pad + sizeof(struct mipv6_mo_nonce_indices);
	}

	if (ops->auth_data) {
		pad = option_pad(MIPV6_OPT_AUTH_DATA, offset);
		nextopt += mipv6_add_pad(nextopt, 
					 8 - ((nextopt + 2 + ops->auth_data->length  - (u8 *)mh) & 7)); 
		memcpy(nextopt, ops->auth_data, ops->auth_data->length + 2);
		nextopt += ops->auth_data->length + 2;
	}
	first = nextopt = NULL;

	return 0;
}

int calculate_mh_opts(struct mipv6_mh_opt *ops, int mh_len)
{
	int offset = mh_len;
	
	if (ops == NULL)
		return 0;

	if (ops->alt_coa)
		offset += sizeof(struct mipv6_mo_alt_coa)
			+ option_pad(MIPV6_OPT_ALTERNATE_COA, offset);

	if (ops->br_advice)
		offset += sizeof(struct mipv6_mo_br_advice)
			+ option_pad(MIPV6_OPT_BIND_REFRESH_ADVICE, offset);

	if (ops->nonce_indices)
		offset += sizeof(struct mipv6_mo_nonce_indices)
			+ option_pad(MIPV6_OPT_NONCE_INDICES, offset);

	if (ops->auth_data) /* 8 - totlen & 7 */
		offset += 8 - ((offset + mh_len + ops->auth_data->length + 2) & 7) + ops->auth_data->length + 2;

	return offset - mh_len;
}

static inline int mh_may_use_hao(u8 type, struct in6_addr *daddr,
				 struct in6_addr *saddr)
{
	if (type == MIPV6_MH_BU || type == MIPV6_MH_HOTI ||
	    type == MIPV6_MH_COTI)
		return 0;
	return mipv6_mn_use_hao(daddr, saddr);
}

static inline int mh_may_use_rth(u8 type)
{
	if (type == MIPV6_MH_BA || type == MIPV6_MH_HOT)
		return 0;
	return 1;
}

/**
 * send_mh - builds and sends a MH msg
 *
 * @daddr : destination address for packet
 * @saddr : source address for packet
 * @msg : MH type specific data
 * @msg_type : type of MH
 * @hoa_opt : home address for home address option
 *
 * Builds MH, appends the type specific msg data to the header and
 * sends the packet with a home address option, if a home address was
 * given. Returns 0, if everything succeeded and a negative error code
 * otherwise.
 **/
int send_mh(struct in6_addr *daddr, 
	    struct in6_addr *saddr, 
	    u8 *msg,
	    u8 msg_len,
	    u8 msg_type,
	    struct in6_addr *hao_addr,
	    struct in6_addr *rth_addr,
	    struct mipv6_mh_opt *ops,
	    struct mipv6_auth_parm *parm)
{
	struct flowi fl;
	struct mipv6_mh *mh; 
	struct sock *sk = mipv6_mh_socket->sk;
	struct ipv6_txoptions *txopt = NULL;
	int tot_len = sizeof(struct mipv6_mh) + msg_len;
	int padded_len = 0, txopt_len = 0;
	struct mipv6_bcache_entry bc_entry;

	DEBUG_FUNC();
	/* Add length of options */
	tot_len += calculate_mh_opts(ops, tot_len);
	/* Needs to be a multiple of 8 octets */
	padded_len = (tot_len & 7) ? tot_len + (8 - (tot_len & 7)) : tot_len;

	mh = sock_kmalloc(sk, padded_len, GFP_ATOMIC);
	if (!mh) {
		DEBUG(DBG_ERROR, "memory allocation failed");
		return -ENOMEM;
	}

	if (hao_addr || mh_may_use_hao(msg_type, daddr, saddr))
		txopt_len += sizeof(struct mipv6_dstopt_homeaddr) + 6;
	if (rth_addr)
		txopt_len += sizeof(struct rt2_hdr);
	if (mh_may_use_rth(msg_type) && rth_addr == NULL) {
		if (mipv6_bcache_get(daddr, saddr, &bc_entry) == 0) {
			rth_addr = &bc_entry.coa;
			txopt_len += sizeof(struct rt2_hdr);
		}
	}

	memset(&fl, 0, sizeof(fl)); 
	fl.proto = IPPROTO_MOBILITY;
	fl.fl6_dst = daddr;
	fl.fl6_src = saddr;
	fl.fl6_flowlabel = 0;
	fl.oif = sk->bound_dev_if;

	if (txopt_len > 0) {
		__u8 *opt_ptr;
		txopt_len += sizeof(*txopt);
		txopt = sock_kmalloc(sk, txopt_len, GFP_ATOMIC);
		if (txopt == NULL) {
			DEBUG(DBG_ERROR, "No socket space left");
			sock_kfree_s(sk, mh, padded_len);
			return -ENOMEM;
		}
		memset(txopt, 0, txopt_len);
		txopt->tot_len = txopt_len;
		opt_ptr = (__u8 *) (txopt + 1);
		if (hao_addr || mh_may_use_hao(msg_type, daddr, saddr)) {
			int holen = sizeof(struct mipv6_dstopt_homeaddr) + 6;
			txopt->dst1opt = (struct ipv6_opt_hdr *) opt_ptr;
			txopt->opt_flen += holen;
			opt_ptr += holen;
			mipv6_append_dst1opts(txopt->dst1opt, saddr, 
					      NULL, holen);
			txopt->mipv6_flags = MIPV6_SND_HAO;
		}
		if (rth_addr) {
			int rtlen = sizeof(struct rt2_hdr);
			txopt->srcrt2 = (struct ipv6_rt_hdr *) opt_ptr;
			txopt->opt_nflen += rtlen;
			opt_ptr += rtlen;
			mipv6_append_rt2hdr(txopt->srcrt2, rth_addr);
		}
	}

	/* Fill in the fields of MH */
	mh->payload = NEXTHDR_NONE;
	mh->length = (padded_len >> 3) - 1;	/* Units of 8 octets - 1 */
	mh->type = msg_type;
	mh->reserved = 0;
	mh->checksum = 0;

	memcpy(mh->data, msg, msg_len);
	prepare_mh_opts(mh, msg_len, ops);
	mipv6_add_pad((u8 *)mh + tot_len, padded_len - tot_len);
	
	if (parm && parm->k_bu && ops && ops->auth_data) {
		/* Calculate the position of the authorization data before adding checksum*/
		mipv6_auth_build(parm->cn_addr, parm->coa, (__u8 *)mh, 
				 (__u8 *)mh + padded_len - MIPV6_RR_MAC_LENGTH, parm->k_bu);
	}
	/* Calculate the MH checksum */
	mh->checksum = csum_ipv6_magic(fl.fl6_src, fl.fl6_dst, 
				       padded_len, IPPROTO_MOBILITY,
				       csum_partial((char *)mh, padded_len, 0));
	ip6_build_xmit(sk, dstopts_getfrag, mh, &fl, padded_len, txopt, 255,
		       MSG_DONTWAIT);
	/* dst cache must be cleared so RR messages can be routed through 
	   different interfaces */
	sk_dst_reset(sk);

	if (txopt_len)
		sock_kfree_s(sk, txopt, txopt_len);
	sock_kfree_s(sk, mh, padded_len);
	return 0;
}

/**
 * mipv6_send_brr - send a Binding Refresh Request 
 * @saddr: source address for BRR
 * @daddr: destination address for BRR
 * @maxdelay: maximum milliseconds before option is sent
 * @ops: mobility options
 *
 * Sends a binding request.  Actual sending may be delayed up to
 * @maxdelay milliseconds.  If 0, request is sent immediately.  On a
 * mobile node, use the mobile node's home address for @saddr.
 * Returns 0 on success, negative on failure.
 **/
int mipv6_send_brr(struct in6_addr *saddr,
	struct in6_addr *daddr,	long maxdelay, 
	struct mipv6_mh_opt *ops)
{
	struct mipv6_mh_brr br;

	memset(&br, 0, sizeof(br));
	/* We don't need to  explicitly add a RH to brr, since it will be 
	 * included automatically, if a binding exists 
	 */
	MIPV6_INC_STATS(n_brr_sent);
	return send_mh(daddr, saddr, (u8 *)&br, sizeof(br),  MIPV6_MH_BRR,
		       NULL, NULL, ops, NULL);
}

/**
 * mipv6_send_ba - send a Binding Acknowledgement 
 * @saddr: source address for BA
 * @daddr: destination address for BA 
 * @coaddr: care-of address of MN
 * @maxdelay: maximun milliseconds before option is sent
 * @status: status field value
 * @sequence: sequence number from BU
 * @lifetime: granted lifetime for binding in seconds
 * @refresh: refresh timeout
 * @ops: mobility options
 *
 * Sends a binding acknowledgement.  
 * Returns 0 on success, negative on failure.
 **/
int mipv6_send_ba(struct in6_addr *saddr, struct in6_addr *daddr, 
		  struct in6_addr *coaddr, long maxdelay, u8 status, 
		  u16 sequence, u32 lifetime, u32 refresh, u8 *k_bu)
{
	struct mipv6_mh_ba ba;
	struct mipv6_auth_parm parm;
	struct mipv6_mh_opt *ops = NULL; 
	int ops_len = 0;

	memset(&ba, 0, sizeof(ba));
	
	ba.status = status;
	ba.sequence = htons(sequence);
	ba.lifetime = htons(lifetime) >> 2;
	
	DEBUG(DBG_INFO, "sending a status %d BA %s authenticator to MN \n"
	      "%x:%x:%x:%x:%x:%x:%x:%x  at care of address \n" 
	      "%x:%x:%x:%x:%x:%x:%x:%x : with lifetime %d",
	      status, k_bu ? "with" : "without", 
	      NIPV6ADDR(daddr), NIPV6ADDR(coaddr), lifetime);

	if (status < 128) {
		MIPV6_INC_STATS(n_ba_sent);
	} else {
		MIPV6_INC_STATS(n_ban_sent);
	}

	memset(&parm, 0, sizeof(parm));
	parm.coa = coaddr;
	parm.cn_addr = saddr;

	if (k_bu) {
		ops_len += sizeof(struct mipv6_mo_bauth_data) + 
			MIPV6_RR_MAC_LENGTH;
		parm.k_bu = k_bu;
	}
	/* Other options (if any) should materialize right about here.
	 * Since MIPV6_OPT_BR_ADVICE is the only other option possible
	 * here (currently) and it is only defined for Home Agent, we
	 * could easily make it into a configurable parameter. */
	if (ops_len) {
		ops = alloc_mh_opts(ops_len);
		if (ops == NULL) {
			DEBUG(DBG_WARNING, "Out of memory");
			return -ENOMEM;
		}
		if (k_bu) {
			if (append_mh_opt(ops, MIPV6_OPT_AUTH_DATA,
					  MIPV6_RR_MAC_LENGTH, NULL) < 0) {
				DEBUG(DBG_WARNING, "Adding BAD failed");
				if (ops)
					kfree(ops);
				return -ENOMEM;
			}
		}
	}

	send_mh(daddr, saddr, (u8 *)&ba, sizeof(ba), MIPV6_MH_BA, 
		NULL, coaddr, ops, &parm);

	if (ops)
		kfree(ops);

	return 0;
}

/**
 * mipv6_send_be - send a Binding Error message
 * @saddr: source address for BE
 * @daddr: destination address for BE
 * @home: Home Address in offending packet (if any)
 *
 * Sends a binding error.  Actual sending may be delayed up to
 * @maxdelay milliseconds.  If 0, request is sent immediately.  On a
 * mobile node, use the mobile node's home address for @saddr.
 * Returns 0 on success, negative on failure.
 **/
int mipv6_send_be(struct in6_addr *saddr, struct in6_addr *daddr, 
		  struct in6_addr *home, __u8 status)
{
	struct sock *sk = mipv6_mh_socket->sk;
	struct mipv6_mh_be *be;
	struct mipv6_mh_opt *ops = NULL;
	int be_len = sizeof(struct mipv6_mh_be);
	int ret;

	/* Future: add options, if there are such */
	be = sock_kmalloc(sk, be_len, GFP_ATOMIC);
	if (!be) {
		DEBUG(DBG_ERROR, "Sending of BE failed: Out of memory");
		return -ENOMEM;
	}
	memset(be, 0, sizeof(*be));
	be->status = status;
	if (home)
		ipv6_addr_copy(&be->home_addr, home);

	ret = send_mh(daddr, saddr, (u8 *)be, be_len,  MIPV6_MH_BE,
		      NULL, NULL, ops, NULL);
	if (!ret) 
		MIPV6_INC_STATS(n_be_sent);
	sock_kfree_s(sk, be, be_len);

	return ret;
}

/**
 * mipv6_send_addr_test_init - send a HoTI or CoTI message
 * @saddr: source address for H/CoTI
 * @daddr: destination address for H/CoTI
 * @msg_type: Identifies whether HoTI or CoTI
 * @test_cookie: the HoT or CoT cookie
 *
 * The message will be retransmitted till we get a HoT or CoT message, since 
 * our caller (mipv6_RR_start) has entered this message in the BUL with
 * exponential backoff retramission set.
 */
int mipv6_send_addr_test_init(struct in6_addr *saddr,
	       struct in6_addr *daddr,
	       u8 msg_type,
	       u8 *test_cookie)
{
	struct mipv6_mh_addr_ti ti;
	struct mipv6_mh_opt *ops = NULL;

	/* Set reserved and copy the cookie from address test init msg */
	ti.reserved = 0;
	memcpy(ti.test_cookie, test_cookie, MIPV6_RR_COOKIE_LENGTH);

	return send_mh(daddr, saddr, (u8 *)&ti, sizeof(ti), msg_type, 
		       NULL, NULL, ops, NULL);
}

/**
 * mipv6_send_addr_test - send a HoT or CoT message
 * @saddr: source address
 * @daddr: destination address
 * @init: HoTI or CoTI message
 * @type: HoT or CoT message
 *
 * Send a reply to HoTI or CoTI message. 
 **/
int mipv6_send_addr_test(struct in6_addr *saddr,
	       struct in6_addr *daddr,
	       struct mipv6_mh_addr_ti *init,
	       int type)
{
	u_int8_t			*addr_cookie = NULL;
	struct mipv6_mh_addr_test       addr_test;      
	struct mipv6_rr_nonce		*nonce;
	struct mipv6_mh_opt *ops = NULL;

	DEBUG_FUNC();

	if ((nonce = mipv6_rr_get_new_nonce())== NULL) {
		DEBUG(DBG_WARNING, "Nonce creation failed");
		return 0;
	} 
	if (mipv6_rr_cookie_create(daddr, &addr_cookie, nonce->index)) {
		DEBUG(DBG_WARNING, "No cookie");
		return 0;
	}
	
	addr_test.nonce_index = nonce->index;
	memcpy(addr_test.test_cookie, init->test_cookie,
			MIPV6_RR_COOKIE_LENGTH);
	memcpy(addr_test.addr_cookie, addr_cookie,
			MIPV6_RR_COOKIE_LENGTH);

	/* No options defined */
	send_mh(daddr, saddr, (u8 *)&addr_test, sizeof(addr_test), type,
		NULL, NULL, ops, NULL);
	
	return 1;
}

extern int mipv6_dad_start(
	struct inet6_ifaddr *ifp, int ifindex,
	struct in6_addr *saddr, struct in6_addr *daddr,
	struct in6_addr *haddr, struct in6_addr *coa,
	__u32 ba_lifetime, int plength, __u16 sequence,
	int single);

/*
 * Lifetime checks. ifp->valid_lft >= ifp->prefered_lft always (see addrconf.c)
 * Returned value is in seconds.
 */

static __u32 get_min_lifetime(struct inet6_ifaddr *ifp, __u32 lifetime)
{
	__u32 rem_lifetime = 0;
	unsigned long now = jiffies;

	if (ifp->valid_lft == 0) {
		rem_lifetime = lifetime;
	} else {
		__u32 valid_lft_left =
		    ifp->valid_lft - ((now - ifp->tstamp) / HZ);
		rem_lifetime =
		    min_t(unsigned long, valid_lft_left, lifetime);
	}

	return rem_lifetime;
}

#define MAX_LIFETIME 1000

/**
 * mipv6_lifetime_check - check maximum lifetime is not exceeded
 * @lifetime: lifetime to check
 *
 * Checks @lifetime does not exceed %MAX_LIFETIME.  Returns @lifetime
 * if not exceeded, otherwise returns %MAX_LIFETIME.
 **/
int mipv6_lifetime_check(int lifetime)
{
	return (lifetime > MAX_LIFETIME) ? MAX_LIFETIME : lifetime;
}


/* Generic routine handling finish of BU processing */
void mipv6_bu_finish(struct inet6_ifaddr *ifp, __u8 ba_status,
		     __u32 ba_lifetime, __u32 ba_refresh,
		     __u32 maxdelay, int plength, __u16 sequence,
		     struct in6_addr *saddr, struct in6_addr *daddr,
		     struct in6_addr *haddr, struct in6_addr *coa,
		     int ifindex, int single, __u8 *k_bu)
{
	struct in6_addr *reply_addr;
	int err;

	if (ba_status >= REASON_UNSPECIFIED) {
		/* DAD failed */
		reply_addr = saddr;
		goto out;
	}

	reply_addr = haddr;
	ba_lifetime = get_min_lifetime(ifp, ba_lifetime);
	ba_lifetime = mipv6_lifetime_check(ba_lifetime);

	if ((err = mipv6_bcache_add(ifindex, daddr, haddr, coa, 
				    ba_lifetime, plength, sequence, 
				    single, HOME_REGISTRATION)) != 0 ) {
		if (err == -ENOMEDIUM) {
			if (ifp)
				in6_ifa_put(ifp);
			return;
		}
		DEBUG(DBG_WARNING, "home reg failed.");

		ba_status = INSUFFICIENT_RESOURCES;
		reply_addr = saddr;
	} else {
		DEBUG(DBG_INFO, "home reg succeeded.");
		/*
		 * refresh MAY be lesser than the lifetime since the cache is
		 * not crash-proof. Set refresh to 80% of lifetime value.  
		 */
		ba_refresh = ba_lifetime * 4 / 5;
	}

	DEBUG(DBG_DATADUMP, "home_addr: %x:%x:%x:%x:%x:%x:%x:%x",
	      NIPV6ADDR(haddr));
	DEBUG(DBG_DATADUMP, "coa: %x:%x:%x:%x:%x:%x:%x:%x",
	      NIPV6ADDR(coa));
	DEBUG(DBG_DATADUMP, "lifet:%d, plen:%d, seq:%d",
	      ba_lifetime, plength, sequence);
      out:
	if (ifp)
		in6_ifa_put(ifp);

	DEBUG(DBG_INFO, "sending ack (code=%d)", ba_status);

	mipv6_send_ba(daddr, haddr, coa, maxdelay, ba_status,
		      sequence, ba_lifetime, ba_refresh, k_bu);
}

void mipv6_bu_add_cache(int ifindex, struct in6_addr *daddr,
			struct in6_addr *haddr, struct in6_addr *coa, 
			__u32 lifetime, int plength, __u16 sequence, 
			int single, int ack, __u32 ba_lifetime, 
			__u32 ba_refresh, __u8 *k_bu)
{
	__u8 ba_status = SUCCESS;

	/* Not "home registration", use 0 as plength */
	plength = 0;
	if (mipv6_bcache_add(ifindex, daddr, haddr, coa, lifetime,
			     plength, sequence, single,
			     CACHE_ENTRY) != 0) {
		DEBUG(DBG_ERROR, "binding failed.");
		ba_status = INSUFFICIENT_RESOURCES;
	} else
		DEBUG(DBG_INFO, "binding succeeded");

	DEBUG(DBG_DATADUMP, "home_addr: %x:%x:%x:%x:%x:%x:%x:%x",
	      NIPV6ADDR(haddr));
	DEBUG(DBG_DATADUMP, "coa: %x:%x:%x:%x:%x:%x:%x:%x",
	      NIPV6ADDR(coa));
	DEBUG(DBG_DATADUMP, "lifet: %d, plen: %d, seq: %d",
	      lifetime, plength, sequence);
	if (ack) {
		DEBUG(DBG_INFO, "sending ack (code=%d)",
		      ba_status);
		mipv6_send_ba(daddr, haddr, coa, 0,
			      ba_status, sequence,
			      ba_lifetime, ba_refresh, k_bu);
	}
}

/*
 * performs the actual processing of a binding update the purpose
 * of which is to add a binding.  
 */
static inline void mipv6_bu_add(int ifindex, struct in6_addr *daddr, 
				struct in6_addr *haddr, struct in6_addr *coa, 
				int ack, int home, int single, int dad, 
				int lladdr, int plength, __u16 sequence, 
				__u32 lifetime, __u8 *k_bu)
{
	__u32 ba_lifetime = lifetime;
	__u32 ba_refresh = lifetime;


	DEBUG_FUNC();

	if (home == 0) {
		mipv6_bu_add_cache(ifindex, daddr, haddr, coa, lifetime,
				   plength, sequence, single, ack, ba_lifetime,
				   ba_refresh, k_bu);
	} else {
		mipv6_bu_add_home(ifindex, haddr, daddr, haddr, coa, lifetime,
				  plength, sequence, single, ack, ba_lifetime,
				  ba_refresh, dad, lladdr, k_bu);
	}
}

/*
 * only called by mipv6_dstopt_process_bu. could well be inlined.
 *
 * performs the actual processing of a binding update the purpose
 * of which is to delete a binding.
 *
 */
static void mipv6_bu_delete(struct in6_addr *daddr,
			    struct in6_addr *haddr, struct in6_addr *coa,
			    int ack, int home, int single, int plength,
			    __u16 sequence, __u32 lifetime, __u8 *k_bu)
{
	__u8 ba_status = SUCCESS;
	__u32 ba_lifetime = 0;
	__u32 ba_refresh = 0;
	__u32 maxdelay = 0;
	int need_ack = ack;

	DEBUG_FUNC();

	if (home == 0) {
		/* Care-of Address entry deletion request */

		if (mipv6_bcache_exists(haddr, daddr) == CACHE_ENTRY) {
			if (mipv6_bcache_delete(haddr, daddr, CACHE_ENTRY) != 0)
				DEBUG(DBG_ERROR, "delete failed.");
			else
				DEBUG(DBG_INFO, "delete succeeded.");

			DEBUG(DBG_DATADUMP, "home_addr: %x:%x:%x:%x:%x:%x:%x:%x",
			      NIPV6ADDR(haddr));
			DEBUG(DBG_DATADUMP, "coa: %x:%x:%x:%x:%x:%x:%x:%x",
			      NIPV6ADDR(coa));
			DEBUG(DBG_DATADUMP, "lifet:%d, plen:%d, seq:%d",
			      lifetime, plength, sequence);
		} else {
			DEBUG(DBG_WARNING, "entry is not in cache");
			ba_status = REASON_UNSPECIFIED;
		}

	}

	mipv6_bcache_delete_wrapper(haddr, daddr, coa, &ba_lifetime,
			&ba_refresh, &need_ack, &ba_status, lifetime,
			plength, sequence, home);

	if (need_ack) {
		mipv6_send_ba(daddr, haddr, coa, maxdelay,
			      ba_status, sequence, ba_lifetime,
			      ba_refresh, k_bu);
	}
}

/*
 *
 * Mobility Header Message handlers
 *
 */

/**
 * parse_mo_tlv - Parse TLV-encoded Mobility Options
 * @mos: pointer to Mobility Options
 * @len: total length of options
 * @opts: structure to store option pointers
 *
 * Parses Mobility Options passed in @mos.  Stores pointers in @opts
 * to all valid mobility options found in @mos.  Unknown options and
 * padding (%MIPV6_OPT_PAD1 and %MIPV6_OPT_PADN) is ignored and
 * skipped.
 **/
int parse_mo_tlv(void *mos, int len, struct mobopt *opts)
{
	struct mipv6_mo *curr = (struct mipv6_mo *)mos;
	int left = len;

	while (left > 0) {
		int optlen = 0;
		if (curr->type == MIPV6_OPT_PAD1)
			optlen = 1;
		else
			optlen = 2 + curr->length;

		if (optlen > left)
			goto bad;

		switch (curr->type) {
		case MIPV6_OPT_PAD1:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_PAD1 at %x", curr);
			break;
		case MIPV6_OPT_PADN:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_PADN at %x", curr);
			break;
		case MIPV6_OPT_ALTERNATE_COA:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_ACOA at %x", curr);
			opts->alt_coa = (struct mipv6_mo_alt_coa *)curr;
			break;
		case MIPV6_OPT_NONCE_INDICES:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_NONCE_INDICES at %x", curr);
			opts->nonce_indices = 
				(struct mipv6_mo_nonce_indices *)curr;
			break;
		case MIPV6_OPT_AUTH_DATA:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_AUTH_DATA at %x", curr);
			opts->auth_data = (struct mipv6_mo_bauth_data *)curr;
			break;
		case MIPV6_OPT_BIND_REFRESH_ADVICE:
			DEBUG(DBG_DATADUMP, "MIPV6_OPT_BIND_REFRESH_ADVICE at %x", curr);
			opts->br_advice = (struct mipv6_mo_br_advice *)curr;
			break;
		default:
			DEBUG(DBG_INFO, "MO Unknown option type %d at %x, ignoring.",
			       curr->type, curr);
			/* unknown mobility option, ignore and skip */
		}

		(u8 *)curr += optlen;
		left -= optlen;
	}

	if (left == 0)
		return 0;
 bad:
	return -1;
}

static int mipv6_handle_mh_HC_testinit(struct in6_addr *cn,
				       struct in6_addr *unused,
				       struct in6_addr *saddr,
				       struct in6_addr *hao,
				       struct mipv6_mh *mh)
{
	struct mipv6_mh_addr_ti *ti = (struct mipv6_mh_addr_ti *)mh->data;
	int type = mh->type;

	DEBUG_FUNC();

	if (mh->length < 1) {
		DEBUG(DBG_INFO, "Mobility Header length less than H/C TestInit");
		return -1;
	}

	if (hao) {
		DEBUG(DBG_INFO, "H/C TestInit has HAO, dropped.");
		return -1;
	}

	if (type == MIPV6_MH_HOTI)
		return mipv6_send_addr_test(cn, saddr, ti, MIPV6_MH_HOT);
	else if (type == MIPV6_MH_COTI)
		return mipv6_send_addr_test(cn, saddr, ti, MIPV6_MH_COT);
	else 
		return -1; /* Impossible to get here */
}

/**
 * mipv6_handle_mh_bu - Binding Update handler
 * @src: care-of address of sender
 * @dst: destination of this packet
 * @haddr: home address of sender
 * @mh: pointer to the beginning of the Mobility Header
 *
 * Handles Binding Update. Packet and offset to option are passed.
 * Returns 0 on success, otherwise negative.
 **/
static int mipv6_handle_mh_bu(struct in6_addr *dst,
			      struct in6_addr *unused,
			      struct in6_addr *haddr, 
			      struct in6_addr *coaddr,
			      struct mipv6_mh *mh)
{
	struct mipv6_mh_bu *bu = (struct mipv6_mh_bu *)mh->data;
	int msg_len = (mh->length << 3) + 2;
	int auth = 0;
	int dereg; /* Is this deregistration? */ 

	struct mipv6_bcache_entry bc_entry;
	struct in6_addr *coa;
	__u8 *key_bu = NULL; /* RR BU authentication key */
	__u8 ack, home, single, dad, ll, plength = 0;
	__u16 sequence;
	__u32 lifetime;
	__u32 ba_refresh = 0;
	__u32 maxdelay = 0;
	
	if (msg_len < (sizeof(*bu))) {
		DEBUG(DBG_INFO, "Mobility Header length less than BU");
		MIPV6_INC_STATS(n_bu_drop.invalid);
		return -1;
	}

	/* If HAO not present, CoA == HAddr */
	if (coaddr == NULL)
		coa = haddr;
	else
		coa = coaddr;

	ack = !!(bu->flags & MIPV6_BU_F_ACK);
	home = !!(bu->flags & MIPV6_BU_F_HOME);
	single = !!(bu->flags & MIPV6_BU_F_SINGLE);
	dad = !!(bu->flags & MIPV6_BU_F_DAD);
	ll = !!(bu->flags & MIPV6_BU_F_LLADDR);


	sequence = ntohs(bu->sequence);
	if (bu->lifetime == 0xffff)
		lifetime = 0xffffffff;
	else
		lifetime = ntohs(bu->lifetime) << 2;

	dereg = (ipv6_addr_cmp(haddr, coa) == 0 || lifetime == 0);

	if ((mipv6_bcache_get(haddr, dst, &bc_entry) == 0)
	    && !modGT65536(sequence, bc_entry.seq)) {
		DEBUG(DBG_INFO,
		      "Sequence number mismatch. Sending BA SEQUENCE_NUMBER_OUT_OF_WINDOW");
		mipv6_send_ba(dst, haddr, coa, maxdelay,
			      SEQUENCE_NUMBER_OUT_OF_WINDOW,
			      bc_entry.seq, lifetime,
			      ba_refresh, key_bu);
		goto out;
	}

	if (msg_len > sizeof(*bu)) {
		struct mobopt opts;
		memset(&opts, 0, sizeof(opts));
		if (parse_mo_tlv(bu + 1, msg_len - sizeof(*bu), &opts) < 0)
			return -1;
		/*
		 * MIPV6_OPT_AUTH_DATA, MIPV6_OPT_NONCE_INDICES, 
		 * MIPV6_OPT_ALT_COA
		 */
		if (opts.alt_coa) {
			coa = &opts.alt_coa->addr;
			dereg = (ipv6_addr_cmp(haddr, coa) == 0 || lifetime == 0);
		}
		if (opts.auth_data && opts.nonce_indices && !home) { 
			u8 ba_status = 0;
			u8 *h_ckie  = NULL, *c_ckie = NULL; /* Home and care-of cookies */
		
			if (mipv6_rr_cookie_create(
				    haddr, &h_ckie, opts.nonce_indices->home_nonce_i) < 0) {
				DEBUG(DBG_WARNING,
				      "mipv6_rr_cookie_create failed for home cookie");
				ba_status = EXPIRED_HOME_NONCE_INDEX;
			}
			/* Don't create the care-of cookie, if MN deregisters */
			if (!dereg && mipv6_rr_cookie_create(
				    coa, &c_ckie,
				    opts.nonce_indices->careof_nonce_i) < 0) {
				DEBUG(DBG_WARNING,
				      "mipv6_rr_cookie_create failed for coa cookie");
				if (ba_status == 0)
					ba_status = EXPIRED_CAREOF_NONCE_INDEX;
				else
					ba_status = EXPIRED_NONCES;
			}
			if (ba_status == 0) {
				if (dereg)
					key_bu = mipv6_rr_key_calc(h_ckie, NULL);
				else
					key_bu = mipv6_rr_key_calc(h_ckie, c_ckie);	       
				mh->checksum = 0;/* TODO: Don't mangle the packet */
				if (key_bu && mipv6_auth_check(
					dst, coa, (__u8 *)mh,  msg_len + sizeof(*mh), opts.auth_data, key_bu) == 0) {
					DEBUG(DBG_INFO, "mipv6_auth_check OK for BU");
					auth = 1;
				} else {
					DEBUG(DBG_WARNING, 
					      "BU Authentication failed");
				}
			}
			if (h_ckie)
				kfree(h_ckie);
			if (c_ckie)
				kfree(c_ckie);
			if (ba_status != 0) {
				MIPV6_INC_STATS(n_bu_drop.auth);
				mipv6_send_ba(dst, haddr, coa, maxdelay,
					      ba_status, sequence, 0, 0, NULL);
				goto out;
			}
		}

	}
	/* Require authorization option for RO, home reg is protected by IPsec */
	if (!home && !auth) {
		MIPV6_INC_STATS(n_bu_drop.auth);
		if (key_bu)
			kfree(key_bu);
		return MH_AUTH_FAILED;
	}
	mipv6_adjust_lifetime(&lifetime);

	if (!dereg) {
		int ifindex;
		struct rt6_info *rt;

		DEBUG(DBG_INFO, "calling bu_add.");
		if ((rt = rt6_lookup(haddr, dst, 0, 0)) != NULL) {
			ifindex = rt->rt6i_dev->ifindex;
			dst_release(&rt->u.dst);
		} else {
			/*
			 * Can't process the BU since the right interface is 
			 * not found.
			 */
			DEBUG(DBG_WARNING, "No route entry found for handling "
			      "a BU request, (using 0 as index)");
			ifindex = 0;
		}
		mipv6_bu_add(ifindex, dst, haddr, 
			     coa, ack, home, single, dad, ll, plength, 
			     sequence, lifetime, key_bu);
	} else {
		DEBUG(DBG_INFO, "calling bu_delete.");
		mipv6_bu_delete(dst, haddr, coa, ack, home,
				single, plength, sequence, lifetime, key_bu);
	}
 out:
	MIPV6_INC_STATS(n_bu_rcvd);
	if (key_bu)
		kfree(key_bu);
	return 0;
}

/*
 * these print the contents of several mobility headers.
 */

static inline void dump_bu(struct mipv6_mh_bu *bu)
{
	if (bu->flags & MIPV6_BU_F_ACK)
		DEBUG(DBG_DATADUMP, " Binding ack requested.");
	if (bu->flags & MIPV6_BU_F_HOME)
		DEBUG(DBG_DATADUMP, " Home registration bit is set.");
	if (bu->flags & MIPV6_BU_F_SINGLE)
		DEBUG(DBG_DATADUMP, " Single Address bit is set.");
	if (bu->flags & MIPV6_BU_F_DAD)
		DEBUG(DBG_DATADUMP, " DAD bit is set.");
	if (bu->flags & MIPV6_BU_F_LLADDR)
		DEBUG(DBG_DATADUMP, " Link-local Address bit is set.");
	DEBUG(DBG_DATADUMP, "Sequence number: %d", bu->sequence);
	DEBUG(DBG_DATADUMP, "Lifetime: %d", ntohs(bu->lifetime) << 2);
}

static inline void dump_ba(struct mipv6_mh_ba *ba)
{
	DEBUG(DBG_DATADUMP, "Status: %d", ba->status);
	DEBUG(DBG_DATADUMP, "Sequence number: %d", ntohs(ba->sequence));
	DEBUG(DBG_DATADUMP, "Lifetime: %d", ntohs(ba->lifetime) << 2);
}

static inline void dump_be(struct mipv6_mh_be *be)
{
	DEBUG(DBG_DATADUMP, "Status: %d", be->status);
}

void dump_mh(struct mipv6_mh *mh)
{
	int len = 0, length;
	int msg_len = (mh->length << 3) + 2;
	struct mobopt ops;

	length = mh->length;

	switch (mh->type) {
	case MIPV6_MH_HOTI:
		DEBUG(DBG_DATADUMP, "Home Test Init (HoTI)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_addr_ti);
		break;
	case MIPV6_MH_COTI:
		DEBUG(DBG_DATADUMP, "Care-of Test Init (CoTI)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_addr_ti);
		break;
	case MIPV6_MH_HOT:
		DEBUG(DBG_DATADUMP, "Home Test (HoT)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_addr_test);
		break;
	case MIPV6_MH_COT:
		DEBUG(DBG_DATADUMP, "Care-of Test (CoT)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_addr_test);
		break;
	case MIPV6_MH_BU:
		DEBUG(DBG_DATADUMP, "Binding Update (BU)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		dump_bu((struct mipv6_mh_bu *)mh->data);
		len = sizeof(struct mipv6_mh_bu);
		break;
	case MIPV6_MH_BA:
		DEBUG(DBG_DATADUMP, "Binding Acknowledgement (BA)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		dump_ba((struct mipv6_mh_ba *)mh->data);
		len = sizeof(struct mipv6_mh_ba);
		break;
	case MIPV6_MH_BRR:
		DEBUG(DBG_DATADUMP, "Binding Refresh Request (BRR)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_brr);
		break;
	case MIPV6_MH_BE:
		DEBUG(DBG_DATADUMP, "Binding Error (BE)"
		      " Message (%d)", mh->type);
		DEBUG(DBG_DATADUMP, " Message Length: %d", length);
		len = sizeof(struct mipv6_mh_be);
		break;
	}

	memset(&ops, 0, sizeof(ops));
	if (msg_len > len && parse_mo_tlv((u8 *)mh->data + len, msg_len - len, &ops) <0)
			return;

	if (ops.alt_coa) {
		DEBUG(DBG_DATADUMP, "MO Alternate Care-of Address %x:%x:%x:%x:%x:%x:%x:%x",
		      NIPV6ADDR(&ops.alt_coa->addr));
	}
	if (ops.nonce_indices) {
		DEBUG(DBG_DATADUMP, "MO Nonce Indices HI %d CI %d", 
		      ntohs(ops.nonce_indices->home_nonce_i), 
		      ntohs(ops.nonce_indices->careof_nonce_i));
	} 
	if (ops.auth_data) {
		DEBUG(DBG_DATADUMP, "MO Binding Authentication Data");
	}
	if (ops.br_advice) {
		DEBUG(DBG_DATADUMP, "MO Binding Refresh Advice %d", 
		      ntohs(ops.br_advice->refresh_interval));
	}
}

static int mipv6_mh_rcv(struct sk_buff *skb)
{
	struct inet6_skb_parm *opt = (struct inet6_skb_parm *)skb->cb;
	struct mipv6_mh *mh;
	struct in6_addr *lhome, *fhome, *lcoa = NULL, *fcoa = NULL;
	int len = ((skb->h.raw[1] + 1)<<3);
	int ret = 0;

	fhome = &skb->nh.ipv6h->saddr;
	lhome = &skb->nh.ipv6h->daddr;

	if (opt->hao != 0) {
		fcoa = (struct in6_addr *)((u8 *)skb->nh.raw + opt->hao);
	}

	if (opt->srcrt2 != 0) {
		struct rt2_hdr *rt2;
		rt2 = (struct rt2_hdr *)((u8 *)skb->nh.raw + opt->srcrt2);
		lcoa = &rt2->addr;
	}

	/* Verify checksum is correct */
	if (skb->ip_summed == CHECKSUM_HW) {
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		if (csum_ipv6_magic(fhome, lhome, skb->len, IPPROTO_MOBILITY,
				    skb->csum)) {
			if (net_ratelimit())
				printk(KERN_WARNING "MIPv6 MH hw checksum failed\n");
			skb->ip_summed = CHECKSUM_NONE;
		}
	}
	if (skb->ip_summed == CHECKSUM_NONE) {
		if (csum_ipv6_magic(fhome, lhome, skb->len, IPPROTO_MOBILITY,
				    skb_checksum(skb, 0, skb->len, 0))) {
			printk(KERN_WARNING "MIPv6 MH checksum failed\n");
			goto bad;
		}
	}

	if (!pskb_may_pull(skb, (skb->h.raw-skb->data) + sizeof(*mh)) ||
	    !pskb_may_pull(skb, (skb->h.raw-skb->data) + len)) {
		DEBUG(DBG_INFO, "MIPv6 MH invalid length");
		kfree_skb(skb);
		return 0;
	}

	mh = (struct mipv6_mh *) skb->h.raw;

	/* Verify there are no more headers after the MH */
	if (mh->payload != NEXTHDR_NONE) {
		DEBUG(DBG_INFO, "MIPv6 MH error");
		goto bad;
	}

	if (mh->type > MIPV6_MH_MAX) {
		/* send binding error */
		printk("Invalid mobility header type (%d)\n", mh->type);
		mipv6_send_be(lhome, fcoa ? fcoa : fhome, fcoa ? fhome : NULL, 
			      MIPV6_BE_UNKNOWN_MH_TYPE);
		goto bad;
	}
	if (mh_rcv[mh->type].func != NULL) {
		ret = mh_rcv[mh->type].func(lhome, lcoa, fhome, fcoa, mh);
	} else {
		DEBUG(DBG_INFO, "No handler for MH Type %d", mh->type);
		goto bad;
	}

	kfree_skb(skb);
	return 0;

bad:
	MIPV6_INC_STATS(n_mh_in_error);
	kfree_skb(skb);
	return 0;

}

#if LINUX_VERSION_CODE >= 0x2052a
struct inet6_protocol mipv6_mh_protocol =
{
	mipv6_mh_rcv,		/* handler		*/
	NULL			/* error control	*/
};
#else
struct inet6_protocol mipv6_mh_protocol = 
{
	mipv6_mh_rcv,		/* handler		*/
	NULL,			/* error control	*/
	NULL,			/* next			*/
	IPPROTO_MOBILITY,	/* protocol ID		*/
	0,			/* copy			*/
	NULL,			/* data			*/
	"MIPv6 MH"	       	/* name			*/
};
#endif

/*
 *
 * Code module init/exit functions
 *
 */

int __init mipv6_mh_common_init(void)
{
	struct sock *sk;
	int err;

	mipv6_mh_socket = sock_alloc();
	if (mipv6_mh_socket == NULL) {
		printk(KERN_ERR
		       "Failed to create the MIP6 MH control socket.\n");
		return -1;
	}
	mipv6_mh_socket->type = SOCK_RAW;

	if ((err = sock_create(PF_INET6, SOCK_RAW, IPPROTO_MOBILITY, 
			       &mipv6_mh_socket)) < 0) {
		printk(KERN_ERR
		       "Failed to initialize the MIP6 MH control socket (err %d).\n",
		       err);
		sock_release(mipv6_mh_socket);
		mipv6_mh_socket = NULL; /* for safety */
		return err;
	}

	sk = mipv6_mh_socket->sk;
	sk->allocation = GFP_ATOMIC;
	sk->sndbuf = SK_WMEM_MAX;
	sk->prot->unhash(sk);

	memset(&mh_rcv, 0, sizeof(mh_rcv));
	mh_rcv[MIPV6_MH_HOTI].func = mipv6_handle_mh_HC_testinit;
	mh_rcv[MIPV6_MH_COTI].func = mipv6_handle_mh_HC_testinit;
	mh_rcv[MIPV6_MH_BU].func =  mipv6_handle_mh_bu;

#if LINUX_VERSION_CODE >= 0x2052a
	if (inet6_add_protocol(&mipv6_mh_protocol, IPPROTO_MOBILITY) < 0) {
		printk(KERN_ERR "Failed to register MOBILITY protocol\n");
		sock_release(mipv6_mh_socket);
		mipv6_mh_socket = NULL;
		return -EAGAIN;
	}
#else
	inet6_add_protocol(&mipv6_mh_protocol);
#endif
	/* To disable the use of dst_cache, 
	 *  which slows down the sending of BUs ??
	 */
	sk->dst_cache=NULL; 

	return 0;
}

void __exit mipv6_mh_common_exit(void)
{
	if (mipv6_mh_socket) sock_release(mipv6_mh_socket);
	mipv6_mh_socket = NULL; /* For safety. */

#if LINUX_VERSION_CODE >= 0x2052a
	inet6_del_protocol(&mipv6_mh_protocol, IPPROTO_MOBILITY);
#else
	inet6_del_protocol(&mipv6_mh_protocol);
#endif
	memset(&mh_rcv, 0, sizeof(mh_rcv));
}
