#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/file.h>
#include <linux/net.h>
#include <net/sock.h>
#include <linux/grsecurity.h>
#include <linux/grinternal.h>
#include <linux/gracl.h>

void
gr_attach_curr_ip(const struct sock *sk)
{
#ifdef CONFIG_GRKERNSEC
	struct task_struct *p;

	if (unlikely(sk->protocol != IPPROTO_TCP))
		return;

	read_lock(&tasklist_lock);
	for_each_task(p) {
		if (likely(!p->used_connect))
			continue;
		if (unlikely(sk->rcv_saddr == p->gr_daddr &&
			     sk->daddr == p->gr_saddr &&
			     ntohs(sk->sport) ==
			     ntohs(p->gr_dport)
			     && ntohs(sk->dport) ==
			     ntohs(p->gr_sport))) {
			current->curr_ip = p->curr_ip;
			current->used_accept = 1;
			p->gr_daddr = 0;
			p->gr_saddr = 0;
			p->gr_sport = 0;
			p->gr_dport = 0;
			p->used_connect = 0;
			read_unlock(&tasklist_lock);
			return;
		}
	}
	read_unlock(&tasklist_lock);

	current->curr_ip = sk->daddr;
	current->used_accept = 1;
#endif
	return;
}

int
gr_handle_sock_all(const int family, const int type, const int protocol)
{
#ifdef CONFIG_GRKERNSEC_SOCKET_ALL
	if (grsec_enable_socket_all && in_group_p(grsec_socket_all_gid) &&
	    (family != AF_UNIX) && (family != AF_LOCAL)) {
		security_alert(GR_SOCK_MSG, family, type,
			       protocol, DEFAULTSECARGS);
		return -EACCES;
	}
#endif
	return 0;
}

int
gr_handle_sock_server(const struct sockaddr *sck)
{
#ifdef CONFIG_GRKERNSEC_SOCKET_SERVER
	if (grsec_enable_socket_server &&
	    in_group_p(grsec_socket_server_gid) &&
	    sck && (sck->sa_family != AF_UNIX) &&
	    (sck->sa_family != AF_LOCAL)) {
		security_alert(GR_BIND_MSG, DEFAULTSECARGS);
		return -EACCES;
	}
#endif
	return 0;
}

int
gr_handle_sock_client(const struct sockaddr *sck)
{
#ifdef CONFIG_GRKERNSEC_SOCKET_CLIENT
	if (grsec_enable_socket_client && in_group_p(grsec_socket_client_gid) &&
	    sck && (sck->sa_family != AF_UNIX) &&
	    (sck->sa_family != AF_LOCAL)) {
		security_alert(GR_CONNECT_MSG, DEFAULTSECARGS);
		return -EACCES;
	}
#endif
	return 0;
}

__u32
gr_cap_rtnetlink(void)
{
#ifdef CONFIG_GRKERNSEC
	if (!gr_acl_is_enabled())
		return current->cap_effective;
	else
		return (current->cap_effective & ~(current->acl->cap_lower));
#else
	return current->cap_effective;
#endif
}
