#include "gradm.h"

static struct proc_acl *
get_default_acl(void)
{
	struct proc_acl *tmp;

	for_each_subject(tmp, def_acl_tmp)
	    if (!strcmp(tmp->filename, "/"))
		return tmp;

	return NULL;
}

int
test_perm(const char *subj, const char *obj)
{
	struct proc_acl *proc;
	struct file_acl *filp;
	char *tmpsubj;
	char *tmpobj;
	__u32 perm = -1;

	if ((tmpsubj = calloc(strlen(subj) + 1, sizeof (char))) == NULL)
		failure("calloc");

	if ((tmpobj = calloc(strlen(obj) + 1, sizeof (char))) == NULL)
		failure("calloc");

	strcpy(tmpsubj, subj);

	do {
		for_each_subject(proc, def_acl_tmp) {
			if (!strcmp(proc->filename, tmpsubj)) {
				strcpy(tmpobj, obj);
				do {
					for_each_object(filp, proc->proc_object) {
						if (!strcmp
						    (filp->filename, tmpobj)) {
							perm = filp->mode;
							goto done;
						}
					}
				} while (parent_dir(obj, &tmpobj));
			}
		}
	} while (parent_dir(subj, &tmpsubj));

      done:
	printf("Allowed access for %s from %s:\n", obj, subj);
	printf("Read: %s\n"
	       "Write: %s\n"
	       "Append: %s\n"
	       "Execute: %s\n"
	       "Hidden: %s\n"
	       "Inherit ACL on exec: %s\n"
	       "Read-only ptrace: %s\n"
	       "Audit reads: %s\n"
	       "Audit writes: %s\n"
	       "Audit execs: %s\n"
	       "Audit appends: %s\n"
	       "Audit finds: %s\n"
	       "Audit inherits: %s\n",
	       (perm & GR_READ) ? "yes" : "no",
	       (perm & GR_WRITE) ? "yes" : "no",
	       (perm & GR_APPEND) ? "yes" : "no",
	       (perm & GR_EXEC) ? "yes" : "no",
	       (perm & GR_FIND) ? "no" : "yes",
	       (perm & GR_INHERIT) ? "yes" : "no",
	       (perm & GR_PTRACERD) ? "yes" : "no",
	       (perm & GR_AUDIT_READ) ? "yes" : "no",
	       (perm & GR_AUDIT_WRITE) ? "yes" : "no",
	       (perm & GR_AUDIT_EXEC) ? "yes" : "no",
	       (perm & GR_AUDIT_APPEND) ? "yes" : "no",
	       (perm & GR_AUDIT_FIND) ? "yes" : "no",
	       (perm & GR_AUDIT_INHERIT) ? "yes" : "no");

	return perm;
}

static int
check_permission(struct proc_acl *def_acl, const char *filename,
		 struct chk_perm *chk)
{
	struct file_acl *tmpf = NULL;
	char *tmpname;

	if (chk->type == CHK_FILE) {
		if ((tmpname =
		     calloc(strlen(filename) + 1, sizeof (char))) == NULL)
			failure("calloc");

		strcpy(tmpname, filename);

		do {
			for_each_object(tmpf, def_acl->proc_object)
			    if (!strcmp(tmpf->filename, tmpname)) {
				if (((chk->w_modes == 0xffff)
				     || (tmpf->mode & chk->w_modes))
				    && ((chk->u_modes == 0xffff)
					|| !(tmpf->mode & chk->u_modes))) {
					free(tmpname);
					return 1;
				} else {
					free(tmpname);
					return 0;
				}
			}
		} while (parent_dir(filename, &tmpname));

		free(tmpname);
	} else if (chk->type == CHK_CAP) {
		if (((chk->w_caps == 0xffffffff)
		     || !(def_acl->cap_drop & chk->w_caps))
		    && ((chk->u_caps == 0xffffffff)
			|| (def_acl->cap_drop & chk->u_caps)))
			return 1;
	}

	return 0;
}

static int
check_subjects(void)
{
	struct proc_acl *tmp;
	struct proc_acl *def_acl;
	struct chk_perm chk;
	unsigned int errs_found = 0;

	def_acl = get_default_acl();

	if (!def_acl)
		return 0;

	chk.type = CHK_FILE;
	chk.u_modes = GR_WRITE;
	chk.w_modes = 0xffff;

	for_each_subject(tmp, def_acl_tmp)
	    if ((tmp->filename[0] == '/') && (tmp->filename[1] != '\0'))
		if (!check_permission(def_acl, tmp->filename, &chk))
			fprintf(stderr,
				"Warning: write access is allowed to your "
				"subject ACL for %s.  Please ensure that the subject is running with less privilege than the default ACL.\n",
				tmp->filename);

	return errs_found;
}

static void
check_default_objects(void)
{
	int def_notfound = 1;
	struct proc_acl *tmp;
	struct file_acl *tmpf;

	for_each_subject(tmp, def_acl_tmp) {
		for_each_object(tmpf, tmp->proc_object)
		    if (!strcmp(tmpf->filename, "/"))
			def_notfound = 0;
		if (def_notfound) {
			fprintf(stderr, "Default ACL object not found for "
				"subject %s\nThe ACL system will "
				"not load until you correct this "
				"error.\n", tmp->filename);
			exit(EXIT_FAILURE);
		}
		def_notfound = 1;
	}

	return;
}

static int
check_lilo_conf(struct proc_acl *def_acl)
{
	FILE *liloconf;
	char buf[PATH_MAX];
	struct stat fstat;
	struct chk_perm chk;
	unsigned int errs_found = 0;
	char *p;

	if ((liloconf = fopen("/etc/lilo.conf", "r")) == NULL)
		return 0;

	chk.type = CHK_FILE;
	chk.u_modes = GR_WRITE;
	chk.w_modes = 0xffff;

	while (fgets(buf, PATH_MAX - 1, liloconf)) {
		if (buf[strlen(buf) - 1] == '\n')
			buf[strlen(buf) - 1] = '\0';
		if ((p = strstr(buf, "image="))) {
			p += 6;
			if (!stat(p, &fstat)
			    && !check_permission(def_acl, p, &chk)) {
				fprintf(stderr,
					"Write access is allowed to %s, a kernel "
					"for your system specified in "
					"/etc/lilo.conf.\n\n", p);
				errs_found++;
			}
		}
	}

	fclose(liloconf);

	return errs_found;
}

static int
check_lib_paths(struct proc_acl *def_acl)
{
	FILE *ldconf;
	char buf[PATH_MAX];
	struct stat fstat;
	struct chk_perm chk;
	unsigned int errs_found = 0;

	if ((ldconf = fopen("/etc/ld.so.conf", "r")) == NULL)
		return 0;

	chk.type = CHK_FILE;
	chk.u_modes = GR_WRITE;
	chk.w_modes = 0xffff;

	while (fgets(buf, PATH_MAX - 1, ldconf)) {
		if (buf[strlen(buf) - 1] == '\n')
			buf[strlen(buf) - 1] = '\0';

		if (!stat(buf, &fstat) && !check_permission(def_acl, buf, &chk)) {
			fprintf(stderr,
				"Write access is allowed to %s, a directory which "
				"holds libraries for your system and is included "
				"in /etc/ld.so.conf.\n\n", buf);
			errs_found++;
		}
	}

	fclose(ldconf);

	return errs_found;
}

static int
check_path_env(struct proc_acl *def_acl)
{
	char *pathstr, *p, *p2;
	struct stat fstat;
	struct chk_perm chk;
	unsigned int errs_found = 0;

	if ((pathstr = getenv("PATH")) == NULL)
		return 0;

	p = pathstr;

	chk.type = CHK_FILE;
	chk.u_modes = GR_WRITE;
	chk.w_modes = 0xffff;

	while ((p2 = strchr(p, ':'))) {
		*p2++ = '\0';
		if (!stat(p, &fstat) && !check_permission(def_acl, p, &chk)) {
			fprintf(stderr,
				"Write access is allowed to %s, a directory which "
				"holds binaries for your system and is included "
				"in the PATH environment variable.\n\n", p);
			errs_found++;
		}
		p = p2;
	}

	if (!stat(p, &fstat) && !check_permission(def_acl, p, &chk)) {
		fprintf(stderr,
			"Write access is allowed to %s, a directory which "
			"holds binaries for your system and is included "
			"in the PATH environment variable.\n\n", p);
		errs_found++;
	}

	return errs_found;
}

static int
handle_notrojan_mode(void)
{
	struct proc_acl *subj, *subj2;
	struct file_acl *obj, *obj2;
	char *objname;
	int ret = 0;

	for_each_subject(subj, def_acl_tmp) {
		if (!(subj->mode & GR_NOTROJAN))
			continue;
		for_each_object(obj, subj->proc_object) {
			if (!(obj->mode & GR_EXEC))
				continue;
			if ((objname =
			     malloc(strlen(obj->filename) + 1)) == NULL)
				failure("malloc");
			strcpy(objname, obj->filename);
			do {
				for_each_subject(subj2, def_acl_tmp) {
					if (subj2 == subj
					    || (subj2->filename[0] != '/'))
						continue;
					for_each_object(obj2,
							subj2->proc_object) {
						if (!strcmp
						    (obj2->filename, objname)) {
							if (obj2->
							    mode & GR_WRITE) {
								ret++;
								fprintf(stderr,
									"\'T\' specified in mode for %s."
									"  %s's executable object %s is "
									"writable by %s, due to its "
									"writable object %s.\nThis would "
									"allow %s to execute trojaned code.\n\n",
									subj->
									filename,
									subj->
									filename,
									obj->
									filename,
									subj2->
									filename,
									obj2->
									filename,
									subj->
									filename);
							}
							break;
						}
					}
				}
			} while (parent_dir(obj->filename, &objname));
			free(objname);
		}
	}

	return ret;
}

void
analyze_acls(void)
{
	struct proc_acl *def_acl;
	struct chk_perm chk;
	unsigned int errs_found = 0;

	if ((def_acl = get_default_acl()) == NULL) {
		fprintf(stderr, "There is no default ACL present in your "
			"configuration.\nPlease read the ACL "
			"documentation and create a default ACL "
			"before attempting to enable the ACL " "system.\n");
		exit(EXIT_FAILURE);
	}

	check_default_objects();
	errs_found += check_subjects();

	chk.type = CHK_FILE;
	chk.u_modes = GR_FIND;
	chk.w_modes = 0xffff;

	if (!check_permission(def_acl, GRSEC_DIR, &chk)) {
		fprintf(stderr,
			"Viewing access is allowed to %s, the directory which "
			"holds ACL and ACL password information.\n\n",
			GRSEC_DIR);
		errs_found++;
	}

	if (!check_permission(def_acl, "/dev/kmem", &chk)) {
		fprintf(stderr,
			"Viewing access is allowed to /dev/kmem.  This could "
			"allow an attacker to modify the code of your "
			"running kernel.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/dev/mem", &chk)) {
		fprintf(stderr,
			"Viewing access is allowed to /dev/mem.  This would "
			"allow an attacker to modify the code of programs "
			"running on your system.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/dev/port", &chk)) {
		fprintf(stderr,
			"Viewing access is allowed to /dev/port.  This would "
			"allow an attacker to modify the code of programs "
			"running on your system.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/proc/kcore", &chk)) {
		fprintf(stderr,
			"Viewing access is allowed to /proc/kcore.  This would "
			"allow an attacker to view the raw memory of processes "
			"running on your system.\n\n");
		errs_found++;
	}

	chk.u_modes = GR_WRITE;

	if (!check_permission(def_acl, "/boot", &chk)) {
		fprintf(stderr,
			"Writing access is allowed to /boot, the directory which "
			"holds boot and kernel information.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/dev", &chk)) {
		fprintf(stderr,
			"Writing access is allowed to /dev, the directory which "
			"holds system devices.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/dev/log", &chk)) {
		fprintf(stderr,
			"Writing access is allowed to /dev/log.  This could in some cases allow an attacker"
			" to spoof learning logs on your system.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/root", &chk)) {
		fprintf(stderr,
			"Writing access is allowed to /root, the directory which "
			"holds shell configurations for the root user.  "
			"If writing is allowed to this directory, an attacker "
			"could modify your $PATH environment to fool you "
			"into executing a trojaned gradm.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/proc/sys", &chk)) {
		fprintf(stderr,
			"Write access is allowed to /proc/sys, the directory which "
			"holds entries that allow modifying kernel variables.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/etc", &chk)) {
		fprintf(stderr,
			"Write access is allowed to /etc, the directory which "
			"holds initialization scripts and configuration files.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/lib", &chk)) {
		fprintf(stderr,
			"Write access is allowed to /lib, a directory which "
			"holds system libraries and loadable modules.\n\n");
		errs_found++;
	}

	if (!check_permission(def_acl, "/usr/lib", &chk)) {
		fprintf(stderr,
			"Write access is allowed to /usr/lib, a directory which "
			"holds system libraries.\n\n");
		errs_found++;
	}

	chk.u_modes = GR_READ;

	if (!check_permission(def_acl, "/dev", &chk)) {
		fprintf(stderr,
			"Reading access is allowed to /dev, the directory which "
			"holds system devices.\n\n");
		errs_found++;
	}

	chk.type = CHK_CAP;
	chk.u_caps =
	    (1 << CAP_SYS_MODULE) | (1 << CAP_SYS_RAWIO) | (1 << CAP_MKNOD);
	chk.w_caps = 0xffffffff;

	if (!check_permission(def_acl, "", &chk)) {
		fprintf(stderr,
			"CAP_SYS_MODULE, CAP_SYS_RAWIO, and CAP_MKNOD are all not "
			"removed for all users.  This would allow an "
			"attacker to modify the kernel by means of a "
			"module or corrupt devices on your system.\n\n");
		errs_found++;
	}

	chk.u_caps = (1 << CAP_SYS_ADMIN);
	chk.w_caps = 0xffffffff;

	if (!check_permission(def_acl, "", &chk)) {
		fprintf(stderr, "CAP_SYS_ADMIN is not "
			"removed for all users.  This would allow an "
			"attacker to mount filesystems to bypass ACLs\n\n");
		errs_found++;
	}

	chk.u_caps = (1 << CAP_NET_ADMIN);
	chk.w_caps = 0xffffffff;

	if (!check_permission(def_acl, "", &chk)) {
		fprintf(stderr, "CAP_NET_ADMIN is not "
			"removed for all users.  This would allow an "
			"attacker to modify your firewall configuration or redirect private information\n\n");
		errs_found++;
	}

	chk.u_caps = (1 << CAP_NET_BIND_SERVICE);
	chk.w_caps = 0xffffffff;

	if (!check_permission(def_acl, "", &chk)) {
		fprintf(stderr, "CAP_NET_BIND_SERVICE is not "
			"removed for all users.  This would allow an "
			"attacker (if he can kill a network daemon) to "
			"launch a trojaned daemon that could steal privileged information\n\n");
		errs_found++;
	}

	chk.u_caps = (1 << CAP_SYS_TTY_CONFIG);
	chk.w_caps = 0xffffffff;

	if (!check_permission(def_acl, "", &chk)) {
		fprintf(stderr, "CAP_SYS_TTY_CONFIG is not "
			"removed for all users.  This would allow an "
			"attacker to hijack terminals of "
			"privileged processes\n\n");
		errs_found++;
	}

	errs_found += check_path_env(def_acl);
	errs_found += check_lib_paths(def_acl);
	errs_found += check_lilo_conf(def_acl);

	errs_found += handle_notrojan_mode();

	if (errs_found) {
		printf("There were %d holes found in your ACL "
		       "configuration.  These must be fixed before the "
		       "ACL system will be allowed to be enabled.\n",
		       errs_found);
		exit(EXIT_FAILURE);
	}

	return;
}
