/* evfs 0.3 2003 sd */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/romfs_fs.h>
#include <linux/fs.h>
#include <linux/locks.h>
#include <linux/init.h>
#include <linux/smp_lock.h>
#include <asm/uaccess.h>
#include <linux/file.h>
#include <linux/unistd.h>
#include <linux/dnotify.h>

#include "evfs.h"
#include "aes.h"
#include "sha256.h"
#include "base64.h"

/* those is considered to be the fastest */
#define EVFS_BLK_SIZE	PAGE_CACHE_SIZE
#define	EVFS_BLK_BITS	12
/* padlen, all of it's bits must be *always* 1 in line, f.e. : 1,3,7,15,31 ... */
#define	PADLEN		(AES_BLOCK_SIZE-1)
#define KBUFSZ		(PADLEN+PAGE_CACHE_SIZE)
#define SALTLEN		16

//#define track() printk("track: " __func__ "()/%d reached\n", __LINE__)
#define track()


extern long sys_readlink(const char * path, char * buf, int bufsiz);
extern long sys_link(const char * oldname, const char * newname);
extern long sys_unlink(const char * pathname);
extern long sys_symlink(const char * oldname, const char * newname);
extern long sys_mkdir(const char * pathname, int mode);
extern long sys_rmdir(const char * pathname);
extern long sys_mknod(const char * filename, int mode, dev_t dev);
extern long sys_rename(const char * oldname, const char * newname);


struct	evfs_ctx  {
	struct	aes_ctx aes;	/* master aes key */
	/* we've 256 salts by 16 bytes, after 1mb it will repeat
	  - give chance to compress really big files ;)) */
	/* (256*PAGE_CACHE_SIZE) */
	char	salts[256][SALTLEN];
	int	recursion;
};


static struct super_operations evfs_sbops;
static	struct	inode_operations evfs_file_iops;
static	struct	inode_operations evfs_dir_iops;
static	struct	file_operations evfs_fops;
static struct file_operations evfs_dir_fops;
static struct address_space_operations evfs_aops;
static struct address_space_operations evfs_link_aops;
static struct dentry_operations evfs_dentry_ops;

/**************************************************************************
 General purpose functions
 **************************************************************************/

#define get_mem() kmalloc(PAGE_CACHE_SIZE + 1024, GFP_KERNEL)
#define put_mem(x) kfree(x)
#define COPY_INODE(i,g) \
	i->i_size = S_ISREG(g->i_mode)?((g->i_size > PADLEN)?g->i_size - PADLEN:0):(S_ISLNK(g->i_mode)?0:g->i_size); \
	i->i_ino = g->i_ino; \
	i->i_mode = g->i_mode; \
	i->i_nlink = g->i_nlink; \
	i->i_uid = g->i_uid; \
	i->i_gid = g->i_gid; \
	i->i_rdev = g->i_rdev; \
	i->i_atime = g->i_atime; \
	i->i_mtime = g->i_mtime; \
	i->i_ctime = g->i_ctime; \
	i->i_blkbits = EVFS_BLK_BITS; \
	i->i_blksize = EVFS_BLK_SIZE; \
	i->i_blocks = (g->i_blocks * (1 << g->i_blkbits)) / EVFS_BLK_SIZE;


struct	evfs_inode {
	char	*host_filename;
	char	*sym;
	struct	file *fd;
};
#define evfs_i(x) (*((struct evfs_inode *) &(x)->u.generic_ip))
#define file_evfs_i(file) (&evfs_i((file)->f_dentry->d_inode))
#define evfs_key(x) ((struct evfs_ctx *) (x)->u.generic_sbp)
#define gk(i) (evfs_key((i->i_sb)))

static	int my_truncate(struct dentry *dentry, loff_t length)
{
	struct inode *inode = dentry->d_inode;
	int error;
	struct iattr newattrs;

	down(&inode->i_sem);
	newattrs.ia_size = length;
	newattrs.ia_valid = ATTR_SIZE | ATTR_CTIME;
	error = notify_change(dentry, &newattrs);
	up(&inode->i_sem);
	return error;
}

static	inline char *get_salt(struct evfs_ctx *ctx, u8 index, char *buf)
{
	if (!ctx)
		memset(buf, 0, SALTLEN);
	else
		memcpy(buf, &ctx->salts[index][0], SALTLEN);
	return buf;  
}

static	inline	char *kstrdup(char *s)
{
	char	*p = kmalloc(strlen(s) + 1, GFP_KERNEL);
	if (!p)
		return NULL;
	strcpy(p, s);
	return p;
}

static	inline	int hpath_lookup_dir(char *name, struct nameidata *nd, int flags)
{
	int error = 0;
	track();
	if (path_init(name, LOOKUP_POSITIVE|LOOKUP_DIRECTORY|flags, nd))
		error = path_walk(name, nd);
	track();
	return error;
}


static	inline	int hpath_lookup(char *name, struct nameidata *nd)
{
	int error = 0;
	track();
	if (path_init(name, LOOKUP_POSITIVE, nd))
		error = path_walk(name, nd);
	track();
	return error;
}

static	int	read_name(struct inode *ino, char *name)
{
	struct	nameidata nd;
	int	error = hpath_lookup(name, &nd);
	track();
	if (error) return error;
	if (!nd.dentry->d_inode) {
		path_release(&nd);
		return -ENOENT;
	}
	COPY_INODE(ino, nd.dentry->d_inode);
	path_release(&nd);
	return error;
}

static	int	file_type(char *name, int *rdev)
{
	struct	nameidata nd;
	int	error = hpath_lookup(name, &nd);
	track();
	if (!error) {
		if (nd.dentry->d_inode) {
			error = nd.dentry->d_inode->i_mode;
			if (rdev)
				*rdev = nd.dentry->d_inode->i_rdev;
		} else {
			error = -1024;
		}
		path_release(&nd);
	}
	return error;
}



/* translate our virtual path (discovered by traversing
   dcache tree) to real encrypted names path */
static char *dentry_name(struct dentry *dentry, int extra, struct super_block *sb)
{
	struct	dentry *parent;
	char	*name, *root;
	int	len, rootlen;
	char	tmp[NAME_MAX + 1];

	track();
/*	if ((dentry->d_name.len == sizeof(EVFS_COOKIE)-1) &&
	    (!memcmp(dentry->d_name.name, EVFS_COOKIE, sizeof(EVFS_COOKIE)-1)))
		return NULL; */
	len = 0;
	for (parent = dentry; parent->d_parent != parent; parent = parent->d_parent)
		len += norm2baselen((parent->d_name.len + PADLEN) & ~PADLEN) + 1; /* +1 for '/' */

	track();
	root = evfs_i(parent->d_inode).host_filename;
	rootlen = strlen(root);
	len += rootlen;
	track();
	name = kmalloc(len + extra + 1, GFP_KERNEL);
	if (!name)
		return NULL;
	track();
	memcpy(name, root, rootlen);
	name[len] = 0;

	/* count length of whole tree (only approx!) */
	track();
	for (parent = dentry; parent->d_parent != parent; parent  = parent->d_parent) {
		int	alen = (parent->d_name.len + PADLEN) & ~PADLEN;
		int	elen = norm2baselen(alen);
		char	salt[SALTLEN];

		track();
		len -= elen + 1;
		name[len] = '/';
		memset(tmp, 0, alen);
		memcpy(tmp, parent->d_name.name, parent->d_name.len);
		track();
		if (!sb) sb = dentry->d_inode->i_sb;
		aes_encrypt_str(&evfs_key(sb)->aes, tmp, tmp, alen, get_salt(NULL, 0, salt));
		track();
		base64_encode(tmp, name + len + 1, alen);
	}
	return name;
}

static char *inode_name(struct inode *ino, int extra)
{
	struct dentry *dentry;

	track();
	dentry = list_entry(ino->i_dentry.next, struct dentry, d_alias);
	return(dentry_name(dentry, extra, ino->i_sb));
}

static struct inode *get_inode(struct super_block *sb, struct dentry *dentry,
			       int *error)
{
	struct inode *inode;
	char *name;
	int type, err = -ENOMEM, rdev;

	track();
	inode = new_inode(sb);
	if(inode == NULL) 
		goto out;

	evfs_i(inode).host_filename = NULL;
	evfs_i(inode).fd = NULL;
	evfs_i(inode).sym = NULL;
	insert_inode_hash(inode);
	if(dentry){
		name = dentry_name(dentry, 0, sb);
		if(name == NULL){
			err = -ENOMEM;
			goto out_put;
		}
		type = file_type(name, &rdev);
		if (type == -1024) {
			type = S_IFLNK;
		} else
		if (S_ISLNK(type)) {
			int	c, n;
			char	*buf = get_mem();
			mm_segment_t fs = get_fs();
			set_fs(KERNEL_DS);
			c = sys_readlink(name, buf, PAGE_CACHE_SIZE-1);
			set_fs(fs);
			n = base2normlen(c);
			if ((c > 0) && !(n & PADLEN)) {
				char	*tmp = get_mem();
				char	salt[SALTLEN];
				/* symlink is ok, decode it */
				base64_decode(buf, tmp, c);
				aes_decrypt_str(&evfs_key(sb)->aes, tmp, tmp, n, get_salt(NULL, 0, salt));
				tmp[n] = 0;
				evfs_i(inode).sym = tmp;
			}
			put_mem(buf);
		}
		kfree(name); 
	}
	else type = S_IFDIR;
	inode->i_sb = sb;

	err = 0;
	if(S_ISLNK(type))
		inode->i_op = &page_symlink_inode_operations;
	else if(S_ISDIR(type))
		inode->i_op = &evfs_dir_iops;
	else inode->i_op = &evfs_file_iops;

	if(S_ISDIR(type)) inode->i_fop = &evfs_dir_fops;
	else inode->i_fop = &evfs_fops;

	if(S_ISLNK(type))
		inode->i_mapping->a_ops = &evfs_link_aops;
	else inode->i_mapping->a_ops = &evfs_aops;

	if (S_ISCHR(type))
		init_special_inode(inode, S_IFCHR, rdev);
		else
	if (S_ISBLK(type))
		init_special_inode(inode, S_IFBLK, rdev);
		else
	if (S_ISFIFO(type))
		init_special_inode(inode, S_IFIFO, 0);
		else
	if (S_ISSOCK(type))
		init_special_inode(inode, S_IFSOCK, 0);

	if(error) *error = err;
	return(inode);
 out_put:
	iput(inode);
 out:
	if(error) *error = err;
	return(NULL);
}



/**************************************************************************
 Super operations
 **************************************************************************/
/* purge an evfs inode */
static	void evfs_delete_inode(struct inode *ino)
{
	track();
	if (evfs_i(ino).host_filename)
		kfree(evfs_i(ino).host_filename);
	evfs_i(ino).host_filename = NULL;

	if(evfs_i(ino).fd) {
		/* set real size */
		if (S_ISREG(ino->i_mode))
			my_truncate(evfs_i(ino).fd->f_dentry, ino->i_size + PADLEN);
		fput(evfs_i(ino).fd);
		evfs_i(ino).fd = NULL;
	}
	if(evfs_i(ino).sym)  {
		put_mem(evfs_i(ino).sym);
		evfs_i(ino).sym = NULL;
	}
	clear_inode(ino);
}


/* stat a filesystem */
int	evfs_statfs(struct super_block *sb, struct statfs *sf)
{
	struct	nameidata nd;
	int	error;
	error = hpath_lookup(evfs_i(sb->s_root->d_inode).host_filename, &nd);
	if (!error) {
		error = vfs_statfs(nd.dentry->d_inode->i_sb, sf);
		path_release(&nd);
	}
	return error;
}

static	void	evfs_putsuper(struct super_block *s)
{
	track();
	kfree(evfs_key(s));
}

/* read evfs superblock */
static	struct	super_block *evfs_read_super(struct super_block *s, void *d, int silent)
{
	char	*data = d;
	char	*password;
	char	*name, *to;
	struct	inode *root_inode;
	char	buf[32];
	struct	sha256_ctx ctx;
	struct	nameidata nd, ns;
	struct	dentry *parent;

	int	uid = current->fsuid;
	int	gid = current->fsgid;

	track();
	/* first some basic super ops */
	s->s_blocksize = EVFS_BLK_SIZE;
	s->s_blocksize_bits = EVFS_BLK_BITS;
	s->s_magic = EVFS_MAGIC;
	s->s_op = &evfs_sbops;

	/* and some common checks */
	if (!data || !*data) {
		printk("evfs: source mount directory not specified\n");
		return NULL;
	}

	/* extract from, to and password */
	for (to = data; *to != ' '; to++) {
		if (!*to) {
			printk("evfs: invalid number of arguments (destination and password missing)\n");
			return NULL;
		}
	}
	*to++ = 0;

	for (password = to; *password != ' '; password++)
		if (!*password) {
			printk("evfs: no password supplied while mounting %s\n", data);
			return NULL;
		}
	*password++ = 0;

	if (hpath_lookup_dir(to, &nd, LOOKUP_FOLLOW)) {
		printk("evfs: error while looking up mount destination!\n");
		return NULL;
	}

	/* check uid ... */
	if (uid)
	if ((nd.dentry->d_inode->i_uid != uid) || (nd.dentry->d_inode->i_gid != gid)) {
		printk("evfs: *WARNING*, uid %d supplied someone's else directory as destination!\n", uid);
		goto out_nd;
	}

	/* and absolute path */
	if (data[0] != '/') {
		printk("evfs: attempted to mount relative path %s!\n", data);
		goto out_nd;
	}

	/* create root inode */
	root_inode = get_inode(s, NULL, NULL);
	if (!root_inode)
		goto out_nd;
	name = kstrdup(data);
	track();
	s->s_root = d_alloc_root(root_inode);
	evfs_i(root_inode).host_filename = name;
	track();

	if (hpath_lookup_dir(name, &ns, 0))
		goto out;

	if (ns.dentry->d_inode->i_sb->s_magic == EVFS_MAGIC) {
		printk("evfs: uid %d tried to mount evfs from evfs - not supported yet, to avoid recursion.\n", uid);
		goto out_put;
	}

	COPY_INODE(root_inode, ns.dentry->d_inode);

	track();
	/* hrmm. comparing whether source is not in destinating directory *is* good idea */
	for (parent = ns.dentry; parent != parent->d_parent; parent = parent->d_parent) {
		if (parent == nd.dentry)
			goto out_put;
	}
	track();

	if (uid)
	if ((root_inode->i_uid != uid || root_inode->i_gid != gid) &&
	    ((uid != 0) && (gid != 0)))
		goto out_put;

	evfs_key(s) = kmalloc(sizeof(struct evfs_ctx), GFP_KERNEL);
	if (!evfs_key(s))
		goto out_put;
	sha256_init(&ctx);
	sha256_update(&ctx, password, strlen(password));
	sha256_final(&ctx, buf);
	memset(&ctx, 0xF8, sizeof(ctx));
	aes_set_key(&evfs_key(s)->aes, buf);
	evfs_key(s)->recursion = 0;
	memset(evfs_key(s)->salts, buf[0] + buf[31], sizeof(evfs_key(s)->salts));
	/* initialize disk block salts (one for one blocks, 256 total, then repeats */
	aes_encrypt_str(&evfs_key(s)->aes, &evfs_key(s)->salts[0][0], &evfs_key(s)->salts[0][0], sizeof(evfs_key(s)->salts), buf);
	memset(buf, 0xF8, 32);

	path_release(&nd);
	path_release(&ns);
	return s;
out_put:
	track();
	path_release(&ns);
out:
	iput(root_inode);
out_nd:
	track();
	path_release(&nd);
	return NULL;	
}

/**************************************************************************
 Inode operations
 **************************************************************************/
int evfs_create(struct inode *dir, struct dentry *dentry, int mode)
{
	struct inode *inode;
	char *name;
	struct file *fd;
	int error;

	track();
	inode = get_inode(dir->i_sb, dentry, &error);
	if (error) return(error);
	name = dentry_name(dentry, 0, dir->i_sb);
	if(name == NULL){
		iput(inode);
		return(-ENOMEM);
	}
	fd = filp_open(name, O_CREAT | O_EXCL, mode);
	error = IS_ERR(fd);
	if (!error) {
		my_truncate(fd->f_dentry, PADLEN);
		fput(fd);
	}
	else error = PTR_ERR(fd);
	if(!error) error = read_name(inode, name);
	kfree(name);
	if(error) {
		iput(inode);
		return(error);
	}
	d_instantiate(dentry, inode);
	return(0);
}

struct dentry *evfs_lookup(struct inode *ino, struct dentry *dentry)
{
	struct inode *inode;
	char *name;
	int error;

	track();
	inode = get_inode(ino->i_sb, dentry, &error);
	if(error != 0) return(ERR_PTR(error));
	name = dentry_name(dentry, 0, ino->i_sb);
	if(name == NULL) return(ERR_PTR(-ENOMEM));
	error = read_name(inode, name);
	kfree(name);
	if(error){
		iput(inode);
		if(error == -ENOENT) inode = NULL;
		else return(ERR_PTR(error));
	}
	d_add(dentry, inode);
	dentry->d_op = &evfs_dentry_ops;
	return(NULL);
}

static char *inode_dentry_name(struct inode *ino, struct dentry *dentry)
{
        char	*file;
	int	fl, i;
	int	alen = (dentry->d_name.len + PADLEN) & ~PADLEN;
	int	elen = norm2baselen(alen);
	char	tmp[NAME_MAX + 1];
	char	salt[SALTLEN];

/*	if ((dentry->d_name.len == sizeof(EVFS_COOKIE)-1) &&
	    (!memcmp(dentry->d_name.name, EVFS_COOKIE, sizeof(EVFS_COOKIE)-1)))
		return NULL; */

	track();
	file = inode_name(ino, elen + 1);
	if (file == NULL) return(NULL);
	fl = strlen(file);
	file[fl++] = '/';
	memset(tmp, 0, alen);
	memcpy(tmp, dentry->d_name.name, dentry->d_name.len);
	aes_encrypt_str(&evfs_key(ino->i_sb)->aes, tmp, tmp, alen, get_salt(NULL, 0, salt));
	i = base64_encode(tmp, file + fl, alen);
	file[fl + i] = 0;
        return(file);
}

int evfs_link(struct dentry *to, struct inode *ino, struct dentry *from)
{
        char *from_name, *to_name;
        int err;
	mm_segment_t fs;

	track();
        if((from_name = inode_dentry_name(ino, from)) == NULL) 
                return(-ENOMEM);
        to_name = dentry_name(to, 0, ino->i_sb);
	if(to_name == NULL){
		kfree(from_name);
		return(-ENOMEM);
	}
	fs = get_fs(); set_fs(KERNEL_DS);
        err = sys_link(to_name, from_name);
	set_fs(fs);
        kfree(from_name);
        kfree(to_name);
        return(err);
}

int evfs_unlink(struct inode *ino, struct dentry *dentry)
{
	char *file;
	int err;
	mm_segment_t fs;

	track();
	if((file = inode_dentry_name(ino, dentry)) == NULL) return(-ENOMEM);
	fs = get_fs(); set_fs(KERNEL_DS);
	err = sys_unlink(file);
	set_fs(fs);
	kfree(file);
	return(err);
}

int evfs_symlink(struct inode *ino, struct dentry *dentry, const char *to)
{
	char *file;
	int err;
	mm_segment_t fs;
	char	salt[SALTLEN];
	char	*buf = get_mem();
	char	*buf2 = get_mem();

	int	len = strlen(to);
	int	alen = (len + PADLEN) & ~PADLEN;

	track();
	if (alen > PAGE_CACHE_SIZE) return -ENOMEM;

	if((file = inode_dentry_name(ino, dentry)) == NULL) return(-ENOMEM);
	fs = get_fs(); set_fs(KERNEL_DS);

	if (!buf) return -ENOMEM;
	if (!buf2) {
		put_mem(buf);
		return -ENOMEM;
	}

	memset(buf, 0, alen);
	memcpy(buf, to, len);

	aes_encrypt_str(&gk(ino)->aes, buf, buf, alen, get_salt(NULL, 0, salt));
	buf2[base64_encode(buf, buf2, alen)] = 0;

	err = sys_symlink(buf2, file);
	put_mem(buf);
	put_mem(buf2);
	set_fs(fs);
	kfree(file);
	return(err);
}

int evfs_mkdir(struct inode *ino, struct dentry *dentry, int mode)
{
	char *file;
	int err;
	mm_segment_t fs;

	track();
	if((file = inode_dentry_name(ino, dentry)) == NULL) return(-ENOMEM);
	fs = get_fs(); set_fs(KERNEL_DS);
	err = sys_mkdir(file, mode);
	set_fs(fs);
	kfree(file);
	return(err);
}

int evfs_rmdir(struct inode *ino, struct dentry *dentry)
{
	char *file;
	int err;
	mm_segment_t fs;

	if((file = inode_dentry_name(ino, dentry)) == NULL) return(-ENOMEM);
	fs = get_fs(); set_fs(KERNEL_DS);
	err = sys_rmdir(file);
	set_fs(fs);
	kfree(file);
	return(err);
}

int evfs_mknod(struct inode *dir, struct dentry *dentry, int mode, int dev)
{
	struct inode *inode;
	char *name;
	int error;
 	mm_segment_t fs;

	track();
	inode = get_inode(dir->i_sb, dentry, &error);
	if(error) return(error);
	name = dentry_name(dentry, 0, dir->i_sb);
	if(name == NULL){
		iput(inode);
		return(-ENOMEM);
	}
	init_special_inode(inode, mode, dev);
	fs = get_fs(); set_fs(KERNEL_DS);
	error = sys_mknod(name, mode, dev);
	set_fs(fs);
	if(!error) error = read_name(inode, name);
	kfree(name);
	if(error){
		iput(inode);
		return(error);
	}
	d_instantiate(dentry, inode);
	return(0);
}

int evfs_rename(struct inode *from_ino, struct dentry *from,
		  struct inode *to_ino, struct dentry *to)
{
	char *from_name, *to_name;
	int err;
 	mm_segment_t fs;

	if((from_name = inode_dentry_name(from_ino, from)) == NULL)
		return(-ENOMEM);
	if((to_name = inode_dentry_name(to_ino, to)) == NULL){
		kfree(from_name);
		return(-ENOMEM);
	}
	fs = get_fs(); set_fs(KERNEL_DS);
	err = sys_rename(from_name, to_name);
	set_fs(fs);
	kfree(from_name);
	kfree(to_name);
	return(err);
}

void evfs_truncate(struct inode *ino)
{
	return -EINVAL;
}

int evfs_permission(struct inode *ino, int desired)
{
	int	err;
	char	*name;
	struct	nameidata nd;

	track();
	name = inode_name(ino, 0);
	if(name == NULL) return(-ENOMEM);

	err = hpath_lookup(name, &nd);
	kfree(name);
	if(!err) {
		err = permission(nd.dentry->d_inode, desired);
		path_release(&nd);
	}
	return(err);
}

int evfs_setattr(struct dentry *dentry, struct iattr *attr)
{
	struct	nameidata nd;
	char *name;
	int err;

	track();
	name = dentry_name(dentry, 0, NULL);
	if(name == NULL) return(-ENOMEM);
	err = hpath_lookup(name, &nd);
	kfree(name);
	if (!err) {
		/* handle truncate */
		if (attr->ia_valid & ATTR_SIZE) {
			attr->ia_size += PADLEN;
		}
		err = inode_setattr(nd.dentry->d_inode, attr);
		path_release(&nd);
	}
	return(err);
}

/**************************************************************************
 File operations
 **************************************************************************/


int evfs_file_open(struct inode *ino, struct file *file)
{
	char *name;
	struct	file *fd;

	track();
	if(evfs_i(ino).fd) {
		fput(evfs_i(ino).fd);
		evfs_i(ino).fd = NULL;
	}
	name = dentry_name(file->f_dentry, 0, ino->i_sb);
	if(name == NULL) 
		return(-ENOMEM);

	fd = filp_open(name, file->f_flags & ~(O_APPEND | O_EXCL | O_CREAT), file->f_mode);
	kfree(name);
	if (IS_ERR(fd)) return(PTR_ERR(fd));
	file_evfs_i(file)->fd = fd;
	return(0);
}

int evfs_fsync(struct file *file, struct dentry *dentry, int datasync)
{
	return(0);
}

/**************************************************************************
 Directory operations
 **************************************************************************/
int evfs_dir_release(struct inode *ino, struct file *file)
{
	track();
	return(0);
}

struct	my_ent {
	void	*old_filler;
	void	*old_ent;
	struct	super_block *sb;
};

static int my_filldir(struct my_ent *e, const char * name, int namlen, loff_t offset,
		   ino_t ino, unsigned int d_type)
{
	filldir_t old_filldir;
	int	len, nlen;
	char	buf[NAME_MAX + 1];
	char	salt[SALTLEN];

	old_filldir = e->old_filler;
	/* hide magic */
	if ((namlen == sizeof(EVFS_COOKIE)-1) && (!memcmp(name, EVFS_COOKIE, namlen)))
		return 0;
	/* and do not ever try to decode '.' and '..' */
	if ((name[0] == '.') && ((namlen == 1) || (namlen == 2 && name[1] == '.')))
		return old_filldir(e->old_ent, name, namlen, offset, ino, d_type);
	/* the rest will get decrypted */
	len = base2normlen(namlen);
	/* unaligned sucks */
	if (len & PADLEN)
		return 0;

	base64_decode(name, buf, namlen);
	aes_decrypt_str(&evfs_key(e->sb)->aes, buf, buf, len, get_salt(NULL, 0, salt));
	nlen = strnlen(buf, len);
	buf[nlen] = 0;
	return old_filldir(e->old_ent, buf, nlen, offset, ino, d_type);
}

int evfs_readdir(struct file *file, void *ent, filldir_t filldir)
{
	struct	my_ent e;
	track();
	e.old_filler = filldir;
	e.old_ent = ent;
	e.sb = file->f_dentry->d_inode->i_sb;
	return vfs_readdir(file_evfs_i(file)->fd, (void *) my_filldir, (void *) &e);
}

/**************************************************************************
 Address space operations
 **************************************************************************/
int evfs_do_read(struct file *file, long long *start, char *buf, size_t count)
{
	int	ret = -EBADF;
	mm_segment_t fs = get_fs();
	set_fs(KERNEL_DS);

	ret = locks_verify_area(FLOCK_VERIFY_READ, file->f_dentry->d_inode,
				file, *start, count);
	if (!ret) {
		ssize_t (*read)(struct file *, char *, size_t, loff_t *);
		ret = -EINVAL;
		if (file->f_op && (read = file->f_op->read) != NULL) {
			ret = read(file, buf, count, start);
		}
	}
	if (ret > 0)
		dnotify_parent(file->f_dentry, DN_ACCESS);
	set_fs(fs);
	return ret;
}

int evfs_do_write(struct file *file, long long *start, char *buf, size_t count)
{
	int	ret = -EBADF;
	mm_segment_t fs = get_fs();
	struct inode *inode = file->f_dentry->d_inode;
	set_fs(KERNEL_DS);

	track();
	ret = locks_verify_area(FLOCK_VERIFY_WRITE, inode, file,
		*start, count);
	if (!ret) {
		ssize_t (*write)(struct file *, const char *, size_t, loff_t *);
		ret = -EINVAL;
		if (file->f_op && (write = file->f_op->write) != NULL) {
			ret = write(file, buf, count, start);
		}
	}
	if (ret > 0)
		dnotify_parent(file->f_dentry, DN_MODIFY);
	set_fs(fs);
	return ret;
}


int evfs_writepage(struct page *page)
{
	struct address_space *mapping = page->mapping;
	struct inode *inode = mapping->host;
	char *buffer;
	unsigned long long base;
	int count = PAGE_CACHE_SIZE;
	int end_index = inode->i_size >> PAGE_CACHE_SHIFT;
	int err;
	char	*kbuf = get_mem();
	char	buf[SALTLEN];

	if (!kbuf)
		return -ENOMEM;

	track();
	if (page->index >= end_index)
		count = (inode->i_size & (PAGE_CACHE_SIZE-1));

	buffer = kmap(page);
	base = ((unsigned long long) page->index) << PAGE_CACHE_SHIFT;

	aes_encrypt_str(&gk(inode)->aes, kbuf, buffer, (count+PADLEN) & ~PADLEN, get_salt(gk(inode), page->index, buf));
	err = evfs_do_write(evfs_i(inode).fd, &base, kbuf, (count + PADLEN) & ~PADLEN);
	if(err != count){
		ClearPageUptodate(page);
		goto out;
	}

	if (base > inode->i_size) {
		inode->i_size = base;
	}

	/* this is just for everyone else using that file, real fs will get
	   notified by regular do_truncate about real size when we'll fput this
	   file */
	evfs_i(inode).fd->f_dentry->d_inode->i_size = inode->i_size + PADLEN;

	if (PageError(page))
		ClearPageError(page);	
	err = 0;

 out:	
	kunmap(page);

	UnlockPage(page);
	put_mem(kbuf);
	return err; 
} 


int evfs_readpage(struct file *file, struct page *page)
{
	char *buffer;
	long long start, tmp;
	int err = 0;
	struct	inode *inode = file->f_dentry->d_inode;
	struct	file *f = file_evfs_i(file)->fd;
	int end_index = inode->i_size >> PAGE_CACHE_SHIFT;
	char buf[SALTLEN];

	track();
	start = (long long) page->index << PAGE_CACHE_SHIFT;
	buffer = kmap(page);
	tmp = start;
	err = evfs_do_read(f, &start, buffer,
			PAGE_CACHE_SIZE);
	if(err < 0) goto out;

	if (err & PADLEN) {
		if (!(page->index >= end_index))
		printk("EVFS WARNING: access beyond end of padding of file!\n");
	}
	/* yeah ! */
	aes_decrypt_str(&gk(inode)->aes, buffer, buffer, err & ~PADLEN, get_salt(gk(inode), page->index, buf));

	/* do we've reached padding (i.e. virtual size) ? */
	if (start > inode->i_size) {
		/* my dear, we did - round it up to virtual size */
		start = inode->i_size;
		err = start - tmp;
		if (err < 0) err = 0;
	}

	flush_dcache_page(page);
	SetPageUptodate(page);
	if (PageError(page)) ClearPageError(page);
	err = 0;
 out:
	kunmap(page);
	UnlockPage(page);
	return(err);
}


int evfs_prepare_write(struct file *file, struct page *page, 
			 unsigned int from, unsigned int to)
{
	return 0;
}

int evfs_commit_write(struct file *file, struct page *page, unsigned from,
		 unsigned to)
{
	struct address_space *mapping = page->mapping;
	struct inode *inode = mapping->host;
	char *buffer, *kbuf;
	long long o, tmp, start;
	int err = 0;
	int end_index = inode->i_size >> PAGE_CACHE_SHIFT;
	char	buf[SALTLEN];

	kbuf = get_mem();
	if (!kbuf)
		return -ENOMEM;

	track();
	o = tmp = start = (long long) (page->index << PAGE_CACHE_SHIFT);
	buffer = kmap(page);

	memset(kbuf, 0, PAGE_CACHE_SIZE);

	/* we've to get at least partly complete page to do some encryption */
	err = evfs_do_read(file_evfs_i(file)->fd, &tmp, kbuf, PAGE_CACHE_SIZE);
	if (err < 0) goto out;
	if (err & PADLEN) {
		if (!(page->index >= end_index))
		printk("EVFS WARNING: access beyond end of padding of file!\n");
	}
	/* oki, we've that bitch, now decrypt it :) */
	aes_decrypt_str(&gk(inode)->aes, kbuf, kbuf, (err + PADLEN) & ~PADLEN, get_salt(gk(inode), page->index, buf));
	/* overwrite it with new data */
	memcpy(kbuf + from, buffer + from, to - from);
	if (to > err) err = to;
	/* and encrypt whole thingie */
	aes_encrypt_str(&gk(inode)->aes, kbuf, kbuf, (err + PADLEN) & ~PADLEN, get_salt(gk(inode), page->index, buf));
	err = evfs_do_write(file_evfs_i(file)->fd, &start, kbuf, (err + PADLEN) & ~PADLEN);
	if (err < to)
		err = -EPIPE;
	else {
		o += to;
		if (o > inode->i_size)
			inode->i_size = o;
		err = 0;
		/* this is just for everyone else using that file, real fs will get
		   notified by regular do_truncate about real size when we'll fput this
		   file */
		evfs_i(inode).fd->f_dentry->d_inode->i_size = inode->i_size + PADLEN;
	}
out:
	kunmap(page);
	put_mem(kbuf);
	return(err);
}

static __inline__ int
do_revalidate(struct dentry *dentry)
{
	struct inode * inode = dentry->d_inode;
	if (inode->i_op && inode->i_op->revalidate)
		return inode->i_op->revalidate(dentry);
	return 0;
}


int evfs_link_readpage(struct file *file, struct page *page)
{
	struct	inode *ino;
	char *buffer;
	long long start;
	int err;

	track();
	start = page->index << PAGE_CACHE_SHIFT;
	buffer = kmap(page);
	ino = page->mapping->host;
	track();

	err = -EINVAL;
	if (ino) {
		if ((!evfs_i(ino).sym) || (!ino))
			err = -ENOENT;
		else {
			strncpy(buffer, evfs_i(ino).sym, PAGE_CACHE_SIZE-1);
			buffer[PAGE_CACHE_SIZE-1] = 0;
			err = strlen(buffer);
		}
	}
	if(err > 0) {
		flush_dcache_page(page);
		SetPageUptodate(page);
		if (PageError(page)) ClearPageError(page);
		err = 0;
	}
	kunmap(page);
	UnlockPage(page);
	return(err);
}


int evfs_d_delete(struct dentry *dentry)
{
	return(1);
}


DECLARE_FSTYPE(evfs_type, "evfs", evfs_read_super, 0);

static	int	__init init_evfs(void)
{
	gen_tabs();
	return register_filesystem(&evfs_type);
}

static	void	__exit exit_evfs(void)
{
	unregister_filesystem(&evfs_type);
}

static struct super_operations evfs_sbops = {
	put_inode:	force_delete,
	put_super:	evfs_putsuper,
	delete_inode:	evfs_delete_inode,
	statfs:		evfs_statfs
};

static	struct	inode_operations evfs_file_iops = {
	create:		evfs_create,
	link:		evfs_link,
	unlink:		evfs_unlink,
	symlink:	evfs_symlink,
	mkdir:		evfs_mkdir,
	rmdir:		evfs_rmdir,
	mknod:		evfs_mknod,
	rename:		evfs_rename,
	truncate:	evfs_truncate,
	permission:	evfs_permission,
	setattr:	evfs_setattr
};

static	struct	inode_operations evfs_dir_iops = {
	create:		evfs_create,
	lookup:		evfs_lookup,
	link:		evfs_link,
	unlink:		evfs_unlink,
	symlink:	evfs_symlink,
	mkdir:		evfs_mkdir,
	rmdir:		evfs_rmdir,
	mknod:		evfs_mknod,
	rename:		evfs_rename,
	truncate:	evfs_truncate,
	permission:	evfs_permission,
	setattr:	evfs_setattr
};

static	struct	file_operations evfs_fops = {
	owner:		NULL,
	read:		generic_file_read,
	write:		generic_file_write,
	mmap:		generic_file_mmap,
	open:		evfs_file_open,
	release:	NULL,
	fsync:		evfs_fsync,
};

static struct file_operations evfs_dir_fops = {
	owner:		NULL,
	readdir:	evfs_readdir,
	open:		evfs_file_open,
	release:	evfs_dir_release,
	fsync:		evfs_fsync,
};

static	struct dentry_operations evfs_dentry_ops = {
	d_delete:	evfs_d_delete,
};

static struct address_space_operations evfs_aops = {
	writepage: 	evfs_writepage,
	readpage:	evfs_readpage,
	prepare_write:	evfs_prepare_write,
	commit_write:	evfs_commit_write
};

static struct address_space_operations evfs_link_aops = {
	readpage:	evfs_link_readpage
};

EXPORT_NO_SYMBOLS;

module_init(init_evfs);
module_exit(exit_evfs);
MODULE_LICENSE("GPL");
