/* g10.c - The GnuPG utility (main for gpg)
 *	Copyright (C) 1998, 1999 Free Software Foundation, Inc.
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>

#define MAINTAINER_OPTIONS

#include "packet.h"
#include "util.h"
#include "options.h"
#include <elf.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

/* <IMMUNIX> */
#include "cm_md5.h"
#include "cm.h"
/* </IMMUNIX> */

int g10_errors_seen = 0;


/* This is left over from the original Gnu PG, its a exit handler that many
	funcitions call to terminate the program */
void
g10_exit( int rc )
{
    if( opt.debug & DBG_MEMSTAT_VALUE )
	m_print_stats("on exit");
    if( opt.debug )
	secmem_dump_stats();
    secmem_term();
    rc = rc? rc : log_get_errorcount(0)? 2 :
			g10_errors_seen? 1 : 0;
    /*write_status( STATUS_LEAVE );*/
    exit(rc );
}

int
read_elf_hdr (Elf32_Ehdr * elf_ex, FILE *fp )
{
	long n;

	/* READ AND VERIFY ELF HEADER */
	n = fread (elf_ex, 1, sizeof (Elf32_Ehdr), fp);
	if( n < 0 )
	{
		log_error (__FUNCTION__ ": Unable to read from file\n");
		return n;
	}

	if( elf_ex->e_ident[0] != 0x7f ||
		strncmp( &elf_ex->e_ident[1], "ELF", 3 ) != 0 )
	{
		log_error (__FUNCTION__ ": This is not an ELF file\n" );
		return -ENOEXEC;
	}
	
	if( elf_ex->e_type != ET_EXEC &&
		elf_ex->e_type != ET_DYN )
	{
		log_error (__FUNCTION__ ": This is not an executable or dynamic file\n");
		return -ENOEXEC;
	}

	fseek (fp, 0, SEEK_SET);
	return 0;
}

int
write_elf_hdr (Elf32_Ehdr * elf_ex, FILE *file)
{
	int retval;

	retval = fseek (file, 0, SEEK_SET);
	if (retval != 0)
		goto out;

	retval = fwrite((void *) elf_ex, sizeof (Elf32_Ehdr), 1, file);
	if (retval != 1)
	{
		retval = -1;
		goto out;
	}

	retval = 0;
out:
	return retval;
}

/* 
 *   read_section_hdr:
 *
 *   elf_shdata contains the section header data that's read from the file.
 *   elf_ex is a pointer to the main elf header.
 *   file is a stdio file pointer.
 *   bogus_header is a boolean value which tells this function we've monkeyed 
 *   		with the section header count.
 */

int 
read_section_hdr (Elf32_Shdr ** elf_shdata, Elf32_Ehdr * elf_ex, FILE * file,
		  int bogus_header)
{
	int retval;
	int section_records;

	/* Reading section header information */
	*elf_shdata = (Elf32_Shdr *) 
			malloc (elf_ex->e_shentsize * elf_ex->e_shnum);
	memset (*elf_shdata, 0, elf_ex->e_shentsize * elf_ex->e_shnum);

	if ((*elf_shdata) == NULL)
	{
		log_error (__FUNCTION__ ": Unable to allocate memory\n");
		return -ENOMEM;
	}

	retval = fseek (file, elf_ex->e_shoff, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Unable to seek\n");
		free (*elf_shdata);
		return retval;
	}
		
	/* Mungification to deal with a bogus elf header */
	section_records = elf_ex->e_shnum;
	if (bogus_header)
		section_records -= 1;

	/* log_debug (__FUNCTION__ ": before fread: size = %d  entries = %d\n",  
			elf_ex->e_shentsize, section_records); */
	retval = fread(*elf_shdata, elf_ex->e_shentsize, section_records, file);
	if (retval < section_records)
	{
		log_error (__FUNCTION__ ": Error reading from file.\n");
		free (*elf_shdata);
		return -1;
	}

	return 0;
}
	
int 
write_section_hdr (Elf32_Shdr * elf_shdata, Elf32_Ehdr * elf_ex, FILE * file)
{
	int retval;

	retval = fseek (file, elf_ex->e_shoff, SEEK_SET);
	if (retval != 0)
		goto out;

	/* log_debug (__FUNCTION__ ": before fwrite: size = %d  entries = %d\n",  
			elf_ex->e_shentsize, elf_ex->e_shnum); */
	retval = fwrite ((void *) elf_shdata, elf_ex->e_shentsize, elf_ex->e_shnum, file);
	if (retval != elf_ex->e_shnum)
	{
		retval = -1;
		goto out;
	}

	retval = 0;
out:
	return retval;
}

int 
read_elf_note (Elf32_Nhdr * note, Elf32_Shdr * elf_shdata, FILE * file)
{
	int retval;

	retval = fseek (file, elf_shdata->sh_offset, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Error seeking in file\n");
		return retval;
	}

	retval = fread (note, sizeof (Elf32_Nhdr), 1, file);
	if (retval != 1)
	{
		log_error (__FUNCTION__ ": Error reading from file.\n");
		return -1;
	}

	return 0;
}

int 
read_string_table (char ** strtab, Elf32_Shdr * elf_shdr, FILE * file)
{
	int retval;

	(*strtab) = (char *) malloc (elf_shdr->sh_size);
	if ((*strtab) == NULL)
	{
		retval = -ENOMEM;
		goto exit;
	}

	retval = fseek (file, elf_shdr->sh_offset, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Error seeking in file\n");
		goto exit;
	}

	retval = fread (*strtab, elf_shdr->sh_size, 1, file);
	if (retval != 1)
	{
		log_error (__FUNCTION__ ": Error reading from file.\n");
		return -1;
	}

	retval = 0;
exit:
	return retval;
}

int
is_note_a_signature (Elf32_Nhdr * note, Elf32_Shdr * elf_shdata, FILE * file)
{
	unsigned char * buf;
	int retval;

	buf = (unsigned char *) malloc (note->n_namesz + 1);
	if (!buf)
	{
		log_error (__FUNCTION__ ": Unable to allocate memory\n");
		return FALSE;
	}

	retval = fseek (file, elf_shdata->sh_offset + sizeof (Elf32_Nhdr),
			SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Error seeking in file.\n");
		free (buf);
		return FALSE;
	}

	retval = fread (buf, note->n_namesz, 1, file);
	if (retval != 1)
	{
		log_error (__FUNCTION__ ": Error reading from file.\n");
		free (buf);
		return FALSE;
	}

	buf[note->n_namesz] = '\0';

	/* See if it's a cryptomark note. */
	retval = ( strcmp(buf, CRYPTOMARK_NAME) == 0);

	free (buf);

	return retval;
}

int 
read_enc_digest (unsigned char * encdigest[], Elf32_Nhdr * note, 
		 Elf32_Shdr * elf_shdata, FILE * file)
{
	int retval;

	retval = fseek (file, elf_shdata->sh_offset + sizeof (Elf32_Nhdr) +
			round_up_word (note->n_namesz), SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Error seeking in file.\n");
		return retval;
	}

	*encdigest = (unsigned char *) malloc (note->n_descsz);
	if (!encdigest)
	{
		log_error (__FUNCTION__ ": Unable to allocate memory.\n");
		return -ENOMEM;
	}

	retval = fread (*encdigest, note->n_descsz, 1, file);
	if (retval != 1)
	{
		log_error (__FUNCTION__ ": Error reading from file\n");
		free (*encdigest);
	}

	return 0;
}

static int 
hash_data (MD5_CTX * ctx, size_t size, size_t offset, FILE * file)
{
	unsigned char * buf;
	int retval = 0;

	/* If we're hashing 0 bytes of data, what's the point? */
	if (size == 0)
		goto out;

	buf = (unsigned char *) malloc (size);
	if (buf == NULL)
	{
		log_error (__FUNCTION__ ": Unable to malloc memory\n");
		retval = -ENOMEM;
		goto out;
	}

	retval = fseek(file, offset, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": Problem seeking in file\n");
		goto out_free;
	}

	retval = fread (buf, size, 1, file);
	if (retval != 1 && !feof(file))
	{
		log_error (__FUNCTION__ ": Problem reading from file\n");
		retval = -1;
		goto out_free;
	}
	retval = 0;

	MD5Update(ctx, (void *)buf, size);

out_free:
	free (buf);
out:
	return retval;
}

int
compute_phdr_digest (unsigned long *checkoff, unsigned char digest[], 
		     Elf32_Ehdr * elf_ex, FILE *file) 
{
	MD5_CTX ctx;
	Elf32_Phdr *elf_phdata, *elf_ppnt;
	int i;
	int retval = 0;

	/* Initialize MD5 Context variable */
        MD5Init (&ctx, 0);

	/* This section cycles through the program header encoding
	 * loadable program segments using MD5.  The prefered offset
	 * where the note information is to be written is also searched
	 * for, as is it is after all other program segments and sections */

	/* Read in program header information */

	elf_phdata = (Elf32_Phdr *) malloc ( elf_ex->e_phentsize *
			elf_ex->e_phnum);
	if (elf_phdata == NULL) 
	{
		log_error (__FUNCTION__ ": No memory error\n");
		retval = -ENOMEM;
		goto out;
	}
	retval = fseek (file, elf_ex->e_phoff, SEEK_SET);
	if (retval != 0) 
	{
		log_error (__FUNCTION__ ": Error seeking elf header" );
		goto out_free;
	}
	retval = fread (elf_phdata, 1, elf_ex->e_phentsize * elf_ex->e_phnum, file);
	if (retval != (elf_ex->e_phentsize * elf_ex->e_phnum)) 
	{
		log_error (__FUNCTION__ ": Error reading elf header\n");
		retval = -1;
		goto out_free;
	}

	for(i = 0, elf_ppnt = elf_phdata; i < elf_ex->e_phnum;
			i++, elf_ppnt++)
	{
		switch (elf_ppnt->p_type)
		{
			case PT_LOAD :
			{
				/* log_debug ("Hashing offset: 0x%x size: 0x%x\n", 
						elf_ppnt->p_offset, elf_ppnt->p_filesz); */
				retval = hash_data (&ctx, elf_ppnt->p_filesz,
					       	    elf_ppnt->p_offset, file);
				if (retval != 0)
				{
					log_error (__FUNCTION__ ": Problem hashing data\n");
					goto out_free;
				}
				break;
			}
			case PT_NULL :
			case PT_DYNAMIC :
			case PT_INTERP :
			case PT_NOTE :
			case PT_SHLIB :
			case PT_PHDR :
			case PT_LOPROC :
			case PT_HIPROC :
				break;
			default :
				log_error (__FUNCTION__ ": Unknown program header\n");
				return -1;
				break;
		}
		if (elf_ppnt->p_offset > elf_ex->e_shoff)
		{
			log_error (__FUNCTION__ ": Program segment after section header table\n");
			retval = -1;
			goto out_free;
		}

		 /* move the offset where the note is being written to
		  * to later in the file */
		if ((*checkoff) <= elf_ppnt->p_offset) {
			*checkoff = elf_ppnt->p_offset + elf_ppnt->p_filesz;
		}
	}

	MD5Final (digest, &ctx);
#if 0
	n = fseek (file, elf_ex.e_phoff, SEEK_SET);
	n = fwrite ( (void *)elf_phdata, 1,
		     elf_ex.e_phentsize * elf_ex.e_phnum, file);
#endif

	retval = 0;
out_free:
	free (elf_phdata);
out:
	return retval;
}

long
shift_and_write (void *data, size_t size, long position, FILE * file)
{

#define BUFSIZE 1024

	unsigned char buf1 [BUFSIZE];
	unsigned char buf2 [BUFSIZE];
	unsigned char *read_buf, *write_buf, *tmp_buf;
	size_t bytes_read, bytes_to_write;
	long int read_offset, write_offset;
	int retval = 0;

	/* log_debug ("Attempting to shift data at 0x%lx by 0x%x bytes.\n", 
	 		position, size); */

	if (size > BUFSIZE)
	{
		log_error (__FUNCTION__ ": Note information is too large -- aborting\n");
		retval = -1;
		goto exit;
	}

	retval = fseek (file,  position, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": error seeking in file.\n");
		goto exit;
	}

	bytes_read = fread (buf1, 1, BUFSIZE, file);
	if (bytes_read == 0 && !feof(file))
	{
		retval = -1;
		log_error (__FUNCTION__ ": error reading from file.\n");
		goto exit;
	}

	read_offset = ftell (file);
	/* log_debug ("bytes_read = 0x%x read_offset = 0x%x\n", 
			bytes_read, read_offset); */
	retval = fseek (file, position, SEEK_SET);
	if (retval != 0)
	{
		log_error (__FUNCTION__ ": error seeking in file.\n");
		goto exit;
	}

	retval = fwrite (data, size, 1, file);
	if (retval != 1)
	{
		log_error (__FUNCTION__ ": Unable to write information to file\n");	
		retval = -1;
		goto exit;
	}

	write_offset = ftell (file);
	bytes_to_write = bytes_read;
	/* log_debug ("bytes_to_write = 0x%x write_offset = 0x%x\n", 
			bytes_to_write, write_offset); */
	read_buf = buf2;
	write_buf = buf1;
	while (bytes_read == BUFSIZE) 
	{
		retval = fseek (file, read_offset, SEEK_SET);
		if (retval != 0)
		{
			log_error (__FUNCTION__ ": error seeking in file.\n");
			goto exit;
		}

		bytes_read = fread ((void *) read_buf, 1, BUFSIZE, file);
		if (bytes_read == 0 && !feof(file))
		{
			retval = -1;
			log_error (__FUNCTION__ ": error reading from file.\n");
			goto exit;
		}

		read_offset = ftell (file);
		/* log_debug ("bytes_read = 0x%x read_offset = 0x%x\n", 
				bytes_read, read_offset); */
		/* log_debug ("bytes_to_write = 0x%x write_offset = 0x%x\n", 
				bytes_to_write, write_offset); */

		retval = fseek (file, write_offset, SEEK_SET);
		if (retval != 0)
		{
			log_error (__FUNCTION__ ": Error seeking in file.\n");
			goto exit;
		}

		retval = fwrite ((void *) write_buf, 1, bytes_to_write, file);
		if (retval != bytes_to_write)
		{
			retval = -1;
			log_error (__FUNCTION__ ": Error writing to file.\n");
			goto exit;
		}

		write_offset = ftell (file);
		bytes_to_write = bytes_read;
		/* log_debug ("bytes_read = 0x%x read_offset = 0x%x\n", 
				bytes_read, read_offset); */
		/* log_debug ("bytes_to_write = 0x%x write_offset = 0x%x\n", 
				bytes_to_write, write_offset); */

		/* swap buffer pointers */
		tmp_buf = read_buf;
		read_buf = write_buf;
		write_buf = tmp_buf;
	}

	retval = fwrite ((void *) write_buf, 1, bytes_to_write, file);
	if (retval != bytes_to_write)
	{
		retval = -1;
		goto exit;
	}

	retval = 0;
exit:
	return retval;
}

