/*======================================================================
*
*  Immunix CryptoMark module
*  Copyright 1998, 1999, 2000 Wirex Communications &
* 			Oregon Graduate Institute
*
*	Greg Kroah-Hartman <greg@wirex.com>
*
*======================================================================*/

/* Module versioning code  */
#include <linux/autoconf.h> 		/* Retrieve the CONFIG_* macros */
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
	#define MODVERSIONS 		/* Force it on */
	#include <linux/modversions.h>
#endif



/* __NO_VERSION__ is needed or else symbols get defined multiply */
#define __NO_VERSION__
#include <linux/version.h>
//#include <linux/module.h>

#include <asm/uaccess.h>
#include <linux/immunix.h>
#include <linux/proc_fs.h>

#include "cryptomark.h"
#include "cryptomark_log.h"
#include "cryptomark_version.h"


#ifdef CONFIG_PROC_FS
static void __proc_init (void);
static void __proc_done (void);

static struct proc_dir_entry	*root;


#endif


unsigned char	*cryptomark_key;	/* loader will set to NULL when module is loaded */
unsigned long	cryptomark_key_length;

int cryptomark_add_key_data (unsigned char *data, unsigned long length)
{
	unsigned long	count;
#ifdef CONFIG_CRYPTOMARK_DEBUG
	int		i;
#endif

	dbg (__FUNCTION__" - length = %ld", length);

	if (cryptomark_key == NULL) {
		cryptomark_key = (unsigned char *)__get_free_page(GFP_KERNEL);
		if (!cryptomark_key) {
			err (__FUNCTION__" - out of memory.");
			return -ENOMEM;
		}
		cryptomark_key_length = 0;
	}

	count = MIN(length, PAGE_SIZE-cryptomark_key_length);

	memcpy (&cryptomark_key[cryptomark_key_length], data, count);

	cryptomark_key_length += count;

#ifdef CONFIG_CRYPTOMARK_DEBUG
	printk (KERN_DEBUG "CryptoMark: "__FUNCTION__" - cryptomark_key_length = %ld, cryptomark_key = ", cryptomark_key_length);
	for (i = 0; i < cryptomark_key_length; ++i) {
		printk ("%.2x ", cryptomark_key[i]);
	}
	printk ("\n");
#endif;

	return count;
}


/*======================================================================
*
*
*======================================================================*/
void proc_init (void)
{
	dbg (__FUNCTION__);
#ifdef CONFIG_PROC_FS
	__proc_init();
#else
	dbg (__FUNCTION__": CONFIG_PROC_FS is not enabled, so no /proc");
#endif
}


/*======================================================================
*
*
*======================================================================*/
void proc_done (void)
{
	dbg (__FUNCTION__);
#ifdef CONFIG_PROC_FS
	__proc_done();
#else
	dbg (__FUNCTION__": CONFIG_PROC_FS is not enabled, so no /proc");
#endif
}

#ifdef CONFIG_PROC_FS


/*======================================================================
*
*
*======================================================================*/
static int proc_version_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;

	len += sprintf (page + len,
			"%s\n"
			"Version " CRYPTOMARK_VERSION "\n"
			"For the Immunix Scaffold " IMMUNIX_SCAFFOLD_VERSION"\n"
			"%s\n"
			"Compiled on " __DATE__ " for kernel " UTS_RELEASE
#ifdef __SMP__
		      " (SMP)"
#endif
#ifdef MODVERSIONS
		      " with versioned symbols"
#endif
#ifdef CONFIG_SECURE_STACK
		      "\n\t\t\tand non-executable stack"
#endif
		      ".\n",
			cryptomark_title, cryptomark_copyright);

	return len;
}


/*======================================================================
*
*
*======================================================================*/
static int proc_key_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;

	len += sprintf (page + len, "Like I would just give up my key to you that easy...\n");
	return len;
}


/*======================================================================
*
*
*======================================================================*/
static int proc_key_write (struct file *file, const char *buffer, unsigned long count, void *data)
{
	unsigned char *key;
	int result;
#ifdef CONFIG_CRYPTOMARK_DEBUG
	int i;
#endif
	
	dbg (__FUNCTION__" - count = %ld", count);
	dbg (__FUNCTION__" - PAGE_SIZE = %ld", PAGE_SIZE);
	
	if (count == 0) {
		return 0;
	}
	
	key = (unsigned char *)__get_free_page(GFP_KERNEL);
	if (!key) {
		err (__FUNCTION__" - out of memory.");
		return -ENOMEM;
	}
	
	copy_from_user(key, buffer, count = MIN(count, PAGE_SIZE));
	
#ifdef CONFIG_CRYPTOMARK_DEBUG
	printk (KERN_DEBUG "CryptoMark: "__FUNCTION__" - data = ");
	for (i = 0; i < count; ++i) {
		printk ("%.2x ", key[i]);
	}
	printk ("\n");
#endif

	result = cryptomark_add_key_data (key, count);
	if (result != count) {
		err(__FUNCTION__" - key too big for internal buffer, things are probably messed up now.");
	}

	free_page ((unsigned long)key);

	return count;
}


/*======================================================================
*
*
*======================================================================*/
static void __proc_init (void)
{
	struct proc_dir_entry	*p;

	root = create_proc_entry ("cryptomark", S_IFDIR | S_IRUGO | S_IXUGO, NULL);
	p = create_proc_entry ("version", S_IFREG | S_IRUGO, root);
	p->read_proc = proc_version_read;

	p = create_proc_entry ("key", S_IFREG | S_IRUGO, root);
	p->read_proc = proc_key_read;
	p->write_proc = proc_key_write;
}


/*======================================================================
*
*
*======================================================================*/
static void __proc_done (void)
{
	remove_proc_entry ("version", root);
	remove_proc_entry ("cryptomark", NULL);

	/* clean up our key if we have created one */
	if (cryptomark_key != NULL) {
		free_page ((unsigned long)cryptomark_key);
		cryptomark_key = NULL;
	}

}



#endif	/* CONFIG_PROC_FS */

