/*
 * Copyright (C) 2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: snapshot.c,v 1.2 2003/03/22 18:41:16 guenter Exp $
 *
 * snapshot implementation
 *
 * code taken from:
 *    totem     Bastien Nocera <hadess@hadess.net>
 *    transcode Copyright (C) Thomas Oestreich - June 2001
 *    enix      enix.berlios.de
 */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <inttypes.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "gtkxine.h"
#include "globals.h"
#include "utils.h"

static void yuy2toyv12 (uint8_t *y, uint8_t *u, uint8_t *v, uint8_t *input,
			int width, int height) {
  int    i, j, w2;
  
  w2 = width / 2;
  
  for (i = 0; i < height; i += 2) {
    for (j = 0; j < w2; j++) {
      /*
       * packed YUV 422 is: Y[i] U[i] Y[i+1] V[i] 
       */
      *(y++) = *(input++);
      *(u++) = *(input++);
      *(y++) = *(input++);
      *(v++) = *(input++);
    }
    
    /*
     * down sampling 
     */
    
    for (j = 0; j < w2; j++) {
      /*
       * skip every second line for U and V 
       */
      *(y++) = *(input++);
      input++;
      *(y++) = *(input++);
      input++;
    }
  }
}

/*
 *   Create rgb data from yv12
 */

#define clip_8_bit(val)                                                        \
{                                                                              \
   if (val < 0)                                                                \
      val = 0;                                                                 \
   else                                                                        \
      if (val > 255) val = 255;                                                \
}

static uint8_t * yv12torgb (uint8_t *src_y, uint8_t *src_u, uint8_t *src_v,
			    int width, int height) {

  int     i, j;
  
  int     y, u, v;
  int     r, g, b;
  
  int     sub_i_uv;
  int     sub_j_uv;

  int     uv_width, uv_height;
  
  guchar *rgb;
  
  uv_width  = width / 2;
  uv_height = height / 2;

  rgb = malloc (width * height * 3);
  if (!rgb)
    return NULL;
  
  for (i = 0; i < height; ++i) {
    /*
     * calculate u & v rows 
     */
    sub_i_uv = ((i * uv_height) / height);
    
    for (j = 0; j < width; ++j) {
      /*
       * calculate u & v columns 
       */
      sub_j_uv = ((j * uv_width) / width);
      
      /***************************************************
       *
       *  Colour conversion from 
       *    http://www.inforamp.net/~poynton/notes/colour_and_gamma/ColorFAQ.html#RTFToC30
       *
       *  Thanks to Billy Biggs <vektor@dumbterm.net>
       *  for the pointer and the following conversion.
       *
       *   R' = [ 1.1644         0    1.5960 ]   ([ Y' ]   [  16 ])
       *   G' = [ 1.1644   -0.3918   -0.8130 ] * ([ Cb ] - [ 128 ])
       *   B' = [ 1.1644    2.0172         0 ]   ([ Cr ]   [ 128 ])
       *
       *  Where in xine the above values are represented as
       *
       *   Y' == image->y
       *   Cb == image->u
       *   Cr == image->v
       *
       ***************************************************/
      
      y = src_y[(i * width) + j] - 16;
      u = src_u[(sub_i_uv * uv_width) + sub_j_uv] - 128;
      v = src_v[(sub_i_uv * uv_width) + sub_j_uv] - 128;
      
      r = (1.1644 * y) + (1.5960 * v);
      g = (1.1644 * y) - (0.3918 * u) - (0.8130 * v);
      b = (1.1644 * y) + (2.0172 * u);
      
      clip_8_bit (r);
      clip_8_bit (g);
      clip_8_bit (b);
      
      rgb[(i * width + j) * 3 + 0] = r;
      rgb[(i * width + j) * 3 + 1] = g;
      rgb[(i * width + j) * 3 + 2] = b;
    }
  }
  
  return rgb;
}

void make_snapshot (char *fname) {

  uint8_t   *yuv, *y, *u, *v, *rgb;
  int        width, height, ratio, format;
  GdkPixbuf *pixbuf, *scaled;
  double     desired_ratio, image_ratio, f;

  gtk_xine_get_current_frame (GTK_XINE(gtx), &width, &height, &ratio, &format, NULL);

  yuv = malloc ((width+8) * (height+1) * 2);

  gtk_xine_get_current_frame (GTK_XINE(gtx), &width, &height, &ratio, &format,
			      yuv);

  /*
   * convert to yv12 if necessary
   */

  switch (format) {
  case XINE_IMGFMT_YUY2:
    {
      uint8_t *yuy2 = yuv;

      yuv = malloc (width * height * 2);
      y = yuv;
      u = yuv + width * height;
      v = yuv + width * height * 5 / 4;

      yuy2toyv12 (y, u, v, yuy2, width, height);
      
      free (yuy2);
    }
    break;
  case XINE_IMGFMT_YV12:
    y = yuv;
    u = yuv + width * height;
    v = yuv + width * height * 5 / 4;

    break;
  default:
    printf ("snapshot: error - format '%.4s' unsupported\n",
	    (char *) &format);
    abort();
  }

  /*
   * convert to rgb
   */

  rgb = yv12torgb (y, u, v, width, height);

  pixbuf = gdk_pixbuf_new_from_data (rgb,
				     GDK_COLORSPACE_RGB, FALSE,
				     8, width, height, 3 * width,
				     (GdkPixbufDestroyNotify) g_free, NULL);

  image_ratio = (double) width / (double) height;
    
  switch (ratio) {
  case XINE_VO_ASPECT_ANAMORPHIC:  /* anamorphic     */
  case XINE_VO_ASPECT_PAN_SCAN:    /* we display pan&scan as widescreen */
    desired_ratio = 16.0 /9.0;
    break;
  case XINE_VO_ASPECT_DVB:         /* 2.11:1 */
    desired_ratio = 2.11/1.0;
    break;
  case XINE_VO_ASPECT_SQUARE:      /* square pels */
  case XINE_VO_ASPECT_DONT_TOUCH:  /* probably non-mpeg stream => don't touch aspect ratio */
    desired_ratio = image_ratio;
    break;
  default:
    printf ("vo_scale: unknown/vorbidden aspect ratio (%d) in stream => using 4:3\n",
	    ratio);
  case XINE_VO_ASPECT_4_3:         /* 4:3             */
    desired_ratio = 4.0 / 3.0;
    break;
  }

  f = desired_ratio / image_ratio;
    
  if (f >= 1.0)
    scaled = gdk_pixbuf_scale_simple (pixbuf,
				      (int) ((double) width * f), height,
				      GDK_INTERP_BILINEAR);
  else
    scaled = gdk_pixbuf_scale_simple (pixbuf,
				      width, (int) ((double) height / f),
				      GDK_INTERP_BILINEAR);
    
  gdk_pixbuf_unref (pixbuf);
    
  /* 
   * save pixbuf to file
   */

  if (!fname) 
    fname = modal_file_dialog ("Enter filename for snapshot:", "*.png");

  if (fname)
    gdk_pixbuf_save (scaled, fname, "png", NULL, NULL);

  gdk_pixbuf_unref (scaled);
  free (yuv);

}
