/* 
 * Copyright (C) 2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: key_events.c,v 1.13 2003/03/14 20:47:01 guenter Exp $
 *
 * key event handler and keymap editor
 */

#include "config.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "key_events.h"

#include "actions.h"
#include "utils.h"
#include "xmlparser.h"

/*
#define LOG
*/

static int             is_visible;
static GtkListStore   *kb_store;
static GtkWidget      *kb_edit_dlg, *tree_view;

typedef struct {
  guint   keyval;
  guint   state;
  gchar  *cmd;
} key_binding_t;

gint key_cb (GtkWidget *win, GdkEventKey *event, gpointer w) {

  GtkTreeIter iter;

  /* small hack for those who want to exit fullscreen mode using ctrl+f */
  if ( (event->keyval==GDK_f) && (event->state & GDK_CONTROL_MASK) ) {
    action_exec ("set_fullscreen();", NULL, NULL);
    return TRUE;
  }

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), &iter)) {

    gint state;

    /* filter out numlock etc */
    state = event->state & ~GDK_LOCK_MASK;
    state = state & ~GDK_MOD2_MASK;
    state = state & ~GDK_MOD5_MASK;

    do {

      GValue v;
      key_binding_t *key_binding;

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store),
				&iter, 2, &v);
      key_binding = g_value_peek_pointer (&v);
      g_value_unset (&v);

      if ( (key_binding->keyval == event->keyval)
	   && (key_binding->state == state) ) {
	action_exec (key_binding->cmd, NULL, NULL);
	return TRUE;
      }

    } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store),
				       &iter));
  }

#ifdef LOG
  printf ("key_events: key %d not bound\n", event->keyval);
#endif

  return FALSE;
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  is_visible = FALSE;
  gtk_widget_unmap (kb_edit_dlg);

  return TRUE;
}

static key_binding_t *find_key_binding (gchar *cmd, GtkTreeIter *iter) {

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), iter)) {

    do {

      GValue v;
      key_binding_t *key_binding;

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store),
				iter, 2, &v);
      key_binding = g_value_peek_pointer (&v);
      g_value_unset (&v);

      if (!strcasecmp (key_binding->cmd, cmd))
	return key_binding;

    } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store),
				       iter));
  }

  return NULL;
}

gchar* kb2str (key_binding_t *kb) {

  static const gchar text_shift[]   = "<Shift>";
  static const gchar text_control[] = "<Control>";
  static const gchar text_mod1[]    = "<Alt>";
  static const gchar text_mod2[]    = "<Mod2>";
  static const gchar text_mod3[]    = "<Mod3>";
  static const gchar text_mod4[]    = "<Mod4>";
  static const gchar text_mod5[]    = "<Mod5>";
  guint              l, p;
  gchar             *name;

  /*
   * calc length of resulting string
   */

  l = 2;
  if (kb->state & GDK_SHIFT_MASK) 
    l += sizeof (text_shift) - 1;
  if (kb->state & GDK_CONTROL_MASK) 
    l += sizeof (text_control) - 1;
  if (kb->state & GDK_MOD1_MASK) 
    l += sizeof (text_mod1) - 1;
  if (kb->state & GDK_MOD2_MASK) 
    l += sizeof (text_mod2) - 1;
  if (kb->state & GDK_MOD3_MASK) 
    l += sizeof (text_mod3) - 1;
  if (kb->state & GDK_MOD4_MASK) 
    l += sizeof (text_mod4) - 1;
  if (kb->state & GDK_MOD5_MASK) 
    l += sizeof (text_mod5) - 1;

  l += strlen (gdk_keyval_name (kb->keyval));

  name = malloc (l);

  p = 0;
  if (kb->state & GDK_SHIFT_MASK) {
    memcpy (name+p, text_shift, sizeof (text_shift)-1);
    p += sizeof (text_shift) - 1;
  }  
  if (kb->state & GDK_CONTROL_MASK) {
    memcpy (name+p, text_control, sizeof (text_control)-1);
    p += sizeof (text_control) - 1;
  }  
  if (kb->state & GDK_MOD1_MASK) {
    memcpy (name+p, text_mod1, sizeof (text_mod1)-1);
    p += sizeof (text_mod1) - 1;
  }  
  if (kb->state & GDK_MOD2_MASK) {
    memcpy (name+p, text_mod2, sizeof (text_mod2)-1);
    p += sizeof (text_mod2) - 1;
  }  
  if (kb->state & GDK_MOD3_MASK) {
    memcpy (name+p, text_mod3, sizeof (text_mod3)-1);
    p += sizeof (text_mod3) - 1;
  }  
  if (kb->state & GDK_MOD4_MASK) {
    memcpy (name+p, text_mod4, sizeof (text_mod4)-1);
    p += sizeof (text_mod4) - 1;
  }  
  if (kb->state & GDK_MOD5_MASK) {
    memcpy (name+p, text_mod5, sizeof (text_mod5)-1);
    p += sizeof (text_mod5) - 1;
  }  

  memcpy (name+p, gdk_keyval_name (kb->keyval), 
	  strlen (gdk_keyval_name (kb->keyval)));
  p += strlen (gdk_keyval_name (kb->keyval));

  *(name+p) = 0;

  return name;
}

#if 0
static gchar *kb2str (key_binding_t *kb) {
  return gdk_keyval_name (kb->keyval);
}
#endif

static void set_key_binding (gchar *cmd, guint keyval, guint state) {

  key_binding_t *key_binding;
  GtkTreeIter    iter;

  key_binding = find_key_binding (cmd, &iter);

  if (!key_binding) {

    /* add new key binding */

    key_binding = malloc (sizeof (key_binding_t));

    gtk_list_store_append (kb_store, &iter);
  }

  key_binding->keyval = keyval;
  key_binding->state  = ((state & ~GDK_LOCK_MASK) & ~GDK_MOD2_MASK) & ~GDK_MOD5_MASK;
  key_binding->cmd    = cmd;
  
  gtk_list_store_set (kb_store, &iter, 0, cmd, 1, kb2str(key_binding), 
		      2, key_binding, -1);

}

static void load_default_kb (void) {

  set_key_binding ("play ();", GDK_Return, 0);
  set_key_binding ("exit ();", GDK_q, 0);
  set_key_binding ("pause ();", GDK_space, 0);
  set_key_binding ("set_fullscreen (0);", GDK_Escape, 0);
  set_key_binding ("set_fullscreen ();", GDK_f, 0);
  set_key_binding ("set_aspect ();", GDK_a, 0);
  set_key_binding ("set_deinterlace ();", GDK_i, 0);
  set_key_binding ("play (0, get_time()-60000);", GDK_Left, 0);
  set_key_binding ("play (0, get_time()+60000);", GDK_Right, 0);
  set_key_binding ("set_speed (get_speed()+1);", GDK_Up, 0);
  set_key_binding ("set_speed (get_speed()-1);", GDK_Down, 0);
  set_key_binding ("input_up ();", GDK_KP_8, 0);
  set_key_binding ("input_down ();", GDK_KP_2, 0);
  set_key_binding ("input_left ();", GDK_KP_4, 0);
  set_key_binding ("input_right ();", GDK_KP_6, 0);
  set_key_binding ("input_select ();", GDK_KP_Enter, 0);
  set_key_binding ("input_menu1 ();", GDK_F1, 0);
  set_key_binding ("input_menu2 ();", GDK_F2, 0);
  set_key_binding ("input_menu3 ();", GDK_F3, 0);
  set_key_binding ("input_previous ();", GDK_KP_Page_Up, 0);
  set_key_binding ("input_next ();", GDK_KP_Page_Down, 0);
  set_key_binding ("play (10, 0);", GDK_1, 0);
  set_key_binding ("play (20, 0);", GDK_2, 0);
  set_key_binding ("play (30, 0);", GDK_3, 0);
  set_key_binding ("play (40, 0);", GDK_4, 0);
  set_key_binding ("play (50, 0);", GDK_5, 0);
  set_key_binding ("play (60, 0);", GDK_6, 0);
  set_key_binding ("play (70, 0);", GDK_7, 0);
  set_key_binding ("play (80, 0);", GDK_8, 0);
  set_key_binding ("play (90, 0);", GDK_9, 0);
  set_key_binding ("play (0, 0);", GDK_0, 0);
  set_key_binding ("playlist_play (playlist_get_item()+1);", GDK_Page_Down, 0);
  set_key_binding ("playlist_play (playlist_get_item()-1);", GDK_Page_Up, 0);
  set_key_binding ("set_zoom (get_zoom()+5);", GDK_Z, GDK_SHIFT_MASK);
  set_key_binding ("set_zoom (get_zoom()-5);", GDK_z, 0);
  set_key_binding ("snapshot ();", GDK_t, 0);

}

void save_key_bindings (void) {

  gchar *fname;
  FILE  *f;

  fname = g_strconcat (g_get_home_dir(), "/.gxine/keybindings", NULL);

  f = fopen (fname, "w");
  if (f) {
    GtkTreeIter iter;

    fprintf (f, "<GXINEKB VERSION=\"1.0\">\n");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL(kb_store), &iter)) {

      do {
      
	GValue v;
	key_binding_t *key_binding;
	
	memset (&v, 0, sizeof (GValue));
	gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store),
				  &iter, 2, &v);
	key_binding = g_value_peek_pointer (&v);
	g_value_unset (&v);
	
	fprintf (f, "  <KEYBINDING>\n");
	fprintf (f, "    <COMMAND>%s</COMMAND>\n", key_binding->cmd);
	fprintf (f, "    <KEYVAL>%d</KEYVAL>\n", key_binding->keyval);
	fprintf (f, "    <STATE>%d</STATE>\n", key_binding->state);
	fprintf (f, "  </KEYBINDING>\n");
      
      } while (gtk_tree_model_iter_next (GTK_TREE_MODEL (kb_store),
					 &iter));
    }
    
    fprintf (f, "</GXINEKB>\n");
    fclose (f);
  } else 
    printf ("key_events: error trying to open file %s for writing\n",
	    fname);

  g_free (fname);
}

static void xml2kb (xml_node_t *node) {

  gchar   *cmd;
  guint    keyval;
  guint    state;    

  cmd    = NULL;
  keyval = 0;
  state  = 0;

  while (node) {

    if (!strcasecmp (node->name, "command"))
      cmd = node->data;
    else if (!strcasecmp (node->name, "keyval"))
      keyval = atoi (node->data);
    else if (!strcasecmp (node->name, "state"))
      state = atoi (node->data);

    node = node->next;
  }

#ifdef LOG
  printf ("key_events: cmd='%s', keyval=%d, state=%d\n",
	  cmd, keyval, state);
#endif

  if (cmd && keyval) 
    set_key_binding (cmd, keyval, state);
}

static void load_key_bindings (void) {

  gchar *fname;
  gchar *kbfile;

  /* 
   * load saved keybindings
   */

  fname = g_strconcat (g_get_home_dir(), "/.gxine/keybindings", NULL);

  kbfile = read_entire_file_ascii (fname);

  if (kbfile) {
    xml_node_t *node;

    xml_parser_init (kbfile, strlen (kbfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)>=0) {

      if (!strcasecmp (node->name, "gxinekb")) {

	node = node->child;
	while (node) {
	  if (!strcasecmp (node->name, "KEYBINDING")) {

	    xml2kb (node->child);

	  }
	  node = node->next;
	}

      } else {
	printf ("key_events: error, %s is not a valid gxine keybindings file\n",
		fname);
	load_default_kb();
      }
    } else {
      printf ("key_events: error, cannot load keybindings file %s (xml parsing failed)\n",
	      fname);
      load_default_kb();
    }

    free (kbfile);
  } else {
    printf ("key_events: error, cannot open keybindings file %s\n",
	    fname);
    load_default_kb();
  }

  g_free(fname);
}

void kb_edit_show (void) {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (kb_edit_dlg);
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (kb_edit_dlg);
    gtk_widget_map (kb_edit_dlg);
  }
}

#define CATCH_MODE_INIT 0
#define CATCH_MODE_OFF  1
#define CATCH_MODE_ON   2

static int         catch_key_mode = CATCH_MODE_INIT;
static GtkTreeIter catch_key_iter;

static void row_activated_cb (GtkTreeView *treeview,
			      gpointer user_data) {

  GtkTreeSelection  *sel;

  switch (catch_key_mode) {
  case CATCH_MODE_INIT:
    catch_key_mode = CATCH_MODE_OFF;
    break;

  case CATCH_MODE_ON:
    {
      GValue v;
      key_binding_t *key_binding;
      
      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store),
				&catch_key_iter, 2, &v);
      key_binding = g_value_peek_pointer (&v);
      g_value_unset (&v);
      
      gtk_list_store_set (kb_store, &catch_key_iter, 1, kb2str (key_binding), -1);
      
      catch_key_mode = CATCH_MODE_OFF;
    }
    break;

  case CATCH_MODE_OFF:

    sel = gtk_tree_view_get_selection (GTK_TREE_VIEW(tree_view));
  
    if (gtk_tree_selection_get_selected (sel, NULL, &catch_key_iter)) {
      gtk_list_store_set (kb_store, &catch_key_iter, 1, "Type a new accelerator", 
			  -1);
      catch_key_mode = CATCH_MODE_ON;
    }
    break;
  }
}

gboolean key_event_cb (GtkWidget *widget,
		       GdkEventKey *event,
		       gpointer user_data) {

  if ( (event->keyval == GDK_Control_L)
       || (event->keyval == GDK_Control_R)
       || (event->keyval == GDK_Meta_L)
       || (event->keyval == GDK_Meta_R)
       || (event->keyval == GDK_Alt_L)
       || (event->keyval == GDK_Alt_R)
       || (event->keyval == GDK_Super_L)
       || (event->keyval == GDK_Super_R)
       || (event->keyval == GDK_Hyper_L)
       || (event->keyval == GDK_Hyper_R)
       || (event->keyval == GDK_Shift_L)
       || (event->keyval == GDK_Shift_R) )
    return FALSE;

  if (catch_key_mode == CATCH_MODE_ON) {

    GValue v;
    key_binding_t *key_binding;
      
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (kb_store),
			      &catch_key_iter, 2, &v);
    key_binding = g_value_peek_pointer (&v);
    g_value_unset (&v);
      
    key_binding->keyval = event->keyval;
    key_binding->state  = event->state;

    gtk_list_store_set (kb_store, &catch_key_iter, 1, kb2str (key_binding), -1);
      
    catch_key_mode = CATCH_MODE_OFF;
  }

  return FALSE;
}

void key_events_init (void) {

  GtkWidget            *b, *scrolled_window;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;
   
  /*
   * init list store
   */
 
  kb_store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  load_key_bindings ();

  /*
   * create (for now invisible) kb_edit dialog
   */

  kb_edit_dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (kb_edit_dlg), "Keybindings editor");
  gtk_window_set_default_size (GTK_WINDOW (kb_edit_dlg), 400, 250);
  g_signal_connect( GTK_OBJECT (kb_edit_dlg), "delete_event",
		      G_CALLBACK (close_cb), NULL );
  b = gtk_dialog_add_button (GTK_DIALOG (kb_edit_dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK (close_cb),
		    kb_edit_dlg);
  g_signal_connect (GTK_OBJECT(kb_edit_dlg), "key_press_event",
		    G_CALLBACK(key_event_cb), NULL);

  /* add a nice tree view widget here */

  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(kb_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
  g_signal_connect (G_OBJECT (tree_view), "cursor-changed",
		    G_CALLBACK (row_activated_cb), NULL);
  

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Action",
						     cell,
						     "text", 0,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes ("Accelerator Key",
						     cell,
						     "text", 1,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(kb_edit_dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  is_visible = FALSE;
}

