/**********************************************************************

 Server Controller

 $$Id: servercontroller.cpp,v 1.149.2.1 2002/12/06 16:11:45 lunakl Exp $$

 Main Server Controller.  Displays server connection window, and makes
 new server connection on demand.

 Signals: NONE

 Slots:

   new_connection(): Creates popup asking for new connection

   new_ksircprocess(QString):
      Args:
         QString: new server name or IP to connect to.
      Action:
	 Creates a new sirc process and window !default connected to the
	 server.  Does nothing if a server connection already exists.

   add_toplevel(QString parent, QString child):
      Args:
	   parent: the server name that the new channel is being joined on
	   child: the new channel name
      Action:
         Adds "child" to the list of joined channles in the main
	 window.  Always call this on new window creation!

   delete_toplevel(QString parent, QString child):
      Args:
         parent: the server name of which channel is closing
	         child: the channle that is closing. IFF Emtpy, parent is
		 deleted.
      Action:
	 Deletes the "child" window from the list of connections.  If
	 the child is Empty the whole tree is removed since it is assumed
         the parent has disconnected and is closing.

   new_channel:  Creates popup asking for new channel name

   new_toplevel(QString str):
      Args:
         str: name of the new channel to be created
      Action:
         Sends a signal to the currently selected server in the tree
         list and join the requested channel.  Does nothing if nothing
         is selected in the tree list.

   recvChangeChanel(QString parent, QString old, QString new):
      Args:
         parent: parent server connection
         old: the old name for the window
         new: the new name for the window
      Action:
          Changes the old window name to the new window name in the tree
          list box.  Call for all name change!

 *********************************************************************/

#include <stdio.h>
#include <unistd.h>

#include "servercontroller.h"
#include "KSOpenkSirc/open_ksirc.h"
#include "NewWindowDialog.h"
#include "ksopts.h"
#include "control_message.h"
#include "FilterRuleEditor.h"
#include "../config.h"
#include "version.h"
#include "KSPrefs/ksprefs.h"
#include "toplevel.h"
#include <stdlib.h>

#include "objFinder.h"

#include <kconfig.h>
#include <kfontdialog.h>
#include <kiconloader.h>
#include <kwin.h>
#include <kdebug.h>
#include <kstdaccel.h>
#include <kstandarddirs.h>
#include <khelpmenu.h>
#include <kstdaction.h>
#include <kaction.h>
#include <knotifydialog.h>
#include <netwm.h>

#include <qfile.h>

#include "displayMgrSDI.h"
#include "displayMgrMDI.h"

#ifdef HAVE_PATHS_H
#include <paths.h>
#endif
#include <klocale.h>
#include <kapplication.h>
#include <kpopupmenu.h>

DisplayMgr *displayMgr;

servercontroller *servercontroller::s_self = 0;

servercontroller::servercontroller( QWidget*, const char* name )
    : KMainWindow( 0, name )
{
  we_are_exiting = false;
  m_notificationCount = 0;
  MenuBar = menuBar();
  KWin::setIcons( winId(), kapp->icon(), kapp->miniIcon() );

  s_self = this;

  switch (ksopts->displayMode)
  {
    case KSOptions::SDI:
        displayMgr = new DisplayMgrSDI();
        break;
    case KSOptions::MDI:
        displayMgr = new DisplayMgrMDI;
        break;
  }

  sci = new scInside(this, QString(name) + "_mainview");
  setCentralWidget(sci);

  sci->setFrameStyle(QFrame::Box | QFrame::Raised);
  ConnectionTree = sci->ConnectionTree;

  connect(ConnectionTree, SIGNAL(clicked( QListViewItem * )),
	  this, SLOT(WindowSelected(QListViewItem *)));

  setFrameBorderWidth(5);

  QPopupMenu *file = new QPopupMenu(this, QString(name) + "_menu_file");
  KStdAction::quit(this, SLOT(endksirc()), actionCollection())->plug(file);
  MenuBar->insertItem(i18n("&File"), file);

  connections = new QPopupMenu(this, QString(name) + "_menu_connections");

  server_id = connections->insertItem(i18n("&New Server..."), this, SLOT(new_connection()), Key_F2 );
  join_id = connections->insertItem(i18n("&Join Channel..."), this, SLOT(new_channel()), Key_F3 );
  connections->setItemEnabled(join_id, FALSE);
  MenuBar->insertItem(i18n("&Connections"), connections);

//  kConfig->setGroup("GlobalOptions");
  options = new QPopupMenu(this, QString(name) + "_menu_options");
  //insertChild(options);
  options->setCheckable(TRUE);

  options->insertItem(SmallIcon( "filter" ), i18n("&Filter Rule Editor..."),
		      this, SLOT(filter_rule_editor()));
  options->insertSeparator();
  KStdAction::configureNotifications(this, SLOT(notification_prefs()), actionCollection())->plug(options);

  KStdAction::preferences(this, SLOT(general_prefs()), actionCollection())->plug(options);

  MenuBar->insertItem(i18n("&Settings"), options);

  KHelpMenu *help = new KHelpMenu( this, kapp->aboutData() );
  MenuBar->insertItem(i18n("&Help"), help->menu() );

  open_toplevels = 0;

  pic_server = UserIcon("server");
  pic_gf = UserIcon("ksirc_a");
  pic_run = UserIcon("mini-run");
  pic_ppl = UserIcon("channels");
  pic_icon = UserIcon("ksirc_b");
  pic_dock = UserIcon("ksirc_dock");
  pic_info = UserIcon("info");

  setCaption( i18n("Server Control") );
  KWin::setIcons(winId(), pic_icon, pic_server);

  resize( 450,200 );

  // Server Controller is done setting up, create Puke interface.

  pukeSocket = locateLocal("socket", QString("ksirc.%1").arg(getpid()));

  /*
  PukeC = new PukeController(pukeSocket, this, "pukecontroller");
  if(PukeC->running == TRUE){
    connect(PukeC, SIGNAL(PukeMessages(QString, int, QString)),
	    this, SLOT(ProcMessage(QString, int, QString)));
    connect(this, SIGNAL(ServMessage(QString, int, QString)),
	    PukeC, SLOT(ServMessage(QString, int, QString)));
  }
  else{
    kdDebug() << "Puke failed" << endl;
  }
  */

  dockWidget = new dockServerController(this, "servercontroller_dock");
  KWin::setSystemTrayWindowFor( dockWidget->winId(), winId() );
  dockWidget->show();
  MenuBar->show(); // dunno why, but we have to show() the menubar for the mac mode.
  hide();
}


servercontroller::~servercontroller()
{
    /*
  if(PukeC != 0x0){
     delete PukeC;
     PukeC = 0x0;
  }
  */
    s_self = 0;
}

void servercontroller::new_connection()
{
  open_ksirc *w = new open_ksirc(); // Create new ksirc popup
  connect(w, SIGNAL(open_ksircprocess(QString)), // connected ok to process
          this, SLOT(new_ksircprocess(QString))); // start
  w->exec();                                       // show the sucker!
  delete w;
}

void servercontroller::new_ksircprocess(QString str)
{
  if(str.isEmpty())  // nothing entered, nothing done
    return;
  if(proc_list[str])   // if it already exists, quit
    return;

  // Insert new base
  QListViewItem *rootItem = new QListViewItem( ConnectionTree, str );
  rootItem->setPixmap( 0, pic_server );
  rootItem->setOpen( true );

  // We do no_channel here since proc emits the signal in the
  // constructor, and we can't connect to before then, so we have to
  // do the dirty work here.
  ProcMessage(str, ProcCommand::addTopLevel, QString("no_channel"));

  KSircProcess *proc = new KSircProcess(pukeSocket, (char *) str.ascii(), 0, (QString(name()) + "_" + str + "_ksp").ascii() ); // Create proc
  //this->insertChild(proc);                           // Add it to out inheritance tree so we can retreive child widgets from it.
  objFinder::insert(proc);
  proc_list.insert(str, proc);                      // Add proc to hash
  connect(proc, SIGNAL(ProcMessage(QString, int, QString)),
	  this, SLOT(ProcMessage(QString, int, QString)));
  connect(this, SIGNAL(ServMessage(QString, int, QString)),
	  proc, SLOT(ServMessage(QString, int, QString)));

  if(!ConnectionTree->currentItem()){   // If nothing's highlighted
    ConnectionTree->setCurrentItem(rootItem);     // highlight it.
  }

  connections->setItemEnabled(join_id, TRUE);
}

void servercontroller::new_channel()
{
  NewWindowDialog w;
  connect(&w, SIGNAL(openTopLevel(QString)), SLOT(new_toplevel(QString)));
  w.exec();
}

void servercontroller::new_toplevel(QString str)
{
  QListViewItem *citem = ConnectionTree->currentItem(); // get item
  if(citem){ // if it exist, ie something is highlighted, continue
    if(proc_list[citem->text(0)]){ // If it's a match with a server, ok
      proc_list[citem->text(0)]->new_toplevel(str);
    }
    // Otherwise, check the parent to see it's perhaps a server.
    else if ( citem->parent() ) {
	if(proc_list[citem->parent()->text(0)]){
	    proc_list[citem->parent()->text(0)]->new_toplevel(str);
	}
    }
  }
}

void servercontroller::ToggleAutoCreate()
{
    ksopts->autoCreateWin = !ksopts->autoCreateWin;
}

void servercontroller::general_prefs()
{
  KSPrefs *kp = new KSPrefs();
  connect(kp, SIGNAL(update(int)),
          this, SLOT(configChange()));
  kp->resize(550, 450);
  kp->show();
}

void servercontroller::notification_prefs()
{
  KNotifyDialog::configure(this, "Notification Configuration Dialog");
}


void servercontroller::filter_rule_editor()
{
  FilterRuleEditor *fe = new FilterRuleEditor();
  connect(fe, SIGNAL(destroyed()),
	  this, SLOT(slot_filters_update()));
  fe->show();
}

void servercontroller::font_update(const QFont &font)
{
  ksopts->defaultFont = font;
/*  configChange(); */

  KConfig *kConfig = kapp->config();
  kConfig->setGroup("GlobalOptions");
  kConfig->sync();
  QApplication::setFont( font, true, "KSirc::TextView" );
}

void servercontroller::configChange()
{
  QDictIterator<KSircProcess> it( proc_list );
  while(it.current()){
    it.current()->filters_update();
    it.current()->getWindowList()["!all"]->control_message(REREAD_CONFIG, "");
    ++it;
  }
}

void servercontroller::ProcMessage(QString server, int command, QString args)
{
  QListViewItem *serverItem = 0L;
  QListViewItem *item = ConnectionTree->firstChild();
  while ( item ) {
      if ( !item->parent() && item->text(0) == server ) {
	  serverItem = item;
	  break;
      }
      item = item->nextSibling();
  }

  if ( !serverItem ) {
      kdDebug() << "* ProcMessage for non-existant server?!" << endl;
      return;
  }


  switch(command){

#if 0
    // fix this... /notify ... add the "Online" item again?

    // Nick offline and online both remove the nick first.
    // We remove the nick in case of an online so that we don't get
    // duplicates.
    // Args == nick comming on/offline.
  case ProcCommand::nickOffline:
    // Add new channel, first add the parent to the path
    path.push(&server);
    path.push(&online);
    path.push(&args);

    item = serverItem;

    // add a new child item with parent as it's parent
    ConnectionTree->removeItem(&path); // Remove the item
    break;
  case ProcCommand::nickOnline:
    // Add new channel, first add the parent to the path
    path.push(&server);
    path.push(&online);
    path.push(&args);
    // Remove old one if it's there
    ConnectionTree->removeItem(&path); // Remove the item
    path.pop();
    // add a new child item with parent as its parent
    ConnectionTree->addChildItem(args, pic_run, &path);
    if (kSircConfig->BeepNotify) {
      KNotifyClient::beep();
    }
    break;
#endif

    /**
      *  Args:
      *	   parent: the server name that the new channel is being joined on
      *    child:  the new channel name
      *  Action:
      *    Adds "child" to the list of joined channles in the main
      *    window.  Always call this on new window creation!
      */
  case ProcCommand::addTopLevel:
    // Add new channel
    if(args[0] == '!')
      args.remove(0, 1); // If the first char is !, it's control, remove it
    // add a new child item with parent as it's parent
    item = new QListViewItem( serverItem, args );
    item->setPixmap( 0, pic_ppl );

    open_toplevels++;
    break;
    /**
      *  Args:
      *    parent: the server name of which channel is closing
      *	   child: the channle that is closing. IFF Emtpy, parent is
      *	   deleted.
      *  Action:
      *	   Deletes the "child" window from the list of connections.  If
      *	   the child is Empty the whole tree is removed since it is assumed
      *    the parent has disconnected and is closing.
      */
  case ProcCommand::deleteTopLevel:
    // If the child is emtpy, delete the whole tree, otherwise just the child
    if(args[0] == '!')
      args.remove(0, 1); // If the first char is !, it's control, remove it

    item = findChild( serverItem, args );
    delete item;
    if ( serverItem->childCount() == 0 )
	delete serverItem;

    open_toplevels--;
    break;

  /**
      *  Args:
      *    parent: parent server connection
      *    old: the old name for the window
      *    new: the new name for the window
      *  Action:
      *    Changes the old window name to the new window name in the tree
      *    list box.  Call for all name change!
      */
  case ProcCommand::changeChannel:
    {
      char *new_s, *old_s;
      new_s = new char[args.length()+1];
      old_s = new char[args.length()+1];
      sscanf(args.ascii(), "%s %s", old_s, new_s);
      //  If the channel has a !, it's a control channel, remove the !
      if(old_s[0] == '!')
        // Even though, we want strlen() -1 characters, strlen doesn't
        // include the \0, so we need to copy one more. -1 + 1 = 0.
	memmove(old_s, old_s+1, strlen(old_s));
      if(new_s[0] == '!')
	memmove(new_s, new_s, strlen(new_s)); // See above for strlen()

      item = findChild( serverItem, old_s );
      delete item;
      item = new QListViewItem( serverItem, new_s );
      item->setPixmap( 0, pic_ppl );

      delete[] new_s;
      delete[] old_s;
    }
    break;
  case ProcCommand::procClose:
    delete serverItem;
    proc_list.remove(server); // Remove process entry while we are at it
    if(proc_list.count() == 0){
      ConnectionTree->clear();
      connections->setItemEnabled(join_id, FALSE);
    }
    break;
  case ProcCommand::turnOffAutoCreate:
    if (ksopts->autoCreateWin) {
      ToggleAutoCreate();
    }
    break;
  case ProcCommand::turnOnAutoCreate:
    if (!ksopts->autoCreateWin) {
      ToggleAutoCreate();
    }
    break;
  default:
    kdDebug() << "Unkown command: [" << command << "] from "
              << server
              << " " << args << endl;
  }
}

void servercontroller::slot_filters_update()
{
  emit ServMessage(QString(), ServCommand::updateFilters, QString());
}

void servercontroller::saveGlobalProperties(KConfig *ksc)
{
  // ksc hos the K Session config
  // ksp == current KSircProcess
  // ksm == current KSircMessageReceiver

  // Ignore all !<name> windows

  QString group = ksc->group();

  ksc->setGroup( "KSircSession" );
  SessionConfigMap::ConstIterator it = m_sessionConfig.begin();
  for (; it != m_sessionConfig.end(); ++it ) {

      ChannelSessionInfoList infoList = *it;

      QStringList channels;
      QStringList desktops;

      for ( ChannelSessionInfoList::ConstIterator sessionInfoIt = infoList.begin();
          sessionInfoIt != infoList.end(); ++sessionInfoIt ) {
	  channels << ( *sessionInfoIt ).name;
	  desktops << QString::number( ( *sessionInfoIt ).desktop );
      }

      KConfigGroup( ksc, "KSircSession" ).writeEntry( it.key(), channels );
      KConfigGroup( ksc, "KSircSessionDesktopNumbers" ).writeEntry( it.key(), desktops );
  }

  ksc->setGroup("ServerController");
  ksc->writeEntry("Docked", !isVisible());
  ksc->writeEntry("Size", geometry());
  ksc->setGroup(group);
}

void servercontroller::readGlobalProperties(KConfig *ksc)
{
    QString group = ksc->group();

    // ksc == K Session Config

    // KMainWindow silently disables our menubar, when we quit in a docked
    // state, so we have to force showing it here.
    menuBar()->show();

    // commented in for testing...
    ksc->setGroup( "KSircSession" );
    QMap<QString,QString> keyMap = ksc->entryMap( ksc->group() );
    QMap<QString,QString>::Iterator it = keyMap.begin();

    while(it != keyMap.end()) {
	QString server = it.key();
	//	debug("%s", it.key().latin1());
	new_ksircprocess( server ); // sets up proc_list
	QStringList channels = ksc->readListEntry( server );

	QStringList desktops = KConfigGroup( ksc, "KSircSessionDesktopNumbers" ).readListEntry( server );

	for(uint i = 0; i < channels.count(); i++){

	    QString channel = channels[ i ];

	    proc_list[ server ]->new_toplevel( channel );

	    KSircTopLevel *topLevel = dynamic_cast<KSircTopLevel *>( proc_list[ server ]->getWindowList()[ channel ] );
	    if ( !topLevel || !topLevel->isTopLevel() )
		continue;

	    QStringList::ConstIterator desktopNumberIt = desktops.at( i );
	    if ( desktopNumberIt == desktops.end() )
		continue;

	    int desktop = ( *desktopNumberIt ).toInt();
	    if ( desktop == -1 )
		continue;

	    NETWinInfo winInfo( qt_xdisplay(), topLevel->winId(), qt_xrootwin(), NET::WMDesktop );
	    winInfo.setDesktop( desktop );
	}
	++it;
    }

    QRect geom;

    ksc->setGroup("ServerController");
    bool docked = ksc->readBoolEntry("Docked", FALSE);
    if ( !docked )
      show();

    geom = ksc->readRectEntry("Size");
    if(! geom.isEmpty())
    	setGeometry(geom);

    ksc->setGroup(group);
}

void servercontroller::saveSessionConfig()
{
    QDictIterator<KSircProcess> ksp(proc_list);
    for (; ksp.current(); ++ksp ) {
	ChannelSessionInfoList channels;

	QDictIterator<KSircMessageReceiver> ksm(ksp.current()->getWindowList());
	for (; ksm.current(); ++ksm )
	    if(ksm.currentKey()[0] != '!') { // Ignore !ksm's (system created)
		ChannelSessionInfo sessionInfo;

		sessionInfo.name = ksm.currentKey();
		KSircTopLevel *topLev = dynamic_cast<KSircTopLevel *>( ksm.current() );
		if ( topLev && topLev->isTopLevel() ) {
		    NETWinInfo winInfo( qt_xdisplay(), topLev->winId(), qt_xrootwin(), NET::WMDesktop );
		    sessionInfo.desktop = winInfo.desktop();
		}

		channels << sessionInfo;
	    }

	if ( !channels.isEmpty() )
	    m_sessionConfig[ ksp.currentKey() ] = channels;
    }
}

void servercontroller::showEvent( QShowEvent *e )
{
  QWidget::showEvent( e );
  if ( !e->spontaneous() )
    saveDockingStatus();
}

void servercontroller::hideEvent( QHideEvent *e )
{
  QWidget::hideEvent( e );
  if ( !e->spontaneous() )
    saveDockingStatus();
  if(QWidget::isMinimized()){
    hide();
    KWin::setState(winId(), NET::SkipTaskbar);
  }
}

void servercontroller::saveDockingStatus()
{
    if ( we_are_exiting ) // we are hidden by closeEvent
	return;

    KConfig *kConfig = kapp->config();
    KConfigGroupSaver s( kConfig, "ServerController" );
    kConfig->writeEntry("Docked", !isVisible());
    kConfig->sync();
}

void servercontroller::endksirc(){
    kapp->config()->sync();
    unlink( QFile::encodeName( pukeSocket ));
    exit(0);
}

void servercontroller::closeEvent( QCloseEvent *e )
{
    we_are_exiting = true;
    saveSessionConfig();
    KMainWindow::closeEvent( e );
}

void servercontroller::WindowSelected(QListViewItem *item)
{
    if ( !item )
	return;

  QListViewItem *parent_server = item->parent();
  if(!parent_server)
    return;

  QString txt = QString(parent_server->text(0)) + "_" + item->text(0) + "_toplevel";
  KSircTopLevel *obj = reinterpret_cast<KSircTopLevel *>( objFinder::find(txt, "KSircTopLevel"));
  if(obj == 0x0){
    txt =QString(parent_server->text(0)) + "_!" + item->text(0) + "_toplevel";
    obj = reinterpret_cast<KSircTopLevel *>( objFinder::find(txt, "KSircTopLevel"));
  }

  if(obj != 0x0){
    displayMgr->raise(obj);
  }
  else {
    kdWarning() << "Did not find widget ptr to raise it" << endl;
  }
}


QListViewItem * servercontroller::findChild( QListViewItem *parent,
					     const QString& text )
{
    if ( !parent || parent->childCount() == 0 ) {
	return 0L;
    }

    QListViewItem *item = parent->firstChild();
    while ( item ) {
	if ( item->text(0) == text ) {
	    return item;
	}
	item = item->nextSibling();
    }

    return 0L;
}

void servercontroller::increaseNotificationCount()
{
    if ( m_notificationCount == 0 )
    {
        // change icon
        dockWidget->setPixmap( pic_info );
    }
    m_notificationCount++;
}

void servercontroller::decreaseNotificationCount()
{
    m_notificationCount--;
    if ( m_notificationCount == 0 )
    {
        // change icon back
        dockWidget->setPixmap( pic_dock );
    }
}

scInside::scInside ( QWidget * parent, const char * name, WFlags
		     f )
  : QFrame(parent, name, f)
{
  ASConn = new QLabel(i18n("Active server connections:"), this, "servercontroller_label");
  QFont asfont = ASConn->font();
  asfont.setBold(TRUE);
  ASConn->setFont(asfont);

  ConnectionTree = new KListView(this, "connectiontree");
  ConnectionTree->addColumn(QString::null);
  ConnectionTree->setRootIsDecorated( true );
  ConnectionTree->setSorting( 0 );
  ConnectionTree->header()->hide();
}

scInside::~scInside()
{
  delete ASConn;
  delete ConnectionTree;
}

void scInside::resizeEvent ( QResizeEvent *e )
{
  QFrame::resizeEvent(e);
  ASConn->setGeometry(10,10, width() - 20,
		      ASConn->fontMetrics().height()+5);
  ConnectionTree->setGeometry(10, 10 + ASConn->height(),
                              width() - 20, height() - 20 - ASConn->height());
}

dockServerController::dockServerController(servercontroller *_sc, const char *_name)
  : KSystemTray(_sc, _name)
{
  sc = _sc;

  KPopupMenu *pop = contextMenu();
  pop->setName("dockServerController_menu_pop");

  pop->insertItem(SmallIcon( "filter" ), i18n("&Filter Rule Editor..."),
                  _sc, SLOT(filter_rule_editor()));

  KStdAction::preferences(_sc, SLOT(general_prefs()),
                  _sc->actionCollection())->plug(pop);

  pop->insertSeparator();
  pop->insertItem(i18n("&New Server..."),
                  _sc, SLOT(new_connection()));

  if ( !_sc->pic_dock.isNull() )
    setPixmap( _sc->pic_dock );

}

dockServerController::~dockServerController()
{
  sc = 0x0;
}

#include "servercontroller.moc"
