/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2003  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <gmime/gmime-message.h>

#include <pan/base/article.h>
#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/server.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/nntp.h>
#include <pan/task-cancel.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE ROUTINES
************/

/***
****  Running
***/

static void
task_cancel_run (Task * task, PanSocket * sock)
{
	TaskCancel * task_cancel = TASK_CANCEL(task);
	StatusItem * status = STATUS_ITEM(task);
	GMimeMessage * message = task_cancel->message;
	const char * message_id = g_mime_message_get_message_id (message);

        const int result = nntp_cancel (status, message_id, sock);

	/* if successful, then remove the article from our local headers */
	if (result == TASK_SUCCESS)
	{
		const char * subject = g_mime_message_get_subject (message);
		const char * newsgroups = g_mime_message_get_header (message, HEADER_NEWSGROUPS);
		char ** groups = g_strsplit (newsgroups, ",", 0);
		guint i;

		for (i=0; groups[i]; ++i) {
			Group * group = server_get_named_group (task->server, groups[i]);
			group_ref_articles (group, status); {
				Article * article = ARTICLE(group_get_article_by_message_id (group, message_id));
				if (article)
					group_remove_article (group, article);
			}
			group_unref_articles (group, status);
		}

		g_strfreev (groups);

		log_add_va (LOG_INFO|LOG_URGENT, _("Article %s (%s) canceled"), subject, message_id);
	}

	/* update the task's state */
	if (result == TASK_SUCCESS || result == TASK_FAIL_HOPELESS)
		task_state_set (&task->state, result);
	else
		task_state_set_need_socket (&task->state, task->server, task_cancel_run);
}

static char*
task_cancel_describe (const StatusItem * status)
{
	return g_strdup (_("Canceling article"));
}


static void
task_cancel_destructor (PanObject * obj)
{
	debug_enter ("task_cancel_destructor");

	/* sanity clause */
	g_return_if_fail (obj!=NULL);

	/* task-cancel dtor */
	g_object_unref (TASK_CANCEL(obj)->message);

	/* destroy parent class */
	task_destructor (obj);

	debug_exit ("task_cancel_destructor");
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_cancel_new (Server * server, GMimeMessage * message)
{
	TaskCancel * cancel = NULL;
	debug_enter ("task_cancel_new");

	/* sanity clause */
	g_return_val_if_fail (server_is_valid(server), NULL);
	g_return_val_if_fail (GMIME_IS_MESSAGE(message), NULL);

	/* create the object */
       	cancel = g_new0 (TaskCancel, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_cancel_new: %p", cancel);

	/* initialize the parent class */
	task_constructor (TASK(cancel), task_cancel_destructor, task_cancel_describe, server, TRUE);
	task_state_set_need_socket (&TASK(cancel)->state, server, task_cancel_run);
	TASK(cancel)->type = TASK_TYPE_POST;

	/* initialize the task-cancel */
	cancel->message = message;
	g_object_ref (message);

	debug_exit ("task_cancel_new");
	return PAN_OBJECT(cancel);
}
