/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  $Id: ksimfsystem.cpp,v 1.55.2.2 2003/03/14 09:53:19 mlaurent Exp $
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <sys/types.h>
#include <sys/param.h>

#ifdef HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#elif defined(HAVE_SYS_STATFS_H)
#include <sys/statfs.h>
#endif
#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#include <stdio.h>
#include <unistd.h>

#include <qlayout.h>
#include <qvaluelist.h>
#include <qtimer.h>
#include <qfile.h>
#include <qregexp.h>

#include <kdebug.h>
#include <klocale.h>
#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <kapplication.h>
#include <kconfig.h>

#include "ksimfsystem.h"
#include "fsystemconfig.h"
#include "fsysscroller.h"
#include <themeloader.h>

KSIM_INIT_PLUGIN(PluginModule);

PluginModule::PluginModule(const char *name)
           : KSim::PluginObject(name)
{
  setConfigFileName(instanceName());
}

PluginModule::~PluginModule()
{
}

KSim::PluginView *PluginModule::createView(const char *className)
{
  return new Fsystem(this, className);
}

KSim::PluginPage *PluginModule::createConfigPage(const char *className)
{
  return new FsystemConfig(this, className);
}

void PluginModule::showAbout()
{
  QString version = kapp->aboutData()->version();

  KAboutData aboutData(instanceName(),
     I18N_NOOP("KSim FileSystem Plugin"), version.latin1(),
     I18N_NOOP("A filesystem plugin for KSim"),
     KAboutData::License_GPL, "(C) 2001 Robbie Ward");

  aboutData.addAuthor("Robbie Ward", I18N_NOOP("Author"),
     "linuxphreak@gmx.co.uk");
  aboutData.addAuthor("Jason Katz-Brown", I18N_NOOP("Some Fixes"),
     "jason@katzbrown.com");
  aboutData.addAuthor("Heitham Omar", I18N_NOOP("FreeBSD ports"),
     "super_ice@ntlworld.com");

  KAboutApplication(&aboutData).exec();
}

Fsystem::Fsystem(KSim::PluginObject *parent, const char *name)
   : KSim::PluginView(parent, name),
   DCOPObject("fsystem")
{
  config()->setGroup("Fsystem");
  (new QVBoxLayout(this))->setAutoAdd(true);

  // Add here any items that should be ignored
  m_ignoreList.append("/proc.*");
  m_ignoreList.append("/dev/pts");

  m_mountEntries = makeList(config()->readListEntry("mountEntries"));
  m_showPercentage = config()->readBoolEntry("ShowPercentage", true);

  int updateVal = config()->readNumEntry("scrollerUpdate", 65);
  stackItems = config()->readBoolEntry("stackItems", false);
  m_scroller = new FSysScroller(stackItems, updateVal, this);

  createFreeInfo();

  m_updateTimer = new QTimer(this);
  connect(m_updateTimer, SIGNAL(timeout()), SLOT(updateFS()));
  m_updateTimer->start(config()->readNumEntry("updateValue", 60) * 1000);
  
  // A little hackish, write a proper fix later on
  QTimer::singleShot(0, m_scroller, SLOT(resizeScroller()));
}

Fsystem::~Fsystem()
{
}

void Fsystem::reparseConfig()
{
  config()->setGroup("Fsystem");

  m_showPercentage = config()->readBoolEntry("ShowPercentage", true);
  bool stack = config()->readBoolEntry("stackItems", false);
  MountEntryList mountEntries = makeList(config()->readListEntry("mountEntries"));

  if (m_mountEntries != mountEntries) {
    m_scroller->clear();
    m_mountEntries = mountEntries;
    createFreeInfo();
  }

  if (stackItems != stack) {
    stackItems = stack;
    m_scroller->setStackScroller(stackItems);
  }

  if (!stackItems) {
    int updateVal = config()->readNumEntry("scrollerUpdate", 65);
    m_scroller->setScrollSpeed(updateVal);
  }

  updateFS();
  // A little hackish, write a proper fix later on
  QTimer::singleShot(0, m_scroller, SLOT(resizeScroller()));
}

int Fsystem::totalFreeSpace() const
{
  int totalSize, totalFree;
  int totalFreeSpace = 0;
  MountEntryList::ConstIterator it;
  for (it = m_mountEntries.begin(); it != m_mountEntries.end(); ++it) {
    getMountInfo((*it).first, totalSize, totalFree);
    totalFreeSpace += totalFree;
  }

  return totalFreeSpace;
}

#ifdef HAVE_STATVFS
#define FSystemStats struct statvfs
#elif defined(HAVE_STATFS)
#define FSystemStats struct statfs
#elif defined(__FreeBSD__)
#define FSystemStats struct statfs
#else
struct ksim_statfs // fall back for (possibly) non-supported systems
{
  int f_blocks;
  int f_bfree;
};

#define FSystemStats struct ksim_statfs
#endif

int fsystemStats(const char *file, FSystemStats &stats)
{
#ifdef HAVE_STATVFS
  return statvfs(file, &stats);
#elif defined(HAVE_STATFS)
  return statfs(file, &stats);
#elif defined(__FreeBSD__)
  return statfs(file, &stats);
#else // fall back for (possibly) non-supported systems
  (void)file;

  stats.f_blocks = 0;
  stats.f_bfree = 0;
  return -1;
#endif
}

void Fsystem::getMountInfo(const QString &mntPoint,
   int &totalBlocks, int &freeBlocks) const
{
  if (isFound(mntPoint)) {
    kdWarning() << "Ignoring entry " << mntPoint << endl;
    totalBlocks = 1;
    freeBlocks = 1;
    return;
  }

  FSystemStats sysStats;
  if (fsystemStats(QFile::encodeName(mntPoint).data(), sysStats) < 0) {
    kdError() << "While reading filesystem information for " << mntPoint << endl;
    totalBlocks = 1;
    freeBlocks = 1;
    return;
  }

  totalBlocks = sysStats.f_blocks;
  freeBlocks = sysStats.f_bfree;
}

void Fsystem::createFreeInfo()
{
  int total, free;
  int i = 0;
  MountEntryList::ConstIterator it;
  for (it = m_mountEntries.begin(); it != m_mountEntries.end(); ++it) {
    // returns the total space and free space from *it
    // (the current mounted partition)
    getMountInfo((*it).first, total, free);
    int percent = 0;
	if( total != 0)
		percent = ((total - free) * 100) / total;
    
	m_scroller->append(i, total, (*it).first);
    m_scroller->setValue(i, total - free);
    if (m_showPercentage)
      m_scroller->setText(i, ((*it).second.isEmpty() ? (*it).first : (*it).second)
         + " - " + QString::number(percent) + "%");
    else
      m_scroller->setText(i, ((*it).second.isEmpty() ? (*it).first : (*it).second));
    i++;
  }

  // hack to make the scrolling smooth
  if (!stackItems && m_mountEntries.count() >= 2)
    m_scroller->insertDummy();
}

void Fsystem::updateFS()
{
  int total, free;
  int i = 0;
  MountEntryList::ConstIterator it;
  for (it = m_mountEntries.begin(); it != m_mountEntries.end(); ++it) {
    // returns the total space and free space from *it
    // (the current mounted partition)
    getMountInfo((*it).first, total, free);
    int percent = 0;
	if( total != 0 )
		percent = ((total - free) * 100) / total;
    m_scroller->setValue(i, total - free);
    if (m_showPercentage)
      m_scroller->setText(i, ((*it).second.isEmpty() ? (*it).first : (*it).second)
         + " - " + QString::number(percent) + "%");
    else
      m_scroller->setText(i, ((*it).second.isEmpty() ? (*it).first : (*it).second));
    i++;
  }

  // hack to make the scrolling smooth
  if (!stackItems && m_mountEntries.count() >= 2)
    m_scroller->insertDummy();
}

bool Fsystem::isFound(const QString &point) const
{
  QStringList::ConstIterator it;
  for (it = m_ignoreList.begin(); it != m_ignoreList.end(); ++ it) {
    QRegExp regexp((*it));
    if (regexp.search(point) != -1)
      return true;
  }

  return false;
}

Fsystem::MountEntryList Fsystem::makeList(const QStringList &list) const
{
  MountEntryList newList;
  QStringList splitList;
  QStringList::ConstIterator it;
  for (it = list.begin(); it != list.end(); ++it) {
    splitList = QStringList::split(":", (*it));
    newList.append(qMakePair(splitList[0], splitList[1]));
  }

  return newList;
}

#include "ksimfsystem.moc"
