/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include <CORBA.h>
#include <mico/template_impl.h>
#include <mico/ir_impl.h>

#include <iostream>
#include <fstream>

#include <string>
#include <list>
#include <string.h>
#include <unistd.h>

CORBA::Repository_var g_vRepo;
string g_strModule;
list<string> g_lstImport;

ostream* pout = &cout;
#define sout (*pout)

const char* keywords[] = { "const", "FALSE", "TRUE", "struct", "switch", "case", "default", "enum",
			   "in", "out", "interface", "class", "immortal", "delete", "while" "do",
			   "for", "if", "else", "main", "foreach", "return", "signal", "emit", "import",
			   "var", "readonly", "attribute", "inout", "raises", 0L }; 

string g_buffer;
const char* escape( const char* _str )
{
  int i = 0;
  while( keywords[i] && strcmp( keywords[i], _str ) != 0L )
    i++;

  if ( keywords[i] )
  {
    g_buffer = _str;
    g_buffer += "_";
    return g_buffer.c_str();
  }

  return _str;
}

void importModule( const char *_id )
{
  string repoid = _id + 4;
  if ( strncmp( repoid.c_str(), "omg.org/" , 8 ) == 0 )
    repoid.assign( repoid, 8 );

  int i = repoid.find( "/" );
  if ( i == -1 )
    return;
  string mod;
  mod.assign( repoid, 0, i );
  if ( mod == g_strModule )
    return;

  // Do not insert dupes
  list<string>::iterator it = g_lstImport.begin();
  for( ; it != g_lstImport.end(); it++ )
    if ( *it == mod )
      return;
  
  g_lstImport.push_back( mod );
}

/******************************
 * Creates a scoped name from a repoid. For example IDL:/omg.org/Trader/Query will be translated
 * to Trader.Query. But if the current module is Trader, then only "Query" is returned.
 *******************************/
string repoid2Ident( const char *_id, bool _strip_module = true, bool _strip_pragma = true )
{
  string repoid = _id + 4;
  repoid.erase( repoid.rfind( ":" ) );
  int pos;
  while ( ( pos = repoid.find( "/" ) ) != -1 )
    repoid[pos] = '.';

  if ( _strip_pragma )
  {    
    if ( strncmp( repoid.c_str(), "omg.org." , 8 ) == 0 )
      repoid.assign( repoid, 8 );
  }
  
  if ( !_strip_module )
    return repoid;
 
  
  string cmp = g_strModule;
  cmp += ".";  
  if ( strncmp( repoid.c_str(), cmp.c_str(), cmp.size() ) == 0 )
    repoid.assign( repoid, cmp.size() );
  return repoid;
}

void print_ir( CORBA::Container_ptr repo );
bool print_type( CORBA::IDLType_ptr t );

/********************************
 * Prints the typecode of typedef
 ********************************/
void print_typedef( CORBA::TypedefDef_ptr t )
{
  CORBA::TypeCode_var tc = t->type();
  sout << "const " << tc->name() << " = \"" << tc->stringify() << "\";" << endl;

  /* sout << "_tc_";
  CORBA::TypeCode_var tc = t->type();
  sout << tc->name();
  string tcstr = tc->stringify();
  sout << " = libpymico.typecode( 23, \"" << tcstr << "\" )" << endl; */
}

/*********************************
 * Prints the typecode of an alias.
 *********************************/
void print_alias( CORBA::AliasDef_ptr t )
{
  CORBA::TypeCode_var tc = t->type();
  sout << "const " << tc->name() << " = \"" << tc->stringify() << "\";" << endl;

  /* sout << "_tc_";
  CORBA::TypeCode_var tc = t->type();
  sout << tc->name();
  string tcstr = tc->stringify();
  sout << " = libpymico.typecode( 23, \"" << tcstr << "\" )" << endl; */
}

/***********************************
 * Prints the typecode of almost everything
 **********************************/
bool print_type( CORBA::IDLType_ptr t )
{
  CORBA::String_var tmp;
  string abs_name;
  
  switch( t->def_kind() )
  {
  case CORBA::dk_Struct:
  case CORBA::dk_Union:
  case CORBA::dk_Enum:
  case CORBA::dk_Alias:
  case CORBA::dk_Interface:
  case CORBA::dk_Exception:
    {
      CORBA::TypeCode_var tc = t->type();
      importModule( tc->id() );
      string ident = repoid2Ident( tc->id() );
      sout << ident;
      break;
    }
  case CORBA::dk_Primitive:
    {
      CORBA::PrimitiveDef_var p = CORBA::PrimitiveDef::_narrow( t );
      assert( !CORBA::is_nil( p ) );
      switch( p->kind() )
	{
	case CORBA::pk_any:
	  sout << "corba.any";
	  break;
	case CORBA::pk_void:
	  sout << "corba.void";
	  return false;
	  break;
	case CORBA::pk_boolean:
	  sout << "corba.boolean";
	  break;
	case CORBA::pk_char:
	  sout << "corba.char";
	  break;
	case CORBA::pk_wchar:
	  sout << "corba.wchar";
	  break;
	case CORBA::pk_octet:
	  sout << "corba.octet";
	  break;
	case CORBA::pk_ushort:
	  sout << "corba.ushort";
	  break;
	case CORBA::pk_ulong:
	  sout << "corba.ulong";
	  break;
	case CORBA::pk_short:
	  sout << "corba.short";
	  break;
	case CORBA::pk_long:
	  sout << "corba.long";
	  break;
	case CORBA::pk_longlong:
	  sout << "corba.longlong";
	  break;
	case CORBA::pk_ulonglong:
	  sout << "corba.ulonglong";
	  break;
	case CORBA::pk_float:
	  sout << "corba.float";
	  break;
	case CORBA::pk_double:
	  sout << "corba.double";
	  break;
	case CORBA::pk_longdouble:
	  sout << "corba.longdouble";
	  break;
	case CORBA::pk_string:
	  sout << "corba.string";
	  break;
	case CORBA::pk_wstring:
	  sout << "corba.wstring";
	  break;
	case CORBA::pk_objref:
	  sout << "corba.Object";
	  break;
	case CORBA::pk_TypeCode:
	  sout << "corba.TypeCode";
	  break;
	case CORBA::pk_Principal:
	  sout << "corba.Principal";
	  break;
	default:
	  assert( 0 );
	}
      break;
    }
  default:
    assert( 0 );
  }

  return true;
}

/*********************************
 * Prints the typecode and struct
 * of a CORBA::Union
 *********************************/
void print_union( CORBA::UnionDef_ptr _union, const char *_indent )
{
  CORBA::TypeCode_var tc = _union->type();

  // The struct
  sout << _indent << "struct " << tc->name() << endl;
  sout << _indent << "{" << endl;

  // Typecode
  sout << _indent << "\tconst typecode = \"" << tc->stringify() << "\";" << endl;

  // TODO: members and discriminator
  sout << _indent << "};" << endl;
}

/*********************************
 * Prints the typecode and struct
 * of a CORBA::Struct
 *********************************/
void print_struct( CORBA::StructDef_ptr _struct, const char *_indent )
{
  CORBA::TypeCode_var tc = _struct->type();

  // The struct
  sout << _indent << "struct " << tc->name() << endl;
  sout << _indent << "{" << endl;

  // Typecode
  sout << _indent << "\tconst typecode = \"" << tc->stringify() << "\";" << endl;
  // Repoid
  sout << _indent << "\tconst repoid = \"" << tc->id() << "\";" << endl;

  CORBA::StructMemberSeq_var members = _struct->members();
  if ( members->length() > 0 )
  {
    sout << _indent << "\tvar ";
    for ( CORBA::ULong l = 0; l < members->length(); l++ )
    {
      sout << members[l].name.in();
      if ( l + 1 < members->length() )
	sout << ", ";
    }
    sout << ";" << endl;
  }

  sout << _indent << "};" << endl << endl;
}

/*********************************
 * Prints the typecode and struct
 * of a CORBA::Exception
 *********************************/
void print_exception( CORBA::ExceptionDef_ptr _exc, const char *_indent )
{
  CORBA::TypeCode_var tc = _exc->type();

  // The struct
  sout << _indent << "struct " << tc->name() << endl;
  sout << _indent << "{" << endl;

  // Typecode
  sout << _indent << "\tconst typecode = \"" << tc->stringify() << "\";" << endl;
  // Repoid
  sout << _indent << "\tconst repoid = \"" << tc->id() << "\";" << endl;

  CORBA::StructMemberSeq_var members = _exc->members();
  if ( members->length() > 0 )
  {
    sout << _indent << "\tvar ";
    for ( CORBA::ULong l = 0; l < members->length(); l++ )
    {
      sout << members[l].name.in();
      if ( l + 1 < members->length() )
	sout << ", ";
    }
    sout << ";" << endl;
  }

  sout << _indent << "};" << endl << endl;
}

/*********************************
 * Prints the typecode and members
 * of a CORBA::Enum
 *********************************/
void print_enum( CORBA::EnumDef_ptr _enum, const char *_indent )
{
  CORBA::TypeCode_var tc = _enum->type();
  sout << _indent << "const " << tc->name() << " = \"" << tc->stringify() << "\";" << endl;

  CORBA::EnumMemberSeq_var members = _enum->members();
 
  for ( CORBA::ULong l = 0; l < members->length(); l++ )
    sout << _indent << "const " << members[l].in() << " = \"" << members[l].in() << "\";" << endl;
  sout << endl;
}

/**********************************
 * Generates kscript proxies for a complete
 * CORBA module.
 **********************************/
void print_module( CORBA::ModuleDef_ptr _mod )
{
  CORBA::String_var mod = _mod->name();
  sout << "******** MODULE " << mod.in() << "**********" << endl;  
  g_strModule = mod.in();
  g_lstImport.clear();
  string tmp = mod.in();
  tmp += ".tmp";
  ofstream out( tmp.c_str() );
  pout = &out;
  print_ir( _mod );
  pout = &cout;
  out.close();

  tmp = mod.in();
  tmp += ".ks";
  ofstream out2( tmp.c_str() );
  out2 << "import corba;" << endl;
  list<string>::iterator it = g_lstImport.begin();
  for( ; it != g_lstImport.end(); ++it )
    out2 << "import " << *it << ";" << endl;

  tmp = mod.in();
  tmp += ".tmp";  
  ifstream in( tmp.c_str() );  
  char buffer[ 4096 ];
  while ( !in.eof() && out2.good() )
  {
    in.read( buffer, 4096 );
    out2.write( buffer, in.gcount() );
  }
  in.close();
  out2.close();
  unlink( tmp.c_str() );
}

/**********************************
 * Generates a kscript proxiy for a
 * CORBA interface
 **********************************/

void print_interface( CORBA::InterfaceDef_ptr _iface )
{
  // Get all operations defined in this interface
  CORBA::InterfaceDef::FullInterfaceDescription_var desc;
  desc = _iface->describe_interface();
  CORBA::InterfaceDefSeq_var supertypes = _iface->base_interfaces();

  /**
   * Stub
   */
  list<string> superlist;

  /////////
  // Declare the class
  /////////
  if ( supertypes->length() == 0 )
    sout << "interface " << desc->name.in() << endl;
  else
  {
    sout << "interface " << desc->name.in() << " : ";
    for( CORBA::ULong s = 0; s < supertypes->length(); s++ )
    {
      // Get all operations defined in this interface
      CORBA::InterfaceDef::FullInterfaceDescription_var superdesc;
      superdesc = supertypes[s]->describe_interface();

      importModule( superdesc->id.in() );
      string name = repoid2Ident( superdesc->id.in() );
      superlist.push_back( name );
      sout << name;
      if ( s + 1 < supertypes->length() )
	sout << ", ";
    }
    sout << endl;
  }
  sout << "{" << endl;

  // TypeCode
  sout << "\tconst typecode = \"" << desc->type->stringify() << "\";" << endl;
  // Repoid
  sout << "\tconst repoid = \"" << desc->type->id() << "\";" << endl;

  //////////////
  // Get the contents of the interface: typecodes, structs, ....
  //////////////
  CORBA::ContainedSeq_var c;
  c = _iface->contents( CORBA::dk_all, 1 );
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    if ( desc->kind == CORBA::dk_Typedef )
    {
      CORBA::TypedefDef_var in = CORBA::TypedefDef::_narrow( c[ j ] );
      sout << "\t";
      print_typedef( in );
    }
    else if ( desc->kind == CORBA::dk_Alias )
    {
      CORBA::AliasDef_var in = CORBA::AliasDef::_narrow( c[ j ] );
      sout << "\t";
      print_alias( in );
    }
    else if ( desc->kind == CORBA::dk_Struct )
    {
      CORBA::StructDef_var in = CORBA::StructDef::_narrow( c[ j ] );
      print_struct( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Union )
    {
      CORBA::UnionDef_var in = CORBA::UnionDef::_narrow( c[ j ] );
      print_union( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Exception )
    {
      CORBA::ExceptionDef_var in = CORBA::ExceptionDef::_narrow( c[ j ] );
      print_exception( in, "\t" );
    }
    else if ( desc->kind == CORBA::dk_Enum )
    {
      CORBA::EnumDef_var in = CORBA::EnumDef::_narrow( c[ j ] );
      print_enum( in, "\t" );
    }
  }

  ///////////////
  // Constructor
  ///////////////
  /* string global_ident = repoid2Ident( desc->id.in(), false );
  int pos;
  while ( ( pos = global_ident.find( "." ) ) != -1 )
    global_ident[pos] = '_';

  sout << "\tdef __init__( self, obj=\"\", ior=\"\", repoid=\"" << desc->id.in() << "\" ):" << endl;
  sout << "\t\tif \"_init_" << global_ident << "_\" in self.__dict__.keys():" << endl;
  sout << "\t\t\treturn" << endl;
  sout << "\t\tself.__dict__[\"_init_" << global_ident << "_\"] = 1" << endl;
  sout << "\t\tif repoid == \"\":" << endl;
  sout << "\t\t\trepoid = \"" << desc->id.in() << "\"" << endl;
  if ( supertypes->length() == 0 )
    sout << "\t\tCORBA.Object.__init__( self, obj, ior, repoid )" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    for( ; it != superlist.end(); ++it )
    {
      sout << "\t\t" << *it << ".__init__( self, obj, ior, repoid )" << endl;
    }
    } */

  // Attributes
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      CORBA::AttributeDef_var adef = CORBA::AttributeDef::_narrow( c[j] );
      CORBA::IDLType_var t = adef->type_def();
      if ( adef->mode() == CORBA::ATTR_READONLY )
      {
	sout << "\treadonly attribute "; print_type( t ); sout << " " << attr.name.in() << ";" << endl;
      }
      else
      {
	sout << "\tattribute "; print_type( t ); sout << " " << attr.name.in() << ";" << endl;
      }
    }
  }
  
  // Methods
  for( CORBA::ULong k = 0; k < desc->operations.length(); k++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ k ];

    CORBA::Contained_var con = g_vRepo->lookup_id( op.id );
    assert( !CORBA::is_nil( con ) );
    CORBA::OperationDef_var opdef = CORBA::OperationDef::_narrow( con );
    assert( !CORBA::is_nil( opdef ) );
    CORBA::IDLType_var idltype = opdef->result_def();

    // Print the return value
    sout << "\t"; print_type( idltype ); sout << " " << op.name.in() << "( ";

    // Print the parameter list
    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      switch( p[l].mode )
	{
	case CORBA::PARAM_INOUT:
	  sout << "inout "; print_type( p[l].type_def ); sout << " " << escape( p[ l ].name.in() );
	  break;
	case CORBA::PARAM_OUT:
	  sout << "out "; print_type( p[l].type_def ); sout << " " << escape( p[ l ].name.in() );
	  break;
	case CORBA::PARAM_IN:
	  sout << "in "; print_type( p[l].type_def ); sout << " " << escape( p[ l ].name.in() );
	  break;
	}
      if ( l + 1 < p.length() )
	sout << ", ";
      else
	sout << " ";
    }
    sout << ")";

    // Exceptions
    if ( op.exceptions.length() > 0 )
    {
      sout << " raises( ";
      for ( CORBA::ULong l = 0; l < op.exceptions.length(); l++ )
      {
	string id = op.exceptions[l].type->id();
	importModule( id.c_str() );
	string ident = repoid2Ident( id.c_str() );
	sout << ident;

	if ( l + 1 < op.exceptions.length() )
	  sout << ", ";
	else
	  sout << " ";
      }
      sout << ")";
    }
    sout << ";" << endl;
  }

  sout << "};" << endl << endl;
  
  /**
   * Skeleton
   */
  /* if ( supertypes->length() == 0 )
    sout << "class " << desc->name.in() << "_skel( CORBA.Server ):" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    sout << "class " << desc->name.in() << "_skel( ";
    while( it != superlist.end() )
    {
      sout << *it++ << "_skel";
      if( it != superlist.end() )
	sout << ", ";
    }
    sout << " ):" << endl;
  }
  
  sout << "\tdef __init__( self, repoid=\"" << desc->id.in() << "\" ):" << endl;

  sout << "\t\tif \"_init_" << global_ident << "_\" in self.__dict__.keys():" << endl;
  sout << "\t\t\treturn" << endl;
  sout << "\t\tself._init_" << global_ident << "_ = 1" << endl;

  if ( supertypes->length() == 0 )
    sout << "\t\tCORBA.Server.__init__( self, repoid )" << endl;
  else
  {
    list<string>::iterator it = superlist.begin();
    for( ; it != superlist.end(); ++it )
    {
      sout << "\t\t" << *it << "_skel.__init__( self, repoid )" << endl;
    }
  }

  // Setting attribute to None
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      sout << "\t\t" << attr.name.in() << " = None" << endl;
    }
  }
  
  // Register every function
  for( CORBA::ULong x = 0; x < desc->operations.length(); x++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ x ];

    CORBA::Contained_var con = g_vRepo->lookup_id( op.id );
    assert( !CORBA::is_nil( con ) );
    CORBA::OperationDef_var opdef = CORBA::OperationDef::_narrow( con );
    assert( !CORBA::is_nil( opdef ) );

    sout << "\t\tlibpymico.register_callback( self._server_, self, \"" << op.name.in()
	 << "\", ";

    CORBA::IDLType_var idltype = opdef->result_def();
    print_type( idltype );

    sout << ", [ ";

    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      sout << "( ";
      print_type( p[l].type_def );
      switch( p[l].mode )
	{
	case CORBA::PARAM_IN:
	  sout << ", CORBA.ARG_IN )";
	  break;
	case CORBA::PARAM_OUT:
	  sout << ", CORBA.ARG_OUT )";
	  break;
	case CORBA::PARAM_INOUT:
	  sout << ", CORBA.ARG_INOUT )";
	  break;
	}
      
      if ( l + 1 < p.length() )
	sout << ", ";
    }
    sout << " ] )" << endl;
  }

  // Function prototypes
  for( CORBA::ULong k = 0; k < desc->operations.length(); k++ )
  {
    CORBA::OperationDescription op;
    op = desc->operations[ k ];
    sout << "\tdef " << op.name.in() << "( self";

    CORBA::ParDescriptionSeq& p = op.parameters;
    for( CORBA::ULong l = 0; l < p.length(); l++ )
    {
      switch( p[l].mode )
	{
	case CORBA::PARAM_INOUT:
	case CORBA::PARAM_IN:
	  sout << ", " << py( p[ l ].name.in() );
	}
    }
    sout << " ):" << endl;
    sout << "\t\traise CORBA.NotImplemented" << endl;
  }

  // Attribute access function prototypes
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    CORBA::AttributeDescription attr;
    if ( desc->kind == CORBA::dk_Attribute && ( desc->value >>= attr ) )
    {
      sout << "\tdef _get_" << attr.name.in() << "( self ):" << endl;
      sout << "\t\treturn " << attr.name.in() << endl;
      if ( attr.mode == CORBA::ATTR_NORMAL )
      {
	sout << "\tdef _set_" << attr.name.in() << "( self, value ):" << endl;
	sout << "\t\t" << attr.name.in() << " = value" << endl;
      }
    }
  }
  */
  sout << endl;
}

void print_ir( CORBA::Container_ptr repo )
{
  CORBA::ContainedSeq_var c;

  // Get the contents of the IR, but only objects of type
  // InterfaceDef and only objects directly contained in the IR
  c = repo->contents( CORBA::dk_all, 1 );
  for( CORBA::ULong j = 0; j < c->length(); j++ )
  {
    CORBA::Contained::Description_var desc = c[j]->describe();
    if ( desc->kind == CORBA::dk_Interface )
    {
      CORBA::InterfaceDef_var in = CORBA::InterfaceDef::_narrow( c[ j ] );
      print_interface( in );
    }
    else if ( desc->kind == CORBA::dk_Module )
    {
      CORBA::ModuleDef_var in = CORBA::ModuleDef::_narrow( c[ j ] );
      print_module( in );
    }
    else if ( desc->kind == CORBA::dk_Typedef )
    {
      CORBA::TypedefDef_var in = CORBA::TypedefDef::_narrow( c[ j ] );
      print_typedef( in );
    }
    else if ( desc->kind == CORBA::dk_Alias )
    {
      CORBA::AliasDef_var in = CORBA::AliasDef::_narrow( c[ j ] );
      print_alias( in );
    }
    else if ( desc->kind == CORBA::dk_Struct )
    {
      CORBA::StructDef_var in = CORBA::StructDef::_narrow( c[ j ] );
      print_struct( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Union )
    {
      CORBA::UnionDef_var in = CORBA::UnionDef::_narrow( c[ j ] );
      print_union( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Exception )
    {
      CORBA::ExceptionDef_var in = CORBA::ExceptionDef::_narrow( c[ j ] );
      print_exception( in, "" );
    }
    else if ( desc->kind == CORBA::dk_Enum )
    {
      CORBA::EnumDef_var in = CORBA::EnumDef::_narrow( c[ j ] );
      print_enum( in, "" );
    }
  }
}

int main( int argc, char *argv[] )
{
  // ORB initialization
  CORBA::ORB_var orb = CORBA::ORB_init( argc, argv, "mico-local-orb" );
  CORBA::BOA_var boa = orb->BOA_init( argc, argv, "mico-local-boa");

  // Create a local IR
  CORBA::Object_var o = orb->resolve_initial_references ("InterfaceRepository");
  assert( !CORBA::is_nil( o ) );
  g_vRepo = CORBA::Repository::_narrow( o );
  assert( !CORBA::is_nil( g_vRepo ) );

  cerr << "Running ..." << endl;
  
  print_ir( g_vRepo );
}
