/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs.h"
#include "prefs_utils.h"
#include "misc_gui.h"
#include "mozilla.h"

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenushell.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtklist.h>
#include <libgnomeui/gnome-color-picker.h>

void
pu_set_config_from_editable (GtkWidget *editable, gchar *config_name)
{
	GConfValue *gcvalue = eel_gconf_get_value (config_name);
	GConfValueType value_type;
	gchar *utf8_value, *value;
	gint ivalue;
	gfloat fvalue;

	if (gcvalue == NULL) {
		/* ugly hack around what appears to be a gconf bug 
		 * it returns a NULL GConfValue for a valid string pref
		 * which is "" by default */
		value_type = GCONF_VALUE_STRING;
	} else {
		value_type = gcvalue->type;
		gconf_value_free (gcvalue);
	}

	/* get all the text into a new string */	
	value = gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1);
	utf8_value = mozilla_locale_to_utf8(value);

	switch (value_type) {
	case GCONF_VALUE_STRING:
		eel_gconf_set_string (config_name, 
				      utf8_value);
		break;
	/* FIXME : handle possible errors in the input for int and float */
	case GCONF_VALUE_INT:
		ivalue = atoi (value);
		eel_gconf_set_integer (config_name, ivalue);
		break;
	case GCONF_VALUE_FLOAT:
		fvalue = strtod (value, (char**)NULL);
		eel_gconf_set_float (config_name, fvalue);
		break;
	default:
		break;
	}

	/* free the allocated strings */
	g_free (utf8_value);
	g_free (value);
}

void
pu_set_config_from_optionmenu (GtkWidget *optionmenu, gchar *config_name)
{
	int index = pu_get_int_from_optionmenu (optionmenu);
	
	eel_gconf_set_integer (config_name, index);
}

void
pu_set_config_from_radiobuttongroup (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

	/* get value from radio button group */
	index = misc_gui_gtk_radio_button_get (GTK_RADIO_BUTTON (radiobutton));

	eel_gconf_set_integer (config_name, index);
}

void
pu_set_config_from_spin_button (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* read the value as an integer */
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinbutton));

	eel_gconf_set_integer (config_name, value);
}

void
pu_set_config_from_togglebutton (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* read the value */
	value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));

	eel_gconf_set_boolean (config_name, value);
}

void
pu_set_config_from_list (GtkWidget *list, gchar *config_name)
{
	GList *children = gtk_container_children (GTK_CONTAINER(list));
	GSList *strings = NULL;

	while (children) 
	{
		gchar           *item_data_string;
		gchar           *utf8_item_data_string;
        
		gtk_label_get (GTK_LABEL(GTK_BIN(children->data)->child), 
			       &item_data_string);
		utf8_item_data_string = mozilla_locale_to_utf8(item_data_string);
		strings = g_slist_append(strings, utf8_item_data_string);
		children=children->next;
	}
	
	/* set the configuration value */
	eel_gconf_set_string_list (config_name, strings);

	g_slist_foreach (strings, (GFunc)g_free, NULL);
	g_list_free (children);
	g_slist_free (strings);
}

void
pu_set_config_from_color (GtkWidget *colorpicker, gchar *config_name)
{
	guint8 r, g, b, a;
	gchar color_string[9];

	/* get color values from color picker */
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   &r, &g, &b, &a);

	/* write into string (bounded size) */
	snprintf (color_string, 9, "#%02X%02X%02X", r, g, b);

	/* set the configuration value */
	eel_gconf_set_string (config_name, color_string);
}

void
pu_set_editable_from_config (GtkWidget *editable, gchar *config_name)
{
	GConfValue *gcvalue = eel_gconf_get_value (config_name);
	GConfValueType value_type;
	gchar *value, *utf8_value;

	if (gcvalue == NULL) {
		/* ugly hack around what appears to be a gconf bug 
		 * it returns a NULL GConfValue for a valid string pref
		 * which is "" by default */
		value_type = GCONF_VALUE_STRING;
	} else {	
		value_type = gcvalue->type;
		gconf_value_free (gcvalue);
	}
  	
	switch (value_type) {
	case GCONF_VALUE_STRING:
		value = eel_gconf_get_string (config_name);
		break;
	case GCONF_VALUE_INT:
		value = g_strdup_printf ("%d",eel_gconf_get_integer (config_name));
		break;
	case GCONF_VALUE_FLOAT:
		value = g_strdup_printf ("%.2f",eel_gconf_get_float (config_name));
		break;
	default:
		value = NULL;
	}
	
	/* set this string value in the widget */
	if (value)
	{
		utf8_value = mozilla_utf8_to_locale(value);
		gtk_entry_set_text(GTK_ENTRY(editable), utf8_value);
		g_free (utf8_value);
	}

	/* free the allocated string */
	g_free (value);
}

void
pu_set_optionmenu_from_config (GtkWidget *optionmenu, gchar *config_name)
{
	gint index;

	/* get the current value from the configuration space */
	index = eel_gconf_get_integer (config_name);

	/* set this option value in the widget */
	gtk_option_menu_set_history (GTK_OPTION_MENU (optionmenu), index);
}

void
pu_set_radiobuttongroup_from_config (GtkWidget *radiobutton, gchar *config_name)
{
	gint index;

        /* get the current value from the configuration space */
        index = eel_gconf_get_integer (config_name);

	/* set it (finds the group for us) */
	misc_gui_gtk_radio_button_set (GTK_RADIO_BUTTON (radiobutton), index);
}

void
pu_set_spin_button_from_config (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_integer (config_name);

	/* set this option value in the widget */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton), value);
}

void
pu_set_togglebutton_from_config (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* get the current value from the configuration space */
	value = eel_gconf_get_boolean (config_name);

	/* set this option value in the widget */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), value);
}

void
pu_set_list_from_config (GtkWidget *list, gchar *config_name)
{
	GSList *strings, *tmp;
	GList *items;

	/* empty the list */
	gtk_list_clear_items (GTK_LIST (list), 0, -1);

	/* get the array of strings */
        strings = eel_gconf_get_string_list (config_name);
	if(strings == NULL)
	{
		return;
	}
	items = NULL;
	tmp = strings;
	while (tmp != NULL)
	{	
		GtkWidget *list_item;
		char *label;
		
		label = mozilla_utf8_to_locale(tmp->data);	
		list_item = gtk_list_item_new_with_label (label);
		g_free (label);
		items = g_list_append (items, list_item);
		gtk_widget_show (list_item);
		tmp = tmp->next;		
	}

	gtk_list_append_items (GTK_LIST (list), items);
	
	/* free the conf strings */
	g_slist_foreach (strings, (GFunc)g_free, NULL);
	g_slist_free (strings);
}

void
pu_set_color_from_config (GtkWidget *colorpicker, gchar *config_name)
{
	gchar *color_string;
	guint r, g, b;

	/* get the string from config */
	color_string = eel_gconf_get_string (config_name);

	if (color_string)
	{
		/* parse it and setup the color picker */
		sscanf (color_string, "#%2X%2X%2X", &r, &g, &b);
		gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (colorpicker), 
					   r, g, b, 0);
		/* free the string */
		g_free (color_string);
	}
}

int pu_get_int_from_optionmenu (GtkWidget *optionmenu)
{
        GtkWidget *menu;
        GList *list;
        gpointer item;
        gint index;

        /* extract the selection */
        menu = GTK_OPTION_MENU(optionmenu)->menu;
        list = GTK_MENU_SHELL(menu)->children;
        item = gtk_menu_get_active (GTK_MENU(menu));
        index = g_list_index (list, item);
        
        return index;
}
