/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti, Matthew Aubury
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "history.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "misc_string.h"
#include "prefs.h"
#include "mozilla.h"

#include <math.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <time.h>


/*
 *  AutoBookmarks
 *
 *  (c) Matthew Aubury 2000
 *
 *  This code implements "AutoBookmarks", bookmarks which are generated
 *  automatically by scanning and scoring the user's history file.
 *  In theory, these represent the most commonly used sites and are
 *  handy for particularly lazy users who otherwise don't get round
 *  to making bookmarks (really, I've seen a lot of them!).
 *
 *  The current code is the result of some considerable tweaking and
 *  refinement but at present is based ONLY on my history file, which
 *  cannot possibly be representative of general usage. I'm interested
 *  in ANY feedback people have about how well the scoring algorithm 
 *  actually works.
 *
 *  The Scoring Algorithm
 *
 *  This works as follows. All visits to a server (e.g. www.foo.com) are
 *  lumped together and counted as one. The root page of the site is taken
 *  as the most popular amongst these. This means that if you visit
 *  lots of news pages linked from a site (say, slashdot.org) the rating
 *  of http://slashdot.org/index.pl should be extremely high.
 *  The score is then scaled according to how long you've been visiting
 *  the site (simply visiting a bunch of pages in one session doesn't
 *  give a high score), and how long it is since you last visited the
 *  site (so if I stop going somewhere it gradually drops down the list --
 *  but if I go again it shoots back up).
 * 
 *  FIXME: at the moment autobookmarks are generated once only at the
 *  beginning of the session. In future they should be regenerated at
 *  "some point" during execution... but when?
 *
 *  -- MattA  <matt@ookypooky.com>  19/12/2000
 */

/* local function prototypes */
static gint compare_hosts (const HistoryHost *a, const HistoryHost *b);
static gdouble score_of_host (const HistoryHost *server);
static GList *find_autobookmarks_roots (void);
static void find_autobookmarks_roots_rec (BookmarkItem *root, GList **l);
static void free_list_of_bookmarks (GList *list);
static void fill_autobookmarks_folder (BookmarkItem *folder, 
				       GList *reversed_list_of_autobookmarks);

/**
 * autobookmarks_generate: generate AutoBookmarks folder from 
 * history elements
 */
void 
autobookmarks_generate (void)
{
	GList *list, *item;
	gint count, maximum;
	gint shorten;
	GList *autobookmarks_roots;
	/* list of generated autobookmarks */
	GList *autobookmarks_list = NULL;

	g_return_if_fail(bookmarks_root != NULL);
	
	autobookmarks_roots = find_autobookmarks_roots ();
		
	if (autobookmarks_roots == NULL)
	{
		/* there's nothing to generate */
		return;
	}

	/* get values from configuration */
	maximum = eel_gconf_get_integer (CONF_BOOKMARKS_AB_COUNT);
	shorten = eel_gconf_get_integer (CONF_BOOKMARKS_AB_SHORTEN);

	/* get the list of hosts */
	list = history_get_host_list ();
	
	/* sort by score */
	list = g_list_sort (list, (GCompareFunc)compare_hosts);

	/* iterate over the first useful elements */
	count = 0;
	for (item = list; item != NULL; item = g_list_next(item), count++)
	{
		HistoryHost *host = (HistoryHost *)item->data;
		HistoryItem *hi = (HistoryItem *)host->dominant_item;
		gchar *title_utf8, *name;
		BookmarkItem *b;

		/* add it if within bounds */
		if (count < maximum && hi != NULL)
		{
			/* create the bookmark */
			/* note that shorten_name takes care of
			 * duplicating the title string for us, even 
			 * when it's not changed */
			title_utf8 = (hi->title_utf8 != NULL ? 
				      hi->title_utf8 : hi->title_locale);

			if (title_utf8 && title_utf8[0] != '\0')
			{
				gchar *tmp;
				tmp = misc_string_shorten_name (title_utf8,
								shorten);
				name = misc_string_strip_newline (tmp);
				g_free (tmp);
			}
			else name = NULL;

			b = bookmarks_new_bookmark (BM_SITE, TRUE, name, 
						    hi->url, NULL, NULL, NULL);
			if (name) g_free (name);
			
			/* add it to the autobookmarks list */
			autobookmarks_list = 
				g_list_append (autobookmarks_list, b);
		}
	}

	/* free the list */
	g_list_free (list);
	
	/* reverse the list, for efficiency when adding them */
	autobookmarks_list = g_list_reverse (autobookmarks_list);

	for (item = autobookmarks_roots; item != NULL; item = item->next)
	{
		fill_autobookmarks_folder (item->data, autobookmarks_list);
	}

	bookmarks_updated ();
	free_list_of_bookmarks (autobookmarks_list);
	g_list_free (autobookmarks_list);
	g_list_free (autobookmarks_roots);
}

/**
 * score_of_host: compute the "score" of a host.
 * This value represents how likely the item is to make a good AutoBookmark. 
 */
static inline gdouble 
score_of_host (const HistoryHost *host)
{
	static gdouble age, age_scale;
 	static gdouble interval_scale;
	static gdouble score;

	/* check */
	if (host->dominant_item == NULL)
	{
		return 0.0;
	}

	/* get some basic values */
	age = (gdouble)(time(NULL) - host->dominant_item->last);

	/* age scaling, falls of exponentially with time */
	age_scale = exp((double)(-age / 1e12));

	/* scale linearly according to how long we've been visiting */
	interval_scale = (gdouble)(host->dominant_item->last - 
				   host->dominant_item->first);

	/* score by scaled number of vists */
	score = (gdouble)host->item.visits * age_scale * interval_scale;

	/* return the score */
	return score;	
}

/**
 * compare_hosts: compare the scores of two hosts
 */
static gint 
compare_hosts (const HistoryHost *a, const HistoryHost *b)
{
	static gdouble score_a, score_b;

	/* compute scores */
	score_a = score_of_host(a);
	score_b = score_of_host(b);

	/* return compared values */
	return (score_b > score_a ? 1 : -1);
}

static void
free_list_of_bookmarks (GList *list)
{
	/* copy the list because removing may modify it if it's the 
	   parent's list*/
	GList *copy = g_list_copy (list);
	GList *l;
	for (l = copy; l != NULL; l = l->next)
	{
		bookmarks_remove_recursively (l->data);
	}
	g_list_free (copy);
}

static GList *
find_autobookmarks_roots (void)
{
	GList *ret = NULL;
	find_autobookmarks_roots_rec (bookmarks_root, &ret);
	return ret;
}

static void
find_autobookmarks_roots_rec (BookmarkItem *root, GList **l)
{
	if (root->type == BM_AUTOBOOKMARKS)
	{
		if (!root->alias_of)
			*l = g_list_prepend (*l, root);
		return;
	}
	if (BOOKMARK_ITEM_IS_FOLDER (root) && !root->alias_of)
	{
		GList *li;
		for (li = root->list; li != NULL; li = li->next)
		{
			BookmarkItem *b = li->data;
			if (BOOKMARK_ITEM_IS_FOLDER (b))
				find_autobookmarks_roots_rec (b, l);
		}
	}
}

static void
fill_autobookmarks_folder (BookmarkItem *folder, 
			   GList *reversed_list_of_autobookmarks)
{
	GList *l;

	/* delete the contents */
	free_list_of_bookmarks (folder->list);
	g_list_free (folder->list);
	folder->list = NULL;
	
	for (l = reversed_list_of_autobookmarks; l != NULL; l = l->next)
	{
		BookmarkItem *copy = bookmarks_copy_bookmark (l->data);
		bookmark_add_child (folder, copy, 0);
	}
}
