/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 * Special thanks go to Mike Cuddy (mcuddy@fensende.rational.com) for his
 * suggestions and code.
 *
 */

#include <mbase.h>
#include <parse.h>


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

   void     main           XARGS( (int, char **) );
   charptr  fpStrip        XARGS( (FParse *) );
   void     lineRelation   XARGS( (FParse *) );
   void     lineHeader     XARGS( (FParse *) );
   void     lineTypedef    XARGS( (FParse *) );
   void     lineField      XARGS( (FParse *) );
   void     lineIndex      XARGS( (FParse *) );
   bool     parseArgs      XARGS( (int, char **) );
   void     displaySummary XARGS( (void) );
   void     createHeader   XARGS( (void) );
   void     cleanup        XARGS( (void) );

#ifndef MSDOS
   void     strupr         XARGS( (char *) );
#endif


/*
 * MACROS ---------------------------------------------------------------------
 *
 */

#define Hprintf  fprintf (stderr, "header %s: ", hdrname);      fprintf
#define Rprintf  fprintf (stderr, "relation %s: ", relname);    fprintf
#define Tprintf  fprintf (stderr, "typedef %s: ", typedefname); fprintf
#define Fprintf  fprintf (stderr, "field %s: ", fldName);       fprintf
#define Iprintf  fprintf (stderr, "index %s: ", idxName);       fprintf

#define Last lastLine
#define Line fpLineno(fp)

#define LBC  '{'         /* These two characters are pulled out of the code, */
#define RBC  '}'         /* so editor brace-matching (as in VI) will work.   */


/*
 * VARIABLES ------------------------------------------------------------------
 *
 */

relation *rel;          /* Relation that we're creating                      */

bool  fQuiet = FALSE;   /* TRUE if output should be limited to errors        */
bool  fHeader = FALSE;  /* TRUE if header should be created by default       */
bool  fSerial = FALSE;  /* TRUE if already supplied a serial field           */

/*
 * A word about "name", below.  This is obviously a temporary buffer, but I
 * need one in main().  So there's no reason to allocate space on the stack
 * and not be able to use it later down.
 *
 */

char  name[128];        /* Temporary name buffer                             */
char  pathname[128];    /* Pathname given to schema                          */
char  relname[32];      /* Relation name, no path or extension               */
char  schemaname[32];   /* Schema name, no path or extension                 */
char  hdrname[128];     /* Header name, full filespec                        */
char  typedefname[32];  /* Typedef value                                     */

char *comment[MAX_FLD]; /* Comments for each field (for header file)         */
int   iComm;            /* Position in comment[] for next comment            */

int   lastLine;         /* Last line number                                  */

char *fldNames[] =
   {
   "string",    /* Should correspond with T_CHAR   */
   "short",     /* Should correspond with T_SHORT  */
   "ushort",    /* Should correspond with T_USHORT */
   "long",      /* Should correspond with T_LONG   */
   "ulong",     /* Should correspond with T_ULONG  */
   "float",     /* Should correspond with T_FLOAT  */
   "double",    /* Should correspond with T_DOUBLE */
   "money",     /* Should correspond with T_MONEY  */
   "time",      /* Should correspond with T_TIME   */
   "date",      /* Should correspond with T_DATE   */
   "serial",    /* Should correspond with T_SERIAL */
   "phone",     /* Should correspond with T_PHONE  */
   "byte",      /* Should correspond with T_BYTE   */
   "mchar",     /* Should correspond with T_MCHAR  */
   "mbyte"      /* Should correspond with T_MBYTE  */
   };

char *szDesc[] =
   {
  "\/\*",
   " \* This header file was created to reflect the structure of the relation",
   " \* %s.rel; header built on %s at %s.",
   " \*",
   " \* MetalBase 5.1 released January 1st, 1993 by Huan-Ti.",
   " \*",
   " \*\/",
   ""
   };

/*
 * PRIMARY ROUTINES -----------------------------------------------------------
 *
 */

void
main  (argc, argv)
int    argc;
char **argv;
{
   FParse  *fp;
   charptr  pch;

   setbuf (stdout, NULL);
   setbuf (stderr, NULL);

   if (! parseArgs (argc, argv))
      {
      fprintf (stderr, "format: build [-q][-h] schema.s\n");
      mb_exit (1);
      }

   if ((rel = mb_new ()) == RNULL)
      {
      fprintf (stderr, "%s\n", mb_error);
      mb_exit (1);
      }

   strcpy (name, pathname);
   strcat (name, schemaname);
   strcat (name, ".s");

   if ((fp = fpInit (name, TRUE)) == NULL)
      {
      fprintf (stderr, "cannot find schema file %s\n", name);
      mb_exit (1);
      }

   lineRelation (fp);

   for (;;)
      {
      pch = fpStrip (fp);

      iComm = -1;  /* In case we never found a comment from the last field */

      if (! strcmp (pch, "end"))
         break;

      if (! strcmp (pch, "header"))   lineHeader (fp);
      else
      if (! strcmp (pch, "typedef"))  lineTypedef (fp);
      else
      if (! strcmp (pch, "field"))    lineField (fp);
      else
      if (! strcmp (pch, "index"))    lineIndex (fp);
      else
         {
         fprintf (stderr, "unexpected keyword '%s' on line %d\n", pch, Line);
         mb_exit (1);
         }
      }

   if (rel->nFld == 0)
      {
      fprintf (stderr, "at least one field must be defined\n");
      mb_exit (1);
      }
   if (rel->nIdx == 0)
      {
      fprintf (stderr, "at least one index must be defined\n");
      mb_exit (1);
      }

   if (! (fHeader && !fQuiet))   /* If they didn't specify "build -h schema" */
      {
      if (! fQuiet)
         {
         displaySummary();
         }

      strcpy (name, pathname);
      strcat (name, relname);
      strcat (name, ".rel");

      if (mb_create (rel, name, 0) != MB_OKAY)
         {
         fprintf (stderr, "%s\n", mb_error);
         mb_exit (1);
         }

      if (! fQuiet)
         {
         fprintf (stderr, "\nRelation created--zero entries\n");
         }
      }

   if (fHeader)
      {
      createHeader();
      }

   cleanup();
   exit (0);
}

void
displaySummary ()
{
   int   i, j;
   char  line[80], buf[40], buf2[10];
   ftype ft;

   printf ("\n");
   printf ("MetalBase version 5.1\n");
   printf ("\n");
   printf ("Fields:\n");

   line[0] = 0;
   for (i = 0; i < rel->nFld; i++)
      {
      strcpy (buf, "......................");
      strncpy (buf, rel->fldName[i], strlen(rel->fldName[i]));

      buf2[0] = 0;
      ft = rel->fldType[i];

      if (ft == T_CHAR)    sprintf (buf2, " [%d]", rel->cbLen[i]);
      if (ft == T_BYTE)    sprintf (buf2, " [%d]", rel->cbLen[i]);
      if (ft == T_SERIAL)  sprintf (buf2, " [%d]", rel->serial);

      strcat (buf, fldNames[(int)ft]);
      strcat (buf, buf2);

      if (! line[0])
         strcpy (line, buf);
      else
         {
         printf ("   %-38.38s%s\n", line, buf);
         line[0] = 0;
         }
      }
   if (line[0])
      {
      printf ("   %s\n", line);
      }

   printf ("\n");
   printf ("Indices:\n");

   for (i = 0; i < rel->nIdx; i++)
      {
      strcpy (line, "...........................");
      strcpy (buf, rel->idxName[i]);
      if (rel->fDups[i])
         strcat (buf, " [dups]");
      strncpy (line, buf, strlen(buf));

      buf[0] = 0;
      for (j = 0; j < rel->nIdxFld[i]; j++)
         {
         if (buf[0])
            strcat (buf, ", ");
         strcat (buf, rel->fldName[ rel->idxFld[i][j] ]);
         }

      printf ("   %s%s\n", line, buf);
      }

   strcpy (name, pathname);
   strcat (name, relname);
   strcat (name, ".rel");

   if (access (name, 0) != -1)
      {
      printf ("\n");
      printf ("WARNING:  This relation already exists!\n");
      printf ("          If you continue, any existing data will be lost.\n");
      }

   printf ("\n");
   printf ("Continue creation of this relation [Y/n] ? ");

   gets (buf);
   if (buf[0] == 'n' || buf[0] == 'N')
      mb_exit (0);

   if (! fHeader)
      {
      printf ("Create header file for this relation [y/N] ? ");

      gets (buf);
      if (buf[0] == 'y' || buf[0] == 'Y')
         fHeader = TRUE;
      }
}

void
createHeader ()
{
   file  fh;
   char  line[128], buf[128];
   int   i;


   if (access (hdrname, 0) != -1)
      unlink (hdrname);

   if ((fh = creatx (hdrname)) == -1)
      {
      fprintf (stderr, "cannot create header file '%s'\n", hdrname);
      return;
      }

/*
 * First, create the top-line comment for the header
 *
 */

   for (i = 0; ; i++)
      {
      if (! *szDesc[i])
         break;

      sprintf (line, szDesc[i], relname, fmt_date (curdate(), 4),
               fmt_time (curtime(), 1));
      strcat  (line, szEOL);

      writx (fh, line, strlen(line));
      }

   writx (fh, szEOL, strlen(szEOL));

/*
 * First comes the #ifndef RELATION_H/#define RELATION_H...
 *
 */

   strcpy (buf, relname);
   strupr (buf);

   sprintf (line, "#ifndef %s_H%s", buf, szEOL);
   writx (fh, line, strlen(line));

   sprintf (line, "#define %s_H%s", buf, szEOL);
   writx (fh, line, strlen(line));

   writx (fh, szEOL, strlen(szEOL));

/*
 * Beginning of the typedef command:
 *
 */

   sprintf (line, "typedef struct%s", szEOL);
   writx (fh, line, strlen(line));

   sprintf (line, "   %c%s", LBC, szEOL);
   writx (fh, line, strlen(line));

/*
 * Lines within the typedef command:
 *
 */

   for (i = 0; i < rel->nFld; i++)
      {
      switch (rel->fldType[i])
         {
         case T_BYTE:   sprintf (buf, "byte     %s[%d];", rel->fldName[i],
                                                       rel->cbLen[i]);   break;
         case T_CHAR:   sprintf (buf, "char     %s[%d];", rel->fldName[i],
                                                       rel->cbLen[i]);   break;
         case T_SHORT:  sprintf (buf, "short    %s;", rel->fldName[i]);  break;
         case T_USHORT: sprintf (buf, "ushort   %s;", rel->fldName[i]);  break;
         case T_LONG:   sprintf (buf, "long     %s;", rel->fldName[i]);  break;
         case T_ULONG:  sprintf (buf, "ulong    %s;", rel->fldName[i]);  break;
         case T_FLOAT:  sprintf (buf, "float    %s;", rel->fldName[i]);  break;
         case T_DOUBLE: sprintf (buf, "double   %s;", rel->fldName[i]);  break;
         case T_MONEY:  sprintf (buf, "double   %s;", rel->fldName[i]);  break;
         case T_TIME:   sprintf (buf, "mb_time  %s;", rel->fldName[i]);  break;
         case T_DATE:   sprintf (buf, "mb_date  %s;", rel->fldName[i]);  break;
         case T_PHONE:  sprintf (buf, "mb_phone %s;", rel->fldName[i]);  break;
         case T_MCHAR:  sprintf (buf, "mchar    %s;", rel->fldName[i]);  break;
         case T_MBYTE:  sprintf (buf, "mbyte    %s;", rel->fldName[i]);  break;
         default:       sprintf (buf, "long     %s;", rel->fldName[i]);  break;
         }

      sprintf (line, "   %-76.76s%s", buf, szEOL);
      line[30] = '/'; line[31] = '*';           line[77] = '*'; line[78] = '/';

      if (comment[i] != NULL)
         {
         sprintf (buf, "%-43.43s", comment[i]);
         }
      else if (rel->fldType[i] == T_CHAR)
         {
         sprintf (buf,
              "field %s type string length %d", rel->fldName[i], rel->cbLen[i]);
         }
      else if (rel->fldType[i] == T_BYTE)
         {
         sprintf (buf,
              "field %s type byte length %d", rel->fldName[i], rel->cbLen[i]);
         }
      else if (rel->fldType[i] == T_SERIAL && rel->serial != 0L)
         {
         sprintf (buf,
              "field %s type serial start %ld", rel->fldName[i], rel->serial);
         }
      else
         {
         sprintf (buf, "field %s type %s", rel->fldName[i],
                            fldNames[ (int)rel->fldType[i] ]);
         }

      strncpy (&line[33], buf, strlen(buf));

      writx (fh, line, strlen(line));
      }

/*
 * End of the typedef command:
 *
 */

   sprintf (line, "   %c %s;%s", RBC, typedefname, szEOL);
   writx (fh, line, strlen(line));

   writx (fh, szEOL, strlen(szEOL));

/*
 * And the final #endif to match #ifdef RELATION_H:
 *
 */

   sprintf (line, "#endif%s", szEOL);
   writx (fh, line, strlen(line));

   writx (fh, szEOL, strlen(szEOL));

   if (! fQuiet)
      {
      fprintf (stderr, "Header '%s' created\n", hdrname);
      }

   close (fh);
}


/*
 * SERVICE ROUTINES -----------------------------------------------------------
 *
 */

bool
parseArgs (argc, argv)
int        argc;
char     **argv;
{
   charptr  pch;

   for (--argc,++argv; argc; --argc,++argv)
      {
      if (*(pch = *argv) != '-')
         break;

      for (pch++; *pch; pch++)
         {
         switch (*pch)
            {
            case 'q':  fQuiet  = TRUE;
                      break;
            case 'h':  fHeader = TRUE;
                      break;
            default:   fprintf (stderr, "unrecognized option '%c'\n", *pch);
                       return FALSE;
                      break;
            }
         }
      }

   if (! argc)
      {
      return FALSE;
      }

   if ((pch = strrchr (*argv, DIRSEP)) == NULL)
      {
      strcpy (schemaname, *argv);
      *pathname = 0;
      }
   else
      {
      pch++;
      strcpy (schemaname, pch);
      *pch = 0;
      strcpy (pathname, *argv);
      }

   if (! strcmp ((pch = &schemaname[ strlen(schemaname)-2 ]), ".s"))
      {
      *pch = 0;
      }

   if (! *schemaname)
      return FALSE;

   relname[0] = 0;

   return TRUE;
}

charptr
fpStrip (fp)
FParse  *fp;
{
   charptr pWord;

   Last = Line;

   while ( * (pWord = fpWord(fp)) == '#' )
      {
      pWord = fpLine (fp);
      if (iComm != -1 && iComm < MAX_FLD && comment[iComm] == NULL)
         {
         while (*pWord == ' ' || *pWord == '\t')
            {
            pWord++;
            }
         if ((comment[iComm] = (char *)malloc (strlen (pWord)+1)) != NULL)
            {
            strcpy (comment[iComm], pWord);
            }
         }
      }

   if (fpEOF (fp))
      {
      fprintf (stderr, "expected 'end' before end of file (line %d)\n", Line);
      mb_exit (1);
      }

   return pWord;
}

void
cleanup ()
{
   int  i;

   for (i = 0; i < MAX_REL; i++)
      {
      if (comment[i] != NULL)
         free (comment[i]);
      comment[i] = NULL;
      }
}


/*
 * LINE-PARSING ROUTINES ------------------------------------------------------
 *
 */

void
lineHeader (fp)
FParse     *fp;
{
   charptr  pch;

   strcpy (name, fpStrip (fp));
   if (strcmp ((pch = &name[ strlen(name)-2 ]), ".h"))
      {
      strcat (pch, ".h");
      }
   if (strchr (name, DIRSEP) != NULL)
      strcpy (hdrname, name);
   else
      {
      strcpy (hdrname, pathname);
      strcat (hdrname, name);
      }

   if (*(fpStrip(fp)) != ';')
      {
      Hprintf (stderr, "terminator missing from line %d\n", Last);
      mb_exit (1);
      }
}


void
lineRelation (fp)
FParse       *fp;
{
   charptr  pch;
   int      i;

   for (i = 0; i < MAX_REL; i++)
      {
      comment[i] = NULL;
      }
   iComm = -1;

   if (! strcmp ("relation", (pch = fpStrip (fp))))
      {
      pch = fpStrip (fp);
      }

   strcpy (relname, pch);
   if (! strcmp (pch = &relname[strlen(relname)-4], ".rel"))
      *pch = 0;

   strcpy (hdrname, pathname);
   strcat (hdrname, relname);
   strcat (hdrname, ".h");

   sprintf (typedefname, "%s_str", relname);
}


void
lineTypedef (fp)
FParse      *fp;
{
   strcpy (typedefname, fpStrip (fp));

   if (*(fpStrip(fp)) != ';')
      {
      Tprintf (stderr, "terminator missing from line %d\n", Last);
      mb_exit (1);
      }
}


void
lineField (fp)
FParse    *fp;
{
   charptr  pch;
   char     fldName[MAX_NAME+1];
   ftype    fldType;
   long     fldArg;

   strcpy (fldName, fpStrip (fp));

   if (fldnum (rel, fldName) != -1)
      {
      fprintf (stderr, "field '%s' declared twice, line %d\n", fldName, Line);
      mb_exit (1);
      }

   if (! strcmp ("type", (pch = fpStrip (fp))))
      {
      pch = fpStrip (fp);
      }

   fldArg = 0L;

   for (fldType= T_CHAR; fldType <= T_MBYTE; fldType=(ftype)(((int)fldType)+1))
      {
      if (! strcmp (pch, fldNames[(int)fldType]))
         {
         break;
         }
      }
   if (! strcmp (pch, "char"))  /* Second name for "string" */
      fldType = T_CHAR;

   if (fldType > T_LASTTYPE)
      {
      Fprintf (stderr, "unknown field type '%s' on line %d\n", pch, Line);
      mb_exit (1);
      }

   if (fldType == T_SERIAL)
      {
      if (fSerial)
         {
         Fprintf (stderr, "multiple serial fields, line %d\n", Line);
         mb_exit (1);
         }
      fSerial = TRUE;
      }


/*
 * We'll set iComm to a valid value here, so fpStrip() will remember the next
 * comment that it comes across.  That way, we can report anything we find
 * in the .H file we'll create later.
 *
 */

   iComm = rel->nFld;

   pch = fpStrip (fp);

/*
 * Check for "[length | *] NNN" or "[start] NNN"
 *
 */

   if (*pch != ';')
      {
      if (! strcmp ("length", pch) || ! strcmp ("*", pch))
         {
         if (fldType != T_CHAR && fldType != T_BYTE)
            {
            Fprintf (stderr, "invalid array for field type, line %d\n", Line);
            mb_exit (1);
            }
         }
      else
      if (! strcmp ("start", pch) || ! strcmp ("@", pch))
         {
         if (fldType != T_SERIAL)
            {
            Fprintf (stderr, "bad serial syntax on line %d\n", Line);
            mb_exit (1);
            }
         }
      else
         {
         Fprintf (stderr, "terminator missing from line %d\n", Last);
         mb_exit(1);
         }

      if ((fldArg = atol (fpStrip (fp))) < 0)
         {
         Fprintf (stderr, "invalid argument on line %d\n", Line);
         mb_exit (1);
         }

      pch = fpStrip (fp);
      }

   if (*pch != ';')
      {
      Fprintf (stderr, "terminator missing from line %d\n", Last);
      mb_exit (1);
      }

   if (mb_addfield (rel, fldName, fldType, fldArg) != MB_OKAY)
      {
      Fprintf (stderr, "%s, line %d\n", mb_error, Line);
      mb_exit (1);
      }
}


void
lineIndex (fp)
FParse    *fp;
{
   charptr  pch;
   char     idxName[MAX_NAME+1], idxDesc[100];
   char     buf[5];
   int      fldNum;
   bool     fDups = TRUE;  /* By default, allow duplicates */

   strcpy (idxName, fpStrip (fp));

   if (idxnum (rel, idxName) != -1)
      {
      fprintf (stderr, "index '%s' declared twice, line %d\n", idxName, Line);
      mb_exit (1);
      }

   if (! strcmp ("on", (pch = fpStrip (fp))))
      {
      pch = fpStrip (fp);
      }

   for (*idxDesc=0; ; )
      {
      if (! strcmp ("with", pch) || ! strcmp ("without", pch) || *pch == ';')
         {
         Iprintf (stderr, "syntax error on line %d\n", Line);
         mb_exit (1);
         }

      if ((fldNum = fldnum (rel, pch)) == -1)
         {
         Iprintf (stderr,"index on undeclared field '%s', line %d\n",pch,Line);
         mb_exit (1);
         }

      sprintf (buf, "%d", fldNum);
      if (*idxDesc != 0)
         strcat (idxDesc, ",");
      strcat (idxDesc, buf);

      if (*(pch = fpStrip (fp)) != ',')  /* If there's no comma, then there */
         break;                          /* shouldn't be any more fields.   */
      pch = fpStrip (fp);
      }

   if (! strcmp ("with", pch) || ! strcmp ("without", pch))
      {
      fDups = (! strcmp ("with", pch)) ? TRUE : FALSE;

      pch = fpStrip (fp);
      if (strcmp ("dups", pch) && strcmp ("duplicates", pch))
         {
         Iprintf (stderr, "dups/nodups syntax error, line %d\n", Line);
         mb_exit (1);
         }
      pch = fpStrip (fp);
      }

   if (*pch != ';')
      {
      Iprintf (stderr, "terminator missing from line %d\n", Last);
      mb_exit (1);
      }

   if (mb_addindex (rel, idxName, fDups, idxDesc) != MB_OKAY)
      {
      Iprintf (stderr, "%s, line %d\n", mb_error, Line);
      mb_exit (1);
      }
}


/*
 * All MS-DOS compilers I've played with already have this routine in their
 * standard C library.  You may not need it.
 *
 */

#ifndef MSDOS

void
strupr (str)
char   *str;
{
   for ( ;*str; str++)
      {
      *str = toupper (*str);
      }
}

#endif

